package com.digiwin.athena.base.application.service.guide;

import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.baomidou.mybatisplus.core.conditions.update.UpdateWrapper;
import com.digiwin.athena.appcore.domain.BaseResultDTO;
import com.digiwin.athena.appcore.exception.BusinessException;
import com.digiwin.athena.appcore.util.JsonUtils;
import com.digiwin.athena.appcore.util.SnowflakeIdWorker;
import com.digiwin.athena.base.application.config.BaseAudcDataSourceConfig;
import com.digiwin.athena.base.infrastructure.config.EnvProperties;
import com.digiwin.athena.base.application.constant.UserGuideConstant;
import com.digiwin.athena.base.application.meta.request.guide.UserGuidePoJo;
import com.digiwin.athena.base.application.meta.response.guide.UserGuideResult;
import com.digiwin.athena.base.infrastructure.mapper.audc.guide.UserGuideInfoMapper;
import com.digiwin.athena.base.infrastructure.mapper.audc.guide.UserGuideLogMapper;
import com.digiwin.athena.base.infrastructure.meta.po.guide.UserGuideInfo;
import com.digiwin.athena.base.infrastructure.meta.po.guide.UserGuideLog;
import net.sf.json.JSONArray;
import net.sf.json.JSONObject;
import org.apache.commons.collections4.MapUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.i18n.LocaleContextHolder;
import org.springframework.core.ParameterizedTypeReference;
import org.springframework.http.HttpEntity;
import org.springframework.http.HttpHeaders;
import org.springframework.http.HttpMethod;
import org.springframework.http.HttpStatus;
import org.springframework.http.ResponseEntity;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.web.client.RestTemplate;

import java.time.LocalDateTime;
import java.util.HashMap;
import java.util.Map;

/**
 * 描述
 *
 * @author gonghongxing
 * @version 1.0
 * @since 2022/05/18 15:30:37
 */
@Service
public class UserGuideServiceImpl implements UserGuideService {

    private static final Logger logger = LoggerFactory.getLogger(UserGuideServiceImpl.class);

    /**
     * 禁用状态
     */
    private static final int GUIDE_STATE_DISABLE = 1;

    /**
     * 启用状态
     */
    private static final int GUIDE_STATE_ENABLE = 0;

    private static final String DISABLE_SOURCE_MANUAL = "manual";

    private static final String DISABLE_SOURCE_AUTO = "auto";

    private final UserGuideInfoMapper userGuideInfoMapper;

    private final UserGuideLogMapper userGuideLogMapper;

    private final EnvProperties envProperties;

    private final RestTemplate restTemplate;

    /**
     * default construct
     *
     * @param userGuideInfoMapper userGuideInfoMapper
     * @param userGuideLogMapper  userGuideLogMapper
     */
    @Autowired
    public UserGuideServiceImpl(UserGuideInfoMapper userGuideInfoMapper, UserGuideLogMapper userGuideLogMapper, EnvProperties envProperties, RestTemplate restTemplate) {
        this.userGuideInfoMapper = userGuideInfoMapper;
        this.userGuideLogMapper = userGuideLogMapper;
        this.envProperties = envProperties;
        this.restTemplate = restTemplate;
    }

    /**
     * 查询状态信息
     *
     * @param pojo 参数
     * @return UserGuideCallResult
     */
    @Override
    @Transactional(transactionManager = BaseAudcDataSourceConfig.BASE_AUDC_DATASOURCE_TRANSACTION_MANAGER_BUSINESS)
    public UserGuideResult queryGuideInfo(UserGuidePoJo pojo) {

        UserGuideResult result = new UserGuideResult();
        //查询调用状态
        UserGuideInfo guideInfo = queryGuideState(pojo);
        //如果查询不到对应的记录，则直接插入一条新的记录
        if (null == guideInfo) {
            try {
                //插入一条新的记录
                guideInfo = addGuideInfo(pojo);
            } catch (Exception e) {
                logger.error("add new guide info error,params:{}", JsonUtils.objectToString(pojo), e);
                throw BusinessException.create(1003, "Add new guide info error.");
            }
        }
        //引导状态不可用，直接返回
        if (GUIDE_STATE_DISABLE == guideInfo.getState()) {
            throw BusinessException.create(1001, "State is disable.");
        }

        Map<String, Object> guideDefine = queryGuideNoteInfo(pojo);
        Map<String, Object> guideDetail = (Map<String, Object>) guideDefine.get(pojo.getSubKey());
        //判断是否包含定义
        if (null == guideDetail) {
            throw BusinessException.create(1009, "Guide Define not contain SubKey");
        }
        String content = MapUtils.getString(guideDetail, "content");
        int maxNum = MapUtils.getIntValue(guideDetail, "maxCount");
        //判断是否有内容信息
        if (StringUtils.isEmpty(content)) {
            throw BusinessException.create(1010, "Guide Define not contain Content");
        }

        //统计记录的访问数
        Long visitCount = queryCount(pojo);
        //需要调用KG的API获取对应的引导内容
        //如果访问次数大于指定次数，则更新状态为不可引导
        if (visitCount >= maxNum) {
            //更新引导状态为不可引导
            pojo.setState(GUIDE_STATE_DISABLE);
            try {
                updateUserGuideState(pojo, DISABLE_SOURCE_AUTO);
            } catch (Exception e) {
                logger.error("update user guide info state error{} ", JsonUtils.objectToString(pojo), e);
                throw BusinessException.create(1004, "update guide info state error.");
            }
            throw BusinessException.create(1002, "Above Max visit number.");
        }
        recordUserGuildLog(pojo);
        result.setData(content);
        return result;
    }

    /**
     * 设置用户引导的状态信息
     *
     * @param pojo 参数信息
     * @return UserGuideCallResult
     */
    @Override
    @Transactional(transactionManager = BaseAudcDataSourceConfig.BASE_AUDC_DATASOURCE_TRANSACTION_MANAGER_BUSINESS)
    public UserGuideResult disableGuideInfo(UserGuidePoJo pojo) {
        UserGuideResult result = new UserGuideResult();
        //更新状态为不可用
        try {
            updateUserGuideState(pojo, DISABLE_SOURCE_MANUAL);
        } catch (Exception e) {
            logger.error("disableGuideInfo error", e);
            throw BusinessException.create(1012, "update guide info state error.");
        }
        return result;
    }


    /**
     * 给指定用户重置引导
     *
     * @param userId userId
     * @return UserGuideCallResult
     */
    @Override
    @Transactional(transactionManager = BaseAudcDataSourceConfig.BASE_AUDC_DATASOURCE_TRANSACTION_MANAGER_BUSINESS)
    public UserGuideResult resetUserGuideInfo(String userId) {
        UserGuideResult result = new UserGuideResult();
        try {
            QueryWrapper<UserGuideLog> logQueryWrapper = new QueryWrapper<>();
            logQueryWrapper.eq("user_id", userId);
            userGuideLogMapper.delete(logQueryWrapper);
            QueryWrapper<UserGuideInfo> infoQueryWrapper = new QueryWrapper<>();
            infoQueryWrapper.eq("user_id", userId);
            userGuideInfoMapper.delete(infoQueryWrapper);
        } catch (Exception e) {
            logger.error("resetUserGuideInfo error", e);
            throw BusinessException.create(1013, "reset user info state error.");
        }
        return result;


    }

    /**
     * 插入一条新增数据
     *
     * @param param param
     * @return UserGuideInfo
     */
    private UserGuideInfo addGuideInfo(UserGuidePoJo param) {
        UserGuideInfo userGuideInfo = new UserGuideInfo();
        userGuideInfo.setUserId(param.getUserId());
        userGuideInfo.setTenantId(param.getTenantId());
        userGuideInfo.setAppCode(param.getAppCode());
        userGuideInfo.setModularCode(param.getModularCode());
        userGuideInfo.setState(GUIDE_STATE_ENABLE);
        userGuideInfo.setCreateDate(LocalDateTime.now());
        userGuideInfo.setTerminalType("web");
        userGuideInfo.setSubKey(param.getSubKey());
        userGuideInfo.setCode(param.getCode());
        userGuideInfoMapper.insert(userGuideInfo);
        return userGuideInfo;
    }

    private Map<String, Object> queryGuideNoteInfo(UserGuidePoJo param) {
        String url = envProperties.getThemeMapUri() + "/restful/service/knowledgegraph/Guide/query";
        HttpHeaders headerMap = new HttpHeaders();
        headerMap.add("Content-Type", "application/json;charset=UTF-8");
        headerMap.add("locale", LocaleContextHolder.getLocale().toString());

        Map<String, Object> map = new HashMap<>();
        map.put("appCode", param.getAppCode());
        map.put("code", param.getCode());
        map.put("type", param.getModularCode());


        HttpEntity httpEntity = new HttpEntity<>(map, headerMap);
        ResponseEntity<BaseResultDTO<JSONObject>> responseEntity;
        try {
            responseEntity = restTemplate.exchange(url, HttpMethod.POST, httpEntity, new ParameterizedTypeReference<BaseResultDTO<JSONObject>>() {
            });
        } catch (Exception e) {
            logger.error("query guide info from km error");
            throw BusinessException.create(1011, "Request KM error,But no data return");
        }

        if ((responseEntity.getStatusCodeValue() == HttpStatus.OK.value()) && responseEntity.getBody() != null) {
            JSONObject object = responseEntity.getBody().getResponse();
            if (null == object) {
                throw BusinessException.create(1008, "Request KM Success,But no data return");
            }

            return parseResult(object);
        } else {
            throw BusinessException.create(1007, "Request KM Fail");
        }

    }


    /**
     * 处理km返回的结果
     *
     * @param jsonObject jsonObject
     * @return map
     */
    private Map<String, Object> parseResult(JSONObject jsonObject) {
        Map<String, Object> retMap = new HashMap<>();
        JSONArray guideArrs = jsonObject.getJSONArray("settings");
        if (null != guideArrs) {
            for (int i = 0; i < guideArrs.size(); i++) {
                JSONObject temp = guideArrs.getJSONObject(i);
                retMap.put(temp.getString("key"), temp);
            }
        }
        return retMap;
    }

    /**
     * 查询用户当前模块的引导状态
     *
     * @param param 查询阐述
     * @return 结果
     */
    private UserGuideInfo queryGuideState(UserGuidePoJo param) {
        QueryWrapper<UserGuideInfo> queryWrapper = new QueryWrapper<>();
        queryWrapper.eq(UserGuideConstant.USER_ID, param.getUserId());
        queryWrapper.eq(UserGuideConstant.TENANT_ID, param.getTenantId());
        queryWrapper.eq(UserGuideConstant.MODULAR_CODE, param.getModularCode());
        queryWrapper.eq(UserGuideConstant.CODE, param.getCode());
        queryWrapper.eq(UserGuideConstant.SUB_KEY, param.getSubKey());
        return userGuideInfoMapper.selectOne(queryWrapper);
    }

    /**
     * 查询访问条数
     *
     * @param pojo 查询条件
     * @return int
     */
    private Long queryCount(UserGuidePoJo pojo) {
        QueryWrapper<UserGuideLog> queryWrapper = new QueryWrapper<>();
        queryWrapper.eq(UserGuideConstant.USER_ID, pojo.getUserId());
        queryWrapper.eq(UserGuideConstant.TENANT_ID, pojo.getTenantId());
        queryWrapper.eq(UserGuideConstant.MODULAR_CODE, pojo.getModularCode());
        queryWrapper.eq(UserGuideConstant.CODE, pojo.getCode());
        queryWrapper.eq(UserGuideConstant.SUB_KEY, pojo.getSubKey());
        return userGuideLogMapper.selectCount(queryWrapper);
    }

    /**
     * 访问记录表中插入一条数据
     *
     * @param record 记录
     * @return int
     */
    private int recordUserGuildLog(UserGuidePoJo record) {
        UserGuideLog guideLog = new UserGuideLog();
        guideLog.setId(SnowflakeIdWorker.getInstance().newId());
        guideLog.setAppCode(record.getAppCode());
        guideLog.setModularCode(record.getModularCode());
        guideLog.setTenantId(record.getTenantId());
        guideLog.setUserId(record.getUserId());
        guideLog.setCode(record.getCode());
        guideLog.setSubKey(record.getSubKey());
        guideLog.setCreateDate(LocalDateTime.now());
        return userGuideLogMapper.insert(guideLog);
    }


    /**
     * 更新是否引导的状态
     *
     * @param guideInfo     guideInfo
     * @param disableSource disableSource
     * @return int
     */
    private int updateUserGuideState(UserGuidePoJo guideInfo, String disableSource) {
        UpdateWrapper<UserGuideInfo> updateWrapper = new UpdateWrapper<>();
        updateWrapper.set("state", guideInfo.getState());
        updateWrapper.set("disable_source", disableSource);
        updateWrapper.set("modify_date", LocalDateTime.now());
        updateWrapper.eq(UserGuideConstant.USER_ID, guideInfo.getUserId());
        updateWrapper.eq(UserGuideConstant.TENANT_ID, guideInfo.getTenantId());
        updateWrapper.eq(UserGuideConstant.MODULAR_CODE, guideInfo.getModularCode());
        updateWrapper.eq(UserGuideConstant.CODE, guideInfo.getCode());
        updateWrapper.eq(UserGuideConstant.APP_CODE, guideInfo.getAppCode());
        updateWrapper.eq(UserGuideConstant.SUB_KEY, guideInfo.getSubKey());
        return userGuideInfoMapper.update(null, updateWrapper);
    }


}
