package com.digiwin.athena.base.application.service.permission;

import com.digiwin.athena.appcore.auth.domain.AuthoredUser;
import com.digiwin.athena.appcore.util.JsonUtils;
import com.digiwin.athena.base.application.constant.BaseGlobalConstant;
import com.digiwin.athena.base.application.meta.response.permission.ActionPermissionDTO;
import com.digiwin.athena.base.application.meta.response.permission.FunctionPermissionDTO;
import com.digiwin.athena.base.infrastructure.manager.iam.model.AppAuthoredDTO;
import com.digiwin.athena.base.infrastructure.manager.iam.service.BaseIamService;
import com.google.common.collect.Lists;
import com.jugg.agile.framework.core.config.JaProperty;
import net.sf.json.JSONArray;
import net.sf.json.JSONObject;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.BooleanUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.redis.core.StringRedisTemplate;
import org.springframework.stereotype.Service;

import javax.annotation.Resource;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Iterator;
import java.util.List;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;

/**
 * PermissionServiceImpl Description
 *
 * @author majianfu
 * @date 2022/5/16
 * @since
 */
@Service
public class PermissionServiceImpl implements PermissionService {
    @Resource
    private BaseIamService baseIamService;

    @Autowired
    private StringRedisTemplate stringRedisTemplate;

    /**
     * {@inheritDoc}
     */
    @Override
    public List<FunctionPermissionDTO> getUserAuthoredAppPermission(AuthoredUser authoredUser) {
        // 获取用户所有授权的应用列表
        List<String> authoredAppCodeList = getUserAuthoredAppCode(authoredUser.getToken());
        if (CollectionUtils.isEmpty(authoredAppCodeList)) {
            return Collections.emptyList();
        }

        // 批量获取应用的权限配置
        JSONArray userAppPermissionList = batchQueryUserAppPermission(authoredUser, authoredAppCodeList);
        if (null == userAppPermissionList || userAppPermissionList.isEmpty()) {
            return Collections.emptyList();
        }

        // 数据变形，构造返回结果列表
        return convertToFunctionPermissionDTO(userAppPermissionList);
    }

    @Override
    public List<FunctionPermissionDTO> getUserAuthoredAppPermissionByAppCode(AuthoredUser authoredUser,String appCode) {
        if (StringUtils.isEmpty(appCode)) {
            return Collections.emptyList();
        }
        JSONArray userAppPermissionList = batchQueryUserAppPermission(authoredUser, Lists.newArrayList(appCode));
        if (null == userAppPermissionList || userAppPermissionList.isEmpty()) {
            return Collections.emptyList();
        }

        // 数据变形，构造返回结果列表
        return convertToFunctionPermissionDTO(userAppPermissionList);
    }

    private List<FunctionPermissionDTO> convertToFunctionPermissionDTO(JSONArray userAppPermissionList) {
        List<FunctionPermissionDTO> resultList = new ArrayList<>();
        for (int idx = 0; idx < userAppPermissionList.size(); idx++) {
            JSONObject userAppPermission = userAppPermissionList.getJSONObject(idx);

            JSONArray permissions = userAppPermission.getJSONArray("permissions");
            if (permissions.isEmpty()) {
                continue;
            }

            String appCode = userAppPermission.getString("sysId");
            resultList.addAll(resolveFunctionPermission(appCode, permissions));
        }
        return resultList;
    }

    private List<FunctionPermissionDTO> resolveFunctionPermission(String appCode, JSONArray permissions) {
        List<FunctionPermissionDTO> resultList = new ArrayList<>();

        for (int idx = 0; idx < permissions.size(); idx++) {
            JSONObject permission = permissions.getJSONObject(idx);
            if (permission.isEmpty() || !permission.containsKey("conditions")) {
                continue;
            }

            JSONObject conditions = permission.getJSONObject("conditions");
            if (conditions.isEmpty()) {
                continue;
            }

            String target = permission.getString("target");
            JSONObject actionJSON = conditions.getJSONObject(target);
            if (actionJSON.isEmpty()) {
                continue;
            }

            FunctionPermissionDTO functionPermission = new FunctionPermissionDTO();
            functionPermission.setAppCode(appCode);
            functionPermission.setAction(new ArrayList<>());

            String moduleCode = permission.getString("moduleId");
            functionPermission.setModuleCode(moduleCode);

            String functionId = permission.getString("id");
            functionPermission.setCode(functionId);

            String effect = permission.getString("effect");

            Iterator keyIterator = actionJSON.keys();
            while (keyIterator.hasNext()) {
                String actionCode = keyIterator.next().toString();
                String actionValue = actionJSON.getString(actionCode);

                ActionPermissionDTO actionPermission = new ActionPermissionDTO();
                actionPermission.setId(actionCode);
                /**
                 * 若 effect="allow", 表示鼎捷云上该选项已勾选，则restriction以actionValue为准；否则表示鼎捷云上该选项未配置，则restriction设置为effect的值
                 */
                actionPermission.setRestriction("allow".equalsIgnoreCase(effect) ? actionValue : effect);
                functionPermission.getAction().add(actionPermission);
            }
            resultList.add(functionPermission);
        }

        return resultList;
    }

    public List<String> getUserAuthoredAppCode(String token) {
        List<AppAuthoredDTO> authoredAppList = baseIamService.getUserAuthoredApp(token);
        if (CollectionUtils.isNotEmpty(authoredAppList)) {
            return authoredAppList.stream()
                    // 未过期
                    .filter(authoredApp -> BooleanUtils.isNotTrue(authoredApp.getExpired()))
                    .map(AppAuthoredDTO::getId)
                    .collect(Collectors.toList());
        } else {
            return Collections.emptyList();
        }
    }


    /**
     * 批量获取应用的权限并放入redis中
     */
    @Override
    public void batchGetAppAccessible(AuthoredUser authoredUser, List<String> appCodes) {
        JSONArray userAppPermissionList = batchQueryUserAppPermission(authoredUser, appCodes);
        if (null == userAppPermissionList || userAppPermissionList.isEmpty()) {
            return ;
        }
        for (int i = 0; i< userAppPermissionList.size(); i++) {
            JSONObject userAppPermission = userAppPermissionList.getJSONObject(i);
            String userId = userAppPermission.getString("userId");
            String appCode = userAppPermission.getString("sysId");
            JSONArray permissionList = userAppPermission.getJSONArray("permissions");
            String redisKey = BaseGlobalConstant.AUDC_CACHE_KEY_PREFIX + authoredUser.getTenantId() + ":" + userId + ":" + appCode;
            stringRedisTemplate.opsForValue().set(redisKey, JsonUtils.objectToString(permissionList), 60, TimeUnit.SECONDS);
        }
    }

    /**
     * 分页获取应用权限
     * @param authoredUser
     * @param authoredAppCodeList
     * @return
     */
    private JSONArray batchQueryUserAppPermission(AuthoredUser authoredUser, List<String> authoredAppCodeList) {
        final JSONArray batchQueryUserAppPermission = new JSONArray();
        final List<List<String>> userPartitionIds = Lists.partition(authoredAppCodeList, JaProperty.getInteger("audc.appPermission.pageSize",100));
        userPartitionIds.forEach(appCodeList -> {
            JSONArray jsonArray = baseIamService.batchQueryUserAppPermission(authoredUser, appCodeList);
            if (CollectionUtils.isNotEmpty(jsonArray)) {
                batchQueryUserAppPermission.addAll(jsonArray);
            }
        });
        return batchQueryUserAppPermission;
    }
}
