package com.digiwin.athena.base.application.service.proxyinfo;

import com.digiwin.athena.appcore.auth.domain.AuthoredUser;
import com.digiwin.athena.appcore.util.MessageUtils;
import com.digiwin.athena.base.application.config.BaseAudcDataSourceConfig;
import com.digiwin.athena.base.application.converter.DataConverter;
import com.digiwin.athena.base.application.eventbus.event.MessageAgentChangeBatchEvent;
import com.digiwin.athena.base.application.eventbus.event.MessageAgentChangeEvent;
import com.digiwin.athena.base.application.eventbus.event.MessageAgentClearBatchEvent;
import com.digiwin.athena.base.application.eventbus.event.MessageAgentClearEvent;
import com.digiwin.athena.base.application.meta.request.proxyinfo.ProxyInfoRemoveReq;
import com.digiwin.athena.base.application.meta.request.proxyinfo.ProxyInfoSaveReq;
import com.digiwin.athena.base.application.meta.request.proxyinfo.SaveSubordinateReq;
import com.digiwin.athena.base.application.service.principal.PrincipalTaskTypeService;
import com.digiwin.athena.base.infrastructure.constant.AudcErrorCodeEnum;
import com.digiwin.athena.base.infrastructure.manager.eoc.BaseEocService;
import com.digiwin.athena.base.infrastructure.manager.eoc.dto.EocEmpInfoDTO;
import com.digiwin.athena.base.infrastructure.meta.bo.principal.ThePrincipalTaskNum;
import com.google.common.eventbus.AsyncEventBus;
import net.sf.json.JSONArray;
import net.sf.json.JSONObject;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.BooleanUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import javax.annotation.Resource;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.stream.Collectors;

/**
 * @author chenpeng
 * @date 2024年8月29日
 */
@Service
public class ProxyInfoServiceImpl implements ProxyInfoService {
    @Autowired
    private PrincipalTaskTypeService principalTaskTypeService;
    @Resource
    private BaseEocService eocService;
    @Resource
    private MessageUtils messageUtils;
    @Autowired
    private AsyncEventBus asyncEventBus;

    /**
     * 保存代理人信息
     *
     * @param proxyInfoSaveReq
     * @param user
     */
    @Override
    @Transactional(rollbackFor = Exception.class, transactionManager = BaseAudcDataSourceConfig.BASE_AUDC_DATASOURCE_TRANSACTION_MANAGER_BUSINESS)
    public void saveProxyInfo(ProxyInfoSaveReq proxyInfoSaveReq, AuthoredUser user) {
        // 校验请求参数
        validateParam(proxyInfoSaveReq);
        if (BooleanUtils.isTrue(proxyInfoSaveReq.getNeedSaveTaskDefCode())) {
            // 保存被代理人的任务类型
            principalTaskTypeService.saveTaskType(proxyInfoSaveReq.getAgentId(), proxyInfoSaveReq.getUserId(), proxyInfoSaveReq.getTaskDefCodes(), user.getTenantId(), proxyInfoSaveReq.getCalendarTaskCodes());
        }
        if (BooleanUtils.isTrue(proxyInfoSaveReq.getNeedSendMessage())) {
            // 调用eoc接口保存代理人和代理时间
            eocService.empAgentChange(DataConverter.INSTANCE.toEmpAgentSaveReq(proxyInfoSaveReq), user);
            // 调用atmc发送工作提醒(Athena用户)
            if (BooleanUtils.isTrue(proxyInfoSaveReq.getIsAthena())) {
                MessageAgentChangeEvent event = new MessageAgentChangeEvent(proxyInfoSaveReq.getUserId(), proxyInfoSaveReq.getAgentId(), proxyInfoSaveReq.getAgentBeginDate(), proxyInfoSaveReq.getAgentEndDate(), user.getToken());
                asyncEventBus.post(event);
            }
        }
    }

    private void validateParam(ProxyInfoSaveReq proxyInfoSaveReq) {
        if (Objects.isNull(proxyInfoSaveReq)) {
            throw AudcErrorCodeEnum.INVALID_REQUEST_PARAM.getBusinessException(messageUtils.getMessage("exception.param.null"));
        }
        if (StringUtils.isEmpty(proxyInfoSaveReq.getUserId())) {
            throw AudcErrorCodeEnum.INVALID_REQUEST_PARAM.getBusinessException(messageUtils.getMessage("exception.param.userId.null"));
        }
        if (StringUtils.isEmpty(proxyInfoSaveReq.getAgentId())) {
            throw AudcErrorCodeEnum.INVALID_REQUEST_PARAM.getBusinessException(messageUtils.getMessage("exception.param.agentId.null"));

        }
        if (StringUtils.isEmpty(proxyInfoSaveReq.getAgentBeginDate()) || StringUtils.isEmpty(proxyInfoSaveReq.getAgentEndDate())) {
            throw AudcErrorCodeEnum.INVALID_REQUEST_PARAM.getBusinessException(messageUtils.getMessage("exception.param.agentDate.null"));
        }
        if (Objects.isNull(proxyInfoSaveReq.getNeedSaveTaskDefCode()) || Objects.isNull(proxyInfoSaveReq.getNeedSendMessage())) {
            throw AudcErrorCodeEnum.INVALID_REQUEST_PARAM.getBusinessException(messageUtils.getMessage("exception.param.flag.null"));
        }
    }


    /**
     * 清除代理人信息
     *
     * @param proxyInfoRemoveReq
     * @param user
     */
    @Override
    @Transactional(rollbackFor = Exception.class, transactionManager = BaseAudcDataSourceConfig.BASE_AUDC_DATASOURCE_TRANSACTION_MANAGER_BUSINESS)
    public void removeProxyInfo(ProxyInfoRemoveReq proxyInfoRemoveReq, AuthoredUser user) {
        if (Objects.isNull(proxyInfoRemoveReq)) {
            throw AudcErrorCodeEnum.INVALID_REQUEST_PARAM.getBusinessException(messageUtils.getMessage("exception.param.null"));
        }
        if (StringUtils.isEmpty(proxyInfoRemoveReq.getUserId())) {
            throw AudcErrorCodeEnum.INVALID_REQUEST_PARAM.getBusinessException(messageUtils.getMessage("exception.param.userId.null"));
        }
        if (StringUtils.isEmpty(proxyInfoRemoveReq.getAgentId())) {
            return;
        }
        // 清除被代理人的任务类型
        principalTaskTypeService.batchDeleteTaskType(Arrays.asList(proxyInfoRemoveReq.getUserId()), user.getTenantId());
        // 调用eoc接口清除代理人
        eocService.empAgentClear(proxyInfoRemoveReq.getUserId(), user);
        // 调用atmc发送工作提醒(Athena用户)
        if (BooleanUtils.isTrue(proxyInfoRemoveReq.getIsAthena())) {
            MessageAgentClearEvent event = new MessageAgentClearEvent(proxyInfoRemoveReq.getUserId(), user.getToken());
            asyncEventBus.post(event);
        }
    }


    /**
     * 查询下属代理人列表
     *
     * @param user
     */
    @Override
    public List<EocEmpInfoDTO> querySubordinate(AuthoredUser user) {
        // 调用eoc查询租户下的部门和子部门
        JSONArray departmentArray = eocService.queryCurrentTenantAllDepartment();
        // 部门为空则直接返回
        if (CollectionUtils.isEmpty(departmentArray)) {
            return departmentArray;
        }
        // 查询部门人员列表
        List<EocEmpInfoDTO> allEmps = queryEocEmpInfoList(departmentArray, user.getUserId());
        // 部门下人员为空直接返回
        if (CollectionUtils.isEmpty(allEmps)) {
            return allEmps;
        }
        // 筛选出agentId有值的用户
        List<EocEmpInfoDTO> agentEmpList = allEmps.stream().filter(emp -> StringUtils.isNotEmpty(emp.getAgentId())).collect(Collectors.toList());
        if (CollectionUtils.isNotEmpty(agentEmpList)) {
            // 筛选出的用户查询不可被代理的任务数量
            List<ThePrincipalTaskNum> taskNumList = principalTaskTypeService.queryTaskTypeNum(agentEmpList, user.getTenantId());
            // 组装不可被代理任务数量
            setTaskNum(allEmps, taskNumList);
        }
        return allEmps;
    }

    /**
     * 设置不可被代理任务数量
     *
     * @param allEmps
     * @param taskNumList
     */
    private void setTaskNum(List<EocEmpInfoDTO> allEmps, List<ThePrincipalTaskNum> taskNumList) {
        if (CollectionUtils.isEmpty(taskNumList)) {
            allEmps.forEach(emp -> {
                if (StringUtils.isNotEmpty(emp.getAgentId())) {
                    emp.setTaskNum(0L);
                }
            });
            return;
        }
        Map<String, Long> taskNumMap = taskNumList.stream().collect(Collectors.toMap(
                e -> e.getThePrincipalId() + "_" + e.getAgentId(),
                ThePrincipalTaskNum::getTaskNum,
                (key1, key2) -> key2));
        for (EocEmpInfoDTO emp : allEmps) {
            if (StringUtils.isNotEmpty(emp.getAgentId())) {
                Long num = taskNumMap.get(emp.getUserId() + "_" + emp.getAgentId());
                emp.setTaskNum(Objects.isNull(num) ? 0L : num);
            }
        }
    }

    /**
     * 查询部门下所有员工信息
     *
     * @param departmentArray
     * @param userId
     * @return
     */
    private List<EocEmpInfoDTO> queryEocEmpInfoList(JSONArray departmentArray, String userId) {
        List<EocEmpInfoDTO> allEmps = new ArrayList<>();
        // 过滤筛选directorId等于当前登录用户id的部门
        for (int i = 0; i < departmentArray.size(); i++) {
            JSONObject department = departmentArray.getJSONObject(i);
            if (!department.containsKey("directorId")) {
                // 过滤没有部门主管的部门
                continue;
            }
            if (Objects.equals(userId, department.getString("directorId"))) {
                // 过滤部门directorId不等于userId的数据
                continue;
            }
            // 部门的sid查询部门下的人员
            if (StringUtils.isNotEmpty(department.getString("sid"))) {
                List<EocEmpInfoDTO> emps = eocService.queryMembersOfTheDepartment(department.getString("sid"));
                if (CollectionUtils.isNotEmpty(emps)) {
                    // 循环人员信息，塞入部门信息
                    emps.forEach(emp -> {
                        emp.setDeptSid(department.getLong("sid"));
                        emp.setDeptid(department.getString("id"));
                        emp.setDeptName(department.getString("name"));
                    });
                    allEmps.addAll(emps);
                }
            }
        }
        return allEmps;
    }

    /**
     * 保存下属代理人信息
     *
     * @param saveSubordinateReq
     * @param user
     */
    @Override
    @Transactional(rollbackFor = Exception.class, transactionManager = BaseAudcDataSourceConfig.BASE_AUDC_DATASOURCE_TRANSACTION_MANAGER_BUSINESS)
    public void saveSubordinate(SaveSubordinateReq saveSubordinateReq, AuthoredUser user) {
        // 参数校验
        if (Objects.isNull(saveSubordinateReq) || CollectionUtils.isEmpty(saveSubordinateReq.getSaveList())) {
            throw AudcErrorCodeEnum.INVALID_REQUEST_PARAM.getBusinessException(messageUtils.getMessage("exception.param.null"));
        }
        for (ProxyInfoSaveReq proxyInfoSaveReq : saveSubordinateReq.getSaveList()) {
            validateParam(proxyInfoSaveReq);
        }
        List<ProxyInfoSaveReq> saveList = saveSubordinateReq.getSaveList();
        // 筛选出任务类型有修改的数据
        List<ProxyInfoSaveReq> needSaveList = saveList.stream().filter(e -> BooleanUtils.isTrue(e.getNeedSaveTaskDefCode())).collect(Collectors.toList());
        if (CollectionUtils.isNotEmpty(needSaveList)) {
            principalTaskTypeService.batchSaveTaskType(needSaveList, user.getTenantId());
        }
        List<String> clearList = saveSubordinateReq.getClearList();
        // 批量清除被代理人的任务类型
        if (CollectionUtils.isNotEmpty(clearList)) {
            principalTaskTypeService.batchDeleteTaskType(saveSubordinateReq.getClearList(), user.getTenantId());
        }
        // 筛选出需要发送工作提醒的数据调用eoc
        List<ProxyInfoSaveReq> needSendMsgToEocList = saveList.stream().filter(e -> BooleanUtils.isTrue(e.getNeedSendMessage())).collect(Collectors.toList());
        if (CollectionUtils.isNotEmpty(needSendMsgToEocList)) {
            eocService.batchEmpAgentChange(needSendMsgToEocList.stream()
                    .map(needSendMsgToEoc -> DataConverter.INSTANCE.toEmpAgentSaveReq(needSendMsgToEoc))
                    .collect(Collectors.toList()), user);
        }
        // 下属代理人信息保存，原有逻辑没有判断是否Athena用户，直接给atmc发送消息提醒
        if (CollectionUtils.isNotEmpty(needSendMsgToEocList)) {
            MessageAgentChangeBatchEvent event = new MessageAgentChangeBatchEvent(needSendMsgToEocList, user.getToken());
            asyncEventBus.post(event);
        }
        // 发送批量清除代理人信息事件
        if (CollectionUtils.isNotEmpty(clearList)) {
            MessageAgentClearBatchEvent batchEvent = new MessageAgentClearBatchEvent(clearList, user);
            asyncEventBus.post(batchEvent);
        }

    }

    /**
     * 删除下属代理人信息
     * @param proxyInfoRemoveReq
     * @param user
     */
    @Override
    @Transactional(rollbackFor = Exception.class, transactionManager = BaseAudcDataSourceConfig.BASE_AUDC_DATASOURCE_TRANSACTION_MANAGER_BUSINESS)
    public void removeSubordinate(ProxyInfoRemoveReq proxyInfoRemoveReq, AuthoredUser user) {
        if (Objects.isNull(proxyInfoRemoveReq)) {
            throw AudcErrorCodeEnum.INVALID_REQUEST_PARAM.getBusinessException(messageUtils.getMessage("exception.param.null"));
        }
        if (StringUtils.isEmpty(proxyInfoRemoveReq.getUserId())) {
            throw AudcErrorCodeEnum.INVALID_REQUEST_PARAM.getBusinessException(messageUtils.getMessage("exception.param.userId.null"));
        }
        // 清除被代理人的任务类型
        principalTaskTypeService.batchDeleteTaskType(Arrays.asList(proxyInfoRemoveReq.getUserId()), user.getTenantId());
        // 调用eoc接口清除代理人
        eocService.empAgentClear(proxyInfoRemoveReq.getUserId(), user);
        // 调用atmc发送工作提醒
        MessageAgentClearEvent event = new MessageAgentClearEvent(proxyInfoRemoveReq.getUserId(), user.getToken());
        asyncEventBus.post(event);
    }
}
