package com.digiwin.athena.base.application.service.cofw;

import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.digiwin.athena.appcore.auth.domain.AuthoredUser;
import com.digiwin.athena.appcore.util.MessageUtils;
import com.digiwin.athena.appcore.util.SnowflakeIdWorker;
import com.digiwin.athena.base.application.config.BaseAudcDataSourceConfig;
import com.digiwin.athena.base.infrastructure.constant.AudcErrorCodeEnum;
import com.digiwin.athena.base.infrastructure.manager.atmc.BaseAtmcService;
import com.digiwin.athena.base.infrastructure.manager.atmc.dto.VerifyAppPermissionDTO;
import com.digiwin.athena.base.infrastructure.manager.atmc.dto.VerifyAppPermissionReq;
import com.digiwin.athena.base.infrastructure.mapper.audc.cofw.CofwMapper;
import com.digiwin.athena.base.infrastructure.meta.po.cofw.Cofw;
import com.google.common.collect.Lists;
import lombok.extern.slf4j.Slf4j;
import net.sf.json.JSONArray;
import net.sf.json.JSONObject;
import org.apache.commons.collections4.CollectionUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.time.LocalDateTime;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

/**
 * @author xuxx
 * @date 2021/8/27
 * @since
 */
@Slf4j
@Service
public class CofwServiceImpl implements CofwService {

    @Autowired
    private MessageUtils messageUtils;

    @Autowired
    private BaseAtmcService baseAtmcService;

    private static final String ATHENA_TABLE = "ATHENA_TABLE";

    /**
     * 需要隐藏工具栏的表格组件类型集合
     */
    private static final List<String> hiddenToolbarComponentType = Lists.newArrayList(
            "TABLE",
            "GRID_TABLE",
            ATHENA_TABLE,
            "GROUP_ORDER");

    /**
     * 快照类型：0任务，1项目，2报表
     */
    public static final int COFW_TYPE_0 = 0;
    /**
     * 快照类型：0任务，1项目，2报表
     */
    public static final int COFW_TYPE_1 = 1;
    /**
     * 快照类型：0任务，1项目，2报表
     */
    public static final int COFW_TYPE_2 = 2;

    private static final String STR_COFW_TYPE_2 = "2";

    @Autowired
    private CofwMapper cofwMapper;

    /**
     * 快照保存
     *
     * @param cofwSaveBody
     */
    @Transactional(transactionManager = BaseAudcDataSourceConfig.BASE_AUDC_DATASOURCE_TRANSACTION_MANAGER_BUSINESS)
    @Override
    public void save(AuthoredUser user, JSONObject cofwSaveBody) {

        try {
            if (cofwSaveBody != null) {
                //业务ID
                String businessId = cofwSaveBody.getString("businessId");
                //快照类型
                String cofwType = cofwSaveBody.getString("cofwType");
                //快照名称
                String cofwName = cofwSaveBody.getString("cofwName");
                //快照内容
                JSONObject cofwContent = cofwSaveBody.getJSONObject("cofwContent");
                Cofw cofw = new Cofw();
                if (cofwContent != null) {
                    //快照当前登录人信息
                    cofw.setId(SnowflakeIdWorker.getInstance().newId());
                    cofw.setUserId(user.getUserId());
                    cofw.setUserName(user.getUserName());
                    cofw.setTenantId(user.getTenantId());
                    //快照存储信息
                    cofw.setCofwType(Integer.parseInt(cofwType));
                    cofw.setBusinessId(Long.parseLong(businessId));
                    cofw.setCofwName(cofwName);
                    cofw.setCreateTime(LocalDateTime.now());
                    //获取 快照备注（即任务/项目来源工单号、任务/项目名称）
                    if (cofwContent.containsKey("cofwMark")) {
                        JSONObject cofwMark = cofwContent.getJSONObject("cofwMark");
                        if (cofwMark != null) {
                            cofw.setCofwMark(cofwMark.toString());
                        }
                    }

                    //报表类型，需特别处理
                    if (cofwType.equals(STR_COFW_TYPE_2)) {
                        //TODO  报表快照逻辑写在这里
                    } else {
                        //解析 获取动态页面组件
                        if (cofwContent.containsKey("dynamicForm")) {
                            JSONObject dynamicForm = cofwContent.getJSONObject("dynamicForm");
                            if (dynamicForm != null) {
                                //修改 可操作的按钮为不可编辑
                                if (dynamicForm.containsKey("layout")) {
                                    dynamicForm.put("actions", new JSONArray());
                                    this.hiddenToolbar(dynamicForm);
                                }
                                cofw.setCofwContent(dynamicForm.toString());
                            } else {
                                cofw.setCofwContent("");
                            }
                        }
                        cofwMapper.insert(cofw);
                    }
                }
            }
        } catch (Exception e) {
            log.error("Collection failed", e);
            throw AudcErrorCodeEnum.COFW_SAVE_FAILED.getBusinessException(messageUtils.getMessage("cofw.save.fail.mes"));
        }
    }

    /**
     * 隐藏工具栏：多条件排序、表格设定
     *
     * @param dynamicForm 动态表单
     */
    private void hiddenToolbar(JSONObject dynamicForm) {
        JSONArray settingList = new JSONArray();
        JSONObject settingFixed = new JSONObject();
        settingList.add("composite-sort");
        settingList.add("setting");
        settingFixed.put("hideDefaultToolbar", settingList);
        settingFixed.put("type", ATHENA_TABLE);
        JSONArray layoutList = dynamicForm.getJSONArray("layout");

        this.hiddenToolbar(layoutList, settingFixed);
    }

    /**
     * 隐藏工具栏：多条件排序、表格设定
     *
     * @param componentArr 组件数组
     * @param settingFixed 固定配置
     */
    private void hiddenToolbar(JSONArray componentArr, JSONObject settingFixed) {
        if (CollectionUtils.isEmpty(componentArr)) {
            return;
        }
        for (int i = 0; i < componentArr.size(); i++) {
            JSONObject component = componentArr.getJSONObject(i);
            String type = component.getString("type");
            if (!hiddenToolbarComponentType.contains(type)) {
                continue;
            }
            if (component.containsKey("detailModel")) {
                JSONArray detailModelArr = component.getJSONArray("detailModel");
                hiddenToolbar(detailModelArr, settingFixed);
            }
            component.put("editable", false);
            component.put("checkbox", false);
            component.put("disabledUserDefined", true);
            component.put("saveColumnsWidth", false);
            JSONObject settingInner = component.getJSONObject("setting");
            settingInner.putAll(settingFixed);
        }
    }


    /**
     * 根据当前用户信息，获取快照列表
     *
     * @param user
     * @return
     */
    public List<Cofw> getCofwList(AuthoredUser user) {
        List<Cofw> list = cofwMapper.getCofwList(user.getUserId(), user.getTenantId());
        // 过滤没有权限的数据
        list = filterNoAppPermissionData(list);
        return list;
    }

    private List<Cofw> filterNoAppPermissionData(List<Cofw> list) {
        List<Cofw> newCofwList = new ArrayList<>();
        if (CollectionUtils.isEmpty(list)) {
            return newCofwList;
        }
        // 根据类型分组
        Map<Integer, List<Cofw>> typeGroup = list.stream().collect(Collectors.groupingBy(Cofw::getCofwType));
        typeGroup.forEach((type, cofwList) -> {
            // 项目卡
            if (1 == type) {
                List<Long> projectIdList = cofwList.stream().map(Cofw::getBusinessId).distinct().collect(Collectors.toList());
                // 校验应用权限，并且返回应用授权
                List<VerifyAppPermissionDTO> projectAppPermissioms = getAppPermissiom(1, projectIdList);
                filterCofwList(newCofwList, cofwList, projectAppPermissioms);
            }
            // 任务卡
            if (0 == type) {
                List<Long> taskIdList = cofwList.stream().map(Cofw::getBusinessId).distinct().collect(Collectors.toList());
                // 校验应用权限，并且返回应用授权
                List<VerifyAppPermissionDTO> taskAppPermissioms = getAppPermissiom(2, taskIdList);
                filterCofwList(newCofwList, cofwList, taskAppPermissioms);
            }
        });
        return newCofwList;

    }

    private void filterCofwList(List<Cofw> newCofwList, List<Cofw> cofwList, List<VerifyAppPermissionDTO> projectAppPermissioms) {

        // 获取有应用授权的卡片
        List<Long> businessIdList = projectAppPermissioms.stream().filter(VerifyAppPermissionDTO::getHasAppPermission).map(VerifyAppPermissionDTO::getBusinessId).distinct().collect(Collectors.toList());

        // 过滤
        cofwList = cofwList.stream().filter(cofw -> businessIdList.contains(cofw.getBusinessId())).collect(Collectors.toList());

        newCofwList.addAll(cofwList);

    }

    private List<VerifyAppPermissionDTO> getAppPermissiom(int type, List<Long> businessIdList) {
        VerifyAppPermissionReq verifyAppPermissionReq = new VerifyAppPermissionReq();
        verifyAppPermissionReq.setType(type);
        verifyAppPermissionReq.setBusinessIdList(businessIdList);
        return baseAtmcService.verifyAppPermission(verifyAppPermissionReq);
    }


    /**
     * 根据快照ID获取某一个快照信息
     */
    public Cofw getOneCofwById(long id) {
        return cofwMapper.getOneCofwById(id);
    }

    /**
     * 根据
     *
     * @param cofw
     */
    @Override
    @Transactional(transactionManager = BaseAudcDataSourceConfig.BASE_AUDC_DATASOURCE_TRANSACTION_MANAGER_BUSINESS)
    public void updateCofwById(Cofw cofw) {
        cofwMapper.updateById(cofw);
    }


    /**
     * 根据租户id清理租户的快照数据
     *
     * @param tenantId 租户id
     * @return 操作结果
     */
    @Override
    @Transactional(transactionManager = BaseAudcDataSourceConfig.BASE_AUDC_DATASOURCE_TRANSACTION_MANAGER_BUSINESS)
    public int deleteByTenant(String tenantId) {
        QueryWrapper<Cofw> deleteWrapper = new QueryWrapper<>();
        deleteWrapper.eq("tenant_id", tenantId);
        int result;
        try {
            result = cofwMapper.delete(deleteWrapper);
        } catch (Exception e) {
            log.error("Delete tenantId:[" + tenantId + "] error", e);
            return -1;

        }
        return result;
    }

}
