package com.digiwin.athena.base.application.service.trial;

import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.baomidou.mybatisplus.core.conditions.update.UpdateWrapper;
import com.digiwin.athena.appcore.util.SnowflakeIdWorker;
import com.digiwin.athena.base.application.config.BaseAudcDataSourceConfig;
import com.digiwin.athena.base.application.meta.request.trial.scene.TrialSceneReq;
import com.digiwin.athena.base.application.meta.request.trial.scene.TrialSceneResp;
import com.digiwin.athena.base.application.meta.request.trial.userScene.TrailUserSceneBindReq;
import com.digiwin.athena.base.application.meta.request.trial.userScene.TrialUserSceneReq;
import com.digiwin.athena.base.application.meta.request.trial.userScene.TrialUserSceneResp;
import com.digiwin.athena.base.infrastructure.mapper.audc.trial.TrialUserSceneMapper;
import com.digiwin.athena.base.infrastructure.meta.po.trial.TrialUserSceneData;
import org.apache.commons.collections4.MapUtils;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.time.LocalDateTime;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;

/**
 * 描述
 *
 * @author gonghongxing
 * @version 1.0
 * @since 2021/09/03 17:22:26
 */
@Service
public class TrialUserSceneServiceImpl implements TrialUserSceneService {

    private final TrialUserSceneMapper trialUserSceneMapper;

    private final TrialSceneService trialSceneService;

    /**
     * 默认构造函数
     *
     * @param trialUserSceneMapper trialUserSceneMapper
     */
    @Autowired
    public TrialUserSceneServiceImpl(TrialUserSceneMapper trialUserSceneMapper, TrialSceneService trialSceneService) {
        this.trialUserSceneMapper = trialUserSceneMapper;
        this.trialSceneService = trialSceneService;
    }

    /**
     * 新增用户场景信息
     *
     * @param req req
     * @return int
     */
    @Override
    @Transactional(transactionManager = BaseAudcDataSourceConfig.BASE_AUDC_DATASOURCE_TRANSACTION_MANAGER_BUSINESS)
    public int add(TrialUserSceneReq req) {
        TrialUserSceneData data = new TrialUserSceneData();
        BeanUtils.copyProperties(req, data);
        data.setId(SnowflakeIdWorker.getInstance().newId());
        data.setCreateDate(LocalDateTime.now());
        data.setModifyDate(LocalDateTime.now());
        return trialUserSceneMapper.insert(data);
    }

    /**
     * 修改用户场景信息
     *
     * @param data data
     * @return int
     */
    @Override
    @Transactional(transactionManager = BaseAudcDataSourceConfig.BASE_AUDC_DATASOURCE_TRANSACTION_MANAGER_BUSINESS)
    public int modify(TrialUserSceneReq data) {
        TrialUserSceneData userSceneData = new TrialUserSceneData();
        userSceneData.setId(data.getId());
        userSceneData.setModifyDate(LocalDateTime.now());
        userSceneData.setRank(data.getRank());
        return trialUserSceneMapper.updateById(userSceneData);
    }

    /**
     * 删除用户对应的场景
     *
     * @param id id
     * @return int
     */
    @Override
    @Transactional(transactionManager = BaseAudcDataSourceConfig.BASE_AUDC_DATASOURCE_TRANSACTION_MANAGER_BUSINESS)
    public int delete(Long id) {
        return trialUserSceneMapper.deleteById(id);
    }

    /**
     * 绑定用户与场景关系
     *
     * @param req req
     */
    @Override
    @Transactional(transactionManager = BaseAudcDataSourceConfig.BASE_AUDC_DATASOURCE_TRANSACTION_MANAGER_BUSINESS)
    public void bindScenes(TrailUserSceneBindReq req) {

        List<Map<String, Object>> scenes = req.getScenes();
        if (scenes != null && !scenes.isEmpty()) {
            //删除用户角色对应的记录
            //删除所有的场景与应用的绑定数据
            UpdateWrapper<TrialUserSceneData> delWrapper = new UpdateWrapper<>();
            delWrapper.eq("user_id", req.getUserId());
            delWrapper.eq("tenant_id", req.getTenantId());
            delWrapper.eq("app_code", req.getRoleId());
            trialUserSceneMapper.delete(delWrapper);
            //重新插入数据
            scenes.forEach(x -> {
                TrialUserSceneData data = new TrialUserSceneData();
                BeanUtils.copyProperties(req, data);
                data.setId(SnowflakeIdWorker.getInstance().newId());
                data.setSceneId(MapUtils.getLongValue(x, "sceneId"));
                data.setRank(MapUtils.getIntValue(x, "rank"));
                data.setSceneKey(MapUtils.getString(x, "sceneKey"));
                data.setCreateDate(LocalDateTime.now());
                data.setModifyDate(LocalDateTime.now());
                trialUserSceneMapper.insert(data);
            });
        }

    }

    /**
     * 查询试用场景列表
     *
     * @return list
     */
    public TrialUserSceneResp listWithScenes(TrailUserSceneBindReq req) {
        QueryWrapper<TrialUserSceneData> wrapper = new QueryWrapper<>();
        wrapper.eq("user_id", req.getUserId());
        wrapper.eq("tenant_id", req.getTenantId());
        wrapper.eq("app_code", req.getAppCode());
        List<TrialUserSceneData> userSceneList = trialUserSceneMapper.selectList(wrapper);
        TrialUserSceneResp resp = new TrialUserSceneResp();
        List<TrialSceneResp> listResp = new ArrayList<>();
        //遍历场景关联的步骤列表
        for (TrialUserSceneData data : userSceneList) {
            TrialSceneReq sceneReq = new TrialSceneReq();
            sceneReq.setSceneKey(data.getSceneKey());
            TrialSceneResp sceneResp = trialSceneService.listWithSteps(sceneReq);
            listResp.add(sceneResp);
        }
        resp.setScenes(listResp);
        return resp;
    }

    /**
     * 删除用户对应的场景
     *
     * @param sceneId sceneId
     * @return int
     */
    @Override
    @Transactional(transactionManager = BaseAudcDataSourceConfig.BASE_AUDC_DATASOURCE_TRANSACTION_MANAGER_BUSINESS)
    public int deleteBindSceneBySceneId(Long sceneId) {
        QueryWrapper<TrialUserSceneData> queryWrapper = new QueryWrapper<>();
        queryWrapper.eq("scene_id",sceneId);
        return trialUserSceneMapper.delete(queryWrapper);
    }


    /**
     * 根据用户id和场景，删除对应的绑定关系
     * @param userId userId
     * @param sceneId sceneId
     * @return int
     */
    @Override
    @Transactional(transactionManager = BaseAudcDataSourceConfig.BASE_AUDC_DATASOURCE_TRANSACTION_MANAGER_BUSINESS)
    public int deleteWithUser(String userId,Long sceneId)
    {
        QueryWrapper<TrialUserSceneData> queryWrapper = new QueryWrapper<>();
        queryWrapper.eq("scene_id",sceneId);
        queryWrapper.eq("user_id",userId);
        return trialUserSceneMapper.delete(queryWrapper);
    }
}
