package com.digiwin.athena.base.infrastructure.manager.thememap;

import com.digiwin.athena.appcore.auth.AppAuthContextHolder;
import com.digiwin.athena.appcore.auth.domain.AuthoredUser;
import com.digiwin.athena.appcore.domain.BaseResultDTO;
import com.digiwin.athena.appcore.exception.BusinessException;
import com.digiwin.athena.appcore.util.HttpUtils;
import com.digiwin.athena.appcore.util.JsonUtils;
import com.digiwin.athena.appcore.util.MessageUtils;
import com.digiwin.athena.base.infrastructure.manager.thememap.dto.AuthorityConfigResp;
import com.digiwin.athena.base.infrastructure.manager.thememap.dto.BaseActivityDefineDTO;
import com.digiwin.athena.base.infrastructure.manager.thememap.dto.BaseItemDto;
import com.digiwin.athena.base.infrastructure.manager.thememap.dto.TmAppDTO;
import com.fasterxml.jackson.core.type.TypeReference;
import lombok.extern.slf4j.Slf4j;
import net.sf.json.JSONObject;
import org.apache.commons.collections.CollectionUtils;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.context.i18n.LocaleContextHolder;
import org.springframework.core.ParameterizedTypeReference;
import org.springframework.http.HttpEntity;
import org.springframework.http.HttpHeaders;
import org.springframework.http.HttpMethod;
import org.springframework.http.HttpStatus;
import org.springframework.http.MediaType;
import org.springframework.http.ResponseEntity;
import org.springframework.stereotype.Service;
import org.springframework.util.StringUtils;
import org.springframework.web.client.HttpServerErrorException;
import org.springframework.web.client.RestTemplate;
import org.springframework.web.util.UriComponentsBuilder;

import javax.annotation.Resource;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;

/**
 * ThemeMapServiceImpl Description
 *
 * @author majianfu
 * @date 2021/8/28
 * @since
 */
@Service
@Slf4j
public class BaseThemeMapServiceImpl implements BaseThemeMapService {
    private static final String GET_ACTIVITIES_BY_PATTERN_URL = "knowledgegraph/task/activitiesByPattern";

    private static final String BASIC_DATA_PATTERN = "DATA_ENTRY";

    public static final String REPORT_PATTERN = "STATEMENT";
    public static final String BASE_DATA_PATTERN = "DATA_ENTRY";

    public static final Integer BASE_DATA_PLAT = 0;

    public static final Integer REPORT_PLAT = 1;

    public static final String APP_EXPIRED_ERROR_CODE = "RELATED_APP_EXPIRED";

    @Resource
    private RestTemplate restTemplate;

    @Value("${themeMap.uri:}")
    private String themeMapUri;

    @Resource
    private MessageUtils messageUtils;

    /**
     * {@inheritDoc}
     */
    @Override
    public List<BaseActivityDefineDTO> getTenantBasicDataActivities() {
        return getTenantActivitiesByPattern(BASIC_DATA_PATTERN);
    }

    /**
     * {@inheritDoc}
     */
    public List<BaseActivityDefineDTO> getTenantActivitiesByPattern(String pattern) {
        String url = UriComponentsBuilder.fromUriString(themeMapUri + GET_ACTIVITIES_BY_PATTERN_URL).queryParam("pattern", pattern).build().toString();
        try {
            ResponseEntity<BaseResultDTO<List<BaseActivityDefineDTO>>> responseEntity = restTemplate.exchange(url, HttpMethod.GET, null, new ParameterizedTypeReference<BaseResultDTO<List<BaseActivityDefineDTO>>>() {
            });
            List<BaseActivityDefineDTO> activityDefineList = responseEntity.getBody().getResponseWithException("");
            if (CollectionUtils.isNotEmpty(activityDefineList)) {
                activityDefineList.stream().forEach(activityDefine -> activityDefine.setPattern(pattern));
                return activityDefineList;
            } else {
                return Collections.emptyList();
            }
        } catch (Exception ex) {
            log.error("url: {}, error: ", url, ex);
            throw ex;
        }
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public List<BaseActivityDefineDTO> getTenantReportActivities() {
        return getTenantActivitiesByPattern(REPORT_PATTERN);
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public List<AuthorityConfigResp> getActivityAuthorityConfig(List<String> tmActivityIdList, String locale) {
        String url = themeMapUri + "knowledgegraph/activity/AuthorityConfigs";

        Map<String, Object> reqBody = new HashMap<>();
        reqBody.put("activityIdList", tmActivityIdList);

        return getAuthorityConfig(url, reqBody, locale);
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public List<AuthorityConfigResp> getTaskAuthorityConfig(List<String> tmTaskIdList, String locale) {
        String url = themeMapUri + "knowledgegraph/task/AuthorityConfigs";

        Map<String, Object> reqBody = new HashMap<>();
        reqBody.put("taskIdList", tmTaskIdList);

        return getAuthorityConfig(url, reqBody, locale);
    }

    /**
     * 获取基础资料录入、报表数据
     *
     * @Date：2021/9/23 17:01
     */
    @Override
    public List<BaseItemDto> getActivitiesByPattern(AuthoredUser user, Integer plat, String url, String pattern) {
        String uri = themeMapUri + (StringUtils.isEmpty(url) ? ("/knowledgegraph/task" + "/activitiesByPattern?pattern=") : url);
        pattern = pattern == null ? "" : pattern;
        //基础资料pattern
        if (Objects.equals(plat, 0)) {
            pattern = BASE_DATA_PATTERN;
        }
        //报表pattern
        else if (Objects.equals(plat, 1)) {
            pattern = StringUtils.hasLength(pattern) ? pattern : REPORT_PATTERN;
        }
        uri = uri + pattern;

        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON);
        headers.set("locale", LocaleContextHolder.getLocale().toString());
        headers.set("token", user.getToken());
        Map<String, String> param = new HashMap<>();
        HttpEntity<?> httpEntity = new HttpEntity<>(headers);
        ResponseEntity<BaseResultDTO<List<BaseItemDto>>> respMap;
        try {
            respMap = restTemplate.exchange(uri, HttpMethod.GET, httpEntity, new ParameterizedTypeReference<BaseResultDTO<List<BaseItemDto>>>() {
            }, param);
            return respMap.getBody().getResponseWithException("");
        } catch (Exception e) {
            log.error("url: {}, error: ", uri, e);
            throw e;
        }
    }

    @Override
    public List<Map<String, Object>> getActivitiesStartProject(AuthoredUser user) {
        String uri = themeMapUri + "/knowledgegraph/task/manualList";

        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON);
        headers.set("locale", LocaleContextHolder.getLocale().toString());
        headers.set("token", user.getToken());
        Map<String, String> param = new HashMap<>();
        HttpEntity<?> httpEntity = new HttpEntity<>(headers);
        ResponseEntity<BaseResultDTO<List<Map<String, Object>>>> respMap;
        try {
            respMap = restTemplate.exchange(uri, HttpMethod.GET, httpEntity, new ParameterizedTypeReference<BaseResultDTO<List<Map<String, Object>>>>() {
            }, param);
            return respMap.getBody().getResponseWithException("");
        } catch (Exception e) {
            log.error("url: {}, error: ", uri, e);
            throw e;
        }
    }

    private List<AuthorityConfigResp> getAuthorityConfig(String url, Map<String, Object> reqBody, String locale) {
        HttpHeaders headers = new HttpHeaders();
        headers.add("locale", locale);
        headers.add("token", getToken());
        HttpEntity<Map> requestEntity = new HttpEntity<>(reqBody, headers);

        try {
            ResponseEntity<BaseResultDTO<List<AuthorityConfigResp>>> responseEntity = restTemplate.exchange(url, HttpMethod.POST, requestEntity, new ParameterizedTypeReference<BaseResultDTO<List<AuthorityConfigResp>>>() {
            });
            return responseEntity.getBody().getResponseWithException("");
        } catch (Exception ex) {
            log.error("body: {}, error: ", JsonUtils.objectToString(reqBody), ex);
            throw ex;
        }
    }

    private String getToken() {
        if (null != AppAuthContextHolder.getContext().getProxyAuthoredUser()) {
            return AppAuthContextHolder.getContext().getProxyAuthoredUser().getToken();
        } else {
            return AppAuthContextHolder.getContext().getAuthoredUser().getToken();
        }
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public Map getActivityWithoutTaskId(String tmActivityId, String pageCode, String locale) {
        String url = themeMapUri + "knowledgegraph/task/activityDefinition?" + "activityId=" + tmActivityId + "&pageCode=" + pageCode;
        return getTmActivityDef(pageCode, locale, url);
    }

    @Override
    public List<TmAppDTO> getAppInfoByCodeAndType(String type, List<String> codeList) {
        String url = themeMapUri + "knowledgegraph/task/applicationRelation";
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON);

        Map<String, Object> params = new HashMap<>();
        params.put("type", type);
        params.put("codes", codeList);
        HttpEntity<?> httpEntity = new HttpEntity<>(params, headers);
        ResponseEntity<BaseResultDTO<List<TmAppDTO>>> respEntity = null;
        try {
            respEntity = this.restTemplate.exchange(url, HttpMethod.POST, httpEntity, new ParameterizedTypeReference<BaseResultDTO<List<TmAppDTO>>>() {
            });
            log.info("[getAppInfoByCodeAndType]  type = {}, codes = {}, result = {}", type, JsonUtils.objectToString(codeList), JsonUtils.objectToString(respEntity));
        } catch (Exception e) {
            log.error("[getAppInfoByCodeAndType] execute km [knowledgegraph/task/applicationRelation] fail, error = {}", e.getMessage());
        }

        if (null != respEntity && null != respEntity.getBody()) {
            return respEntity.getBody().getResponseWithException("");
        }
        return null;
    }

    @Override
    public Map getGroupConfig() {
        String url = themeMapUri + "knowledgegraph/task/cardConfig";
        HttpHeaders headers = new HttpHeaders();
        headers.add("token", getToken());
        HttpEntity<String> requestEntity = new HttpEntity<>(null, headers);
        ResponseEntity<BaseResultDTO<Map>> respEntity = restTemplate.exchange(url, HttpMethod.GET, requestEntity, new ParameterizedTypeReference<BaseResultDTO<Map>>() {
        });
        Map cardConfig = respEntity.getBody().getResponseWithException("");
        if (null == cardConfig) {
            throw BusinessException.create(String.format(messageUtils.getMessage("exception.tm.null"), url));
        }
        return cardConfig;
    }

    private Map getTmActivityDef(String pageCode, String locale, String url) {
        HttpHeaders headers = new HttpHeaders();
        headers.add("locale", locale);
        headers.add("token", getToken());
        HttpEntity<String> requestEntity = new HttpEntity<>(null, headers);
        try {
            ResponseEntity<BaseResultDTO<Map>> respEntity = restTemplate.exchange(url, HttpMethod.GET, requestEntity, new ParameterizedTypeReference<BaseResultDTO<Map>>() {
            });
            // 应用过期
            Map tmActivity = respEntity.getBody().getResponseWithException("");
            if (null == tmActivity) {
                throw BusinessException.create(String.format(messageUtils.getMessage("exception.tm.null"), url));
            }
            return tmActivity;
        } catch (HttpServerErrorException ex) {
            // 500
            if (HttpStatus.INTERNAL_SERVER_ERROR.value() == ex.getStatusCode().value()) {
                // 尝试反序列化
                BaseResultDTO baseResultDTO = null;
                try {
                    // 尝试反序列化
                    baseResultDTO = JsonUtils.jsonToObject(ex.getResponseBodyAsString(), new TypeReference<BaseResultDTO>() {
                    });
                } catch (Exception exception) {
                    // 反序列化失败，说明不是应用过期
                    baseResultDTO = null;
                }

                // 应用过期
                if (null != baseResultDTO && isAppExpiredErrorResp(baseResultDTO.getStatus(), baseResultDTO.getErrorCode())) {
                    // 应用过期
                    String errMsg = messageUtils.getMessageWithFormat("exception.app.expired", baseResultDTO.getErrorMessage());
                    // 重新生成状态码：错误码整改之后，如果还是使用错误码RELATED_APP_EXPIRED返回的话，前端会优化显示，不能直观显示出异常
                    throw BusinessException.create(baseResultDTO.getStatus(), "P.KM.600.11111", errMsg, errMsg, errMsg, null);
                }
                // 非应用过期
                else {
                    throw ex;
                }
            }
            // 非500
            else {
                throw ex;
            }
        }
    }

    private boolean isAppExpiredErrorResp(int statusCode, String errorCode) {
        return HttpStatus.INTERNAL_SERVER_ERROR.value() == statusCode && APP_EXPIRED_ERROR_CODE.equals(errorCode);
    }

    @Override
    public Map<String, Object> getQueryCondition(String pageCode, String conditionType) {
        String url = themeMapUri + "knowledgegraph/task/queryCondition";
        HttpHeaders headers = new HttpHeaders();
        headers.add("token", getToken());
        JSONObject jsonObject = new JSONObject();
        jsonObject.putIfAbsent("pageCode", pageCode);
        jsonObject.putIfAbsent("configType", conditionType);
        Map<String, Object> result = new HashMap<>();
        JSONObject body = new JSONObject();
        body.putIfAbsent("cardConditionQuery", jsonObject);
        HttpEntity<?> requestEntity = new HttpEntity<>(body, headers);

        ResponseEntity<Object> respEntity = this.restTemplate.exchange(url, HttpMethod.POST, requestEntity, new ParameterizedTypeReference<Object>() {
        });
        return HttpUtils.descResponseBody(url, null, respEntity, new TypeReference<Map<String, Object>>() {
        });

    }

}
