package com.digiwin.athena.base.infrastructure.manager.eoc;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.digiwin.athena.appcore.auth.domain.AuthoredUser;
import com.digiwin.athena.appcore.exception.BusinessException;
import com.digiwin.athena.appcore.util.ExceptionUtil;
import com.digiwin.athena.appcore.util.JsonUtils;
import com.digiwin.athena.appcore.util.MessageUtils;
import com.digiwin.athena.base.infrastructure.constant.AudcErrorCodeEnum;
import com.digiwin.athena.base.infrastructure.manager.eoc.dto.EmpAgentSaveReqDTO;
import com.digiwin.athena.base.infrastructure.manager.eoc.dto.EmpInfoDTO;
import com.digiwin.athena.base.infrastructure.manager.eoc.dto.EocBaseResultDTO;
import com.digiwin.athena.base.infrastructure.manager.eoc.dto.EocEmpInfoDTO;
import com.digiwin.athena.base.infrastructure.manager.eoc.dto.EocResultDTO;
import com.digiwin.athena.base.infrastructure.manager.eoc.dto.EocTenantAllUserDTO;
import lombok.extern.slf4j.Slf4j;
import net.sf.json.JSONArray;
import org.apache.commons.lang.StringUtils;
import org.apache.http.HttpStatus;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.context.i18n.LocaleContextHolder;
import org.springframework.core.ParameterizedTypeReference;
import org.springframework.http.HttpEntity;
import org.springframework.http.HttpHeaders;
import org.springframework.http.HttpMethod;
import org.springframework.http.MediaType;
import org.springframework.http.ResponseEntity;
import org.springframework.stereotype.Service;
import org.springframework.web.client.RestTemplate;

import javax.annotation.Resource;
import java.util.Collections;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;

/**
 * EocServiceImpl Description
 *
 * @author majianfu
 * @date 2022/8/1
 * @since
 */
@Slf4j
@Service
public class BaseEocServiceImpl implements BaseEocService {
    @Value("${eoc.uri:}")
    private String eocUri;

    @Resource
    private RestTemplate restTemplate;

    @Resource
    private MessageUtils messageUtils;

    /**
     * {@inheritDoc}
     */
    @Override
    public List<EmpInfoDTO> queryEmpDeptInfoByUserId(List<String> userIds, int pageNum, int pageSize) {
        //http://172.16.2.141:22694/html/web/controller/share/share.html#5f110cf741fcbe5bc65089cd

        String url = eocUri + "/api/eoc/v2/emp/simple?pageNum=1&pageSize={pageSize}&params={params}";

        Map<String, Object> params = new HashMap<>();
        params.put("userIds", userIds);
        params.put("bindUser", true);

        Map<String, String> param = new HashMap<>();
        param.put("pageNum", String.valueOf(pageNum));
        param.put("pageSize", String.valueOf(pageSize));
        param.put("params", JsonUtils.objectToString(params));

        HttpHeaders httpHeaders = new HttpHeaders();
        httpHeaders.setContentType(MediaType.APPLICATION_JSON);

        HttpEntity<?> httpEntity = new HttpEntity<>(httpHeaders);
        ResponseEntity<EocBaseResultDTO<EocTenantAllUserDTO>> responseEntity = restTemplate.exchange(url, HttpMethod.GET, httpEntity,
                new ParameterizedTypeReference<EocBaseResultDTO<EocTenantAllUserDTO>>() {
                }, param);

        Integer code = responseEntity.getBody().getCode();
        // code 为 200
        try {
            if (StringUtils.equals(String.valueOf(code), String.valueOf(HttpStatus.SC_OK))) {
                return responseEntity.getBody().getData().getList();
            } else {
                String errorMessage = messageUtils.getMessage("eoc.v2.emp.simple");
                log.error("{}, {}, {}", AudcErrorCodeEnum.EOC_V2_EMP_SIMPLE.getErrCode(), errorMessage, "status code: " + HttpStatus.SC_OK + ", code: " + String.valueOf(code) + ", message: " + String.valueOf(responseEntity.getBody().getMessage()));
                throw BusinessException.create(AudcErrorCodeEnum.EOC_V2_EMP_SIMPLE.getErrCode(), errorMessage);
            }
        } catch (Exception ex) {
            throw ExceptionUtil.wrap(AudcErrorCodeEnum.EOC_V2_EMP_SIMPLE.getErrCode(), ex);
        }
    }

    /**
     * 调用eoc接口保存代理人和代理时间
     *
     * @param proxyInfoSaveReq
     * @param user
     */
    @Override
    public void empAgentChange(EmpAgentSaveReqDTO proxyInfoSaveReq, AuthoredUser user) {
        String url = eocUri + "/api/eoc/v2/emp/agent/change";

        Map<String, Object> param = new HashMap<>();
        param.put("userId", proxyInfoSaveReq.getUserId());
        param.put("agentId", proxyInfoSaveReq.getAgentId());
        param.put("agentBeginDate", proxyInfoSaveReq.getAgentBeginDate());
        param.put("agentEndDate", proxyInfoSaveReq.getAgentEndDate());

        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON);
        headers.add("token", user.getToken());
        HttpEntity<?> httpEntity = new HttpEntity<>(param, headers);
        try {
            log.info("/api/eoc/v2/emp/agent/change request param", JsonUtils.objectToString(param));
            ResponseEntity<EocBaseResultDTO> responseEntity = restTemplate.exchange(url, HttpMethod.POST, httpEntity, new ParameterizedTypeReference<EocBaseResultDTO>() {
            });
            log.info("/api/eoc/v2/emp/agent/change response", JsonUtils.objectToString(responseEntity));
            if (Objects.isNull(responseEntity) || Objects.isNull(responseEntity.getBody())
                    || !StringUtils.equals(String.valueOf(responseEntity.getBody().getCode()), String.valueOf(HttpStatus.SC_OK))) {
                String errorMessage = messageUtils.getMessage("eoc.v2.emp.agent.change");
                log.error("{}, {}, {}", AudcErrorCodeEnum.EOC_V2_EMP_AGENT_CHANGE.getErrCode(), errorMessage, responseEntity);
                throw BusinessException.create(AudcErrorCodeEnum.EOC_V2_EMP_AGENT_CHANGE.getErrCode(), errorMessage);
            }
        } catch (Exception ex) {
            throw ExceptionUtil.wrap(AudcErrorCodeEnum.EOC_V2_EMP_AGENT_CHANGE.getErrCode(), ex);
        }
    }

    /**
     * 批量保存代理人信息
     *
     * @param needSendMsgToEocList
     * @param user
     */
    @Override
    public void batchEmpAgentChange(List<EmpAgentSaveReqDTO> needSendMsgToEocList, AuthoredUser user) {
        String url = eocUri + "/api/eoc/v2/emp/agent/change/batch";

        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON);
        headers.add("token", user.getToken());
        HttpEntity<?> httpEntity = new HttpEntity<>(JsonUtils.objectToString(needSendMsgToEocList), headers);
        try {
            log.info("/api/eoc/v2/emp/agent/change/batch request param", JsonUtils.objectToString(needSendMsgToEocList));
            ResponseEntity<EocBaseResultDTO> responseEntity = restTemplate.exchange(url, HttpMethod.POST, httpEntity, new ParameterizedTypeReference<EocBaseResultDTO>() {
            });
            log.info("/api/eoc/v2/emp/agent/change response", JsonUtils.objectToString(responseEntity));
            if (Objects.isNull(responseEntity) || Objects.isNull(responseEntity.getBody())
                    || !StringUtils.equals(String.valueOf(responseEntity.getBody().getCode()), String.valueOf(HttpStatus.SC_OK))) {
                String errorMessage = messageUtils.getMessage("eoc.v2.emp.agent.change");
                log.error("{}, {}, {}", AudcErrorCodeEnum.EOC_V2_EMP_AGENT_CHANGE.getErrCode(), errorMessage, responseEntity);
                throw BusinessException.create(AudcErrorCodeEnum.EOC_V2_EMP_AGENT_CHANGE.getErrCode(), errorMessage);
            }
        } catch (Exception ex) {
            throw ExceptionUtil.wrap(AudcErrorCodeEnum.EOC_V2_EMP_AGENT_CHANGE.getErrCode(), ex);
        }
    }

    /**
     * 调用eoc接口清除代理人信息
     *
     * @param userId
     * @param user
     */
    @Override
    public void empAgentClear(String userId, AuthoredUser user) {
        String url = eocUri + "/api/eoc/v2/emp/agent/clear";

        Map<String, Object> param = new HashMap<>();
        param.put("userId", userId);

        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON);
        headers.add("token", user.getToken());
        HttpEntity<?> httpEntity = new HttpEntity<>(param, headers);
        try {
            log.info("/api/eoc/v2/emp/agent/clear request param", JsonUtils.objectToString(param));
            ResponseEntity<EocBaseResultDTO> responseEntity = restTemplate.exchange(url, HttpMethod.POST, httpEntity, new ParameterizedTypeReference<EocBaseResultDTO>() {
            });
            log.info("/api/eoc/v2/emp/agent/clear response", JsonUtils.objectToString(responseEntity));
            if (Objects.isNull(responseEntity) || Objects.isNull(responseEntity.getBody())
                    || !StringUtils.equals(String.valueOf(responseEntity.getBody().getCode()), String.valueOf(HttpStatus.SC_OK))) {
                String errorMessage = messageUtils.getMessage("eoc.v2.emp.agent.clear");
                log.error("{}, {}, {}", AudcErrorCodeEnum.EOC_V2_EMP_AGENT_CLEAR.getErrCode(), errorMessage, responseEntity);
                throw BusinessException.create(AudcErrorCodeEnum.EOC_V2_EMP_AGENT_CLEAR.getErrCode(), errorMessage);
            }
        } catch (Exception ex) {
            throw ExceptionUtil.wrap(AudcErrorCodeEnum.EOC_V2_EMP_AGENT_CLEAR.getErrCode(), ex);
        }
    }

    /**
     * 根据token获取对应租户的所有部门及子部门
     *
     * @return
     */
    @Override
    public JSONArray queryCurrentTenantAllDepartment() {
        String uri = eocUri + "/api/eoc/v2/dept/list?pageNum=1&pageSize={pageSize}";
        Map<String, Object> params = new HashMap<>();
        params.put("pageSize", 9999);

        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON);
        headers.add("locale", LocaleContextHolder.getLocale().toString());

        HttpEntity<?> httpEntity = new HttpEntity<>(headers);
        ResponseEntity<EocResultDTO> responseEntity = restTemplate.exchange(uri, HttpMethod.GET, httpEntity, EocResultDTO.class, params);
        EocResultDTO result = responseEntity.getBody();
        if (result != null && result.isSuccess()) {
            return JSONArray.fromObject(((LinkedHashMap) result.getData()).get("list"));
        }
        return new JSONArray();
    }

    /**
     * 根据部门Sid获取部门下的人员
     *
     * @param departmentSid
     * @return
     */
    @Override
    public List<EocEmpInfoDTO> queryMembersOfTheDepartment(String departmentSid) {
        String uri = eocUri + "/api/eoc/v2/dept/{departmentSid}/emps?pageNum=1&pageSize={pageSize}&params={statusParam}";
        Map<String, Object> params = new HashMap<>();
        params.put("departmentSid", departmentSid);
        params.put("pageSize", 9999);
        Map<String, Object> status = new HashMap<>();
        status.put("status", true);
        params.put("statusParam", JSON.toJSONString(status));
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON);
        headers.add("locale", LocaleContextHolder.getLocale().toString());

        HttpEntity<?> httpEntity = new HttpEntity<>(headers);
        ResponseEntity<EocResultDTO> responseEntity = restTemplate.exchange(uri, HttpMethod.GET, httpEntity, EocResultDTO.class, params);
        EocResultDTO result = responseEntity.getBody();
        if (result != null && result.isSuccess()) {
            return JSONObject.parseArray(JSONObject.toJSONString(((LinkedHashMap) result.getData()).get("list")), EocEmpInfoDTO.class);
        }
        return Collections.emptyList();
    }
}
