package com.digiwin.athena.base.infrastructure.manager.iam.service;

import com.digiwin.athena.appcore.auth.GlobalConstant;
import com.digiwin.athena.appcore.auth.domain.AuthoredUser;
import com.digiwin.athena.appcore.util.ExceptionUtil;
import com.digiwin.athena.appcore.util.JsonUtils;
import com.digiwin.athena.base.infrastructure.constant.Constants;
import com.digiwin.athena.base.infrastructure.constant.AudcErrorCodeEnum;
import com.digiwin.athena.base.infrastructure.constant.TenantEnum;
import com.digiwin.athena.base.infrastructure.manager.iam.model.AppAuthoredDTO;
import com.digiwin.athena.base.infrastructure.manager.iam.model.CommonResultDTO;
import com.digiwin.athena.base.infrastructure.manager.iam.model.TenantAuthDTO;
import com.digiwin.athena.base.infrastructure.manager.iam.model.TenantDetailDTO;
import com.github.pagehelper.util.StringUtil;
import lombok.extern.slf4j.Slf4j;
import net.sf.json.JSONArray;
import net.sf.json.JSONObject;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.compress.utils.Lists;
import org.apache.http.HttpStatus;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.core.ParameterizedTypeReference;
import org.springframework.http.HttpEntity;
import org.springframework.http.HttpHeaders;
import org.springframework.http.HttpMethod;
import org.springframework.http.MediaType;
import org.springframework.http.ResponseEntity;
import org.springframework.stereotype.Service;
import org.springframework.web.client.RestTemplate;

import javax.annotation.Resource;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;

/**
 * IamServiceImpl Description
 *
 * @author majianfu
 * @date 2021/9/1
 * @since
 */
@Service
@Slf4j
public class IamServiceImpl implements BaseIamService {
    @Value("${athena.auth.uri:}")
    private String iamUri;

    @Resource
    private RestTemplate restTemplate;

    private static final String USER_PERMISSION_V2 = "/api/iam/v2/permission/user";

    private static final String USER_AUTHORED_PERMISSION_URL = "/api/iam/v2/user/application";

    private static final String ALL_USER_PERMISSION_URL = "/api/iam/v2/permission/users/apps/all";

    private static final String TENANT_APPLICATION_LIST_URL = "/api/iam/v2/tenant/application/list";

    // 根据邮箱查询用户归属的租户以及指定应用授权信息
    // 接口文档：http://172.16.2.141:22694/html/web/controller/share/share.html#637478992d6d101906486069
    private static final String QUERY_TENANT_AUTH_BY_EMAIL_URL = "/api/iam/v2/user/email/tenants/auth";

    // 创建个人租户并且自动授权应用
    // 接口文档：http://172.16.2.141:22694/html/web/controller/share/share.html#6342a02d2d6d101906485f6b
    private static final String CREATE_INDIVIDUAL_TENANT_AUTH_URL = "/api/iam/v2/user/individual/tenant/register";

    // 给指定租户下的用户授权指定应用
    // 接口文档：http://172.16.2.141:22694/html/web/controller/share/share.html#6374795a2d6d10190648606b
    private static final String CREATE_TENANT_APP_AUTH_URL = "/api/iam/v2/user/tenant/user/auth";

    // 查询批量用户的单项元数据设置
    private static final String BATCH_QUERY_METADATA_ITEM_URL = "/api/iam/v2/usermetadata/user/get/batch";

    @Override
    public JSONObject getAllUserPermissionV2(AuthoredUser authoredUser, String appCode) {
        HttpHeaders headers = new HttpHeaders();
        headers.add("Content-Type", "application/json");
        headers.add("digi-middleware-auth-user", authoredUser.getToken());

        Map<String, Object> requestMap = new HashMap<>();
        requestMap.put("userId", authoredUser.getUserId());
        requestMap.put("sysId", appCode);
        HttpEntity<Map> requestEntity = new HttpEntity<>(requestMap, headers);

        String url = iamUri + USER_PERMISSION_V2;
        try {
            ResponseEntity<JSONObject> respEntity = restTemplate.exchange(url, HttpMethod.POST, requestEntity, JSONObject.class);
            return respEntity.getBody();
        } catch (Exception ex) {
            log.error("从IAM获取租户（{}）的应用（{}）权限配置出错, url：{}", authoredUser.getTenantId(), appCode, url);
            return null;
        }
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public List<AppAuthoredDTO> getUserAuthoredApp(String token) {
        HttpHeaders headers = new HttpHeaders();
        headers.add("Content-Type", "application/json");
        headers.add("digi-middleware-auth-user", token);

        HttpEntity<Map> reqEntity = new HttpEntity<>(null, headers);
        String url = iamUri + USER_AUTHORED_PERMISSION_URL;
        try {
            ResponseEntity<List<AppAuthoredDTO>> respEntity = restTemplate.exchange(url, HttpMethod.GET, reqEntity,
                    new ParameterizedTypeReference<List<AppAuthoredDTO>>() {
                    });
            return respEntity.getBody();
        } catch (Exception ex) {
            log.error("{}，获取用户已授权应用列表失败， 用户token: {}, 错误内容：", AudcErrorCodeEnum.IAM_V2_USER_APPLICATION.getErrCode(), token, ex);
            throw ExceptionUtil.wrap(AudcErrorCodeEnum.IAM_V2_USER_APPLICATION.getErrCode(), ex);
        }
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public JSONArray batchQueryUserAppPermission(AuthoredUser authoredUser, List<String> authoredAppCodeList) {
        HttpHeaders headers = new HttpHeaders();
        headers.add("Content-Type", "application/json");
        headers.add("digi-middleware-auth-user", authoredUser.getToken());

        Map<String, Object> requestMap = new HashMap<>();
        requestMap.put("userIds", Collections.singletonList(authoredUser.getUserId()));
        requestMap.put("sysIds", authoredAppCodeList);

        HttpEntity<Map> reqEntity = new HttpEntity<>(requestMap, headers);
        String url = iamUri + ALL_USER_PERMISSION_URL;

        try {
            ResponseEntity<JSONArray> respEntity = restTemplate.exchange(url, HttpMethod.POST, reqEntity, JSONArray.class);
            return respEntity.getBody();
        } catch (Exception ex) {
            log.error("{}，批量获取用户授权应用权限配置失败，用户token: {}，请求body体：{}，错误内容：", AudcErrorCodeEnum.IAM_V2_PERMISSION_USERS_APPS_ALL.getErrCode(), authoredUser.getToken(),
                    JsonUtils.objectToString(requestMap), ex);
            throw ExceptionUtil.wrap(AudcErrorCodeEnum.IAM_V2_PERMISSION_USERS_APPS_ALL.getErrCode(), ex);
        }
    }

    /**
     * @param
     * @return status的值
     * -1 调用接口失败
     * 0 没有个人租户
     * 1 没有授权
     * 2 有个人租户并且有授权
     * @description: 根据邮箱、号码获取个人租户
     * @author: sunyfa
     */
    @Override
    public Map<String, Object> getTenantAuth(AuthoredUser authoredUser, String email, String telephone, String appId,
                                             Integer tenantType) {
        // 邮箱手机，不能全为空
        if (StringUtil.isEmpty(email) && StringUtil.isEmpty(telephone)) {
            return null;
        }

        Map<String, Object> resultMap = new HashMap<>();

        // 接口地址
        String url = iamUri + QUERY_TENANT_AUTH_BY_EMAIL_URL;

        // 头部数据
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON);
        headers.add(GlobalConstant.IAM_USER_TOKEN, authoredUser.getToken());

        // 参数
        Map<String, Object> param = new HashMap<>();
        param.put("email", email);
        param.put("telephone", telephone);
        // 查询授权的应用id
        param.put("appId", StringUtil.isNotEmpty(appId) ? appId : Constants.ATHENA_APPID);
        // 租户类型，0-当前用户创建的个人租户，1-用户加入的企业租户，不传默认查询全部租户类型
        param.put("tenantType", tenantType != null ? tenantType : TenantEnum.INDIVIDUAL.getType());
        HttpEntity<?> httpEntity = new HttpEntity<>(param, headers);

        try {
            // 调用接口
            log.info("getTenantAuth post iam begin, URL:{}, param:{}", url, JsonUtils.objectToString(httpEntity));
            ResponseEntity<CommonResultDTO> responseEntity = restTemplate.exchange(url, HttpMethod.POST, httpEntity,
                    CommonResultDTO.class, new HashMap<>());
            log.info("getTenantAuth post iam end, result:{}", JsonUtils.objectToString(responseEntity));

            // 调用失败
            if (Objects.isNull(responseEntity)
                    || Objects.isNull(responseEntity.getBody())
                    || Objects.isNull(responseEntity.getBody().getData())
                    || responseEntity.getStatusCodeValue() != HttpStatus.SC_OK) {
                resultMap.put("status", -1);
                log.error("获取用户归属的租户以及指定应用授权信息失败");
                return resultMap;
            }

            // 解析租户信息和用户信息
            TenantAuthDTO tenantAuthDTO = JsonUtils.jsonToObject(
                    JsonUtils.objectToString(responseEntity.getBody().getData()), TenantAuthDTO.class);
            // 租户信息
            List<TenantDetailDTO> tenantDTOS = tenantAuthDTO.getTenants();
            // 没有个人租户
            if (CollectionUtils.isEmpty(tenantDTOS)) {
                resultMap.put("status", 0);
                log.info("获取用户归属的租户以及指定应用授权信息失败：没有个人租户");
                return resultMap;
            }

            // 取第一条个人租户
            TenantDetailDTO tenantDTO = tenantDTOS.get(0);
            // 判断有没有授权
            if (!tenantDTO.getIsTenantAuth() || !tenantDTO.getIsUserAuth()) {
                resultMap.put("status", 1);
                resultMap.put("userId", tenantAuthDTO.getUserId());
                resultMap.put("tenantId", tenantDTO.getId());
                resultMap.put("isTenantAuth", tenantDTO.getIsTenantAuth());
                resultMap.put("isUserAuth", tenantDTO.getIsUserAuth());
                log.info("获取用户归属的租户以及指定应用授权信息失败（没有授权）：{}",
                        JsonUtils.objectToString(tenantAuthDTO));
                return resultMap;
            }

            // 获取用户id和租户id
            resultMap.put("status", 2);
            resultMap.put("userId", tenantAuthDTO.getUserId());
            resultMap.put("tenantId", tenantDTO.getId());
            resultMap.put("isTenantAuth", tenantDTO.getIsTenantAuth());
            resultMap.put("isUserAuth", tenantDTO.getIsUserAuth());
        } catch (Exception e) {
            resultMap.put("status", -1);
            log.error("获取用户归属的租户以及指定应用授权信息失败", e);
        }
        return resultMap;
    }

    /**
     * @param
     * @return
     * @description: 创建个人租户以及授权
     * @author: sunyfa
     */
    @Override
    public Map<String, String> registerTenant(AuthoredUser authoredUser, String email, String telephone, String appId,
                                              String strategySid, String strategyCode) {
        // 邮箱手机，不能全为空
        if (StringUtil.isEmpty(email) && StringUtil.isEmpty(telephone)) {
            return null;
        }

        Map<String, String> resultMap = new HashMap<>();

        // 接口地址
        String url = iamUri + CREATE_INDIVIDUAL_TENANT_AUTH_URL;

        // 头部数据
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON);
        headers.add(GlobalConstant.IAM_USER_TOKEN, authoredUser.getToken());

        // 参数
        Map<String, Object> param = new HashMap<>();
        param.put("email", email);
        param.put("telephone", telephone);
        param.put("appId", StringUtil.isNotEmpty(appId) ? appId : Constants.ATHENA_APPID);
        param.put("strategyCode", StringUtil.isNotEmpty(strategyCode) ? strategyCode : "athena_light_01");
        HttpEntity<?> httpEntity = new HttpEntity<>(param, headers);

        try {
            // 调用接口
            log.info("registerTenant post iam begin, URL:{}, param:{}", url, JsonUtils.objectToString(httpEntity));
            ResponseEntity<CommonResultDTO> responseEntity = restTemplate.exchange(url, HttpMethod.POST, httpEntity,
                    CommonResultDTO.class, new HashMap<>());
            log.info("registerTenant post iam end, result:{}", JsonUtils.objectToString(responseEntity));
            if (Objects.nonNull(responseEntity)
                    && Objects.nonNull(responseEntity.getBody())
                    && Objects.nonNull(responseEntity.getBody().getData())
                    && responseEntity.getStatusCodeValue() == HttpStatus.SC_OK) {
                Object data = responseEntity.getBody().getData();
                resultMap = JsonUtils.jsonToObject(JsonUtils.objectToString(data), Map.class);
            }
        } catch (Exception e) {
            log.error("创建个人租户以及授权失败", e);
        }
        return resultMap;
    }

    /**
     * @param
     * @return
     * @description: 给指定租户下的用户授权指定应用
     * @author: sunyfa
     */
    @Override
    public Boolean authorizeUser(AuthoredUser authoredUser, String tenantId, String email, String telephone,
                                 String appId, String strategySid, String strategyCode, Boolean isTenantAuth) {
        Boolean result = true;
        //邮箱和租户id不能为空
        if (StringUtil.isEmpty(email) && StringUtil.isEmpty(tenantId)) {
            return null;
        }

        //接口地址
        String url = iamUri + CREATE_TENANT_APP_AUTH_URL;

        //头部数据
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON);

        //参数
        Map<String, Object> param = new HashMap<>();
        param.put("tenantId", tenantId);
        param.put("email", email);
        param.put("telephone", telephone);
        param.put("appId", StringUtil.isNotEmpty(appId) ? appId : Constants.ATHENA_APPID);
        param.put("strategyCode", StringUtil.isNotEmpty(strategyCode) ? strategyCode : "athena_light_01");
        param.put("isTenantAuth", isTenantAuth);
        HttpEntity<?> httpEntity = new HttpEntity<>(param, headers);

        try {
            //调用接口
            log.info("authorizeUser post iam begin, URL:{}, param:{}", url, JsonUtils.objectToString(httpEntity));
            ResponseEntity<CommonResultDTO> responseEntity = restTemplate.exchange(url, HttpMethod.POST, httpEntity,
                    CommonResultDTO.class, new HashMap<>());
            log.info("authorizeUser post iam end, result:{}", JsonUtils.objectToString(responseEntity));

            if (Objects.nonNull(responseEntity)
                    && Objects.nonNull(responseEntity.getBody())
                    && Objects.nonNull(responseEntity.getBody().getData())
                    && responseEntity.getStatusCodeValue() == HttpStatus.SC_OK) {
                result = Boolean.valueOf(responseEntity.getBody().getData().toString());
            }
        } catch (Exception e) {
            log.error("给指定租户下的用户授权指定应用失败", e);
        }
        return result;
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public List<Map> batchQueryUserMetadataItem(List<String> userIdList, String tenantSid, String key, String catalogId) {
        // 接口地址
        String url = iamUri + BATCH_QUERY_METADATA_ITEM_URL;

        // 头部数据
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON);

        // 参数
        Map<String, Object> param = new HashMap<>();
        param.put("userIdList", userIdList);
        param.put("tenantSid", tenantSid);
        param.put("key", key);
        param.put("catalogId", catalogId);
        HttpEntity<?> httpEntity = new HttpEntity<>(param, headers);

        List<Map> resultList = new ArrayList<>();
        try {
            // 调用接口
            ResponseEntity<CommonResultDTO> responseEntity = restTemplate.exchange(url, HttpMethod.POST, httpEntity,
                    CommonResultDTO.class, new HashMap<>());
            if (Objects.nonNull(responseEntity)
                    && Objects.nonNull(responseEntity.getBody())
                    && Objects.nonNull(responseEntity.getBody().getData())
                    && responseEntity.getStatusCodeValue() == HttpStatus.SC_OK) {
                Object data = responseEntity.getBody().getData();
                resultList = JsonUtils.jsonToListObject(JsonUtils.objectToString(data), Map.class);
            }
        } catch (Exception e) {
            log.error("创建个人租户以及授权失败", e);
        }
        return null != resultList ? resultList : Lists.newArrayList();
    }

    @Override
    public Integer queryUserType(String userId, String token) {
        // 获取员工ID
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON);
        headers.add(GlobalConstant.IAM_USER_TOKEN, token);
        // post body
        Map<String, String> param = new HashMap<>();
        param.put("userId", userId);
        HttpEntity<?> httpEntity = new HttpEntity<>(param, headers);
        String url = iamUri + "/api/iam/v2/query/user";
        try {
            ResponseEntity<Map> responseEntity = restTemplate.postForEntity(url, httpEntity, Map.class);
            Map body = responseEntity.getBody();
            if (body == null) {
                return null;
            } else {
                return (Integer) body.get("userType");
            }
        } catch (Exception ex) {
            log.error("{}, body: {}, error: ", AudcErrorCodeEnum.IAM_V2_QUERY_USER.getErrCode(), JsonUtils.objectToString(param), ex);
        }
        return null;
    }
}
