package com.digiwin.athena.base.infrastructure.mongo;

import com.digiwin.athena.appcore.auth.domain.AuthoredUser;
import com.digiwin.athena.appcore.util.SnowflakeIdWorker;
import com.digiwin.athena.base.sdk.audc.infrastructure.meta.po.userdefined.mongo.UserDefinedPO;
import com.digiwin.athena.base.sdk.audc.infrastructure.meta.po.userdefined.mongo.UserGridPO;
import com.digiwin.athena.base.sdk.audc.infrastructure.mongo.QueryUserDefinedMapper;
import net.sf.json.JSONArray;
import net.sf.json.JSONObject;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang.StringUtils;
import org.springframework.data.mongodb.core.BulkOperations;
import org.springframework.data.mongodb.core.MongoTemplate;
import org.springframework.data.mongodb.core.query.Criteria;
import org.springframework.data.mongodb.core.query.Query;
import org.springframework.data.mongodb.core.query.Update;
import org.springframework.stereotype.Repository;

import javax.annotation.Resource;
import java.time.LocalDateTime;
import java.util.List;

/**
 * 用户自定义库
 *
 * @author XUXX
 * @date 2022/1/4
 * @since
 */
@Repository
public class UserDefinedMongoMapper {
    @Resource
    private QueryUserDefinedMapper queryUserDefinedMapper;

    //字段排序
    private String PURPOSE_FIELDSEQBYUSER = "FieldSeqByUser";

    //数据排序条件
    private String PURPOSE_ORDERCONDITIONS = "OrderConditions";
    private String PURPOSE_SEARCH_CONDITIONS = "SearchConditions";

    private String COLLECTION_NAME = "user_defined_center";

    private String SHRINK_NAME = "user_shrink_center";

    private String GRID_COLLECTION_NAME = "user_grid_items";

    public MongoTemplate getUserDefinedMongoTemplate() {
        return queryUserDefinedMapper.getUserdefinedMongoTemplate();
    }

    /**
     * 保存用户自定义属性
     *
     * @param userDefined
     */
    public void save(UserDefinedPO userDefined) {
        userDefined.setPurpose(PURPOSE_FIELDSEQBYUSER);
        UserDefinedPO getOneCheck = getFieldSeqList(userDefined);
        if (getOneCheck != null && getOneCheck.getMetadataTags() != null) {
            Query query = getQueryByUniqueId(getOneCheck);
            Update update = new Update();
            update.set("metadataTags", userDefined.getMetadataTags());
            update.set("updateDate", LocalDateTime.now());
            this.queryUserDefinedMapper.getUserdefinedMongoTemplate().upsert(query, update, COLLECTION_NAME);
        } else {
            userDefined.setCreateDate(LocalDateTime.now());
            userDefined.setUniqueId(SnowflakeIdWorker.getInstance().newId());
            userDefined.setPurpose(PURPOSE_FIELDSEQBYUSER);
            BulkOperations operations = this.queryUserDefinedMapper.getUserdefinedMongoTemplate().bulkOps(BulkOperations.BulkMode.UNORDERED, COLLECTION_NAME);
            operations.insert(userDefined);
            operations.execute();
        }
    }


    /**
     * 获取用户自定义栏位属性
     *
     * @param userDefined
     * @return
     */
    public UserDefinedPO getFieldSeqList(UserDefinedPO userDefined) {
        Query query = getQuery(userDefined);
        if (StringUtils.isNotEmpty(userDefined.getTableSchema())) {
            query = getQueryInTable(userDefined);
        }
        List<UserDefinedPO> list = this.queryUserDefinedMapper.getUserdefinedMongoTemplate().find(query, UserDefinedPO.class, COLLECTION_NAME);
        if (CollectionUtils.isNotEmpty(list) && list.size() > 0) {
            UserDefinedPO userDefinedNew = list.get(0);
            if (CollectionUtils.isNotEmpty(userDefinedNew.getMetadataTags())) {
                userDefinedNew.getMetadataTags().stream().forEach(tag -> tag.setTableSchema(userDefinedNew.getTableSchema()));
            }
            if (PURPOSE_FIELDSEQBYUSER.equals(userDefined.getPurpose())) {
                if (list.size() > 1) {
                    for (int i = 1; i < list.size(); i++) {
                        if (CollectionUtils.isNotEmpty(list.get(i).getMetadataTags())) {
                            int finalI = i;
                            list.get(i).getMetadataTags().stream().forEach(tag -> tag.setTableSchema(list.get(finalI).getTableSchema()));
                        }
                        userDefinedNew.getMetadataTags().addAll(list.get(i).getMetadataTags());
                    }
                }
            } else {
                if (StringUtils.isNotEmpty(userDefinedNew.getTableSchema())) {
                    JSONObject settingWithTable = new JSONObject();
                    settingWithTable.put(userDefinedNew.getTableSchema(), userDefinedNew.getSettings());
                    userDefinedNew.setSettings(settingWithTable);
                    if (list.size() > 1) {
                        for (int i = 1; i < list.size(); i++) {
                            UserDefinedPO oneUserDefined = list.get(i);
                            if (StringUtils.isNotEmpty(oneUserDefined.getTableSchema())) {
                                settingWithTable.put(oneUserDefined.getTableSchema(), oneUserDefined.getSettings());
                            }
                        }
                    }
                } else {//兼容没有tableSchema情况
                    if (list.size() > 1) {
                        JSONArray options = userDefinedNew.getSettings().getJSONObject("order").getJSONArray("options");
                        for (int i = 1; i < list.size(); i++) {
                            options.addAll(list.get(i).getSettings().getJSONObject("order").getJSONArray("options"));
                        }
                    }
                }
            }
            return userDefinedNew;
        }
        return new UserDefinedPO();
    }

    /**
     * 重置用户自定义
     *
     * @param userDefined
     */
    public void remove(UserDefinedPO userDefined) {
        Query query = getQuery(userDefined);
        if (StringUtils.isNotEmpty(userDefined.getTableSchema())) {
            query = getQueryInTable(userDefined);
        }
        this.queryUserDefinedMapper.getUserdefinedMongoTemplate().remove(query, COLLECTION_NAME);
    }

    /**
     * 组装查询条件
     * 获取当前用户、当前租户、当前任务中所有表格栏位属性
     *
     * @param userDefined
     * @return
     */
    private Query getQuery(UserDefinedPO userDefined) {
        Query query = new Query();
        query.addCriteria(Criteria.where("purpose").is(userDefined.getPurpose()));
        query.addCriteria(Criteria.where("userId").is(userDefined.getUserId()));
        query.addCriteria(Criteria.where("tenantId").is(userDefined.getTenantId()));
        query.addCriteria(Criteria.where("pageCode").is(userDefined.getPageCode()));
        query.addCriteria(Criteria.where("tmProjectId").is(userDefined.getTmProjectId()));
        query.addCriteria(Criteria.where("tmActivityId").is(userDefined.getTmActivityId()));
        if (StringUtils.isNotBlank(userDefined.getPageUniqueCode())) {
            query.addCriteria(Criteria.where("pageUniqueCode").is(userDefined.getPageUniqueCode()));
        }
        return query;
    }

    /**
     * 组装查询条件--new
     * 获取某一个指定表格的栏位属性
     *
     * @param userDefined
     * @return
     */
    private Query getQueryInTable(UserDefinedPO userDefined) {
        Query query = new Query();
        query.addCriteria(Criteria.where("purpose").is(userDefined.getPurpose()));
        query.addCriteria(Criteria.where("tableSchema").is(userDefined.getTableSchema()));
        query.addCriteria(Criteria.where("userId").is(userDefined.getUserId()));
        query.addCriteria(Criteria.where("tenantId").is(userDefined.getTenantId()));
        query.addCriteria(Criteria.where("pageCode").is(userDefined.getPageCode()));
        query.addCriteria(Criteria.where("tmProjectId").is(userDefined.getTmProjectId()));
        query.addCriteria(Criteria.where("tmActivityId").is(userDefined.getTmActivityId()));
        return query;
    }

    private Query getQueryByUniqueId(UserDefinedPO userDefined) {
        Query query = new Query();
        query.addCriteria(Criteria.where("uniqueId").is(userDefined.getUniqueId()));
        return query;
    }

    /**
     * 保存用户自定义数据排序条件
     *
     * @param userDefined
     */
    public void saveOrderConditions(UserDefinedPO userDefined) {
        userDefined.setPurpose(PURPOSE_ORDERCONDITIONS);
        UserDefinedPO getOneCheck = getFieldSeqList(userDefined);
        if (getOneCheck != null && getOneCheck.getSettings() != null) {
            Query query = getQueryByUniqueId(getOneCheck);
            Update update = new Update();
            update.set("settings", userDefined.getSettings());
            update.set("updateDate", LocalDateTime.now());
            this.queryUserDefinedMapper.getUserdefinedMongoTemplate().upsert(query, update, COLLECTION_NAME);
        } else {
            userDefined.setCreateDate(LocalDateTime.now());
            userDefined.setUniqueId(SnowflakeIdWorker.getInstance().newId());
            userDefined.setPurpose(PURPOSE_ORDERCONDITIONS);
            BulkOperations operations = this.queryUserDefinedMapper.getUserdefinedMongoTemplate().bulkOps(BulkOperations.BulkMode.UNORDERED, COLLECTION_NAME);
            operations.insert(userDefined);
            operations.execute();
        }
    }

    /**
     * 保存当前用户的租户下高级查询常用条件
     *
     * @param userDefined
     * @return
     */
    public Long saveSearchConditions(UserDefinedPO userDefined) {
        userDefined.setPurpose(PURPOSE_SEARCH_CONDITIONS);
        userDefined.setCreateDate(LocalDateTime.now());
        userDefined.setUniqueId(SnowflakeIdWorker.getInstance().newId());
        BulkOperations operations = this.queryUserDefinedMapper.getUserdefinedMongoTemplate().bulkOps(BulkOperations.BulkMode.UNORDERED, COLLECTION_NAME);
        operations.insert(userDefined);
        operations.execute();
        return userDefined.getUniqueId();
    }

    /**
     * 查询当前用户的租户下的高级查询常用条件
     *
     * @param userDefined
     * @return
     */
    public List<UserDefinedPO> getSearchConditions(UserDefinedPO userDefined) {
        return queryUserDefinedMapper.getSearchConditions(userDefined);
    }

    /**
     * 删除高级查询常用条件
     *
     * @param user
     * @param uniqueId
     */
    public void delSearchConditions(AuthoredUser user, Long uniqueId) {
        Query query = new Query();
        query.addCriteria(Criteria.where("purpose").is(PURPOSE_SEARCH_CONDITIONS));
        query.addCriteria(Criteria.where("userId").is(user.getUserId()));
        query.addCriteria(Criteria.where("tenantId").is(user.getTenantId()));
        query.addCriteria(Criteria.where("uniqueId").is(uniqueId));
        this.queryUserDefinedMapper.getUserdefinedMongoTemplate().remove(query, COLLECTION_NAME);
    }

    public UserDefinedPO getUserShrink(UserDefinedPO userDefined) {
        Query query = new Query();
        query.addCriteria(Criteria.where("userId").is(userDefined.getUserId()));
        query.addCriteria(Criteria.where("tenantId").is(userDefined.getTenantId()));
        query.addCriteria(Criteria.where("tmActivityId").is(userDefined.getTmActivityId()));
        List<UserDefinedPO> list = this.queryUserDefinedMapper.getUserdefinedMongoTemplate().find(query, UserDefinedPO.class, SHRINK_NAME);
        if (list.size() > 0) {
            return list.get(0);
        }
        return new UserDefinedPO();
    }

    public void saveUserShrink(UserDefinedPO userDefined) {
        UserDefinedPO getOneCheck = getUserShrink(userDefined);
        if (getOneCheck != null && getOneCheck.getSettings() != null) {
            Query query = new Query();
            query.addCriteria(Criteria.where("userId").is(userDefined.getUserId()));
            query.addCriteria(Criteria.where("tenantId").is(userDefined.getTenantId()));
            query.addCriteria(Criteria.where("tmActivityId").is(userDefined.getTmActivityId()));
            Update update = new Update();
            update.set("settings", userDefined.getSettings());
            update.set("updateDate", LocalDateTime.now());
            this.queryUserDefinedMapper.getUserdefinedMongoTemplate().upsert(query, update, SHRINK_NAME);
        } else {
            userDefined.setCreateDate(LocalDateTime.now());
            userDefined.setUniqueId(SnowflakeIdWorker.getInstance().newId());
            BulkOperations operations = this.queryUserDefinedMapper.getUserdefinedMongoTemplate().bulkOps(BulkOperations.BulkMode.UNORDERED, SHRINK_NAME);
            operations.insert(userDefined);
            operations.execute();
        }
    }

    public void saveGridItems(UserGridPO userGridPO) {
        UserGridPO getOneCheck = queryUserDefinedMapper.getUserGrid(userGridPO);
        if (getOneCheck != null && getOneCheck.getItems() != null) {
            Query query = new Query();
            query.addCriteria(Criteria.where("code").is(userGridPO.getCode()));
            query.addCriteria(Criteria.where("tenantId").is(userGridPO.getTenantId()));
            query.addCriteria(Criteria.where("userId").is(userGridPO.getUserId()));
            query.addCriteria(Criteria.where("pageCode").is(userGridPO.getPageCode()));
            Update update = new Update();
            update.set("items", userGridPO.getItems());
            update.set("updateDate", LocalDateTime.now());
            this.queryUserDefinedMapper.getUserdefinedMongoTemplate().upsert(query, update, GRID_COLLECTION_NAME);
        } else {
            userGridPO.setCreateDate(LocalDateTime.now());
            userGridPO.setUniqueId(SnowflakeIdWorker.getInstance().newId());
            BulkOperations operations = this.queryUserDefinedMapper.getUserdefinedMongoTemplate().bulkOps(BulkOperations.BulkMode.UNORDERED, GRID_COLLECTION_NAME);
            operations.insert(userGridPO);
            operations.execute();
        }
    }

    public void delGridByUniqueId(Long id, AuthoredUser user) {
        Query query = new Query();
        query.addCriteria(Criteria.where("userId").is(user.getUserId()));
        query.addCriteria(Criteria.where("tenantId").is(user.getTenantId()));
        query.addCriteria(Criteria.where("uniqueId").is(id));
        this.queryUserDefinedMapper.getUserdefinedMongoTemplate().remove(query, GRID_COLLECTION_NAME);
    }

}
