package com.digiwin.athena.base.infrastructure.mongo;

import com.digiwin.athena.base.infrastructure.constant.Constants;
import com.digiwin.athena.base.infrastructure.meta.po.usertrack.mongo.UserTrackExtendDTO;
import com.digiwin.athena.base.infrastructure.util.DataFilterUtils;
import org.springframework.data.mongodb.core.BulkOperations;
import org.springframework.data.mongodb.core.MongoTemplate;
import org.springframework.data.mongodb.core.query.Query;
import org.springframework.stereotype.Repository;

import javax.annotation.Resource;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.TimeUnit;

/**
 * UserTrackMongoMapper Description
 *
 * @author majianfu
 * @date 2021/8/27
 * @since
 */
@Repository
public class UserTrackMongoMapper extends AbstractMongoMapper {
    private static final String COLLECTION_NAME_SUFFIX = "_track";

    @Resource
    private MongoTemplate trackMongoTemplate;

    private static final Map<String, Object> trackCollectIndex = new HashMap<>();

    /**
     * {@inheritDoc}
     */
    public void safeSaveTrackData(String tenantId, List<UserTrackExtendDTO> tenantTrackList) {
        // 创建集合、索引，索引ttl为100天
        String collectionName = getCollectionName(tenantId);
        trackCollectIndex.put("uniqueId", -1);
        this.safeCreateCollectionWithIndex(this.trackMongoTemplate, collectionName, trackCollectIndex, Constants.USER_TRACK_DATA_TTL, TimeUnit.DAYS);

        // 并行批量插入
        this.safeSaveData(collectionName, tenantTrackList);
    }

    /**
     * {@inheritDoc}
     */
    public void saveData(String collection, List<UserTrackExtendDTO> dataList) {
        // 并行批量插入
        BulkOperations operations = this.trackMongoTemplate.bulkOps(BulkOperations.BulkMode.UNORDERED, collection);
        operations.insert(dataList);
        operations.execute();
    }

    /**
     * {@inheritDoc}
     */
    public void safeSaveData(String collection, List<UserTrackExtendDTO> dataList) {
        try {
            this.saveData(collection, dataList);
        } catch (IllegalArgumentException ex) {
            // 踢掉每行数据中不符合mongo命名规范的键值对
            for (UserTrackExtendDTO data : dataList) {
                DataFilterUtils.removeMongoIllegalNameField(data.getWorkContent());
                DataFilterUtils.removeMongoIllegalNameField(data.getAttachData());
            }
            this.saveData(collection, dataList);
        }
    }

    /**
     * {@inheritDoc}
     */
    public void dropCollection(String collection) {
        // 直接删除集合
        this.trackMongoTemplate.dropCollection(collection);
    }

    /**
     * 查询埋点数据
     *
     * @return
     */
    public List<UserTrackExtendDTO> findByParam(String collectionName, Query query) {
        return this.trackMongoTemplate.find(query, UserTrackExtendDTO.class, collectionName);
    }

    private String getCollectionName(String tenantId) {
        return tenantId + COLLECTION_NAME_SUFFIX;
    }
}
