/*
 * 文 件 名:  ExcelParser.java
 * 版    权:
 * 描    述:   <描述>
 * 版    本：       <版本号>
 * 创 建 人:  <创建人>
 * 创建时间: 2018年3月15日
 *
 */
package com.digiwin.athena.atdm.importstatistics.util.excel;

import com.digiwin.athena.appcore.exception.BusinessException;
import com.digiwin.athena.appcore.util.JsonUtils;
import com.digiwin.athena.atdm.constant.ErrorCodeEnum;
import com.digiwin.athena.atdm.importstatistics.entity.valueobject.ApiDataFieldLocaleMetadataDTO;
import com.digiwin.athena.atdm.importstatistics.entity.valueobject.GetActionLocaleResponseDTO;
import com.digiwin.athena.atdm.importstatistics.entity.valueobject.MetaDataType;
import com.digiwin.athena.base.sdk.common.application.util.MessageUtil;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.poi.hssf.usermodel.HSSFWorkbook;
import org.apache.poi.ss.usermodel.*;
import org.apache.poi.ss.util.CellRangeAddress;
import org.apache.poi.ss.util.CellRangeAddressList;
import org.apache.poi.xssf.usermodel.*;
import org.springframework.context.i18n.LocaleContextHolder;

import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.text.MessageFormat;
import java.text.SimpleDateFormat;
import java.util.*;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Collectors;

@Slf4j
@SuppressWarnings({"findsecbugs:PATH_TRAVERSAL_IN"})
public class ExcelUtil {

    private static String handleLocale(String locale) {
        if (StringUtils.isEmpty(locale)) {
            return "zh_CN";
        } else if (locale.equals("en")) {
            return "en_US";
        }
        return locale;
    }

    public static XSSFRichTextString addRequiredMark(XSSFWorkbook wb, String s) {
        XSSFFont font = wb.createFont();
        font.setColor(XSSFFont.COLOR_RED);
        font.setBold(true);
        XSSFRichTextString str = new XSSFRichTextString("*" + s);
        str.applyFont(0, 1, font);
        return str;
    }

    /**
     * @return
     * @Author zhuangli
     * @Description 约束单元格数据长度
     * @Date 10:53 2021/11/15
     * @Param [locale] 多语言类别 [sheet]excel工作簿 [min]长度最小值 [max]长度最大值 [firstRow]起始行 [lastRow]结束行65536
     * [firstCol]起始列 [lastCol]结束列
     **/
    public static void excelRuleStringLength(String locale, Sheet sheet, int min, int max, int firstRow, int lastRow, int firstCol, int lastCol) {
        /*Row row = sheet.getRow(0);
        Cell cell = row.getCell(firstCol);
        String r = ((XSSFCell) cell).getCTCell().getR();
        r = r.substring(0, 1);*/
        locale = handleLocale(locale);
        DataValidationHelper helper = sheet.getDataValidationHelper();
        CellRangeAddressList addressList = new CellRangeAddressList(firstRow, lastRow, firstCol, lastCol);
        //唯一
        DataValidationConstraint constraint = helper.createNumericConstraint(
                DataValidationConstraint.ValidationType.TEXT_LENGTH,
                DataValidationConstraint.OperatorType.BETWEEN, String.valueOf(min), String.valueOf(max));
        DataValidation dataValidation = helper.createValidation(constraint, addressList);
        String title = MessageUtil.getMessageByLocale("excel.string.title.length", locale, max);
        String content = MessageUtil.getMessageByLocale("excel.string.content.length", locale, max);
        dataValidation.createPromptBox(title, content);
        dataValidation.createErrorBox(title, content);
        dataValidation.setShowErrorBox(true);
        dataValidation.setEmptyCellAllowed(true);
        dataValidation.setSuppressDropDownArrow(true);
        dataValidation.setShowPromptBox(true);
        dataValidation.setErrorStyle(DataValidation.ErrorStyle.STOP);
        sheet.addValidationData(dataValidation);
    }

    public static void excelRuleDateFormat(String locale, Sheet sheet, String start, String end, String format, int firstRow, int lastRow, int firstCol, int lastCol) {
        /*Row row = sheet.getRow(0);
        Cell cell = row.getCell(firstCol);
        String r = ((XSSFCell) cell).getCTCell().getR();
        r = r.substring(0, 1);*/
        locale = handleLocale(locale);
        DataValidationHelper helper = sheet.getDataValidationHelper();
        CellRangeAddressList addressList = new CellRangeAddressList(firstRow, lastRow, firstCol, lastCol);
        //唯一
        DataValidationConstraint constraint = helper.createDateConstraint(DataValidationConstraint.OperatorType.GREATER_OR_EQUAL, start, end, format);
        DataValidation dataValidation = helper.createValidation(constraint, addressList);
        String content = MessageUtil.getMessageByLocale("excel.date.content.enterCorrect", locale, format);
        dataValidation.createPromptBox("", content);
        dataValidation.createErrorBox(MessageUtil.getMessageByLocale("excel.date.title.enterCorrect", locale), content);
        dataValidation.setShowErrorBox(true);
        dataValidation.setEmptyCellAllowed(true);
        dataValidation.setSuppressDropDownArrow(true);
        dataValidation.setShowPromptBox(true);
        dataValidation.setErrorStyle(DataValidation.ErrorStyle.STOP);
        sheet.addValidationData(dataValidation);
    }

    public static void excelRuleUniqueue(String locale, Sheet sheet, int firstRow, int lastRow, int firstCol, int lastCol) {
        locale = handleLocale(locale);
        Row row = sheet.getRow(0);
        Cell cell = row.getCell(firstCol);
        String r = ((XSSFCell) cell).getCTCell().getR();
        r = r.substring(0, 1);
        DataValidationHelper helper = sheet.getDataValidationHelper();
        CellRangeAddressList addressList = new CellRangeAddressList(firstRow, lastRow, firstCol, lastCol);
        //唯一
        DataValidationConstraint constraint = helper.createCustomConstraint(MessageFormat.format("COUNTIF({0}:{0},{0}2)=1", r));
        DataValidation dataValidation = helper.createValidation(constraint, addressList);
        dataValidation.createErrorBox("错误：", "赋值属性列不允许重复");
        dataValidation.setShowErrorBox(true);
        dataValidation.setEmptyCellAllowed(true);
        dataValidation.setSuppressDropDownArrow(true);
        dataValidation.setShowPromptBox(true);
        dataValidation.setErrorStyle(DataValidation.ErrorStyle.STOP);

        sheet.addValidationData(dataValidation);
    }

    public static void excelRuleSelect(String locale, Sheet sheet, String[] rule, int firstRow, int lastRow, int firstCol, int lastCol) {
        locale = handleLocale(locale);
        DataValidationHelper helper = sheet.getDataValidationHelper();
        CellRangeAddressList addressList = new CellRangeAddressList(firstRow, lastRow, firstCol, lastCol);
        DataValidationConstraint constraint = helper.createExplicitListConstraint(rule);
        DataValidation dataValidation = helper.createValidation(constraint, addressList);
        String itemStr = MessageUtil.getMessageByLocale("excel.select.item.selectCorrect", locale, Arrays.asList(rule).toString());
        String title = MessageUtil.getMessageByLocale("excel.select.title.selectCorrect", locale);
        String content = MessageUtil.getMessageByLocale("excel.select.content.selectCorrect", locale, Arrays.asList(rule).toString());
        dataValidation.createPromptBox(itemStr, itemStr);
        dataValidation.createErrorBox(title, content);
        if (dataValidation instanceof XSSFDataValidation) {
            dataValidation.setSuppressDropDownArrow(true);
            dataValidation.setShowErrorBox(true);
        } else {
            dataValidation.setSuppressDropDownArrow(false);
        }
        dataValidation.setEmptyCellAllowed(true);
        dataValidation.setShowPromptBox(true);
        dataValidation.setErrorStyle(DataValidation.ErrorStyle.STOP);
        sheet.addValidationData(dataValidation);
    }

    public static void excelRuleNumberBetween(String locale, Sheet sheet, int min, int max, int firstRow, int lastRow, int firstCol, int lastCol) {
        locale = handleLocale(locale);
        DataValidationHelper helper = sheet.getDataValidationHelper();
        CellRangeAddressList addressList = new CellRangeAddressList(firstRow, lastRow, firstCol, lastCol);
        //设置数据
        DataValidationConstraint constraint = helper.createIntegerConstraint(DataValidationConstraint.OperatorType.BETWEEN,
                String.valueOf(min), String.valueOf(max));
        DataValidation dataValidation = helper.createValidation(constraint, addressList);
        dataValidation.createPromptBox("", MessageUtil.getMessage("delivery.inputValidData"));
        dataValidation.createErrorBox(MessageUtil.getMessage("delivery.dataValueError"), MessageUtil.getMessage("delivery.inputValueTips", min, max));
        //处理Excel兼容性问题
        if (dataValidation instanceof XSSFDataValidation) {
            dataValidation.setSuppressDropDownArrow(true);
            dataValidation.setShowErrorBox(true);
        } else {
            dataValidation.setSuppressDropDownArrow(false);
        }
        dataValidation.setShowErrorBox(true);
        dataValidation.setShowPromptBox(true);
        dataValidation.setEmptyCellAllowed(true);
        dataValidation.setSuppressDropDownArrow(true);
        dataValidation.setErrorStyle(DataValidation.ErrorStyle.STOP);
        sheet.addValidationData(dataValidation);
    }

    public static void excelRuleIsNumber(String locale, Sheet sheet, int firstRow, int lastRow, int firstCol, int lastCol) {
        locale = handleLocale(locale);
        String isNumber = "=ISNUMBER(INDIRECT(ADDRESS(ROW(),COLUMN())))";
        DataValidationHelper helper = sheet.getDataValidationHelper();
        CellRangeAddressList addressList = new CellRangeAddressList(firstRow, lastRow, firstCol, lastCol);
        //设置数据
        DataValidationConstraint constraint = helper.createCustomConstraint(isNumber);
        DataValidation dataValidation = helper.createValidation(constraint, addressList);
        dataValidation.createPromptBox("", MessageUtil.getMessageByLocale("excel.number.content.isNumber", locale));
        dataValidation.createErrorBox(MessageUtil.getMessageByLocale("excel.number.title.isNumber", locale), MessageUtil.getMessageByLocale("excel.number.content.isNumber", locale));
        //处理Excel兼容性问题
        if (dataValidation instanceof XSSFDataValidation) {
            dataValidation.setSuppressDropDownArrow(true);
            dataValidation.setShowErrorBox(true);
        } else {
            dataValidation.setSuppressDropDownArrow(false);
        }
        dataValidation.setShowErrorBox(true);
        dataValidation.setShowPromptBox(true);
        dataValidation.setEmptyCellAllowed(true);
        dataValidation.setSuppressDropDownArrow(true);
        dataValidation.setErrorStyle(DataValidation.ErrorStyle.STOP);
        sheet.addValidationData(dataValidation);
    }

    public static ExcelParserBean parser(String filePath) throws IOException {
        final ExcelParserBean parserBean = new ExcelParserBean();
        parserBean.setFilePath(filePath);
        final File file = new File(filePath);
        if (!file.exists()) {
            throw new IllegalArgumentException(MessageUtil.getMessage("delivery.excelPathError"));
        }
        InputStream inputStream = new FileInputStream(new File(filePath));
        parserBean.setInput(inputStream);
        ExcelParserBean parser = ExcelUtil.parser(parserBean);
        if (inputStream != null) {
            inputStream.close();
        }
        return parser;
    }

    public static ExcelParserBean parser(InputStream input, String fileName) throws IOException {
        final ExcelParserBean parserBean = new ExcelParserBean();
        parserBean.setInput(input);
        parserBean.setFilePath(fileName);
        return ExcelUtil.parser(parserBean);
    }

    public static ExcelParserBean parser(ExcelParserBean parserBean) {
        return parser(parserBean, null, 0);
    }

    public static ExcelParserBean parser(ExcelParserBean parserBean, List<CellType> cellTypes, int skipHeaders) {
        ExcelUtil.setDefaultExcelType(parserBean);
        final Workbook workbook = ExcelUtil.createWorkbook(parserBean);
        Map<String, List<List<Object>>> parseSheet;
        if (null == cellTypes) {
            parseSheet = ExcelUtil.parseSheet(workbook);
        } else {
            parseSheet = ExcelUtil.parseSheet(workbook, cellTypes, skipHeaders);
        }
        parserBean.setData(parseSheet);
        /*if ((parseSheet != null) && (parseSheet.size() > 0)) {
            parserBean.setHeaders(parseSheet.get(0));
        }*/
        ExcelUtil.realeaseResources(parserBean, workbook);
        return parserBean;
    }

    /**
     * sheet页数据
     *
     * @param parserBean
     * @param cellTypes
     * @param skipHeaders
     * @return
     */
    public static ExcelParserBean parserNew(ExcelParserBean parserBean, List<CellType> cellTypes, int skipHeaders) {
        ExcelUtil.setDefaultExcelType(parserBean);
        final Workbook workbook = ExcelUtil.createWorkbook(parserBean);
        Map<String, List<List<Object>>> parseSheet;
        if (null == cellTypes) {
            parseSheet = ExcelUtil.parseSheet(workbook);
        } else {
            parseSheet = ExcelUtil.parseSheet(workbook, cellTypes, skipHeaders);
        }
        parserBean.setData(parseSheet);
        /*if ((parseSheet != null) && (parseSheet.size() > 0)) {
            parserBean.setHeaders(parseSheet.get(0));
        }*/
        ExcelUtil.realeaseResources(parserBean, workbook);
        return parserBean;
    }

    public static ExcelParserBean parser(ExcelParserBean parserBean, GetActionLocaleResponseDTO getActionLocaleResponseDTO, int skipHeaders, int flag) {
        ExcelUtil.setDefaultExcelType(parserBean);
        final Workbook workbook = ExcelUtil.createWorkbook(parserBean);
        Map<String, List<List<Object>>> parseSheet;
        try {
            if (null == getActionLocaleResponseDTO) {
                parseSheet = ExcelUtil.parseSheet(workbook);
            } else {
                parseSheet = ExcelUtil.parseSheet(workbook, skipHeaders, getActionLocaleResponseDTO);
            }
        } catch (Exception e) {
            log.error("解析Excel报错！e:", e);
            throw BusinessException.create(ErrorCodeEnum.NUM_500_0116.getErrCode(), MessageUtil.getMessageByLocale("upload.template.error", LocaleContextHolder.getLocale().toString()));
        }
        parserBean.setData(parseSheet);
        ExcelUtil.realeaseResources(parserBean, workbook);
        return parserBean;
    }

    public static ExcelParserBean parserHeader(ExcelParserBean parserBean, int headerRow) {
        ExcelUtil.setDefaultExcelType(parserBean);
        final Workbook workbook = ExcelUtil.createWorkbook(parserBean);
        final Map<String, List<List<Object>>> data = new HashMap<>();
        final Iterator<Sheet> sheetIterator = workbook.iterator();
        int sheetCount = 0;
        // 遍历excel多个sheet
        while (sheetIterator.hasNext()) {
            final List<List<Object>> sheetData = new ArrayList<>();
            final Sheet sheet = sheetIterator.next();
            final Iterator<Row> rowIterator = sheet.iterator();
            int rowNumber = 0;
            // 遍历行
            while (rowIterator.hasNext()) {
                // 多个表单头行 一样
                if ((sheetCount > 1) && (rowNumber == 0)) {
                    continue;
                }
                final Row row = rowIterator.next();
                List<Object> rowData;
                if (headerRow > rowNumber + 1) {
                    rowNumber++;
                } else if (headerRow == rowNumber + 1) {
                    rowData = ExcelUtil.parseRow(row);
                    sheetData.add(rowData);
                    rowNumber++;
                } else {
                    break;
                }
            }
            data.put(sheet.getSheetName(), sheetData);
            sheetCount++;
        }
        parserBean.setData(data);
        /*if ((data != null) && (data.size() > 0)) {
            parserBean.setHeaders(data.get(0));
        }*/
        //后续注意释放资源
        //ExcelUtil.realeaseResources(parserBean, workbook);
        return parserBean;
    }

    public static void setDefaultExcelType(ExcelParserBean parserBean) {
        // 获取文件后缀
        final String filePath = parserBean.getFilePath();
        final String fileExtension = ExcelUtil.getFileExtension(filePath);

        ExcelTypeEnum excelType = parserBean.getExcelType();
        // 默认为xls格式excel
        if ((excelType == null) && (fileExtension == null)) {
            excelType = ExcelTypeEnum.XLS;
        } else if ((excelType == null) && (fileExtension != null)) {
            if (fileExtension.equalsIgnoreCase(ExcelTypeEnum.XLS.name())) {
                excelType = ExcelTypeEnum.XLS;
            } else if (fileExtension.equalsIgnoreCase(ExcelTypeEnum.XLSX.name())) {
                excelType = ExcelTypeEnum.XLSX;
            }
        }
        parserBean.setExcelType(excelType);
    }

    public static Workbook createWorkbook(ExcelParserBean parserBean) {
        Workbook wb = null;
        final ExcelTypeEnum excelType = parserBean.getExcelType();
        try {
            if (ExcelTypeEnum.XLS.equals(excelType)) {
                wb = new HSSFWorkbook(parserBean.getInput());
            } else if (ExcelTypeEnum.XLSX.equals(excelType)) {
                wb = new XSSFWorkbook(parserBean.getInput());
            } else {
                wb = new HSSFWorkbook(parserBean.getInput());
            }
        } catch (final IOException e) {
            log.error(e.toString());
        }
        return wb;
    }

    protected static Map<String, List<List<Object>>> parseSheet(Workbook workbook) {
        return parseSheet(workbook, null, 0);
    }

    /**
     * 解析表格
     *
     * @param workbook
     * @return
     */
    protected static Map<String, List<List<Object>>> parseSheet(Workbook workbook, List<CellType> cellTypes, int skipHeaders) {
        final Map<String, List<List<Object>>> data = new HashMap<>();
        final Iterator<Sheet> sheetIterator = workbook.iterator();
        int sheetCount = 0;
        // 遍历excel多个sheet
        while (sheetIterator.hasNext()) {
            final List<List<Object>> sheetData = new ArrayList<>();
            final Sheet sheet = sheetIterator.next();
            //将存在合并单元格的列记录入put进hashmap并返回
            Map<String, Integer[]> map = getMergedRegionMap(sheet);
            //去除空行返回新的sheet
            Sheet sheetExcepNullRow = getAccuracyContextNum(sheet, map);
            final Iterator<Row> rowIterator = sheetExcepNullRow.iterator();
            //final Iterator<Row> rowIterator = sheet.iterator();
            int rowNumber = 0;
            // 遍历行
            while (rowIterator.hasNext()) {
                // 多个表单头行 一样
                if ((sheetCount > 1) && (rowNumber == 0)) {
                    break;//仅处理第一个sheet
                }
                final Row row = rowIterator.next();
                List<Object> rowData;
                if (skipHeaders <= rowNumber + 1) {
                    rowData = ExcelUtil.parseRow(row);
                } else if (null == cellTypes) {
                    rowData = ExcelUtil.parseRow(row);
                } else {
                    rowData = ExcelUtil.parseRow(row, cellTypes);
                }
                sheetData.add(rowData);
                rowNumber++;
            }
            data.put(sheetExcepNullRow.getSheetName(), sheetData);
            sheetCount++;
        }
        return data;
    }

    //在解析sheet的时候，不需要在区分sheet1，返回map的key都是 metadata的data_name.
    protected static Map<String, List<List<Object>>> parseSheet(Workbook workbook, int skipHeaders, GetActionLocaleResponseDTO getActionLocaleResponseDTO) {
        // 使用jackson深拷贝，不影响后续对元数据GetActionLocaleResponseDTO的使用
        List<ApiDataFieldLocaleMetadataDTO> apiDataFieldLocaleMetadataDTOS = JsonUtils.jsonToListObject(JsonUtils.objectToString(getActionLocaleResponseDTO.getRequest()
                .getParameters().get(0).getField()), ApiDataFieldLocaleMetadataDTO.class);
        final Map<String, List<List<Object>>> data = new LinkedHashMap();
        final Iterator<Sheet> sheetIterator = workbook.iterator();
        int sheetCount = 0;
        // 遍历excel多个sheet
//        int totalNum = workbook.getNumberOfSheets();
        while (sheetIterator.hasNext()) {
            final Sheet sheet = sheetIterator.next();
            String sheetName = getSheetNameInParens(sheet.getSheetName());
            if (data.containsKey(sheetName)) {
                continue;
            }
            doParseSheet(workbook, sheetCount, skipHeaders, sheet, data, apiDataFieldLocaleMetadataDTOS);
            sheetCount++;
        }
        return data;
    }

    public static void doParseSheet(Workbook workbook, int sheetCount, int skipHeaders, Sheet sheet, Map<String, List<List<Object>>> data, List<ApiDataFieldLocaleMetadataDTO> apiDataFieldLocaleMetadataDTOS) {
        final List<List<Object>> sheetData = new ArrayList<>();
        //将存在合并单元格的列记录入put进hashmap并返回
        Map<String, Integer[]> map = getMergedRegionMap(sheet);
        //去除空行后的sheet页
        Sheet sheetExcepNullRow = getAccuracyContextNum(sheet, map);
        String sheetName = getSheetNameInParens(sheetExcepNullRow.getSheetName());
        final Iterator<Row> rowIterator = sheetExcepNullRow.iterator();
        int rowNumber = 0;
        // 遍历行
        List<Object> headerList = new ArrayList<>();

        //给单身sheet页添加bk关联
//            if (!MAIN_SHEET_NAME.equals(sheet.getSheetName())) {
        while (rowIterator.hasNext()) {
            //TODO 处理单身sheet
                /*if ((sheetCount > 1) && (rowNumber == 0)) {
                    continue;
                }*/
            final Row row = rowIterator.next();
            List<Object> rowData;
            if (skipHeaders > rowNumber + 1) {
                rowData = ExcelUtil.parseRow(row);
            } else if (skipHeaders == rowNumber + 1) {
                rowData = ExcelUtil.parseRow(row);
                headerList = rowData;
            } else {
                //解析式带上sheetName,用于判断单头单身
                //获取请求参数列表
                rowData = ExcelUtil.parseRow(sheetCount > 0 ? sheetName : MAIN_SHEET_NAME, row, apiDataFieldLocaleMetadataDTOS, headerList);
            }
            sheetData.add(rowData);
            rowNumber++;
        }
        //针对旧的excel可能还存在sheet1这种名称做特殊处理，sheet name变为metadata的main key
        if (MAIN_SHEET_NAME.equals(sheetExcepNullRow.getSheetName())) {
//            String mainKey = getActionLocaleResponseDTO.getRequest().getParameters().get(0).getData_name();
//            data.put(mainKey, sheetData);
        } else {
            data.put(sheetName, sheetData);
        }

        int totalNum = workbook.getNumberOfSheets();
        if (totalNum > sheetCount + 1) {
            Sheet nextSheet = workbook.getSheetAt(sheetCount + 1);
            Sheet nextSheetExcepNullRow = getAccuracyContextNum(nextSheet, map);
            String nextSheetName = getSheetNameInParens(nextSheetExcepNullRow.getSheetName());
            for (ApiDataFieldLocaleMetadataDTO metadataDTO : apiDataFieldLocaleMetadataDTOS) {
                // 栏位和sheetName匹配
                if (metadataDTO.getData_name().equals(nextSheetName) && metadataDTO.getIs_array()) {
                    List<ApiDataFieldLocaleMetadataDTO> businessKeyFields = getBusinessKeyFields(apiDataFieldLocaleMetadataDTOS);
                    List<ApiDataFieldLocaleMetadataDTO> bodyMetadataDtoList = metadataDTO.getField();
                    bodyMetadataDtoList.addAll(businessKeyFields);
                    sheetCount++;
                    doParseSheet(workbook, sheetCount, skipHeaders, nextSheet, data, bodyMetadataDtoList);
                }
            }
//            // 所有的单身
//            List<ApiDataFieldLocaleMetadataDTO> arrayMetadaDtoList = apiDataFieldLocaleMetadataDTOS.stream().filter(item -> item.getIs_array()).collect(Collectors.toList());
//            for(ApiDataFieldLocaleMetadataDTO arrayMetadaDto : arrayMetadaDtoList){
//                // 如果单身和sheetName相等，则继续解析
//                if(arrayMetadaDto.getData_name().equals(nextSheetName)){
//                    List<ApiDataFieldLocaleMetadataDTO> businessKeyFields = getBusinessKeyFields(apiDataFieldLocaleMetadataDTOS);
//                    apiDataFieldLocaleMetadataDTOS = arrayMetadaDto.getField();
//                    apiDataFieldLocaleMetadataDTOS.addAll(businessKeyFields);
//                    sheetCount++;
//                    doParseSheet(workbook,sheetCount,skipHeaders,nextSheet,data,apiDataFieldLocaleMetadataDTOS);
//                } else if(totalNum > sheetCount +1){
//                    Sheet nextSheet1 = workbook.getSheetAt(sheetCount +1);
//                    doParseSheet(workbook,sheetCount,skipHeaders,nextSheet1,data,apiDataFieldLocaleMetadataDTOS);
//                }
//            }
        }
    }

    /**
     * 导入时读取sheet页中的数据
     *
     * @param sheet
     * @param apiDataFieldLocaleMetadataDTOS
     * @return
     */
    public static List<Map> readDataForImport(Sheet sheet, List<ApiDataFieldLocaleMetadataDTO> apiDataFieldLocaleMetadataDTOS) {
        //将存在合并单元格的列记录入put进hashmap并返回
        Map<String, Integer[]> map = ExcelUtil.getMergedRegionMap(sheet);
        //去除空行后的sheet页
        Sheet sheetExcepNullRow = ExcelUtil.getAccuracyContextNum(sheet, map);

        if (CollectionUtils.isEmpty(apiDataFieldLocaleMetadataDTOS)) {
            throw BusinessException.create(ErrorCodeEnum.NUM_500_0086.getErrCode(), MessageUtil.getMessage("basedata.apiNoDF"));
        }
        //将参数列表转为map
        Map<String, String> metaMap = apiDataFieldLocaleMetadataDTOS.stream()
                .collect(Collectors.toMap(ApiDataFieldLocaleMetadataDTO::getData_name,
                        ApiDataFieldLocaleMetadataDTO::getData_type, (item1, item2) -> item1));
        List<CellTypeHolder> cellTypeHolders = new LinkedList<>();

        //第一二行为headers
        List<String> headerKeys = new LinkedList<>();
        Row headRow2 = sheetExcepNullRow.getRow(1);
        int cellStartIdx = 0;
        while (null != headRow2.getCell(cellStartIdx)) {
            String key = headRow2.getCell(cellStartIdx).getStringCellValue();
            headerKeys.add(key);
            cellStartIdx++;
        }

        //遍历表头字段名
        for (int i = 0; i < headerKeys.size(); i++) {
            //仅将元数据描述包含的入参字段加入
            if (null != metaMap.get(headerKeys.get(i))) {
                CellTypeHolder cellTypeHolder = new CellTypeHolder();
                cellTypeHolder.setHeaderKey(headerKeys.get(i));
                cellTypeHolder.setIndex(i);
                cellTypeHolder.setMetaDataType(MetaDataType.valueOf(metaMap.get(headerKeys.get(i)).toUpperCase()));
                switch (MetaDataType.valueOf(metaMap.get(headerKeys.get(i)).toUpperCase())) {
                    case NUMERIC:
                    case NUMBER:
                        cellTypeHolder.setCellType(CellType.NUMERIC);
                        cellTypeHolders.add(cellTypeHolder);
                        break;
                    case BOOLEAN:
                        cellTypeHolder.setCellType(CellType.BOOLEAN);
                        cellTypeHolders.add(cellTypeHolder);
                        break;
                    case STRING:
                    default:
                        cellTypeHolder.setCellType(CellType.STRING);
                        cellTypeHolders.add(cellTypeHolder);
                        break;
                }
            }
        }

        // 修复bug:76100
        if (CollectionUtils.isEmpty(cellTypeHolders)) {
            log.error("解析Excel的栏位为空！");
            throw BusinessException.create(ErrorCodeEnum.NUM_500_0116.getErrCode(), MessageUtil.getMessageByLocale("upload.template.error", LocaleContextHolder.getLocale().toString()));
        }

        List<Map> data = new LinkedList<>();
        int rowStartIdx = 2;
        while (null != sheetExcepNullRow.getRow(rowStartIdx)) {
            Map<String, Object> rowData = new HashMap<>();
            Row row = sheetExcepNullRow.getRow(rowStartIdx);
            for (int i = 0; i < cellTypeHolders.size(); i++) {
                CellTypeHolder cellTypeHolder = cellTypeHolders.get(i);
                //因为数据和根据index获取对应列
                final Cell cell = row.getCell(cellTypeHolder.getIndex());
                Object cellObj = null;
                if (cell != null) {
                    cellObj = ExcelUtil.parseCell(cell, cellTypeHolder.getCellType(), cellTypeHolder.getMetaDataType());
                }
                rowData.put(cellTypeHolder.getHeaderKey(), cellObj);
            }
            data.add(rowData);
            rowStartIdx++;
        }
        return data;
    }

    /**
     * 将存在合并单元格的列记录入put进hashmap并返回
     *
     * @param sheet
     * @return
     */
    public static Map<String, Integer[]> getMergedRegionMap(Sheet sheet) {

        Map<String, Integer[]> result = new HashMap<String, Integer[]>();

        //获取excel中的所有合并单元格信息
        int sheetMergeCount = sheet.getNumMergedRegions();

        //遍历处理
        for (int i = 0; i < sheetMergeCount; i++) {

            //拿到每个合并单元格，开始行，结束行，开始列，结束列
            CellRangeAddress range = sheet.getMergedRegion(i);
            int firstColumn = range.getFirstColumn();
            int lastColumn = range.getLastColumn();
            int firstRow = range.getFirstRow();
            int lastRow = range.getLastRow();

            //构造一个开始行和开始列组成的数组
            Integer[] firstRowNumberAndCellNumber = new Integer[]{firstRow, firstColumn};

            //遍历，将单元格中的所有行和所有列处理成由行号和下划线和列号组成的key，然后放在hashmap中
            for (int currentRowNumber = firstRow; currentRowNumber <= lastRow; currentRowNumber++) {

                for (int currentCellNumber = firstColumn; currentCellNumber <= lastColumn; currentCellNumber++) {
                    result.put(currentRowNumber + "_" + currentCellNumber, firstRowNumberAndCellNumber);
                }
            }
        }
        return result;
    }

    /**
     * 去除空行后的sheet页
     *
     * @param sheetOld
     * @param mergedRegionMap
     * @return
     */
    public static Sheet getAccuracyContextNum(Sheet sheetOld, Map<String, Integer[]> mergedRegionMap) {
        // 取第一个sheet
        Sheet sheet = sheetOld;
        // 删除空行
        for (int i = 0; i <= sheet.getLastRowNum(); i++) {
            Row row = sheet.getRow(i);
            // 处理Excel中的空行，并上移空行之后的所有行以覆盖空行
            if (row == null) {
                // 删除空行
                int lastRowNum = sheet.getLastRowNum();
                if (i >= 0 && i < lastRowNum) {
                    sheet.shiftRows(i + 1, lastRowNum, -1);// 将行号为i+1一直到行号为lastRowNum的单元格全部上移一行，以便删除i行
                }
                i--;
                continue;
            }
            int cellNum = row.getLastCellNum();
            boolean flag = false;
            for (int j = 0; j < cellNum; j++) {
                Integer[] firstRowNumberAndCellNumber = mergedRegionMap.get(i + "_" + j);
                if (firstRowNumberAndCellNumber != null) {
                    flag = true;
                }
            }
            if (isRowEmpty(row) && !flag) {
                // 删除空行
                if (isRowEmpty(row)) {
                    int lastRowNum = sheet.getLastRowNum();
                    if (i >= 0 && i < lastRowNum) {
                        sheet.shiftRows(i + 1, lastRowNum, -1);// 将行号为i+1一直到行号为lastRowNum的单元格全部上移一行，以便删除i行
                    }
                    if (i == lastRowNum) {
                        if (row != null) {
                            sheet.removeRow(row);
                        }
                    }
                    i--;
                }
            }
        }
        return sheet;
    }

    /**
     * 判断是否有空行
     *
     * @param row
     * @return
     */
    protected static boolean isRowEmpty(Row row) {
        //行不存在
        if (row == null) {
            return true;
        }
        //第一个列位置
        int firstCellNum = row.getFirstCellNum();
        //最后一列位置
        int lastCellNum = row.getLastCellNum();
        //空列数量
        int nullCellNum = 0;
        for (int c = firstCellNum; c < lastCellNum; c++) {
            Cell cell = row.getCell(c);
            if (null == cell || cell.getCellType() == CellType.BLANK) {
                nullCellNum++;
                continue;
            }
            if (CellType.STRING.equals(cell.getCellTypeEnum())) {
                String cellValue = cell.getStringCellValue().trim();
                if (StringUtils.isEmpty(cellValue)) {
                    nullCellNum++;
                }
            }
        }
        //所有列都为空
        if (nullCellNum == (lastCellNum - firstCellNum)) {
            return true;
        }
        return false;
    }

    protected static List<Object> parseRow(Row row) {
        return parseRow(row, null);
    }

    protected static List<Object> parseRow(Row row, List<CellType> cellTypes) {
        final List<Object> rowData = new ArrayList<>();
        if (null != cellTypes && row.getLastCellNum() != cellTypes.size()) {
            throw BusinessException.create(ErrorCodeEnum.NUM_500_0083.getErrCode(), MessageUtil.getMessage("delivery.dataTypeNotMatchUnit"));
        }
        for (int i = 0; i < row.getLastCellNum(); i++) {
            final Cell cell = row.getCell(i);
            Object cellObj = null;
            if (cell != null) {
                if (null == cellTypes) {
                    cellObj = ExcelUtil.parseCell(cell);
                } else {
                    cellObj = ExcelUtil.parseCell(cell, cellTypes.get(i));
                }
            }
            rowData.add(cellObj);
        }
        /*
         * // 迭代 一行的各个单元格 Iterator<Cell> cellIterator = row.iterator(); // 遍历一行多列 while (cellIterator.hasNext()) { Cell
         * cell = cellIterator.next(); Object cellObj = parseCell(cell); rowData.add(cellObj); }
         */
        return rowData;
    }

    public static final String MAIN_SHEET_NAME = "sheet1";

    /**
     * 解析行
     *
     * @param row
     * @return
     */
    protected static List<Object> parseRow(String sheetName, Row row, List<ApiDataFieldLocaleMetadataDTO> apiDataFieldLocaleMetadataDTOS, List headerList) {
        if (CollectionUtils.isEmpty(headerList)) {
            throw BusinessException.create(ErrorCodeEnum.NUM_500_0084.getErrCode(), MessageUtil.getMessage("delivery.headerListEmpty"));
        }
        //单身处理
        if (!MAIN_SHEET_NAME.equals(sheetName)) {
            List<String> businessKeys = getBusinessKeys(apiDataFieldLocaleMetadataDTOS);
            if (CollectionUtils.isEmpty(businessKeys)) {
                throw BusinessException.create(ErrorCodeEnum.NUM_500_0085.getErrCode(), MessageUtil.getMessage("basedata.apiNoBusinessKey"));
            }
        }
        if (CollectionUtils.isEmpty(apiDataFieldLocaleMetadataDTOS)) {
            throw BusinessException.create(ErrorCodeEnum.NUM_500_0086.getErrCode(), MessageUtil.getMessage("basedata.apiNoDF"));
        }
        //将参数列表转为map
        Map<String, String> metaMap = apiDataFieldLocaleMetadataDTOS.stream()
                .collect(Collectors.toMap(ApiDataFieldLocaleMetadataDTO::getData_name,
                        ApiDataFieldLocaleMetadataDTO::getData_type, (item1, item2) -> item1));
        List<CellTypeHolder> cellTypeHolders = new LinkedList<>();
        //遍历表头字段名
        for (int i = 0; i < headerList.size(); i++) {
            //仅将元数据描述包含的入参字段加入
            if (null != metaMap.get(headerList.get(i))) {
                CellTypeHolder cellTypeHolder = new CellTypeHolder();
                cellTypeHolder.setIndex(i);
                cellTypeHolder.setMetaDataType(MetaDataType.valueOf(metaMap.get(headerList.get(i)).toUpperCase()));
                switch (MetaDataType.valueOf(metaMap.get(headerList.get(i)).toUpperCase())) {
                    case NUMERIC:
                    case NUMBER:
                        cellTypeHolder.setCellType(CellType.NUMERIC);
                        cellTypeHolders.add(cellTypeHolder);
                        break;
                    case BOOLEAN:
                        cellTypeHolder.setCellType(CellType.BOOLEAN);
                        cellTypeHolders.add(cellTypeHolder);
                        break;
                    case STRING:
                    default:
                        cellTypeHolder.setCellType(CellType.STRING);
                        cellTypeHolders.add(cellTypeHolder);
                        break;
                }
            }
        }

        // 修复bug:76100
        if (CollectionUtils.isEmpty(cellTypeHolders)) {
            log.error("解析Excel的栏位为空！");
            throw BusinessException.create(ErrorCodeEnum.NUM_500_0116.getErrCode(), MessageUtil.getMessageByLocale("upload.template.error", LocaleContextHolder.getLocale().toString()));
        }

        List<Object> rowData = new ArrayList<>();
        for (int i = 0; i < cellTypeHolders.size(); i++) {
            //因为数据和根据index获取对应列
            final Cell cell = row.getCell(cellTypeHolders.get(i).getIndex());
            Object cellObj = null;
            if (cell != null) {
                cellObj = ExcelUtil.parseCell(cell, cellTypeHolders.get(i).getCellType(), cellTypeHolders.get(i).getMetaDataType());
            }
            rowData.add(cellObj);
        }
        /*
         * // 迭代 一行的各个单元格 Iterator<Cell> cellIterator = row.iterator(); // 遍历一行多列 while (cellIterator.hasNext()) { Cell
         * cell = cellIterator.next(); Object cellObj = parseCell(cell); rowData.add(cellObj); }
         */
        return rowData;
    }

    private static List<ApiDataFieldLocaleMetadataDTO> getBusinessKeyFields(List<ApiDataFieldLocaleMetadataDTO> apiDataFieldLocaleMetadataDTOS) {
        return apiDataFieldLocaleMetadataDTOS.stream().filter(item -> item.getIs_businesskey()).collect(Collectors.toList());
    }

    private static List<ApiDataFieldLocaleMetadataDTO> getSheetField(String sheetName, List<ApiDataFieldLocaleMetadataDTO> apiDataFieldLocaleMetadataDTOS) {
        if (CollectionUtils.isEmpty(apiDataFieldLocaleMetadataDTOS)) {
            return null;
        }
        apiDataFieldLocaleMetadataDTOS = apiDataFieldLocaleMetadataDTOS.stream().filter(item -> item.getData_name().equals(sheetName)).collect(Collectors.toList());
        if (!CollectionUtils.isEmpty(apiDataFieldLocaleMetadataDTOS)) {
            return apiDataFieldLocaleMetadataDTOS.get(0).getField();
        }
        return null;
    }

    private static List<String> getBusinessKeys(List<ApiDataFieldLocaleMetadataDTO> apiDataFieldLocaleMetadataDTOS) {
        List<String> businessKeys = new LinkedList<>();
        for (ApiDataFieldLocaleMetadataDTO apiDataFieldLocaleMetadataDTO : apiDataFieldLocaleMetadataDTOS) {
            if (apiDataFieldLocaleMetadataDTO.getIs_businesskey()) {
                businessKeys.add(apiDataFieldLocaleMetadataDTO.getData_name());
            }
        }
        return businessKeys;
    }

    protected static Object parseCell(Cell cell, CellType metaCellType, MetaDataType metaType) {
        Object obj;
        final CellType cellType = cell.getCellTypeEnum();
        switch (cellType) {
            case FORMULA:
                /*
                 * if (cell != null) { cell.setCellType(Cell.CELL_TYPE_FORMULA); obj = cell.getStringCellValue(); }
                 */
                try {
                    /*
                     * 此处判断使用公式生成的字符串有问题，因为HSSFDateUtil.isCellDateFormatted(cell) 判断过程中cell
                     * .getNumericCellValue();方法会抛出java.lang.NumberFormatException异常
                     */
                    if (DateUtil.isCellDateFormatted(cell)) {
                        final Date date = cell.getDateCellValue();
                        obj = (date.getYear() + 1900) + "-" + (date.getMonth() + 1) + "-" + date.getDate();
                        break;
                    } else {
                        obj = String.valueOf(cell.getNumericCellValue());
                    }
                } catch (final IllegalStateException e) {
                    obj = String.valueOf(cell.getRichStringCellValue());
                }
                break;
            case BOOLEAN:
                obj = cell.getBooleanCellValue();
                break;
            case NUMERIC:
                // 处理日期格式、时间格式
                int number58 = 58;
                SimpleDateFormat sdf;
                switch (metaType) {
                    case DATE:
                        sdf = new SimpleDateFormat("yyyy-MM-dd");
                        break;
                    case DATETIME:
                        sdf = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss");
                        break;
                    case TIME:
                        sdf = new SimpleDateFormat("HH:mm:ss");
                        break;
                    default:
                        sdf = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss");
                        break;
                }

                if (DateUtil.isCellDateFormatted(cell)) {
                    final Date date = cell.getDateCellValue();
                    obj = sdf.format(date);
                } else if (cell.getCellStyle().getDataFormat() == number58) {
                    // 处理自定义日期格式：m月d日(通过判断单元格的格式id解决，id的值是58)
                    final double value = cell.getNumericCellValue();
                    final Date date = DateUtil.getJavaDate(value);
                    obj = sdf.format(date);
                } else {
                    final double dValue = cell.getNumericCellValue();
                    obj = dValue;
                    return obj;
                /*final CellStyle style = cell.getCellStyle();
                final DecimalFormat format = new DecimalFormat();
                final String temp = style.getDataFormatString();
                // 单元格设置成常规
                if ("General".equals(temp)) {
                    format.applyPattern("#");
                }
                obj = format.format(value);*/
                }
                break;
            default:
                obj = cell.getStringCellValue();
                break;
        }
        //return obj;
        if (null == metaCellType) {
            return obj;
        }
        switch (metaCellType) {
            case STRING:
                return null == obj ? "" : obj.toString();
            default:
                return obj;
        }
    }

    public static Object parseCell(Cell cell, CellType metaDataType) {
        Object obj;
        final CellType cellType = cell.getCellTypeEnum();
        switch (cellType) {
            case FORMULA:
                /*
                 * if (cell != null) { cell.setCellType(Cell.CELL_TYPE_FORMULA); obj = cell.getStringCellValue(); }
                 */
                try {
                    /*
                     * 此处判断使用公式生成的字符串有问题，因为HSSFDateUtil.isCellDateFormatted(cell) 判断过程中cell
                     * .getNumericCellValue();方法会抛出java.lang.NumberFormatException异常
                     */
                    if (DateUtil.isCellDateFormatted(cell)) {
                        final Date date = cell.getDateCellValue();
                        obj = (date.getYear() + 1900) + "-" + (date.getMonth() + 1) + "-" + date.getDate();
                        break;
                    } else {
                        obj = String.valueOf(cell.getNumericCellValue());
                    }
                } catch (final IllegalStateException e) {
                    obj = String.valueOf(cell.getRichStringCellValue());
                }
                break;
            case BOOLEAN:
                obj = cell.getBooleanCellValue();
                break;
            case NUMERIC:
                // 处理日期格式、时间格式
                int number58 = 58;
                if (DateUtil.isCellDateFormatted(cell)) {
                    final SimpleDateFormat sdf = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss");
                    final Date date = cell.getDateCellValue();
                    obj = sdf.format(date);
                } else if (cell.getCellStyle().getDataFormat() == number58) {
                    // 处理自定义日期格式：m月d日(通过判断单元格的格式id解决，id的值是58)
                    final SimpleDateFormat sdf = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss");
                    final double value = cell.getNumericCellValue();
                    final Date date = DateUtil.getJavaDate(value);
                    obj = sdf.format(date);
                } else {
                    final double value = cell.getNumericCellValue();
                    obj = value;
                /*final CellStyle style = cell.getCellStyle();
                final DecimalFormat format = new DecimalFormat();
                final String temp = style.getDataFormatString();
                // 单元格设置成常规
                if ("General".equals(temp)) {
                    format.applyPattern("#");
                }
                obj = format.format(value);*/
                }
                break;
            default:
                obj = cell.getStringCellValue();
                break;
        }
        if (null == metaDataType) {
            return obj;
        }
        switch (metaDataType) {
            case STRING:
                return null == obj ? "" : obj.toString();
            default:
                return obj;
        }
    }

    /**
     * 解析单元格
     *
     * @param cell
     * @return
     */
    protected static Object parseCell(Cell cell) {
        return parseCell(cell, null);
    }

    private static void realeaseResources(ExcelParserBean parserBean, Workbook workbook) {
        try {
            final InputStream inputStream = parserBean.getInput();
            if (inputStream != null) {
                inputStream.close();
            }
            if (workbook != null) {
                workbook.close();
            }
        } catch (final IOException e) {
            log.error(e.toString());
        }
    }

    private static String getFileExtension(String filePath) {
        if (StringUtils.isEmpty(filePath)) {
            return null;
        }
        final int lastIndexOf = filePath.lastIndexOf(".");
        if (lastIndexOf != -1) {
            final String extension = filePath.substring(lastIndexOf + 1);
            return extension;
        }
        return null;
    }


    public static String getSheetNameInParens(String sheetName) {
        String regex = "[\\(（](.*?)[\\)）]";
        Pattern pattern = Pattern.compile(regex);
        Matcher matcher = pattern.matcher(sheetName);
        if (matcher.find()) {
            // 使用group(1)获取第一个括号中的内容
            return matcher.group(1);
        } else {
            return sheetName;
        }
    }

}
