package com.digiwin.athena.base.infrastructure.mongo;

import com.digiwin.athena.base.infrastructure.util.DistributeLocker;
import com.mongodb.client.model.IndexOptions;
import org.apache.commons.collections4.MapUtils;
import org.apache.tomcat.util.buf.StringUtils;
import org.bson.Document;
import org.springframework.data.mongodb.core.MongoTemplate;
import org.springframework.data.redis.core.RedisTemplate;

import javax.annotation.Resource;
import java.util.Collections;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.TimeUnit;

/**
 * AbstractMongoMapper Description
 *
 * @author majianfu
 * @date 2021/4/26
 * @since
 */
public abstract class AbstractMongoMapper {
    public static final String LOCK_NAME_SUFFIX = ":createMongoCollection:lock";
    public static final String LOCK_NAME_PREFIX = "audc:";

    @Resource(name = "redisTemplate")
    private RedisTemplate lockRedisTemplate;

    protected ConcurrentHashMap<String, String> existCollections = new ConcurrentHashMap<>();

    /**
     * 安全地创建mongo集合，并为集合创建索引
     *
     * @param mongoTemplate
     * @param collectionName 集合名称
     * @param indexMap       索引项，如{"gid": -1}，意思为：gid降序
     */
    protected void safeCreateCollectionWithIndex(MongoTemplate mongoTemplate, String collectionName, Map<String, Object> indexMap) {
        safeCreateCollectionWithIndex(mongoTemplate, collectionName, indexMap, null, null);
    }

    /**
     * 安全地创建mongo集合，并为集合创建索引
     *
     * @param mongoTemplate
     * @param collectionName 集合名称
     * @param indexMap       索引项，如{"gid": -1}，意思为：gid降序
     * @param expireAfter    ttl
     * @param timeUnit
     */
    protected void safeCreateCollectionWithIndex(MongoTemplate mongoTemplate, String collectionName, Map<String, Object> indexMap, Long expireAfter, TimeUnit timeUnit) {
        if (existCollections.contains(collectionName)) {
            return;
        }

        if (mongoTemplate.collectionExists(collectionName)) {
            existCollections.put(collectionName, "1");
            return;
        }

        String tmpLockName = LOCK_NAME_PREFIX + collectionName + LOCK_NAME_SUFFIX;
        DistributeLocker.tryLock30s(lockRedisTemplate, tmpLockName, () -> createCollectionWithIndex(mongoTemplate, collectionName, indexMap, expireAfter, timeUnit));
    }

    /**
     * 创建mongo集合，并为集合创建索引
     *
     * @param mongoTemplate
     * @param collectionName 集合名称
     * @param indexMap       索引项，如{"gid": -1}，意思为：gid降序
     */
    protected void createCollectionWithIndex(MongoTemplate mongoTemplate, String collectionName, Map<String, Object> indexMap, Long expireAfter, TimeUnit timeUnit) {
        createCollection(mongoTemplate, collectionName);
        if (MapUtils.isNotEmpty(indexMap)) {
            createIndex(mongoTemplate, collectionName, indexMap, expireAfter, timeUnit);
        }
    }

    /**
     * 创建mongo集合
     *
     * @param mongoTemplate
     * @param collectionName 集合名称
     */
    protected void createCollection(MongoTemplate mongoTemplate, String collectionName) {
        if (existCollections.contains(collectionName)) {
            return;
        }

        if (mongoTemplate.collectionExists(collectionName)) {
            existCollections.put(collectionName, "1");
            return;
        }

        // 创建集合
        mongoTemplate.createCollection(collectionName);
        existCollections.put(collectionName, "1");

    }

    /**
     * 创建mongo 集合索引，ex:
     * <p>
     * indexOptions indexOptions = new IndexOptions()
     * .unique(true)
     * .background(true)
     * .name("UNIQUE_IDX_GID");
     * <p>
     * // 在gid上创建hashed类型索引
     * mongoTemplate.getCollection(collectionName).createIndex(new Document("gid", -1), indexOptions);
     *
     * @param mongoTemplate
     * @param collectionName 集合名称（表名）
     * @param indexMap       索引项，如{"gid": -1}，意思为：gid降序
     */
    protected void createIndex(MongoTemplate mongoTemplate, String collectionName, Map<String, Object> indexMap, Long expireAfter, TimeUnit timeUnit) {
        // 索引列名以下划线连接起来作为索引名
        String indexName = StringUtils.join(indexMap.keySet(), '_');

        // 该参数为索引的属性配置
        IndexOptions indexOptions = new IndexOptions()
                .unique(true)
                .background(true)
                .name(indexName);
        if (null != expireAfter) {
            indexOptions.expireAfter(expireAfter, timeUnit);
        }

        // 在gid上创建hashed类型索引
        mongoTemplate.getCollection(collectionName).createIndex(new Document(indexMap), indexOptions);

        //bug105831 时间戳增加索引
        indexOptions = new IndexOptions();
        indexOptions.background(true);
        mongoTemplate.getCollection(collectionName).createIndex(new Document(Collections.singletonMap("timestamp",-1)),indexOptions);
    }
}
