package com.digiwin.athena.base.presentation.server.web.trial;

import com.digiwin.athena.appcore.util.JsonUtils;
import com.digiwin.athena.appcore.util.ResponseEntityWrapper;
import com.digiwin.athena.base.application.constant.SceneDataConstant;
import com.digiwin.athena.base.application.service.trial.TrialSceneService;
import com.digiwin.athena.base.application.util.FileUtils;
import com.fasterxml.jackson.databind.JsonNode;
import com.github.fge.jsonschema.core.report.ProcessingMessage;
import com.github.fge.jsonschema.core.report.ProcessingReport;
import lombok.extern.slf4j.Slf4j;
import net.sf.json.JSONObject;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;
import org.springframework.web.multipart.MultipartFile;

import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;

/**
 * 描述往AUDC相关数据中导入数据
 *
 * @author gonghongxing
 * @version 1.0
 * @since 2021/10/11 16:17:36
 */
@Slf4j
@RestController
@RequestMapping(value = "/api/trial/data")
public class TrialImportController {

    private final TrialSceneService trialSceneService;

    @Autowired
    public TrialImportController(TrialSceneService trialSceneService) {
        this.trialSceneService = trialSceneService;
    }

    private static final Logger logger = LoggerFactory.getLogger(TrialImportController.class);

    /**
     * 获取场景或者步骤对应的步骤id
     *
     * @return ResponseEntity
     */
    @PostMapping(value = "/uploadScenes", produces = "application/json;charset=UTF-8")
    public ResponseEntity<?> uploadScene(@RequestParam("file") MultipartFile file) {

        String parseStr = "";
        try {
            InputStream fis = file.getInputStream();
            parseStr = FileUtils.StreamToString(fis);

            //校验是否符合schema要求
            ProcessingReport report = FileUtils.validate(parseStr, SceneDataConstant.SCENE_SCHEMA_JSON);

            if (!report.isSuccess()) {
                List<JsonNode> list = new ArrayList<>();
                Iterator it = report.iterator();
                while (it.hasNext()) {
                    ProcessingMessage message = (ProcessingMessage) it.next();
                    list.add(message.asJson());
                }
                return ResponseEntityWrapper.wrapperOk(list);
            }
        } catch (IOException e) {
            logger.error("Parse Upload File error", e);
            return ResponseEntityWrapper.wrapperFail(-1, "Parse Upload File error." + e.getMessage());
        }

        try {
            JSONObject sceneObj = JsonUtils.jsonToObject(parseStr, JSONObject.class);
            if (sceneObj.containsKey("key")) {
                String sceneKey = sceneObj.getString("key");
                boolean sceneKeyExist = trialSceneService.isSceneExist(sceneKey);
                if (!sceneKeyExist) {
                    return ResponseEntityWrapper.wrapperFail(-1, "Maintenance Scenario Key First");
                }

                int updateResult = trialSceneService.updateDetail(sceneKey, parseStr);
                return ResponseEntityWrapper.wrapperOk(updateResult);
            } else {
                return ResponseEntityWrapper.wrapperFail(-1, "Key information missing");
            }
        } catch (Exception e) {
            logger.error("Parse Upload File Content error", e);
            return ResponseEntityWrapper.wrapperFail(-1, "Parse Upload Content error." + e.getMessage());
        }
    }
}
