package com.digiwin.athena.base.presentation.server.web.commonused;

import cn.hutool.json.JSONArray;
import cn.hutool.json.JSONObject;
import cn.hutool.json.JSONUtil;
import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.conditions.update.LambdaUpdateWrapper;
import com.digiwin.athena.appcore.auth.GlobalConstant;
import com.digiwin.athena.appcore.auth.domain.AuthoredUser;
import com.digiwin.athena.appcore.exception.BusinessException;
import com.digiwin.athena.appcore.util.ResponseEntityWrapper;
import com.digiwin.athena.base.application.config.BaseAudcDataSourceConfig;
import com.digiwin.athena.base.application.meta.dto.commonused.CardGroupGroup;
import com.digiwin.athena.base.application.meta.dto.commonused.CategoriesDto;
import com.digiwin.athena.base.application.meta.dto.commonused.ItemsDto;
import com.digiwin.athena.base.application.meta.dto.commonused.SourceType;
import com.digiwin.athena.base.application.meta.dto.commonused.SystemWSConditionVO;
import com.digiwin.athena.base.application.meta.dto.commonused.WSConditionVO;
import com.digiwin.athena.base.application.meta.request.commonused.BatchAddItemReq;
import com.digiwin.athena.base.application.meta.request.commonused.BatchSortItemReq;
import com.digiwin.athena.base.application.meta.request.commonused.CategoryReq;
import com.digiwin.athena.base.application.meta.request.commonused.CheckUsedReq;
import com.digiwin.athena.base.application.meta.request.commonused.ItemReq;
import com.digiwin.athena.base.application.meta.request.commonused.MenuAuthorityReq;
import com.digiwin.athena.base.application.meta.request.commonused.TaskCardGroupReq;
import com.digiwin.athena.base.application.meta.request.commonused.TaskCardListConfigReq;
import com.digiwin.athena.base.application.meta.response.commonused.CateAndItemResp;
import com.digiwin.athena.base.application.meta.response.commonused.CategoryResp;
import com.digiwin.athena.base.application.meta.response.commonused.CheckUsedResp;
import com.digiwin.athena.base.application.meta.response.commonused.ItemResp;
import com.digiwin.athena.base.application.service.commonused.CategoryService;
import com.digiwin.athena.base.application.service.commonused.CommonUsedService;
import com.digiwin.athena.base.application.service.commonused.DataMigrationService;
import com.digiwin.athena.base.application.service.commonused.ItemService;
import com.digiwin.athena.base.application.service.commonused.MenuService;
import com.digiwin.athena.base.application.service.commonused.TaskCardGroupService;
import com.digiwin.athena.base.application.service.commonused.TaskCardListConfigService;
import com.digiwin.athena.base.application.util.LanguageUtil;
import com.digiwin.athena.base.infrastructure.constant.AudcErrorCodeEnum;
import com.digiwin.athena.base.infrastructure.manager.atmc.BaseAtmcService;
import com.digiwin.athena.base.infrastructure.manager.thememap.BaseThemeMapService;
import com.digiwin.athena.base.infrastructure.mapper.audc.commonUsed.TaskCardListConfigMapper;
import com.digiwin.athena.base.infrastructure.meta.po.commonused.CategoryData;
import com.digiwin.athena.base.infrastructure.meta.po.commonused.CustomTaskCardGroupData;
import com.digiwin.athena.base.infrastructure.meta.po.commonused.ItemData;
import com.digiwin.athena.base.infrastructure.meta.po.commonused.TaskCardListConfigData;
import com.google.common.collect.Lists;
import com.jugg.agile.spring.util.JaSpringBeanUtil;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.collections.MapUtils;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.i18n.LocaleContextHolder;
import org.springframework.http.ResponseEntity;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.ObjectUtils;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestAttribute;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;
import org.springframework.web.method.HandlerMethod;
import org.springframework.web.servlet.mvc.method.RequestMappingInfo;
import org.springframework.web.servlet.mvc.method.annotation.RequestMappingHandlerMapping;

import javax.validation.Valid;
import java.time.LocalDateTime;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.stream.Collectors;


@Slf4j
@RestController
@RequestMapping(value = "/api/amtc/v1/commonUsed")
public class CommonUsedController {

    @Autowired
    private CommonUsedService commonUsedService;

    @Autowired
    private CategoryService categoryService;

    @Autowired
    private ItemService itemService;

    @Autowired
    private MenuService menuService;

    @Autowired
    private DataMigrationService dataMigrationService;

    /**
     * 查询常用的分类以及分类下的子项
     *
     * @param user   登录用户信息
     * @param plat   所属分类0:基础资料录入 1：报表，后续有新增则依次递增
     * @param cateId 分类id，-1为根节点
     * @return ResponseEntity
     */
    @GetMapping(value = "/list", produces = "application/json;charset=UTF-8")
    public ResponseEntity<?> cateAndItemList(@RequestAttribute(value = GlobalConstant.AUTH_USER) AuthoredUser user, @RequestParam(value = "plat") Integer plat, @RequestParam(value = "cateId", required = false) Long cateId) {

        CateAndItemResp resp = commonUsedService.cateAndItemList(user, plat, cateId);
        String defaultUnfold = categoryService.queryDefaultUnfold(user, plat);
        resp.setDefaultUnfold(defaultUnfold);
        return ResponseEntityWrapper.wrapperOk(resp);
    }


    /**
     * 新增常用分类
     *
     * @param user    用户信息
     * @param cateReq 请求信息
     * @return 返回信息
     */
    @PostMapping(value = "/cate/add", produces = "application/json;charset=UTF-8")
    public ResponseEntity<?> addCategory(@RequestAttribute(value = GlobalConstant.AUTH_USER) AuthoredUser user, @RequestBody CategoryReq cateReq) {

        CategoriesDto cateDto = new CategoriesDto();
        BeanUtils.copyProperties(cateReq, cateDto);
        //校验新增分组是否存在
        if (categoryService.isCategoryExist(user, cateDto)) {
            throw BusinessException.create(-1, "New Category Is Exist");
        }
        CategoryData data = categoryService.addCategory(user, cateDto);
        CategoryResp cateResp = new CategoryResp();
        if (null != data) {
            BeanUtils.copyProperties(data, cateResp);
        }
        return ResponseEntityWrapper.wrapperOk(cateResp);
    }

    /**
     * 删除分类,并将该分类下的子项移动到默认分组中
     *
     * @param user 用户信息
     * @param plat 所属模块
     * @param id   待删除分类id
     * @return 返回信息
     */
    @GetMapping(value = "/cate/moveAndDel", produces = "application/json;charset=UTF-8")
    public ResponseEntity<?> deleteCategoryAndMoveItems(@RequestAttribute(value = GlobalConstant.AUTH_USER) AuthoredUser user, @RequestParam(value = "plat") Integer plat, @RequestParam(value = "id") Long id) {

        //根节点不允许删除
        if (id == -1) {
            return ResponseEntityWrapper.wrapperFail(-1, "This Category is Root,Deletion not allowed");
        }
        int result = commonUsedService.deleteCateAndMoveItems(user, id, plat);
        return ResponseEntityWrapper.wrapperOk(result);
    }


    /**
     * 删除分类
     *
     * @param user 用户信息
     * @param plat 所属模块
     * @param id   待删除分类id
     * @return 返回信息
     */
    @GetMapping(value = "/cate/delete", produces = "application/json;charset=UTF-8")
    public ResponseEntity<?> deleteCategory(@RequestAttribute(value = GlobalConstant.AUTH_USER) AuthoredUser user, @RequestParam(value = "plat") Integer plat, @RequestParam(value = "id") Long id) {

        //根节点不允许删除
        if (id == -1) {
            return ResponseEntityWrapper.wrapperFail(-1, "This Category is Root,Deletion not allowed");
        }
        int result = commonUsedService.deleteCommUsedCate(user, id, plat);
        return ResponseEntityWrapper.wrapperOk(result);
    }


    /**
     * 修改分类名称
     *
     * @param user 用户信息
     * @param id   待删除分类id
     * @return 返回信息
     */
    @GetMapping(value = "/cate/rename", produces = "application/json;charset=UTF-8")
    public ResponseEntity<?> cateRename(@RequestAttribute(value = GlobalConstant.AUTH_USER) AuthoredUser user, @RequestParam(value = "id") Long id, @RequestParam(value = "cateName") String cateName) {
        CategoryData cateData = categoryService.detail(id);
        if (null == cateData) {
            throw AudcErrorCodeEnum.COMMON_USED_CATEGORY_NOT_EXIST.getBusinessException("Category Info Not Exist");
        }
        CategoriesDto dto = new CategoriesDto();
        dto.setCateName(cateName);
        dto.setParentId(cateData.getParentId());
        dto.setPlat(cateData.getPlat());
        //校验新的命名是否存在s
        if (categoryService.isCategoryExist(user, dto)) {
            throw BusinessException.create(-1, "Change CategoryName Is Exist");
        }
        int result = categoryService.renameCategory(user, id, cateName);
        return ResponseEntityWrapper.wrapperOk(result);
    }


    /**
     * 新增常用子项
     *
     * @param user    用户信息
     * @param itemReq 请求信息
     * @return 返回信息
     */
    @PostMapping(value = "/item/add", produces = "application/json;charset=UTF-8")
    public ResponseEntity<?> addItem(@RequestAttribute(value = GlobalConstant.AUTH_USER) AuthoredUser user, @RequestBody ItemReq itemReq) {

        ItemsDto itemDto = new ItemsDto();
        BeanUtils.copyProperties(itemReq, itemDto);
        ItemResp resp = new ItemResp();
//        //检查该子项是否已经存在
//        if (itemService.isItemExist(user, itemDto)) {
//            throw new BusinessException(-1, "Item Code Is Exist");
//        }
        ItemData data = itemService.addItem(user, itemDto);
        if (null != data) {
            BeanUtils.copyProperties(data, resp);
        }
        return ResponseEntityWrapper.wrapperOk(resp);
    }


    /**
     * 移动常用子项
     *
     * @param user       用户信息
     * @param id         需要移动的item id
     * @param destCateId 目标分类id -1则为根节点
     * @return 返回信息
     */
    @GetMapping(value = "/item/move", produces = "application/json;charset=UTF-8")
    public ResponseEntity<?> moveItem(@RequestAttribute(value = GlobalConstant.AUTH_USER) AuthoredUser user, @RequestParam(value = "id") Long id, @RequestParam(value = "destCateId") Long destCateId) {

        int result = itemService.moveSingleItem(id, destCateId);
        return ResponseEntityWrapper.wrapperOk(result);
    }


    /**
     * 设置常用
     *
     * @param user    用户信息
     * @param itemReq 需要设置常用的item信息
     * @return 返回信息
     */
    @PostMapping(value = "/item/used", produces = "application/json;charset=UTF-8")
    public ResponseEntity<?> changeUsed(@RequestAttribute(value = GlobalConstant.AUTH_USER) AuthoredUser user, @RequestBody ItemReq itemReq) {
        //常用入口的节点，默认均在跟节点下
        itemReq.setCateId(-1L);
        return addItem(user, itemReq);
    }


    /**
     * 修改常用子项
     *
     * @param user 用户信息
     * @param id   需要修改状态的itemId
     * @return 返回信息
     */
    @GetMapping(value = "/item/unused", produces = "application/json;charset=UTF-8")
    public ResponseEntity<?> changeUsed(@RequestAttribute(value = GlobalConstant.AUTH_USER) AuthoredUser user, @RequestParam(value = "id", required = false) Long id, @RequestParam(value = "itemCode", required = false) String itemCode, @RequestParam(value = "plat", required = false) Integer plat) {
        int result;
        if (null == id) {
            result = itemService.delete(user, plat, itemCode);
        } else {
            result = itemService.delete(user, id);
        }

        return ResponseEntityWrapper.wrapperOk(result);
    }


    /**
     * 查询常用子项详情
     *
     * @param user     用户信息
     * @param itemCode z
     * @return 返回信息
     */
    @GetMapping(value = "/item/detail", produces = "application/json;charset=UTF-8")
    public ResponseEntity<?> itemDetail(@RequestAttribute(value = GlobalConstant.AUTH_USER) AuthoredUser user, @RequestParam(value = "itemCode") String itemCode, @RequestParam(value = "plat") Integer plat) {

        ItemData data = itemService.detail(user, itemCode, plat);

        ItemResp resp = new ItemResp();
        if (null != data) {
            Map<String, String> baseItemMaps = commonUsedService.queryEntryData(user, plat);
            BeanUtils.copyProperties(data, resp);
            resp.setItemName(MapUtils.getString(baseItemMaps, resp.getItemCode(), ""));
        }
        return ResponseEntityWrapper.wrapperOk(resp);
    }


    /**
     * 检查子项是否为常用
     *
     * @param user     用户信息
     * @param plat     所属分类0:基础资料录入 1：报表，后续有新增则依次递增
     * @param itemCode 子项编码
     * @return 返回信息
     */
    @GetMapping(value = "/item/checkUsed", produces = "application/json;charset=UTF-8")
    public ResponseEntity<?> checkUsed(@RequestAttribute(value = GlobalConstant.AUTH_USER) AuthoredUser user, @RequestParam(value = "plat") Integer plat, @RequestParam(value = "itemCode") String itemCode) {

        boolean result = itemService.checkUsed(user, plat, itemCode);
        return ResponseEntityWrapper.wrapperOk(result);
    }

    /**
     * 子项打开记录
     *
     * @param user    用户信息
     * @param itemReq 需要设置常用的item信息
     * @return 返回信息
     */
    @PostMapping(value = "/item/operate/record", produces = "application/json;charset=UTF-8")
    public ResponseEntity<?> operateRecord(@RequestAttribute(value = GlobalConstant.AUTH_USER) AuthoredUser user, @RequestBody ItemReq itemReq) {
        //打开子项的时候，调用该接口，记录子项的打开记录
        ItemsDto itemDto = new ItemsDto();
        BeanUtils.copyProperties(itemReq, itemDto);
        int result = itemService.itemRecord(user, itemDto, 0);
        return ResponseEntityWrapper.wrapperOk(result);
    }


    /**
     * 批量新增常用子项
     *
     * @param user     用户信息
     * @param itemReqs 请求信息
     * @return 返回信息
     */
    @PostMapping(value = "/item/batch/add", produces = "application/json;charset=UTF-8")
    public ResponseEntity<?> batchAddItems(@RequestAttribute(value = GlobalConstant.AUTH_USER) AuthoredUser user, @RequestBody List<ItemReq> itemReqs) {

        if (CollectionUtils.isNotEmpty(itemReqs)) {
            for (ItemReq itemReq : itemReqs) {
                ItemsDto itemDto = new ItemsDto();
                BeanUtils.copyProperties(itemReq, itemDto);
//                //检查该子项是否已经存在
//                if (itemService.isItemExist(user, itemDto)) {
//                    logger.warn("item data:{} is exist", itemDto);
//                    continue;
//                }
                ItemData data = itemService.addItem(user, itemDto);
                ItemResp resp = new ItemResp();
                if (null != data) {
                    BeanUtils.copyProperties(data, resp);
                }
            }
            return ResponseEntityWrapper.wrapperOk(1);

        } else {
            return ResponseEntityWrapper.wrapperOk(new ArrayList<>());
        }
    }


    /**
     * 批量新增常用子项
     *
     * @param user     用户信息
     * @param batchReq 请求信息
     * @return 返回信息
     */
    @PostMapping(value = "/item/batch/V2/add", produces = "application/json;charset=UTF-8")
    public ResponseEntity<?> batchAddItemsV2(@RequestAttribute(value = GlobalConstant.AUTH_USER) AuthoredUser user, @RequestBody BatchAddItemReq batchReq) {

        //获取需要添加的分类id
        Long cateId = batchReq.getCateId();
        //分类id不存在，则认为需要新增该分类
        if (Objects.isNull(cateId)) {
            CategoriesDto categoryDto = new CategoriesDto();
            categoryDto.setCateName(batchReq.getCateName());
            categoryDto.setPlat(batchReq.getPlat());
            categoryDto.setParentId(-1L);
            if (categoryService.isCategoryExist(user, categoryDto)) {
                throw BusinessException.create(-1, "Category Name Is Exist");
            }
            CategoryData insertResult = categoryService.addCategory(user, categoryDto);
            cateId = insertResult.getId();
        }
        if (null != batchReq.getItemList() && !batchReq.getItemList().isEmpty()) {
            for (ItemReq itemReq : batchReq.getItemList()) {
                ItemsDto itemDto = new ItemsDto();
                BeanUtils.copyProperties(itemReq, itemDto);
                itemDto.setPlat(batchReq.getPlat());
                itemDto.setCateId(cateId);
                ItemData data = itemService.addItem(user, itemDto);
                ItemResp resp = new ItemResp();
                if (null != data) {
                    BeanUtils.copyProperties(data, resp);
                }
            }
            return ResponseEntityWrapper.wrapperOk(1);

        } else {
            return ResponseEntityWrapper.wrapperOk(new ArrayList<>());
        }
    }


    /**
     * 检查子项是否为常用
     *
     * @param user 用户信息
     * @param req  子项编码 plat     所属分类0:基础资料录入 1：报表，后续有新增则依次递增
     * @return 返回信息
     */
    @PostMapping(value = "/item/batch/checkUsed", produces = "application/json;charset=UTF-8")
    public ResponseEntity<?> checkUsed(@RequestAttribute(value = GlobalConstant.AUTH_USER) AuthoredUser user, @RequestBody CheckUsedReq req) {

        int plat = req.getPlat();
        List<String> codeLists = req.getItemCodes();
        CheckUsedResp resp = new CheckUsedResp();
        resp.setPlat(plat);
        Map<String, Object> retMap = new HashMap<>();
        for (String str : codeLists) {
            boolean result = itemService.checkUsed(user, plat, str);
            retMap.put(str, result);
        }

        resp.setCheckResult(retMap);
        return ResponseEntityWrapper.wrapperOk(resp);
    }

    /**
     * 获取TM所有基础资料录入/报表数据
     *
     * @Author：SYQ
     * @Date：2021/9/26 9:18
     */
    @GetMapping(value = "/tmDataList", produces = "application/json;charset=UTF-8")
    public ResponseEntity<?> tmDataList(@RequestAttribute(value = GlobalConstant.AUTH_USER) AuthoredUser user, Integer plat
            , @RequestParam(value = "allowRepeat", required = false) Boolean allowRepeat, @RequestParam(required = false) String pattern
            , @RequestParam(value = "effect", required = false) String effect) {
        //如果没有设置允许重复，则默认允许重复的作业code
        if (null == allowRepeat) {
            allowRepeat = true;
        }
        return ResponseEntityWrapper.wrapperOk(commonUsedService.tmDataList(user, plat, allowRepeat, true, null, pattern, effect));
    }

    /**
     * 获取发起项目列表
     *
     * @param user
     * @return
     */
    @GetMapping(value = "/tmStartProject", produces = "application/json;charset=UTF-8")
    public ResponseEntity<?> tmStartProject(@RequestAttribute(value = GlobalConstant.AUTH_USER) AuthoredUser user) {
        return ResponseEntityWrapper.wrapperOk(commonUsedService.startProjectList(user, true, null));
    }


    @GetMapping(value = "/tmTest", produces = "application/json;charset=UTF-8")
    public ResponseEntity<?> tmTest(@RequestAttribute(value = GlobalConstant.AUTH_USER) AuthoredUser user) {
        RequestMappingHandlerMapping mapping = JaSpringBeanUtil.getBean("requestMappingHandlerMapping", RequestMappingHandlerMapping.class);
        Map<RequestMappingInfo, HandlerMethod> map = mapping.getHandlerMethods();
        Map<String, String> retMap = new HashMap<>();
        for (Map.Entry<RequestMappingInfo, HandlerMethod> entry : map.entrySet()) {
            String parttenUrl = entry.getKey().getPatternsCondition().getPatterns().toString();
            String fianlStr = parttenUrl.substring(1, parttenUrl.length() - 1);
            retMap.put(fianlStr, entry.getValue().getBeanType().getSimpleName());
        }

        return ResponseEntityWrapper.wrapperOk(retMap);
    }

    /**
     * 常用分组,默认只查一层目录，cateId不传，默认-1
     *
     * @Author：SYQ
     * @Date：2021/10/13 17:00
     */
    @GetMapping(value = "/getCategory", produces = "application/json;charset=UTF-8")
    public ResponseEntity<?> getCategory(@RequestAttribute(value = GlobalConstant.AUTH_USER) AuthoredUser user, @RequestParam(value = "plat") Integer plat, @RequestParam(value = "cateId", required = false) Long cateId) {
        if (Objects.isNull(cateId)) {
            cateId = -1L;
        }
        return ResponseEntityWrapper.wrapperOk(commonUsedService.finCategory(user, plat, cateId));
    }


    /**
     * 查询置顶的模块信息
     *
     * @param user 用户信息
     * @return 查询结果
     */
    @GetMapping(value = "/menu/query", produces = "application/json;charset=UTF-8")
    public ResponseEntity<?> menuTopQuery(@RequestAttribute(value = GlobalConstant.AUTH_USER) AuthoredUser user) {
        return ResponseEntityWrapper.wrapperOk(menuService.query(user));
    }


    /**
     * 设置指定分组为置顶
     *
     * @param user   用户信息
     * @param plat   '所属模块，0:基础资料录入 1：报表，后续有新增则依次递增
     * @param cateId 分组的id
     * @return 查询结果
     */
    @GetMapping(value = "/cate/setTop", produces = "application/json;charset=UTF-8")
    public ResponseEntity<?> setCateTop(@RequestAttribute(value = GlobalConstant.AUTH_USER) AuthoredUser user, @RequestParam(value = "plat") Integer plat, @RequestParam(value = "cateId", required = false) Long cateId) {

        return ResponseEntityWrapper.wrapperOk(categoryService.setCateTop(user, plat, cateId));
    }

    /**
     * 设置指定模块为置顶
     *
     * @param user     用户信息
     * @param moduleNo 模块信息
     * @return 查询结果
     */
    @GetMapping(value = "/menu/setTop", produces = "application/json;charset=UTF-8")
    public ResponseEntity<?> setMenuTop(@RequestAttribute(value = GlobalConstant.AUTH_USER) AuthoredUser user, @RequestParam(value = "moduleNo") String moduleNo) {

        return ResponseEntityWrapper.wrapperOk(menuService.setMenuTop(user, moduleNo));
    }

    /**
     * 设置指定模块为展开
     *
     * @param user     用户信息
     * @param moduleNo 模块信息
     * @return 查询结果
     */
    @GetMapping(value = "/menu/setUnFold", produces = "application/json;charset=UTF-8")
    public ResponseEntity<?> setUnFold(@RequestAttribute(value = GlobalConstant.AUTH_USER) AuthoredUser user, @RequestParam(value = "moduleNo") String moduleNo, @RequestParam(value = "flag") Integer flag) {

        return ResponseEntityWrapper.wrapperOk(menuService.setMenuUnfold(user, moduleNo, flag));
    }


    /**
     * 设置分组是否展开
     *
     * @param user   用户信息
     * @param cateId 分类id
     * @param flag   0:收起 1：展开
     * @return 查询结果
     */
    @GetMapping(value = "/cate/unfold", produces = "application/json;charset=UTF-8")
    public ResponseEntity<?> cateUnfold(@RequestAttribute(value = GlobalConstant.AUTH_USER) AuthoredUser user, @RequestParam(value = "cateId", required = false) Long cateId, @RequestParam(value = "flag", required = true) Integer flag, @RequestParam(value = "plat", required = true) Integer plat) {

        //如果是默认分组，则使用redis来存储分组展开信息
        if (cateId == -1) {
            return ResponseEntityWrapper.wrapperOk(categoryService.cateDefaultUnfold(user, plat, flag));
        }
        //非默认分组，则使用分组的状态来显示展开信息
        else {
            return ResponseEntityWrapper.wrapperOk(categoryService.cateUnfold(user, cateId, flag));
        }
    }

    /**
     * 获取常用数据
     *
     * @param user
     * @param itemReq
     * @return
     */
    @PostMapping(value = "/itemList", produces = "application/json;charset=UTF-8")
    public ResponseEntity<?> queryItemList(@RequestAttribute(value = GlobalConstant.AUTH_USER) AuthoredUser user, @RequestBody ItemReq itemReq) {
        return ResponseEntityWrapper.wrapperOk(itemService.queryItemList(user, itemReq.getPlat(), itemReq.getCateId()));
    }

    /**
     * 冲刺7临时接口，用于将常用功能的数据迁移到audc，冲刺8可删除
     *
     * @return
     */
    @GetMapping(value = "/dataMigration/insert", produces = "application/json;charset=UTF-8")
    public ResponseEntity<?> dataMigration() {
        return dataMigrationService.insertAllCommonUserData();
    }


    /**
     * 基础资料录入、报表的用户自定义排序
     *
     * @return
     */
    @PostMapping(value = "/sort", produces = "application/json;charset=UTF-8")
    public ResponseEntity<?> sort(@RequestAttribute(value = GlobalConstant.AUTH_USER) AuthoredUser user, @RequestBody @Valid BatchSortItemReq itemReq) {

        commonUsedService.saveCustomSort(user, itemReq);
        return ResponseEntityWrapper.wrapperOk();
    }


    @Autowired
    private TaskCardListConfigService taskCardListConfigService;

    @GetMapping(value = "/getTaskCardListConfig")
    public ResponseEntity<?> getTaskCardListConfig(@RequestAttribute(value = GlobalConstant.AUTH_USER) AuthoredUser user, @RequestParam(value = "groupId") String groupId) {

        TaskCardListConfigData taskCardListConfigData = taskCardListConfigService.getBaseMapper().selectOne(new LambdaQueryWrapper<TaskCardListConfigData>().eq(TaskCardListConfigData::getTenantId, user.getTenantId()).eq(TaskCardListConfigData::getUserId, user.getUserId()).eq(TaskCardListConfigData::getGroupId, groupId));

        return ResponseEntityWrapper.wrapperOk(taskCardListConfigData == null ? null : taskCardListConfigData.getConfig());

    }

    @GetMapping(value = "/getTaskCardListConfigByGroupType")
    public ResponseEntity<?> getTaskCardListConfigByGroupType(@RequestAttribute(value = GlobalConstant.AUTH_USER) AuthoredUser user, @RequestParam(value = "groupType") Integer groupType, @RequestParam(value = "pageCode", required = false) String pageCode) {

        List<TaskCardListConfigData> list = taskCardListConfigService.getBaseMapper().selectList(new LambdaQueryWrapper<TaskCardListConfigData>().eq(TaskCardListConfigData::getTenantId, user.getTenantId()).eq(TaskCardListConfigData::getUserId, user.getUserId()).eq(TaskCardListConfigData::getGroupType, groupType).eq(TaskCardListConfigData::getPageCode, pageCode));

        HashMap<String, String> result = new HashMap<>();
        list.forEach(x -> {
            result.put(x.getGroupId(), x.getConfig());
        });
        return ResponseEntityWrapper.wrapperOk(ObjectUtils.isEmpty(result) ? null : result);

    }

    @Autowired
    TaskCardListConfigMapper taskCardListConfigMapper;

    @PostMapping(value = "saveTaskCardListConfig")
    public ResponseEntity<?> saveTaskCardListConfig(@RequestAttribute(value = GlobalConstant.AUTH_USER) AuthoredUser user, @RequestBody TaskCardListConfigReq taskCardListConfigReq) {
        TaskCardListConfigData taskCardListConfigData = new TaskCardListConfigData();
        taskCardListConfigData.setConfig(taskCardListConfigReq.getConfig());
        taskCardListConfigData.setUserId(user.getUserId());
        taskCardListConfigData.setTenantId(user.getTenantId());
        taskCardListConfigData.setGroupId(taskCardListConfigReq.getGroupId());
        taskCardListConfigData.setGroupType(taskCardListConfigReq.getGroupType());
        taskCardListConfigData.setPageCode(taskCardListConfigReq.getPageCode());

        try {
            TaskCardListConfigData one = taskCardListConfigMapper.getOne(taskCardListConfigData.getUserId(), taskCardListConfigData.getTenantId(), taskCardListConfigData.getGroupId(), taskCardListConfigData.getGroupType(), taskCardListConfigData.getPageCode());
            if (one == null) {
                taskCardListConfigService.save(taskCardListConfigData);
            } else {
                taskCardListConfigMapper.updateConfig(user.getUserId(), user.getTenantId(), taskCardListConfigData.getGroupId(), taskCardListConfigData.getGroupType(), taskCardListConfigData.getConfig(), taskCardListConfigData.getPageCode());
            }
        } catch (Exception e) {
            log.error("saveTaskCardListConfig:{},userId:{}", e.getMessage(), user.getUserId());
        }

        return ResponseEntityWrapper.wrapperOk();
    }


    @Autowired
    private TaskCardGroupService taskCardGroupService;

    @Transactional(transactionManager = BaseAudcDataSourceConfig.BASE_AUDC_DATASOURCE_TRANSACTION_MANAGER_BUSINESS)
    @PostMapping(value = "saveSwitchGroupRecord")
    public ResponseEntity<?> saveSwitchGroupRecord(@RequestAttribute(value = GlobalConstant.AUTH_USER) AuthoredUser user, @RequestBody TaskCardGroupReq taskCardGroupReq) {

        CustomTaskCardGroupData customTaskCardGroupData = taskCardGroupService.getBaseMapper().selectOne(new LambdaQueryWrapper<CustomTaskCardGroupData>().eq(CustomTaskCardGroupData::getUserId, user.getUserId()).eq(CustomTaskCardGroupData::getTenantId, user.getTenantId()).eq(CustomTaskCardGroupData::getGroupKey, taskCardGroupReq.getGroupKey()).eq(CustomTaskCardGroupData::getGroupName, taskCardGroupReq.getGroupName()));
        if (customTaskCardGroupData == null || customTaskCardGroupData.getGroupName() == null) {
            CustomTaskCardGroupData insertData = new CustomTaskCardGroupData();
            insertData.setGroupName(taskCardGroupReq.getGroupName());
            insertData.setCheckTime(LocalDateTime.now());
            insertData.setUserId(user.getUserId());
            insertData.setGroupKey(taskCardGroupReq.getGroupKey());
            insertData.setPageCode("task");
            insertData.setTenantId(user.getTenantId());
            taskCardGroupService.save(insertData);
        } else {
            taskCardGroupService.update(null, new LambdaUpdateWrapper<CustomTaskCardGroupData>().eq(CustomTaskCardGroupData::getUserId, user.getUserId()).eq(CustomTaskCardGroupData::getGroupName, taskCardGroupReq.getGroupName()).eq(CustomTaskCardGroupData::getTenantId, user.getTenantId()).eq(CustomTaskCardGroupData::getGroupKey, taskCardGroupReq.getGroupKey()).set(CustomTaskCardGroupData::getPageCode, "task").set(CustomTaskCardGroupData::getCheckTime, LocalDateTime.now()));
        }
        return ResponseEntityWrapper.wrapperOk();
    }

    @Autowired
    BaseThemeMapService baseThemeMapService;

    @Autowired
    BaseAtmcService baseAtmcService;

    public static final String TASK_ON_OFF = "taskOnOff";

    public static final String TASK_CARD_WHITE_LIST = "taskCardWhiteList";

    /**
     * @param user
     * @return
     */
    @GetMapping(value = "getGroupRecord")
    public ResponseEntity<?> getGroupRecord(@RequestAttribute(value = GlobalConstant.AUTH_USER) AuthoredUser user) {
        Map groupConfig = new HashMap();
        try {
            groupConfig = baseThemeMapService.getGroupConfig();
        } catch (Exception e) {
            log.error("KM business Error: {}", e.getMessage());
            return ResponseEntityWrapper.wrapperOk(generatedDefaultGroup());
        }
        Map<String, Object> groupRule = (Map<String, Object>) groupConfig.get("groupRule");
        JSONArray jsonArray = new JSONArray(groupRule.get("showGroup"));
        List<CustomTaskCardGroupData> list = taskCardGroupService.getBaseMapper().selectList(new LambdaQueryWrapper<CustomTaskCardGroupData>().eq(CustomTaskCardGroupData::getUserId, user.getUserId()).eq(CustomTaskCardGroupData::getTenantId, user.getTenantId()).and(i -> i.eq(CustomTaskCardGroupData::getPageCode, "task").or().isNull(CustomTaskCardGroupData::getPageCode)).orderByDesc(CustomTaskCardGroupData::getCheckTime).last("LIMIT 1"));

        List<String> codes = jsonArray.stream().map(e -> ((JSONObject) e).get("code").toString()).collect(Collectors.toList());

        List<JSONObject> enabledList = jsonArray.stream().filter(e -> "true".equals(((JSONObject) e).get("enabled").toString())).map(e -> (JSONObject) e).collect(Collectors.toList());
        List<JSONObject> isDefault = enabledList.stream().filter(e -> "true".equals(e.get("defaultEffective").toString())).collect(Collectors.toList());
        if (isDefault.isEmpty()) {
            enabledList.stream().filter(e -> "groupByProjectType".equals(e.get("code"))).forEach(e -> e.set("defaultEffective", "true"));
        }

        Boolean aTrue = baseAtmcService.verifyConfig(TASK_ON_OFF, "true");
        Boolean aBoolean = baseAtmcService.verifyConfig(TASK_CARD_WHITE_LIST, user.getTenantId());
        if (!aTrue && !aBoolean) {
            enabledList = enabledList.stream().filter(e -> "system".equals(e.get("type"))).collect(Collectors.toList());
            return ResponseEntityWrapper.wrapperOk(enabledList);
        }

        if (list.isEmpty()) {
            return ResponseEntityWrapper.wrapperOk(enabledList);
        }

        CustomTaskCardGroupData customTaskCardGroupData = list.get(0);
        // 如果是新格式的用户记忆则转换
        convertMemoryConfig(customTaskCardGroupData);
        String groupKey = customTaskCardGroupData.getGroupKey();

        if (!codes.contains(groupKey)) {
            return ResponseEntityWrapper.wrapperOk(enabledList);
        } else {
            enabledList.forEach(e -> {
                if ("true".equals(e.get("defaultEffective").toString())) {
                    e.set("defaultEffective", false);
                }
                if (groupKey.equals(e.get("code").toString())) {
                    e.set("defaultEffective", true);
                }
            });
            return ResponseEntityWrapper.wrapperOk(enabledList);
        }
    }

    private void convertMemoryConfig(CustomTaskCardGroupData customTaskCardGroupData) {
        try {
            String memoryConfig = customTaskCardGroupData.getMemoryConfig();
            if (JSONUtil.isTypeJSON(memoryConfig)) {
                JSONObject jsonObject = JSONUtil.parseObj(memoryConfig);
                // JSONObject data = jsonObject.get("data", JSONObject.class);
                String model = jsonObject.getStr("model");
                // 上次web端选中的是常规，则转换成历史数据结构给移动端
                if (Objects.equals(model, "normal")) {
                    JSONArray normalValue = jsonObject.getJSONArray("normalValue");
                    if (CollectionUtils.isNotEmpty(normalValue)) {
                        Object o = normalValue.get(0);
                        JSONObject object = JSONUtil.parseObj(JSONUtil.toJsonStr(o));
                        String code = object.getStr("code");
                        customTaskCardGroupData.setGroupKey(code);
                        customTaskCardGroupData.setGroupName(object.getStr("name"));
                    }
                }
            }
        } catch (Exception e) {
            log.error("convertMemoryConfig error,", e);
        }
    }


    private Map<String, Object> generatedDefaultGroupV2(String pageCode) {
        Map<String, Object> map = new HashMap<>();
        String locale = LocaleContextHolder.getLocale().toString();
        if (pageCode.equals("project-card")) {
            List<WSConditionVO> projectGroupWSConditionVO = SystemWSConditionVO.projectGroupWSConditionVO;
            try {
                LanguageUtil.processLocaleLanguage(projectGroupWSConditionVO, locale);
            } catch (IllegalAccessException e) {
                log.error("processLocaleLanguage error" + projectGroupWSConditionVO);
            }
            map.put("defaultCondition", projectGroupWSConditionVO);
        } else if (pageCode.equals("task-card")) {
            try {

                net.sf.json.JSONArray groupDimension = baseAtmcService.getGroupDimension();
                JSONArray cardGroupGroups = JSONUtil.parseArray(JSONUtil.toJsonStr(groupDimension));
                try {
                    LanguageUtil.processLocaleLanguage(cardGroupGroups, locale);
                } catch (IllegalAccessException e) {
                    log.error("processLocaleLanguage error" + cardGroupGroups);
                }
                List<WSConditionVO> conditionVOS = cardGroupGroups.stream().map(v -> {
                    WSConditionVO wsConditionVO = JSONUtil.toBean(JSONUtil.toJsonStr(v), WSConditionVO.class);
                    CardGroupGroup cardGroupGroup = JSONUtil.toBean(JSONUtil.toJsonStr(v), CardGroupGroup.class);
                    wsConditionVO.setAliasName(cardGroupGroup.getName());
                    wsConditionVO.setAliasCode(cardGroupGroup.getCode());
                    wsConditionVO.setSourceType(SourceType.system);
                    wsConditionVO.setDefaultEffective(cardGroupGroup.isDefaultEffective());
                    wsConditionVO.setEnable(true);

                    return wsConditionVO;
                }).collect(Collectors.toList());

                map.put("defaultCondition", conditionVOS);
            } catch (Exception e) {
                log.error("call atmc group record error,", e);
            }
        }
        return map;
    }

    private JSONArray generatedDefaultGroup() {
        net.sf.json.JSONArray groupDimension = baseAtmcService.getGroupDimension();
        JSONArray result = new JSONArray();
        for (Object o : groupDimension) {
            JSONObject jsonObject = JSONUtil.parseObj(o);
            Object lang = jsonObject.get("lang");
            JSONObject langJson = JSONUtil.parseObj(lang);
            Object name = langJson.get("name");
            JSONObject nameJson = JSONUtil.parseObj(name);
            String locale = LocaleContextHolder.getLocale().toString();
            if (locale != null) {
                String nameStr = nameJson.getStr(locale);
                jsonObject.set("name", nameStr);
            } else {
                String zh_cn = nameJson.getStr("zh_CN");
                jsonObject.set("name", zh_cn);
            }
            jsonObject.set("enabled", true);
            if ("groupByProjectType".equals(jsonObject.getStr("code"))) {
                jsonObject.set("defaultEffective", true);
            } else {
                jsonObject.set("defaultEffective", false);
            }
            result.add(jsonObject);
        }
        return result;
    }

    /**
     * @param user
     * @return
     */
    @GetMapping(value = "groupRecord/{pageCode}")
    public ResponseEntity<?> getGroupRecordPageCode(@PathVariable("pageCode") String oriPageCode, @RequestAttribute(value = GlobalConstant.AUTH_USER) AuthoredUser user) {
        Map<String, Object> result = new HashMap<>();
        String pageCode;
        Set<String> removeKey = new HashSet<>();
        boolean isTask = oriPageCode.equals("task");
        if (isTask || oriPageCode.equals("team-tasks")) {
            pageCode = "task-card";
            if (oriPageCode.equals("task")) {

                removeKey.add("groupByMember");
                removeKey.add("groupByDepartment");
            }
        } else if (oriPageCode.equals("project") || oriPageCode.equals("team-projects")) {
            pageCode = "project-card";
            if (oriPageCode.equals("project")) {
                removeKey.add("groupByMember");
                removeKey.add("groupByDepartment");
            }
        } else {
            return ResponseEntityWrapper.wrapperOk(result);
        }
        Map<String, Object> groupConfig = new HashMap<>();
        try {
            groupConfig = baseThemeMapService.getQueryCondition(pageCode, "groups");
        } catch (Exception e) {
            log.error("KM business Error: {}", e.getMessage());
            groupConfig = generatedDefaultGroupV2(pageCode);
        }
        JSONObject jsonObject = new JSONObject(groupConfig);
        // JSONArray condition = jsonObject.getJSONArray("condition");
        JSONArray defaultCondition = jsonObject.getJSONArray("defaultCondition");

        List<JSONObject> objects = convertCondition(result, removeKey, defaultCondition);
        ArrayList<JSONObject> jsonObjects = Lists.newArrayList(objects);
        result.put("options", jsonObjects);

        JSONArray condition = jsonObject.getJSONArray("condition");

        List<JSONObject> objectsCondition = convertCondition(result, removeKey, condition);
        result.put("allOptions", objectsCondition);
        // 原先的逻辑是只能选择一条，这边兼容历史数据
        try {

            CustomTaskCardGroupData cardGroupData;
            if (isTask) {
                cardGroupData = taskCardGroupService.getBaseMapper().selectOne(new LambdaQueryWrapper<CustomTaskCardGroupData>().eq(CustomTaskCardGroupData::getUserId, user.getUserId()).eq(CustomTaskCardGroupData::getTenantId, user.getTenantId()).and(i -> i.eq(CustomTaskCardGroupData::getPageCode, "task").or().isNull(CustomTaskCardGroupData::getPageCode)).orderByDesc(CustomTaskCardGroupData::getCheckTime).last("LIMIT 1"));

            } else {
                cardGroupData = taskCardGroupService.getBaseMapper().selectOne(new LambdaQueryWrapper<CustomTaskCardGroupData>().eq(CustomTaskCardGroupData::getUserId, user.getUserId()).eq(CustomTaskCardGroupData::getTenantId, user.getTenantId()).eq(CustomTaskCardGroupData::getPageCode, oriPageCode).orderByDesc(CustomTaskCardGroupData::getCheckTime).last("LIMIT 1"));

            }
            if (cardGroupData != null) {
                String memoryConfig = cardGroupData.getMemoryConfig();
                if (JSONUtil.isTypeJSONObject(memoryConfig)) {
                    result.put("effective", JSONUtil.parseObj(memoryConfig));
                } else {
                    if (oriPageCode.equals("task")) {
                        JSONObject jsonObject1 = new JSONObject();
                        jsonObject1.putIfAbsent("model", "normal");
                        JSONArray value = new JSONArray();
                        JSONObject e = new JSONObject();
                        value.add(e);
                        e.putIfAbsent("code", cardGroupData.getGroupKey());
                        e.putIfAbsent("name", cardGroupData.getGroupName());
                        e.putIfAbsent("sort", "asc");
                        jsonObject1.putIfAbsent("normalValue", value);
                        result.put("effective", jsonObject1);
                    } else {
                        result.put("effective", Collections.emptyMap());
                    }

                }
            } else {
                result.put("effective", Collections.emptyMap());
            }
        } catch (Exception e) {
            log.error("custom group record data parse error,", e);
        }
        return ResponseEntityWrapper.wrapperOk(result);

    }

    private List<JSONObject> convertCondition(Map<String, Object> result, Set<String> removeKey, JSONArray defaultCondition) {
        if (CollectionUtils.isNotEmpty(defaultCondition)) {

            List<JSONObject> normalList = defaultCondition.stream().map(e -> new JSONObject(JSONUtil.toJsonStr(e))).filter(e -> "true".equals(e.getStr("enable"))).collect(Collectors.toList());
            normalList.forEach(o -> {
                o.putIfAbsent("normal", true);

                if ("groupByProjectTime".equals(o.getStr("aliasCode"))) {
                    JSONArray pointInTime = o.getJSONArray("pointInTime");
                    JSONArray dimensionality = o.getJSONArray("dimensionality");
                    JSONObject jsonObject1 = new JSONObject();
                    jsonObject1.putIfAbsent("time", pointInTime);
                    jsonObject1.putIfAbsent("date", dimensionality);
                    o.putIfAbsent("configs", jsonObject1);
                }
            });
            List<JSONObject> objects = normalList.stream().filter(e -> !removeKey.contains(e.getStr("aliasCode"))).map(o -> {
                JSONObject object = new JSONObject();
                String aliasCode = o.getStr("aliasCode");
                object.putIfAbsent("code", aliasCode);
                String aliasName = o.getStr("aliasName");
                object.putIfAbsent("name", aliasName);
                String sourceType = o.getStr("sourceType");
                object.putIfAbsent("type", sourceType);
                Boolean defaultGroup = o.getBool("defaultGroup");
                object.putIfAbsent("defaultEffective", defaultGroup);
                String sort = o.getStr("sort");
                object.putIfAbsent("sort", sort);
                JSONObject configs = o.getJSONObject("configs");
                if (configs != null) {
                    object.putIfAbsent("configs", configs);
                }
                return object;
            }).collect(Collectors.toList());
            return objects;
        } else {
            return Collections.emptyList();
        }
    }


    @PostMapping(value = "groupRecord")
    public ResponseEntity<?> groupRecord(@RequestBody JSONObject jsonObject, @RequestAttribute(value = GlobalConstant.AUTH_USER) AuthoredUser user) {
        String pageCode = jsonObject.getStr("pageCode");
        CustomTaskCardGroupData customTaskCardGroupData = taskCardGroupService.getBaseMapper().selectOne(new LambdaQueryWrapper<CustomTaskCardGroupData>().eq(CustomTaskCardGroupData::getUserId, user.getUserId()).eq(CustomTaskCardGroupData::getPageCode, pageCode).eq(CustomTaskCardGroupData::getTenantId, user.getTenantId()).orderByDesc(CustomTaskCardGroupData::getCheckTime).last("LIMIT 1"));
        if (customTaskCardGroupData == null || customTaskCardGroupData.getPageCode() == null) {
            CustomTaskCardGroupData insertData = new CustomTaskCardGroupData();
            insertData.setCheckTime(LocalDateTime.now());
            insertData.setUserId(user.getUserId());
            insertData.setTenantId(user.getTenantId());
            insertData.setPageCode(pageCode);
            insertData.setMemoryConfig(JSONUtil.toJsonStr(jsonObject.getJSONObject("data"), 0));
            taskCardGroupService.save(insertData);
        } else {
            taskCardGroupService.update(null, new LambdaUpdateWrapper<CustomTaskCardGroupData>().eq(CustomTaskCardGroupData::getUserId, user.getUserId()).eq(CustomTaskCardGroupData::getTenantId, user.getTenantId()).eq(CustomTaskCardGroupData::getPageCode, pageCode).set(CustomTaskCardGroupData::getMemoryConfig, JSONUtil.toJsonStr(jsonObject.getJSONObject("data"), 0)).set(CustomTaskCardGroupData::getCheckTime, LocalDateTime.now()));
        }
        return ResponseEntityWrapper.wrapperOk();

    }


    /**
     * 获取所有系统菜单
     *
     * @return
     */
    @GetMapping(value = "/getSystemMenu", produces = "application/json;charset=UTF-8")
    public ResponseEntity<?> getSystemMenu() {
        return ResponseEntityWrapper.wrapperOk(menuService.getAllSystemMenus());
    }


    /**
     * 获取用户级权限的系统菜单和模组
     *
     * @return
     */
    @GetMapping(value = "/v1/getAuthorityMenu", produces = "application/json;charset=UTF-8")
    public ResponseEntity<?> getAuthorityMenuOld(@RequestAttribute(value = GlobalConstant.AUTH_USER) AuthoredUser user) {
        return ResponseEntityWrapper.wrapperOk(menuService.getAuthorityMenu(user));
    }


    /**
     * 获取用户级权限的系统菜单和模组
     *
     * @return
     */
    @PostMapping(value = "/getAuthorityMenu", produces = "application/json;charset=UTF-8")
    public ResponseEntity<?> getAuthorityMenu(@RequestAttribute(value = GlobalConstant.AUTH_USER) AuthoredUser user, @RequestBody List<MenuAuthorityReq> menuList) {
        return ResponseEntityWrapper.wrapperOk(menuService.getAuthorityMenu(user, menuList));
    }


}
