package com.digiwin.athena.base.presentation.server.web.trial;

import com.digiwin.athena.appcore.util.JsonUtils;
import com.digiwin.athena.appcore.util.ResponseEntityWrapper;
import com.digiwin.athena.base.application.meta.request.trial.scene.TrialSceneReq;
import com.digiwin.athena.base.application.meta.request.trial.scene.TrialSceneResp;
import com.digiwin.athena.base.application.meta.request.trial.sceneStep.TrialSceneStepBindReq;
import com.digiwin.athena.base.application.service.trial.TrialSceneService;
import com.digiwin.athena.base.application.service.trial.TrialUserSceneService;
import com.digiwin.athena.base.infrastructure.meta.po.trial.TrialSceneData;
import lombok.extern.slf4j.Slf4j;
import net.sf.json.JSONObject;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;

import java.util.List;

/**
 * 描述
 * 场景的相关操作
 *
 * @author gonghongxing
 * @version 1.0
 * @since 2021/09/03 16:47:48
 */
@Slf4j
@RestController
@RequestMapping(value = "/api/trial/scene")
public class TrialSceneController {
    private final TrialSceneService trialSceneService;

    private final TrialUserSceneService trialUserSceneService;

    /**
     * 构造函数注入对象
     *
     * @param trialSceneService trialSceneService
     */
    public TrialSceneController(TrialSceneService trialSceneService,TrialUserSceneService trialUserSceneService) {
        this.trialSceneService = trialSceneService;
        this.trialUserSceneService = trialUserSceneService;
    }

    /**
     * 新增场景
     *
     * @return ResponseEntity
     */
    @PostMapping(value = "/add", produces = "application/json;charset=UTF-8")
    public ResponseEntity<?> add(@RequestBody TrialSceneReq req) {
        Long result = trialSceneService.add(req);
        return ResponseEntityWrapper.wrapperOk(result);
    }

    /**
     * 修改场景
     *
     * @return ResponseEntity
     */
    @PostMapping(value = "/modify", produces = "application/json;charset=UTF-8")
    public ResponseEntity<?> modify(@RequestBody TrialSceneReq req) {

        int result = trialSceneService.modify(req);
        return ResponseEntityWrapper.wrapperOk(result);
    }

    /**
     * 删除场景
     *
     * @return ResponseEntity
     */
    @GetMapping(value = "/delete/{id}", produces = "application/json;charset=UTF-8")
    public ResponseEntity<?> deleteById(@PathVariable("id") long id) {

        int result = trialSceneService.deleteById(id);
        return ResponseEntityWrapper.wrapperOk(result);
    }

    /**
     * 查询场景列表
     *
     * @return ResponseEntity
     */
    @GetMapping(value = "/queryBySceneKey", produces = "application/json;charset=UTF-8")
    public ResponseEntity<?> queryBySceneKey(@RequestParam(value = "sceneKey", required = false) String sceneKey) {
        TrialSceneReq req = new TrialSceneReq();
        req.setSceneKey(sceneKey);
        List<TrialSceneResp> respList = trialSceneService.list(req);
        return ResponseEntityWrapper.wrapperOk(respList);
    }

    /**
     * 场景与步骤绑定
     *
     * @return ResponseEntity
     */
    @PostMapping(value = "/bindSteps", produces = "application/json;charset=UTF-8")
    public ResponseEntity<?> bindSteps(@RequestBody TrialSceneStepBindReq req) {
        trialSceneService.bindSteps(req);
        return ResponseEntityWrapper.wrapperOk();
    }

    /**
     * 查询场景场景以及场景绑定的步骤列表
     *
     * @return ResponseEntity
     */
    @GetMapping(value = "/queryWithSteps", produces = "application/json;charset=UTF-8")
    public ResponseEntity<?> queryWithSteps(@RequestParam(value = "sceneKey", required = false) String sceneKey) {
        TrialSceneReq req = new TrialSceneReq();
        req.setSceneKey(sceneKey);
        //查询出对应的场景
        TrialSceneResp result = trialSceneService.listWithSteps(req);
        return ResponseEntityWrapper.wrapperOk(result);
    }


    /**
     * 场景明细详情
     *
     * @return ResponseEntity
     */
    @GetMapping(value = "/detail/{sceneKey}", produces = "application/json;charset=UTF-8")
    public ResponseEntity<?> detail(@PathVariable("sceneKey") String sceneKey) {

        TrialSceneData data  = trialSceneService.sceneDetail(sceneKey);
        return ResponseEntityWrapper.wrapperOk(JsonUtils.jsonToObject(data.getDetail(), JSONObject.class));
    }


    /**
     * 删除场景（包括被用户使用的场景）
     *
     * @return ResponseEntity
     */
    @GetMapping(value = "/delete/user/{id}", produces = "application/json;charset=UTF-8")
    public ResponseEntity<?> deleteSceneById(@PathVariable("id") long id) {

        trialUserSceneService.deleteBindSceneBySceneId(id);
        int result = trialSceneService.deleteById(id);
        return ResponseEntityWrapper.wrapperOk(result);
    }

}
