package com.digiwin.athena.base.presentation.server.web.importstatistics;

import com.digiwin.athena.appcore.auth.GlobalConstant;
import com.digiwin.athena.appcore.auth.domain.AuthoredUser;
import com.digiwin.athena.appcore.domain.log.LogDto;
import com.digiwin.athena.appcore.util.JsonUtils;
import com.digiwin.athena.atdm.importstatistics.common.api.DapResponse;
import com.digiwin.athena.atdm.importstatistics.dto.ActionInfoDTO;
import com.digiwin.athena.atdm.importstatistics.dto.ActivityStatisticsDTO;
import com.digiwin.athena.atdm.importstatistics.dto.BatchImportListRespDTO;
import com.digiwin.athena.atdm.importstatistics.dto.DownloadBaseDataDTO;
import com.digiwin.athena.atdm.importstatistics.dto.DownloadBaseDataParamDTO;
import com.digiwin.athena.atdm.importstatistics.dto.DownloadExportFileParamDTO;
import com.digiwin.athena.atdm.importstatistics.dto.DownloadTemplateDTO;
import com.digiwin.athena.atdm.importstatistics.dto.DownloadTemplateReqDTO;
import com.digiwin.athena.atdm.importstatistics.dto.ErrorTableDTO;
import com.digiwin.athena.atdm.importstatistics.dto.ExportBasicDataRespDTO;
import com.digiwin.athena.atdm.importstatistics.dto.ExportStatisticsDTO;
import com.digiwin.athena.atdm.importstatistics.dto.GetRecordsParamDTO;
import com.digiwin.athena.atdm.importstatistics.dto.ImportStatisticsDTO;
import com.digiwin.athena.atdm.importstatistics.dto.UploadParamDTO;
import com.digiwin.athena.atdm.importstatistics.dto.UserStatisticsDTO;
import com.digiwin.athena.base.application.service.importstatistics.BaseDataEntryApplicationServiceImpl;
import com.digiwin.athena.base.infrastructure.manager.abt.AbtService;
import com.digiwin.athena.base.sdk.common.application.util.MessageUtil;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.HttpStatus;
import org.springframework.web.bind.annotation.CrossOrigin;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestAttribute;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.ResponseBody;
import org.springframework.web.bind.annotation.RestController;
import org.springframework.web.multipart.MultipartFile;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.util.List;
import java.util.Set;

/**
 * @ClassName BaseDataEntryController
 * @Description 基础数据录入接口层
 * @Author zhuangli
 * @Date 2021/4/1 15:23
 * @Version 1.0
 **/
@RestController
@RequestMapping("baseDataEntry")
@CrossOrigin
@Slf4j
public class BaseDataEntryInterface {

    @Autowired
    BaseDataEntryApplicationServiceImpl baseDataEntryService;

    private final String LOCALE = "locale";

    @Autowired
    AbtService abtService;

    @PostMapping("upload")
    @ResponseBody
    public DapResponse upload(@RequestParam String actionId, @RequestParam String activityId, MultipartFile file) {
        UploadParamDTO uploadParam = new UploadParamDTO();
        uploadParam.setActionId(actionId);
        uploadParam.setActivityId(activityId);
        return baseDataEntryService.dealUpload(uploadParam, file);
    }

    @PostMapping("uploadBatch")
    @ResponseBody
    public DapResponse uploadBatch(@RequestParam String actionId, @RequestParam String activityId, MultipartFile[] files, @RequestParam(required = false) String actionInfo, @RequestParam(required = false) Set<String> requiredFields) {
        DapResponse response;
        UploadParamDTO uploadParam = new UploadParamDTO();
        uploadParam.setActionId(actionId);
        uploadParam.setActivityId(activityId);
        if (CollectionUtils.isNotEmpty(requiredFields)) {
            LogDto logDto = new LogDto("单头必填项入参:", JsonUtils.objectToString(requiredFields));
            log.info(logDto.toString());
            uploadParam.setRequiredFields(requiredFields);
        }
        if (StringUtils.isNotBlank(actionInfo)) {
            uploadParam.setActionInfo(JsonUtils.jsonToObject(actionInfo, ActionInfoDTO.class));
        }
        // 循环导入多个Excel
        for (MultipartFile file : files) {
            response = baseDataEntryService.dealUpload(uploadParam, file);
            // 若存在导入失败的文件，则直接返回
            if (!response.getStatus().equals(HttpStatus.OK.value())) {
                return response;
            }
        }
        return DapResponse.ok(MessageUtil.getMessage("delivery.success"));
    }


    /**
     * 批量异步上传文件
     *
     * @param paramDTOS 多个上传参数
     * @return void
     */
    @PostMapping("uploadAsync")
    @ResponseBody
    public DapResponse uploadAsync(@RequestBody List<UploadParamDTO> paramDTOS) {
        baseDataEntryService.uploadAsync(paramDTOS);
        return DapResponse.ok();
    }


    @PostMapping("republish")
    public DapResponse republish(@RequestBody UploadParamDTO uploadParamDTO, @RequestAttribute(value = GlobalConstant.AUTH_USER, required = false) AuthoredUser user) {
        final String userId = String.valueOf(user.getSid());
        uploadParamDTO.setUserId(userId);
        baseDataEntryService.republish(uploadParamDTO);
        return DapResponse.ok();
    }

    @PostMapping("downloadErrorTable/{masterId:.+}")
    public void downloadErrorTable(@PathVariable String masterId, @RequestBody DownloadTemplateDTO downloadTemplateDTO, HttpServletRequest request, HttpServletResponse response) {
        String locale = request.getHeader(LOCALE);
        baseDataEntryService.downloadErrorTable(masterId, locale, downloadTemplateDTO.getRequiredFields(), response);
    }


    @GetMapping("downloadErrorTable/{masterId}")
    public void downloadErrorTable(@PathVariable String masterId, HttpServletRequest request, HttpServletResponse response) {
        String locale = request.getHeader(LOCALE);
        baseDataEntryService.downloadErrorTable(masterId, locale, response);
    }

    @PostMapping("downloadTemplate/{actionId:.+}")
    public void downloadTemplate(@PathVariable String actionId, @RequestBody DownloadTemplateDTO downloadTemplateDTO, HttpServletRequest request, HttpServletResponse response) {
        String fileName = downloadTemplateDTO.getFileName();
        String locale = request.getHeader(LOCALE);
        baseDataEntryService.downloadTemplate(actionId, locale, fileName, downloadTemplateDTO, response);
    }

    /**
     * 批量下载模板
     *
     * @param params   下载模板参数
     * @param request  请求
     * @param response void
     */
    @PostMapping("downloadTemplateBatch")
    public void downloadTemplate(@RequestBody List<DownloadTemplateReqDTO> params, HttpServletRequest request, HttpServletResponse response) {
        baseDataEntryService.downloadTemplateBatch(params, response, request.getHeader(LOCALE));
    }

    @GetMapping("downloadTemplate/{actionId:.+}")
    public void downloadTemplate(@PathVariable String actionId, HttpServletRequest request, HttpServletResponse response) {
        String locale = request.getHeader(LOCALE);
        baseDataEntryService.downloadTemplate(actionId, locale, "", response);
    }

    @Deprecated
    @PostMapping("downloadBaseData/{actionId:.+}")
    public void downloadBaseData(@PathVariable String actionId, @RequestBody DownloadBaseDataDTO downloadBaseDataDTO, HttpServletRequest request, HttpServletResponse response) {
        String loc = request.getHeader(LOCALE);
        String locale = StringUtils.isEmpty(loc) ? "zh_CN" : loc;
        baseDataEntryService.downloadBaseData(actionId, downloadBaseDataDTO, locale, response);
    }

    @GetMapping("getErrorTable/{masterId}")
    public DapResponse getErrorTable(@PathVariable String masterId, HttpServletRequest request) {
        String locale = request.getHeader(LOCALE);
        try {
            ErrorTableDTO errorTableDTO = baseDataEntryService.getErrorTableByMasterId(masterId, locale);
            return DapResponse.ok(errorTableDTO);
        } catch (Exception e) {
            log.error("获取文件异常", e);
            return DapResponse.failed(MessageUtil.getMessage("basedata.nofile"));
        }

    }

    @GetMapping("getActivityStatistics/{activityCode}")
    public DapResponse getActivityStatistics(@PathVariable String activityCode, @RequestAttribute(value = GlobalConstant.AUTH_USER, required = false) AuthoredUser user) {
        final String userId = String.valueOf(user.getSid());
        ActivityStatisticsDTO activityStatisticsDTO = baseDataEntryService.getActivityStatistics(activityCode, userId);
        return DapResponse.ok(activityStatisticsDTO);
    }

    @GetMapping("getImportStatistics/{masterId}")
    public DapResponse getImportStatistics(@PathVariable String masterId) {
        ImportStatisticsDTO importStatisticsDTO = baseDataEntryService.getImportStatistics(masterId);
        return DapResponse.ok(importStatisticsDTO);
    }

    @GetMapping("getProcessingNum")
    public DapResponse getProcessingNum(@RequestAttribute(value = GlobalConstant.AUTH_USER, required = false) AuthoredUser user) {
        final String userId = String.valueOf(user.getSid());
        UserStatisticsDTO userStatisticsDTO = baseDataEntryService.getProcessingNum(userId, user.getTenantId());
        return DapResponse.ok(userStatisticsDTO);
    }

    @GetMapping("getRecords")
    @Deprecated
    public DapResponse getRecords(@RequestAttribute(value = GlobalConstant.AUTH_USER, required = false) AuthoredUser user, HttpServletRequest request) {
        String locale = request.getHeader(LOCALE);
        List<ImportStatisticsDTO> importStatisticsDTOList = baseDataEntryService.getRecordsByUserId(user, locale, null);
        return DapResponse.ok(importStatisticsDTOList);
    }

    @PostMapping("getRecords")
    public DapResponse getRecords(@RequestAttribute(value = GlobalConstant.AUTH_USER, required = false) AuthoredUser user, HttpServletRequest request, @RequestBody GetRecordsParamDTO getRecordsParamDTO) {
        String locale = request.getHeader(LOCALE);
        List<ImportStatisticsDTO> importStatisticsDTOList = baseDataEntryService.getRecordsByUserId(user
                , locale
                , getRecordsParamDTO);
        return DapResponse.ok(importStatisticsDTOList);
    }

    @PostMapping(value = {"downloadBaseData", "open/downloadBaseData"})
    public void downloadBaseData(@RequestBody DownloadBaseDataParamDTO downloadBaseDataParamDTO, HttpServletRequest request, HttpServletResponse response) {
        baseDataEntryService.downloadBaseData(downloadBaseDataParamDTO, response);
    }

    /**
     * 发起下载
     *
     * @param downloadBaseDataParamDTO
     * @param request
     * @param response
     * @return
     */
    @PostMapping("startDownload")
    public DapResponse startDownload(@RequestBody DownloadBaseDataParamDTO downloadBaseDataParamDTO, HttpServletRequest request, HttpServletResponse response) {
        try {
            if (Boolean.TRUE.equals(baseDataEntryService.startDownload(downloadBaseDataParamDTO))) {
                return DapResponse.ok();
            } else {
                return DapResponse.failed(MessageUtil.getMessage("delivery.exportDataError"));
            }
        } catch (Exception e) {
            log.error("baseDataEntryStartDownloadEx", e);
            return DapResponse.failed(MessageUtil.getMessage("delivery.exportDataError"));
        }
    }


    /**
     * 发起批量异步下载
     *
     * @param downloadBaseDataParamDTO 多个下载参数
     * @return void
     */
    @PostMapping("startDownloadBatch")
    public DapResponse startDownloadBatch(@RequestBody List<DownloadBaseDataParamDTO> downloadBaseDataParamDTO) {
        baseDataEntryService.startDownloadBatch(downloadBaseDataParamDTO);
        return DapResponse.ok();
    }


    /**
     * 查询导出记录
     *
     * @param actionId 导出actionId
     * @return 导出可选表头
     */
    @GetMapping("getTableHeaders")
    public DapResponse getTableHeaders(String actionId) {
        return DapResponse.ok(baseDataEntryService.getTableHeader(actionId));
    }


    /**
     * 查询导出记录
     *
     * @return 批量导入列表页面数据
     */
    @GetMapping("/import/basicDataList")
    public DapResponse getImportBasicDataList() {
        List<BatchImportListRespDTO> importAndExportConfig = baseDataEntryService.getImportAndExportConfig();
        return DapResponse.ok(importAndExportConfig);
    }


    /**
     * 查询导出记录
     *
     * @return 批量导出列表页面数据
     */
    @GetMapping("/export/basicDataList")
    public DapResponse getExportBasicDataList() {
        List<ExportBasicDataRespDTO> exportBasicDataList = baseDataEntryService.getExportBasicDataList();
        return DapResponse.ok(exportBasicDataList);
    }


    /**
     * 下载导出文件
     *
     * @param downloadExportFileParamDTO
     * @param request
     * @param response
     */
    @PostMapping("downloadExportFile")
    public void downloadExportFile(@RequestBody DownloadExportFileParamDTO downloadExportFileParamDTO, HttpServletRequest request, HttpServletResponse response) {
        baseDataEntryService.downloadExportFile(downloadExportFileParamDTO, response);
    }

    /**
     * 查询导出记录
     *
     * @param user
     * @param request
     * @param getRecordsParamDTO
     * @return
     */
    @PostMapping("queryExportRecords")
    public DapResponse queryExportRecords(@RequestAttribute(value = GlobalConstant.AUTH_USER, required = false) AuthoredUser user, HttpServletRequest request, @RequestBody GetRecordsParamDTO getRecordsParamDTO) {
        String locale = request.getHeader(LOCALE);
        List<ExportStatisticsDTO> exportStatisticsDTOList = baseDataEntryService.getExportRecordsByUserId(user, locale, getRecordsParamDTO);
        return DapResponse.ok(exportStatisticsDTOList);
    }

    /**
     * 查询导出中数量
     *
     * @param user
     * @return
     */
    @GetMapping("getExportingNum")
    public DapResponse getExportingNum(@RequestAttribute(value = GlobalConstant.AUTH_USER, required = false) AuthoredUser user
            , String type
            , @RequestParam(required = false,value = "types") List<String> types) {
        final String userId = String.valueOf(user.getSid());
        UserStatisticsDTO userStatisticsDTO = baseDataEntryService.getExportingNum(userId, user.getTenantId(), type, types);
        return DapResponse.ok(userStatisticsDTO);
    }


    /**
     * 子页面下载使用接口
     *
     * @param masterId
     * @return
     */
    @GetMapping("retryDownload/{masterId}")
    public DapResponse retryDownload(@PathVariable(value = "masterId") String masterId) {
        DownloadExportFileParamDTO downloadExportFileParamDTO = new DownloadExportFileParamDTO();
        downloadExportFileParamDTO.setMasterId(masterId);
        baseDataEntryService.retryDownload(downloadExportFileParamDTO);
        return DapResponse.ok();
    }

}
