package com.digiwin.athena.base.presentation.server.web.attachment;

import com.digiwin.athena.appcore.util.ResponseEntityWrapper;
import com.digiwin.athena.base.application.meta.request.attachment.AttachUploadedCountDTO;
import com.digiwin.athena.base.application.meta.request.attachment.AttachmentDownloadReqDTO;
import com.digiwin.athena.base.application.meta.request.attachment.AttachmentReqDTO;
import com.digiwin.athena.base.application.meta.request.attachment.DeleteAttachmentReqDTO;
import com.digiwin.athena.base.application.meta.request.attachment.UploadOrDeleteAttachmentReqDTO;
import com.digiwin.athena.base.application.meta.request.attachment.UploadParamDTO;
import com.digiwin.athena.base.application.service.attachment.AttachmentService;
import com.digiwin.athena.base.sdk.aam.application.meta.request.QueryAttachmentReqDTO;
import com.digiwin.athena.base.sdk.aam.application.meta.response.AttachmentRespDTO;
import com.digiwin.athena.base.sdk.aam.application.meta.validator.ValidateGroup;
import com.digiwin.athena.base.sdk.common.application.annotation.ElementNotBlank;
import com.jugg.agile.framework.core.dapper.meta.Dapper;
import io.swagger.annotations.Api;
import org.apache.commons.collections4.CollectionUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.ResponseEntity;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;

import javax.validation.Valid;
import java.util.List;

@Api(tags = "附件CRUD、上传、下载等")
@RestController
@RequestMapping(value = "/api/aam/v1")
@Validated
public class AttachmentController {
    @Autowired
    private AttachmentService attachmentService;

    /**
     * 添加附件信息
     *
     * @param attachmentFile 附件信息
     * @return
     */
    @PostMapping(value = {"/uploadAttachment", "/open/uploadAttachment"})
    public ResponseEntity<?> uploadAttachmentFile(@RequestBody @Validated(ValidateGroup.Save.class) AttachmentReqDTO attachmentFile) {

        // 校验当前上传的数量是否超过允许的数量
        if (attachmentService.checkUploadCount(attachmentFile)) {
            return ResponseEntityWrapper.wrapperOk(-1);
        }
        attachmentService.saveAttachment(attachmentFile);
        return ResponseEntityWrapper.wrapperOk();
    }

    /**
     * 根据id、category（可为空）、categoryId（可为空）删除附件信息
     *
     * @param attachmentFile 要删除的附件信息
     * @return
     */
    @PostMapping(value = {"/deleteAttachment", "/open/deleteAttachment"})
    public ResponseEntity<?> deleteAttachmentFile(@RequestBody @Validated(ValidateGroup.Delete.class) AttachmentReqDTO attachmentFile) {
        attachmentService.deleteAttachment(attachmentFile);
        return ResponseEntityWrapper.wrapperOk();
    }

    /**
     * 根据活动Id、租户id、目录列表和rowDataKey列表查询附件文档信息
     *
     * @param tenantId   租户id，优先使用tenantId，为null则使用token解析出的tenantId
     * @param taskId
     * @param projectId
     * @param category
     * @param rowDataKey
     * @return
     */
    @GetMapping("/attachment")
    public ResponseEntity<?> queryAttachmentFile(@RequestParam(value = "tenantId", required = false) String tenantId,
                                                 @RequestParam("taskId") String taskId,
                                                 @RequestParam(value = "projectId", required = false) String projectId,
                                                 @ElementNotBlank @RequestParam("category") List<String> category,
                                                 @ElementNotBlank @RequestParam("rowDataKey") List<String> rowDataKey) {
        List<AttachmentRespDTO> attachmentEntities = attachmentService.queryByTaskIdAndRowDataKeyList(tenantId, taskId, projectId, category, rowDataKey);
        return ResponseEntityWrapper.wrapperOk(attachmentEntities);
    }

    /**
     * 根据活动Id、租户id、目录列表和rowDataKey列表查询附件文档信息
     *
     * @param queryAttachmentReqDTO
     */
    @PostMapping("/attachment")
    public ResponseEntity<?> queryAttachmentFileV2(@Valid @RequestBody QueryAttachmentReqDTO queryAttachmentReqDTO) {
        List<AttachmentRespDTO> attachmentEntities = attachmentService.queryByTaskIdAndRowDataKeyList(queryAttachmentReqDTO);
        return ResponseEntityWrapper.wrapperOk(attachmentEntities);
    }

    /**
     * 上传保存or删除附件信息
     *
     * @param uploadOrDeleteAttachmentReqDTO
     * @return
     */
    @Dapper(skip = true)
    @PostMapping("/uploadOrDeleteAttachment")
    public ResponseEntity<?> uploadOrDeleteAttachment(@RequestBody UploadOrDeleteAttachmentReqDTO uploadOrDeleteAttachmentReqDTO) {
        attachmentService.uploadOrDeleteAttachment(uploadOrDeleteAttachmentReqDTO);
        return ResponseEntityWrapper.wrapperOk();
    }

    /**
     * 删除附件信息
     *
     * @param deleteAttachmentReqDTO
     * @return
     */
    @PostMapping("/deleteAttachmentBatch")
    public ResponseEntity<?> deleteAttachmentBatch(@RequestBody DeleteAttachmentReqDTO deleteAttachmentReqDTO) {
        attachmentService.deleteAttachmentBatch(deleteAttachmentReqDTO);
        return ResponseEntityWrapper.wrapperOk();
    }

    /**
     * 上传附件信息到文档信息
     *
     * @param uploadParamDTO
     * @return
     */
    @Dapper(skip = true)
    @PostMapping("/uploadAndShareAttachmentByUrl")
    public ResponseEntity<?> uploadAndShareAttachmentByUrl(@RequestBody UploadParamDTO uploadParamDTO) {
        return ResponseEntityWrapper.wrapperOk(attachmentService.uploadAttachmentByUrl(uploadParamDTO));
    }

    /**
     * 上传报表文档
     *
     * @param uploadParamDTO
     * @return
     */
    @Dapper(skip = true)
    @PostMapping("/uploadAgileReport")
    public ResponseEntity<?> uploadAgileReportByUrl(@RequestBody UploadParamDTO uploadParamDTO) {
        return ResponseEntityWrapper.wrapperOk(attachmentService.uploadAgileReport(uploadParamDTO));
    }

    /**
     * 获取报表文档信息
     *
     * @param fileId
     * @return
     */
    @GetMapping("/getAgileReportInfo")
    public ResponseEntity<?> getAgileReportInfo(@RequestParam(value = "fileId") String fileId) {
        return ResponseEntityWrapper.wrapperOk(attachmentService.getFile(fileId));
    }

    /**
     * 删除文档
     *
     * @param fileId
     * @return
     */
    @GetMapping("/deleteAgileReportInfo")
    public ResponseEntity<?> deleteAgileReportInfo(@RequestParam(value = "fileId") String fileId) {
        return ResponseEntityWrapper.wrapperOk(attachmentService.deleteFile(fileId));
    }

    /**
     * 统计已经上传的文档数量
     *
     * @param attachUploadedCountDTO
     * @return
     */
    @PostMapping("/attachUploadedCount")
    public ResponseEntity<?> attachUploadedCount(@RequestBody AttachUploadedCountDTO attachUploadedCountDTO) {
        return ResponseEntityWrapper.wrapperOk(attachmentService.attachUploadedCount(attachUploadedCountDTO));
    }


    /**
     * 加密、水印下载附件
     *
     * @param attachmentDownloadReqDTO
     * @return
     */
    @PostMapping("/download")
    public Object download(@RequestBody AttachmentDownloadReqDTO attachmentDownloadReqDTO) {
        if (CollectionUtils.isNotEmpty(attachmentDownloadReqDTO.getFileIds())) {
            return attachmentService.batchDownload(attachmentDownloadReqDTO);
        } else {
            return attachmentService.download(attachmentDownloadReqDTO);
        }
    }

    /**
     * 下载图片带水印附件
     * 前端图片预览时，前端会先调用下载接口，获取到图片的base64，再进行图片预览
     *
     * @param attachmentDownloadReqDTOList
     * @return
     */
    @PostMapping("/downloadImageWatermarkBase64")
    public Object downloadImageWatermarkBase64(@RequestBody List<AttachmentDownloadReqDTO> attachmentDownloadReqDTOList) {
        return attachmentService.downloadImageWatermarkBase64(attachmentDownloadReqDTOList);
    }

}
