package com.digiwin.athena.base.presentation.server.web.datasource;


import com.digiwin.athena.appcore.auth.GlobalConstant;
import com.digiwin.athena.appcore.auth.domain.AuthoredUser;
import com.digiwin.athena.appcore.util.ResponseEntityWrapper;
import com.digiwin.athena.atdm.adsc.CommonAdscService;
import com.digiwin.athena.atdm.datasource.DataQueryServiceWrapper;
import com.digiwin.athena.atdm.datasource.datasource.process.AtmcBatchDataSourceService;
import com.digiwin.athena.atdm.datasource.domain.QueryResult;
import com.digiwin.athena.atdm.datasource.dto.DataAddTaskInfoDTO;
import com.digiwin.athena.atdm.datasource.dto.DataQueryDTO;
import com.digiwin.athena.atdm.datasource.dto.QueryDataByActionDTO;
import com.digiwin.athena.atdm.datasource.dto.QueryDataByActionIdDTO;
import com.digiwin.athena.atdm.datasource.dto.QueryDataByDataSourceDTO;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.collections4.ListUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestAttribute;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RestController;

import javax.annotation.Resource;
import javax.servlet.http.HttpServletRequest;
import java.util.Map;
import java.util.Objects;

@Slf4j
@RestController
@RequestMapping(value = "/api/atdm/v1/data/")
public class DataQueryController {
    @Autowired
    private CommonAdscService adscService;

    @Resource
    private AtmcBatchDataSourceService atmcBatchDataSourceService;

    @Autowired
    private DataQueryServiceWrapper dataQueryServiceWrapper;


    /**
     * 根据代办的数据来查询获取数据
     *
     * @param request
     * @param dataQueryDTO
     * @return
     */
    @PostMapping("query/by/backlog/withmetadata")
    public ResponseEntity<?> queryWithMetaData(HttpServletRequest request, @RequestBody DataQueryDTO dataQueryDTO) {
        return ResponseEntityWrapper.wrapperOk(dataQueryServiceWrapper.queryWithMetaData(request, dataQueryDTO));
    }

    /**
     * 根据代办的数据来查询获取数据
     *
     * @param request
     * @param dataQueryDTO
     * @return
     */
    @PostMapping("query/by/backlog")
    public ResponseEntity<?> queryByBacklog(HttpServletRequest request, @RequestBody DataQueryDTO dataQueryDTO) {
        return ResponseEntityWrapper.wrapperOk(dataQueryServiceWrapper.queryByBacklog(request, dataQueryDTO));
    }


    /**
     * 根据数据源来查询获取数据
     *
     * @param request
     * @param dataQueryDTO
     * @return
     */
    @PostMapping(value = {"query/page/by/dataset", "open/query/page/by/dataset"})
    public ResponseEntity<?> queryPageDataByDatasource(HttpServletRequest request, @RequestBody DataQueryDTO dataQueryDTO) {
        return ResponseEntityWrapper.wrapperOk(dataQueryServiceWrapper.queryPageDataByDatasource(request, dataQueryDTO));
    }

    /**
     * 根据数据源来查询获取数据（筛选/排序/分页）
     * 注意：暂时只支持SCS相关api
     *
     * @param request
     * @param dataQueryDTO
     * @return
     */
    @PostMapping("query/page/filterData")
    @Deprecated
    public ResponseEntity<?> queryPageDataWithCondition(HttpServletRequest request, @RequestBody DataQueryDTO dataQueryDTO) {
        dataQueryDTO.getExecuteContext().setDataStateCode(dataQueryDTO.getDataStateCode());
        return queryPageDataByDatasource(request, dataQueryDTO);
    }


    /**
     * 根据数据源来查询获取数据（筛选/排序/分页）-分页筛选项查询
     * 注意：暂时只支持SCS相关api
     *
     * @param dataQueryDTO
     * @return
     */
    @PostMapping("query/filterCondition")
    public ResponseEntity<?> queryFilterCondition(@RequestBody DataQueryDTO dataQueryDTO) {
        dataQueryDTO.getExecuteContext().setDataStateCode(dataQueryDTO.getDataStateCode());
        Map result = adscService.queryConditions(dataQueryDTO.getExecuteContext(), dataQueryDTO.getConditionFeildId(), dataQueryDTO.getSearchInfo());
        return ResponseEntityWrapper.wrapperOk(result);
    }


    /**
     * 根据代办的数据来查询获取数据
     *
     * @param request
     * @param dataQueryDTO
     * @return
     */
    @PostMapping("size/by/backlog")
    public ResponseEntity<?> querySize(HttpServletRequest request, @RequestBody DataQueryDTO dataQueryDTO) {
        return ResponseEntityWrapper.wrapperOk(dataQueryServiceWrapper.querySize(request, dataQueryDTO));
    }

    /**
     * 根据代办的数据来查询获取数据
     *
     * @param request
     * @param dataQueryDTO
     * @return
     */
    @PostMapping("size/by/dataSource")
    public ResponseEntity<?> querySizeByDataSize(HttpServletRequest request, @RequestBody DataQueryDTO dataQueryDTO) {
        return ResponseEntityWrapper.wrapperOk(dataQueryServiceWrapper.querySizeByDataSize(request, dataQueryDTO));
    }

    /**
     * 根据数据源的定义来获取数据
     *
     * @param queryDataByActionVO
     * @return
     */
    @PostMapping("query/by/datasource")
    public ResponseEntity<?> queryByDatasource(HttpServletRequest request, @RequestBody QueryDataByDataSourceDTO queryDataByActionVO, @RequestAttribute(value = GlobalConstant.AUTH_USER, required = false) AuthoredUser user) {
        return ResponseEntityWrapper.wrapperOk(dataQueryServiceWrapper.queryByDatasource(request, queryDataByActionVO, user));
    }

    /**
     * 根据tmACTION来获取数据
     *
     * @param queryDataByActionDTO
     * @return
     */
    @PostMapping(value = {"query/by/action", "open/query/by/action"})
    public ResponseEntity<?> actionData(HttpServletRequest request, @RequestBody QueryDataByActionDTO queryDataByActionDTO, @RequestAttribute(value = GlobalConstant.AUTH_USER, required = false) AuthoredUser user) {
        return ResponseEntityWrapper.wrapperOk(dataQueryServiceWrapper.actionData(request, queryDataByActionDTO, user));
    }

    /**
     * 根据tmACTION来获取数据
     *
     * @param queryDataByActionDTO
     * @return
     */
    @PostMapping(value = {"query/by/actionId", "open/query/by/actionId"})
    public ResponseEntity<?> queryByActionId(HttpServletRequest request, @RequestBody QueryDataByActionIdDTO queryDataByActionDTO, @RequestAttribute(value = GlobalConstant.AUTH_USER, required = false) AuthoredUser user) {
        return ResponseEntityWrapper.wrapperOk(dataQueryServiceWrapper.queryByActionId(request, queryDataByActionDTO, user));
    }


    /**
     * 根据代办的数据来查询获取数据
     *
     * @param request
     * @param dataQueryDTO
     * @return
     */
    @PostMapping("size/by/dataSource/abnormal")
    public ResponseEntity<?> querySizeByDataSizeAbnormal(HttpServletRequest request, @RequestBody DataQueryDTO dataQueryDTO) {
        return ResponseEntityWrapper.wrapperOk(dataQueryServiceWrapper.querySizeByDataSizeAbnormal(request, dataQueryDTO));
    }


    /**
     * @param request            HttpServletRequest
     * @param dataAddTaskInfoDTO dataAddTaskInfoDTO
     * @return ResponseEntity<?>
     * @description FCI：批量处理应用数据，给数据增加待办信息activity__backLog__data
     */
    @PostMapping("process/batch")
    @Deprecated
    public ResponseEntity<?> atmcBathDataSourceServreOld(HttpServletRequest request, @RequestBody DataAddTaskInfoDTO dataAddTaskInfoDTO) {
        return atmcBathDataSourceServ(request, dataAddTaskInfoDTO);
    }

    /**
     * @param request            HttpServletRequest
     * @param dataAddTaskInfoDTO DataQueryDTO
     * @return ResponseEntity<?>
     * @description FCI：批量处理应用数据，给数据增加待办信息activity__backLog__data
     */
    @PostMapping("process/batch/addTaskInfo")
    public ResponseEntity<?> atmcBathDataSourceServ(HttpServletRequest request, @RequestBody DataAddTaskInfoDTO dataAddTaskInfoDTO) {
        if (Objects.isNull(dataAddTaskInfoDTO.getDataSourceProcessor())) {
            throw new IllegalArgumentException("dataSourceProcessor cannot be null");
        }
        if (!"atmcBatchDataService".equals(dataAddTaskInfoDTO.getDataSourceProcessor().getServiceName())) {
            throw new IllegalArgumentException("dataSourceProcessor must be a name of atmcBatchDataService");
        }
        if (CollectionUtils.isEmpty(dataAddTaskInfoDTO.getData())) {
            return ResponseEntityWrapper.wrapperOk(CollectionUtils.emptyCollection());
        }
        long start = System.currentTimeMillis();
        QueryResult queryResult = new QueryResult();
        queryResult.setData(dataAddTaskInfoDTO.getData());
        atmcBatchDataSourceService.handelData(null, dataAddTaskInfoDTO.getExecuteContext(), dataAddTaskInfoDTO.getDataSourceProcessor(), queryResult);
        log.info("[{}]应用调用批量增加待办信息接口,数据量[{}]条,耗时[{}]ms", dataAddTaskInfoDTO.getClientCode(), dataAddTaskInfoDTO.getData().size(), System.currentTimeMillis() - start);
        return ResponseEntityWrapper.wrapperOk(queryResult.getData());
    }
}
