package com.digiwin.athena.base.presentation.server.web.usertrack;

import com.digiwin.athena.appcore.util.ResponseEntityWrapper;
import com.mongodb.client.ListIndexesIterable;
import com.mongodb.client.MongoCursor;
import com.mongodb.client.model.IndexOptions;
import org.apache.commons.collections4.map.HashedMap;
import org.bson.Document;
import org.springframework.data.mongodb.core.MongoTemplate;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;

import javax.annotation.Resource;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;

/**
 * 修复数据的controller，避免每次都需要运维操作
 * 该controller不需要维护（勿删）
 */
@RestController
@RequestMapping(value = "/user/track/repair")
public class RepairUserTrackController {
    @Resource
    private MongoTemplate trackMongoTemplate;

    /**
     * 查询数据的集合信息
     *
     * @return
     */
    @GetMapping("/query/queryCollections")
    public ResponseEntity queryAllCollectionInfo() {
        //获取所有的collection信息
        Set<String> collectionInfo = trackMongoTemplate.getCollectionNames();
        return ResponseEntityWrapper.wrapperOk(collectionInfo);
    }

    /**
     * 获取索引信息
     *
     * @param collectionName 集合信息
     * @return
     */
    @GetMapping("/query/getIndexInfo")
    public ResponseEntity queryIndexInfo(@RequestParam(value = "collection", required = false) String collectionName) {

        ListIndexesIterable<Document> currentIndexes = trackMongoTemplate.getCollection(collectionName).listIndexes();
        MongoCursor<Document> cursor = currentIndexes.iterator();
        Set<Object> indexNames = new HashSet<>();
        while (cursor.hasNext()) {
            Object next = cursor.next();
            indexNames.add(next);
        }
        return ResponseEntityWrapper.wrapperOk(indexNames);

    }


    /**
     * 批量设置多个集合的索引信息
     *
     * @param collectionNames 集合信息
     * @return
     */
    @PostMapping("/handle/batchSetIndex")
    public ResponseEntity batchSetIndex(@RequestParam(value = "collections", required = false) String collectionNames, @RequestBody Document document) {
        String[] collArrs = collectionNames.split(",");
        Map<String, String> result = new HashedMap();
        for (String col : collArrs) {
            IndexOptions indexOptions = new IndexOptions();
            indexOptions.background(true);
            String tempResult = trackMongoTemplate.getCollection(col).createIndex(document, indexOptions);
            result.put(col, tempResult);
        }
        return ResponseEntityWrapper.wrapperOk(result);
    }


    /**
     * 删除索引信息
     *
     * @param collectionName 集合信息
     * @return
     * @Param indexName 索引名称
     */
    @GetMapping("/handle/dropIndex")
    public ResponseEntity dropIndex(@RequestParam(value = "collection", required = false) String collectionName, @RequestParam(value = "indexName", required = false) String indexName) {
        try {
            trackMongoTemplate.getCollection(collectionName).dropIndex(indexName);
        } catch (Exception e) {
            return ResponseEntityWrapper.wrapperOk("删除索引异常");
        }

        return ResponseEntityWrapper.wrapperOk("删除索引成功");
    }

    /**
     * 批量查询并设置索引
     *
     * @param document 集合信息
     * @return
     * @Param indexName 索引名称
     */
    @PostMapping("/handle/batchQuerySetIndex")
    public ResponseEntity batchQuerySetIndex(@RequestBody Document document) {
        //获取所有的collection信息
        Set<String> totalColls = trackMongoTemplate.getCollectionNames();
        Set<String> failColls = new HashSet<>();
        for (String col : totalColls) {
            try {
                IndexOptions indexOptions = new IndexOptions();
                indexOptions.background(true);
                String tempResult = trackMongoTemplate.getCollection(col).createIndex(document, indexOptions);
            } catch (Exception e) {
                failColls.add(col);
            }
        }
        Map<String, Object> result = new HashMap<>();
        result.put("totalColls", totalColls);
        result.put("failColls", failColls);
        return ResponseEntityWrapper.wrapperOk(result);
    }

    /**
     * 获取集合的文档总数
     *
     * @param collectionName 集合信息
     * @return
     * @Param indexName 索引名称
     */
    @PostMapping("/query/count")
    public ResponseEntity countData(@RequestParam(value = "collection", required = false) String collectionName, @RequestBody Document document) {
        if (null == document) {
            return ResponseEntityWrapper.wrapperOk(trackMongoTemplate.getCollection(collectionName).countDocuments());
        }
        return ResponseEntityWrapper.wrapperOk(trackMongoTemplate.getCollection(collectionName).countDocuments(document));
    }

    /**
     * 执行mango的执行计划命令行
     *
     * @return
     */
    @PostMapping(value = "/query/explain", produces = "application/json;charset=UTF-8")
    public ResponseEntity<?> queryExplain(@RequestBody Document document) {
        Document explainDoc = new Document();
        explainDoc.put("explain", document);
        return ResponseEntityWrapper.wrapperOk(trackMongoTemplate.executeCommand(explainDoc));
    }

}
