package com.digiwin.athena.atdm.mongodb;

import com.digiwin.athena.appcore.util.JsonUtils;
import com.mongodb.client.model.IndexOptions;
import org.apache.tomcat.util.buf.StringUtils;
import org.bson.Document;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.mongodb.core.MongoTemplate;
import org.springframework.data.mongodb.core.index.IndexInfo;
import org.springframework.data.mongodb.core.query.Query;
import org.springframework.data.mongodb.core.query.Update;
import org.springframework.data.redis.core.RedisTemplate;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Set;

/**
 * @author：SYQ
 * @date：2021/12/28
 */
@Service
public class CommonMongodbServiceImpl implements CommonMongodbService {

    @Autowired
    private RedisTemplate redisTemplate;

    private static final String LOCK_NAME_SUFFIX = ":createMongoCollection:lock";
    private static final String LOCK_NAME_PREFIX = "atdm:";
    private List<String> existCollectionList = new ArrayList<>();

    /**
     * 创建集合（加锁）
     * @Author：SYQ
     * @Date：2021/12/28 14:52
     */
    @Override
    public void createCollectionLock(MongoTemplate mongoTemplate, String collectionName,Map<String,Object> indexMap){
        //判断内存中集合是否存在，存在不需要创建
        if(existCollectionList.contains(collectionName)){
            return;
        }
        //判断mongoDB中集合是否存在，存在不需要创建
        if (mongoTemplate.collectionExists(collectionName)) {
            existCollectionList.add(collectionName);
            return;
        }

        String tmpLockName = LOCK_NAME_PREFIX + collectionName + LOCK_NAME_SUFFIX;
        DistributeLocker.tryLock30s(redisTemplate, tmpLockName, () -> createCollectionAndIndex(mongoTemplate,collectionName,indexMap));
    }

    /**
     * 创建集合和索引
     * @Author：SYQ
     * @Date：2021/12/28 15:00
     */
    private void createCollectionAndIndex(MongoTemplate mongoTemplate, String collectionName, Map<String,Object> indexMap){
        //创建集合
        mongoTemplate.createCollection(collectionName);
        existCollectionList.add(collectionName);

        //创建索引
        if(indexMap != null && indexMap.size() > 0){
            //索引名称
            String indexName = StringUtils.join(indexMap.keySet(), '_');
            // 该参数为索引的属性配置
            IndexOptions indexOptions = new IndexOptions();
            indexOptions.background(true);
            indexOptions.name(indexName);
            //创建索引
            mongoTemplate.getCollection(collectionName).createIndex(new Document(indexMap),indexOptions);
        }
    }


    /**
     * 插入数据
     * @Author：SYQ
     * @Date：2021/12/28 15:48
     */
    @Override
    public void insert(MongoTemplate mongoTemplate, Map data,String collectionName){
        mongoTemplate.insert(data,collectionName);
    }

    /**
     * 更新
     * @Author：SYQ
     * @Date：2021/12/28 16:26
     */
    @Override
    public void updateFirst(MongoTemplate mongoTemplate, String collectionName,Query query,Update update){
        mongoTemplate.updateFirst(query,update,collectionName);
    }

    /**
     * 执行查询
     * @param collectionName
     * @param query
     * @return
     */
    @Override
    public List<Map> query(MongoTemplate mongoTemplate, String collectionName,Query query){
        return mongoTemplate.find(query, Map.class, collectionName);
    }

    /**
     * 执行查询命令
     * @param jsonCommand
     * @return 查询的结果
     */
    public List<Map> executeQueryCommand(MongoTemplate mongoTemplate, String jsonCommand){
        Map document = mongoTemplate.executeCommand(jsonCommand);
        Map cursor = (Map)document.get("cursor");
        List<Map> mapList = (List<Map>) cursor.get("firstBatch");
        return mapList;
    }

    /**
     * 执行查询命令
     * @param jsonCommand
     * @return 查询的结果
     */
    public List<Map> executeQueryCommand(MongoTemplate mongoTemplate, Map jsonCommand){
        return executeQueryCommand(mongoTemplate, JsonUtils.objectToString(jsonCommand));
    }

    /**
     * 删除索引
     * @Author：SYQ
     * @Date：2022/1/26 13:46
     */
    public void deleteIdexFromCollections(MongoTemplate mongoTemplate, String deleteIndex){
        Set<String> collectionNameSet = mongoTemplate.getCollectionNames();
        if(CollectionUtils.isEmpty(collectionNameSet)){
           return;
        }
        for(String collectionName : collectionNameSet){
            List<IndexInfo> indexList = mongoTemplate.indexOps(collectionName).getIndexInfo();
            List<String> indexNameList = new ArrayList<>();
            if(!CollectionUtils.isEmpty(indexList)){
                for(IndexInfo indexInfo : indexList){
                    indexNameList.add(indexInfo.getName());
                }
            }
            if(indexNameList.contains(deleteIndex)){
                mongoTemplate.indexOps(collectionName).dropIndex(deleteIndex);
            }
        }
    }

    /**
     * 添加索引
     * @Author：SYQ
     * @Date：2022/1/26 14:07
     */
    public void insertIndex(MongoTemplate mongoTemplate, Map<String,Object> indexMap){
        Set<String> collectionNameSet = mongoTemplate.getCollectionNames();
        if(CollectionUtils.isEmpty(collectionNameSet)){
            return;
        }
        for(String collectionName : collectionNameSet){
            //创建索引
            if(indexMap != null && indexMap.size() > 0){
                //索引名称
                String indexName = StringUtils.join(indexMap.keySet(), '_');
                // 该参数为索引的属性配置
                IndexOptions indexOptions = new IndexOptions();
                indexOptions.background(true);
                indexOptions.name(indexName);
                //已存在的索引
                List<IndexInfo> indexList = mongoTemplate.indexOps(collectionName).getIndexInfo();
                List<String> indexNameList = new ArrayList<>();
                if(!CollectionUtils.isEmpty(indexList)){
                    for(IndexInfo indexInfo : indexList){
                        indexNameList.add(indexInfo.getName());
                    }
                }
                if(CollectionUtils.isEmpty(indexNameList) || !indexNameList.contains(indexName)){
                    //创建索引
                    mongoTemplate.getCollection(collectionName).createIndex(new Document(indexMap),indexOptions);
                }
            }
        }
    }


    /**
     * 更新多条数据
     * @Author：SYQ
     * @Date：2021/12/28 16:26
     */
    public void updateMulti(MongoTemplate mongoTemplate, String collectionName,Query query,Update update){
        mongoTemplate.updateMulti(query,update,collectionName);
    }



}
