package com.digiwin.athena.atdm.smartdata;

import com.digiwin.athena.appcore.auth.domain.AuthoredUser;
import com.digiwin.athena.appcore.domain.BaseResultDTO;
import com.digiwin.athena.appcore.util.DataUtils;
import com.digiwin.athena.appcore.util.JsonUtils;
import com.digiwin.athena.atdm.RemoteProperties;
import com.digiwin.athena.atdm.datasource.domain.Action;
import com.digiwin.athena.atdm.smartdata.dto.RetractCompareReqDTO;
import com.digiwin.athena.atdm.util.UriUtils;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.InitializingBean;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.core.ParameterizedTypeReference;
import org.springframework.http.HttpEntity;
import org.springframework.http.HttpHeaders;
import org.springframework.http.HttpMethod;
import org.springframework.http.MediaType;
import org.springframework.http.ResponseEntity;
import org.springframework.stereotype.Service;
import org.springframework.web.client.RestTemplate;

import java.net.URI;
import java.text.MessageFormat;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;

@Service
@Slf4j
public class CommonSmartDataServiceImpl implements CommonSmartDataService, InitializingBean {
    @Autowired
    private RemoteProperties envProperties;

    @Autowired
    private RestTemplate restTemplate;

    private String dataComparisonHttpHost;

    private String executionEngineHttpHost;

    private String dataFootPrintHttpHost;

    @Override
    public void afterPropertiesSet() throws Exception {
        dataComparisonHttpHost = UriUtils.parseHttpHost(envProperties.getDataComparisonUri());
        dataFootPrintHttpHost = UriUtils.parseHttpHost(envProperties.getDataFootPrintUri());
        executionEngineHttpHost = UriUtils.parseHttpHost(envProperties.getExecutionEngineUri());
    }

    @Override
    public HashMap<String, Object> execute(String tenantId, Action action) {
        String url = executionEngineHttpHost + "/restful/service/ExecutionEngine/execute";
        HashMap body = new HashMap();
        body.put("tenantId", tenantId);
        body.put("actionId", action.getActionId());
        Optional<Map<String, Object>> optionalStringObjectMap = DataUtils.mergeMap(action.getParas());
        optionalStringObjectMap.ifPresent(stringObjectMap -> body.put("param", stringObjectMap));
        body.put("eocMap", action.getBusinessUnit());
        HttpHeaders headerMap = new HttpHeaders();
        headerMap.add("Content-Type", "application/json;charset=UTF-8");
        HttpEntity httpEntity = new HttpEntity<Map>(body, headerMap);
        try {
            ResponseEntity<BaseResultDTO<HashMap<String, Object>>> responseEntity = restTemplate.exchange(url, HttpMethod.POST, httpEntity, new ParameterizedTypeReference<BaseResultDTO<HashMap<String, Object>>>() {
            });
            return responseEntity.getBody().getResponseWithException("");
        } catch (Exception e) {
            log.error("[execute] body: {}, error: ", JsonUtils.objectToString(body), e);
            throw e;
        }
    }

    @Override
    public HashMap<String, Object> execute(String tenantId, String actionId, Map<String, Object> paras, Map eoc, Map extendedFields) {
        String url = executionEngineHttpHost + "/restful/service/ExecutionEngine/execute";
        HashMap body = new HashMap();
        body.put("tenantId", tenantId);
        body.put("actionId", actionId);
        body.put("param", paras);
        body.put("eocMap", eoc);
        HttpHeaders headerMap = new HttpHeaders();
        headerMap.add("Content-Type", "application/json;charset=UTF-8");
        setHeaderAppCode(extendedFields, headerMap);
        HttpEntity httpEntity = new HttpEntity<Map>(body, headerMap);
        try {
            ResponseEntity<BaseResultDTO> responseEntity = restTemplate.exchange(url, HttpMethod.POST, httpEntity, new ParameterizedTypeReference<BaseResultDTO>() {
            });
            Object response = responseEntity.getBody().getResponseWithException("");
            // 比如：eoc_DutyToId_duty_qc 的response类型为List，去列表的第一个元素
            if (response instanceof List) {
                List respList = (List) response;
                if (CollectionUtils.isNotEmpty(respList)) {
                    return (HashMap<String, Object>) respList.get(0);
                } else {
                    return null;
                }
            } else if (response instanceof Map) {
                return (HashMap<String, Object>) response;
            } else {
                return null;
            }
        } catch (Exception e) {
            log.error("[execute] body: {}, error: ", JsonUtils.objectToString(body), e);
            throw e;
        }
    }

    @Override
    public Object executeWithParas(String tenantId, String actionId, Map[] paras, Map eoc, Map extendedFields) {
        String url = executionEngineHttpHost + "/restful/service/ExecutionEngine/execute";
        HashMap body = new HashMap();
        body.put("tenantId", tenantId);
        body.put("actionId", actionId);
        body.put("paras", paras);
        body.put("eocMap", eoc);
        HttpHeaders headerMap = new HttpHeaders();
        headerMap.add("Content-Type", "application/json;charset=UTF-8");
        if (extendedFields != null && extendedFields.containsKey("traceId") && extendedFields.get("traceId") != null) {
            headerMap.add("digi-dap-service-chain-info", MessageFormat.format("0,0:ScheduleManager^{0}", extendedFields.get("traceId")));
        }
        setHeaderAppCode(extendedFields, headerMap);
        HttpEntity httpEntity = new HttpEntity<Map>(body, headerMap);
        try {
            ResponseEntity<BaseResultDTO> responseEntity = restTemplate.exchange(url, HttpMethod.POST, httpEntity, new ParameterizedTypeReference<BaseResultDTO>() {
            });
            return responseEntity.getBody().getResponseWithException("");
        } catch (Exception e) {
            log.error("[executeWithParas] body: {}, error: ", JsonUtils.objectToString(body), e);
            throw e;
        }
    }

    @Override
    public Object executeForManualProject(String tenantId, String actionId, Map[] paras, Map eocInfo, Map extendedFields) {
        String url = executionEngineHttpHost + "/restful/service/ExecutionEngine/execute";
        HashMap body = new HashMap();
        body.put("tenantId", tenantId);
        body.put("actionId", actionId);
        body.put("paras", paras);
        body.put("eocMap", eocInfo);
        HttpHeaders headerMap = new HttpHeaders();
        headerMap.add("Content-Type", "application/json;charset=UTF-8");
        setHeaderAppCode(extendedFields, headerMap);
        HttpEntity httpEntity = new HttpEntity<Map>(body, headerMap);
        try {
            ResponseEntity<BaseResultDTO> responseEntity = restTemplate.exchange(url, HttpMethod.POST, httpEntity, new ParameterizedTypeReference<BaseResultDTO>() {
            });
            return responseEntity.getBody().getResponseWithException("");
        } catch (Exception e) {
            log.error("[executeForManualProject] body: {}, error: ", JsonUtils.objectToString(body), e);
            throw e;
        }
    }

    private void setHeaderAppCode(Map<String, Object> extendedFields, HttpHeaders headerMap) {
        if (extendedFields != null) {
            // 添加 application 到 headerMap
            Object application = extendedFields.get("application");
            if (Objects.nonNull(application)) {
                headerMap.add("application", application.toString());
            }
            // 获取任务相关信息 taskCode 和 taskType
            String taskCode = Optional.ofNullable(extendedFields.get("tmActivityId")).map(Object::toString).orElse(null);
            String taskType = Optional.ofNullable(extendedFields.get("pageCode")).map(Object::toString).map(pageCode -> {
                switch (pageCode) {
                    case "task-card":
                    case "task-detail":
                        return "TASK";
                    case "basic-data":
                    case "browse-page":
                    case "edit-page":
                        return "DATA_ENTRY";
                    default:
                        return null;
                }
            }).orElse(null);
            // 如果 taskType 和 taskCode 都不为空，添加到 headerMap
            if (StringUtils.isNotBlank(taskType) && StringUtils.isNotBlank(taskCode)) {
                headerMap.add("taskType", taskType);
                headerMap.add("taskCode", taskCode);
            }
        }
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public Object clearFootprintData(AuthoredUser user, String instanceId, String activityCode, String entityName, Map<String, Object> eocMap, List<Map> bkList) {
        String uri = dataFootPrintHttpHost + "/restful/service/DataFootprint/activity/stop";

        Map<String, Object> body = new HashMap<>();
        body.put("tenantId", user.getTenantId());
        body.put("eocId", eocMap);

        List<Map> bkInfoList = new ArrayList<>();
        for (Map bkMap : bkList) {
            Map bkInfo = new HashMap();
            bkInfo.put("entityName", entityName);
            bkInfo.put("bk", bkMap);

            bkInfoList.add(bkInfo);
        }
        body.put("bkInfo", bkInfoList);
        body.put("actInsId", instanceId);
        body.put("actTmpId", activityCode);

        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON);
        HttpEntity httpEntity = new HttpEntity(body, headers);
        try {
            ResponseEntity<BaseResultDTO> respEntity = this.restTemplate.exchange(uri, HttpMethod.POST, httpEntity, new ParameterizedTypeReference<BaseResultDTO>() {
            });
            return respEntity.getBody().getResponseWithException("");
        } catch (Exception ex) {
            log.error("body: {}, error: ", JsonUtils.objectToString(body), ex);
            throw ex;
        }
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public Object clearComparisonData(AuthoredUser user, String instanceId, String activityCode, String entityName, Map<String, Object> eocMap, List<Map> bkList) {
        // 截取域名，https://smartdata-paas.apps.digiwincloud.com.cn/restful/service/ ===> https://smartdata-paas.apps.digiwincloud.com.cn
        String uri = dataComparisonHttpHost + "/restful/standard/datacomparison/comparison/v1/stop";

        Map<String, Object> body = new HashMap<>();
        body.put("tenantId", user.getTenantId());
        body.put("eocMap", eocMap);
        body.put("instanceId", instanceId);
        body.put("tempId", activityCode);
        body.put("bks", bkList);
        body.put("entityName", entityName);

        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON);
        HttpEntity httpEntity = new HttpEntity(body, headers);
        try {
            ResponseEntity<BaseResultDTO> respEntity = this.restTemplate.exchange(uri, HttpMethod.POST, httpEntity, new ParameterizedTypeReference<BaseResultDTO>() {
            });
            return respEntity.getBody().getResponseWithException("");
        } catch (Exception ex) {
            log.error("clearComparisonData: {}, url: {}, error: ", JsonUtils.objectToString(body), uri, ex);
            throw ex;
        }
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void retractComparisonData(RetractCompareReqDTO retractCompareReqDTO) {
        log.info("retractComparisonData:{}", retractCompareReqDTO);
        String uri = dataComparisonHttpHost + "/restful/standard/datacomparison/comparison/v1/retract";

        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON);
        HttpEntity httpEntity = new HttpEntity(retractCompareReqDTO, headers);
        try {
            ResponseEntity<BaseResultDTO> respEntity = this.restTemplate.exchange(uri, HttpMethod.POST, httpEntity, new ParameterizedTypeReference<BaseResultDTO>() {
            });
            respEntity.getBody().getResponseWithException("");
        } catch (Exception ex) {
            log.error("retractComparisonData: {}, url: {}, error: ", retractCompareReqDTO, uri, ex);
            throw ex;
        }
    }

    @Override
    public Object recoverComparisonData(Map<String, Object> paramMap) {
        log.info("recoverComparisonData:{}", paramMap);
        String uri = dataComparisonHttpHost + "/restful/standard/datacomparison/comparison/v1/recover";
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON);
        HttpEntity httpEntity = new HttpEntity(paramMap, headers);
        try {
            ResponseEntity<BaseResultDTO> respEntity = this.restTemplate.exchange(uri, HttpMethod.POST, httpEntity, new ParameterizedTypeReference<BaseResultDTO>() {
            });
            return respEntity.getBody().getResponseWithException("");
        } catch (Exception e) {
            log.error("body: {}, error: ", JsonUtils.objectToString(paramMap), e);
            throw e;
        }
    }

    @Override
    public Object giveupComparisonData(Map<String, Object> paramMap) {
        log.info("giveupComparisonData:{}", paramMap);
        String uri = dataComparisonHttpHost + "/restful/standard/datacomparison/comparison/v1/giveup";
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON);
        HttpEntity httpEntity = new HttpEntity(paramMap, headers);
        try {
            ResponseEntity<BaseResultDTO> respEntity = this.restTemplate.exchange(uri, HttpMethod.POST, httpEntity, new ParameterizedTypeReference<BaseResultDTO>() {
            });
            return respEntity.getBody().getResponseWithException("");
        } catch (Exception e) {
            log.error("body: {}, error: ", JsonUtils.objectToString(paramMap), e);
            throw e;
        }
    }

    @Override
    public boolean existsComparison(Map<String, Object> paramMap) {
        log.info("existsComparison:{}", paramMap);
        String uri = dataComparisonHttpHost + "/restful/standard/datacomparison/comparison/v1/exists";
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON);
        HttpEntity httpEntity = new HttpEntity(paramMap, headers);
        try {
            ResponseEntity<BaseResultDTO> respEntity = this.restTemplate.exchange(uri, HttpMethod.POST, httpEntity, new ParameterizedTypeReference<BaseResultDTO>() {
            });
            boolean exist = false;
            if (respEntity.getBody().getResponseWithException("") != null) {
                exist = Boolean.valueOf(respEntity.getBody().getResponse().toString());
            }
            return exist;
        } catch (Exception e) {
            log.error("body: {}, error: ", JsonUtils.objectToString(paramMap), e);
            throw e;
        }
    }

    @Override
    public Object batchGiveupComparisonData(Map<String, Object> paramMap) {
        log.info("batchGiveupComparisonData:{}", paramMap);
        String uri = dataComparisonHttpHost + "/restful/standard/datacomparison/comparison/v1/batch/giveup";
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON);
        HttpEntity httpEntity = new HttpEntity(paramMap, headers);
        try {
            ResponseEntity<BaseResultDTO> respEntity = this.restTemplate.exchange(uri, HttpMethod.POST, httpEntity, new ParameterizedTypeReference<BaseResultDTO>() {
            });
            return respEntity.getBody().getResponseWithException("");
        } catch (Exception e) {
            log.error("body: {}, error: ", JsonUtils.objectToString(paramMap), e);
            throw e;
        }
    }

}
