package com.digiwin.athena.atdm.datasource.datasource;

import cn.hutool.core.map.MapUtil;
import com.digiwin.athena.appcore.exception.BusinessException;
import com.digiwin.athena.appcore.util.SpringUtil;
import com.digiwin.athena.atdm.UiBotConstants;
import com.digiwin.athena.atdm.constant.ErrorCodeEnum;
import com.digiwin.athena.atdm.datasource.BuildPageDataProcessService;
import com.digiwin.athena.atdm.datasource.domain.*;
import com.digiwin.athena.atdm.datasource.dto.DataStatus;
import com.digiwin.athena.atdm.datasource.dto.PageInfo;
import com.fasterxml.jackson.annotation.JsonIgnore;
import lombok.Data;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.collections4.MapUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.*;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.TimeoutException;

@Data
public class DataSourceSet {
    private Logger logger = LoggerFactory.getLogger(DataSourceSet.class);

    public static final Map<String, Object> COMMON_FIELD_MAP = MapUtil
            .builder(new HashMap<String, Object>())
            .put("uibot_data_status", "not-changed") //标记数据状态  未修改not-changed   已修改changed   删除delete； 新增数据new-data。
            .build();
    /**
     * 查询的数据源
     */
    private List<DataSourceBase> dataSourceList;

    private String mainDatasource;

    /**
     * 全局数据处理器
     */
    private List<DataSourceProcessor> dataProcess;

    public static DataSourceSet create(DataSourceBase dataSourceBase) {
        DataSourceSet dataSourceSet = new DataSourceSet();
        List<DataSourceBase> dataSourceList = new ArrayList<>();
        dataSourceList.add(dataSourceBase);
        dataSourceSet.setDataSourceList(dataSourceList);
        return dataSourceSet;
    }

    public DataSourceSet append(DataSourceBase dataSourceBase) {
        if (dataSourceList == null) {
            this.dataSourceList = new ArrayList<>();
        }
        dataSourceList.add(dataSourceBase);
        return this;
    }

    /**
     * 查询数据和元数据
     *
     * @param executeContext 执行上下文
     * @param parameter      执行参数
     * @return
     */
    public QueryResultSet queryWithMetaData(ExecuteContext executeContext, Map<String, Object> parameter, PageInfo pageInfo, List<Map> sortInfo, List<Map> searchInfo) {
        if (CollectionUtils.isEmpty(dataSourceList)) {
            return QueryResultSet.empty();
        }
        List<QueryResult> queryResults = new ArrayList<>();
        if (dataSourceList.size() == 1) {
            queryResults.add(dataSourceList.get(0).queryWithMetaData(executeContext, parameter, pageInfo, sortInfo, searchInfo));
        } else {
            try {
                queryResults = ParallelQueryTaskUtils.queryWithMetaData(executeContext, parameter, dataSourceList, pageInfo, sortInfo, searchInfo);
            } catch (ExecutionException | TimeoutException | InterruptedException e) {//NOSONAR

                if (e.getCause() instanceof BusinessException) {
                    throw (BusinessException) e.getCause();
                }
                logger.error("多线程处理出问题:", e);
                throw BusinessException.create(ErrorCodeEnum.NUM_500_0047.getErrCode(), "多线程处理出问题", e);
            }
        }
        QueryResultSet queryResultSet = getQueryResultSet(executeContext, queryResults);
        return queryResultSet;
    }

    /**
     * 查询数据
     *
     * @param executeContext 执行上下文
     * @param parameter      参数
     * @return
     */
    public QueryResultSet query(ExecuteContext executeContext, Map<String, Object> parameter, PageInfo pageInfo, List<Map> sortInfo, List<Map> searchInfo) {
        if (CollectionUtils.isEmpty(dataSourceList)) {
            return QueryResultSet.empty();
        }
        List<QueryResult> queryResults = new ArrayList<>();

        if (dataSourceList.size() == 1) {
            queryResults.add(dataSourceList.get(0).query(executeContext, parameter, pageInfo, sortInfo, searchInfo));
        } else {
            try {
                queryResults = ParallelQueryTaskUtils.query(executeContext, parameter, dataSourceList, pageInfo, sortInfo, searchInfo);
            } catch (ExecutionException | TimeoutException | InterruptedException e) {//NOSONAR
                if (e.getCause() instanceof BusinessException) {
                    throw (BusinessException) e.getCause();
                }
                logger.error("多线程处理出问题:", e);
                throw BusinessException.create(ErrorCodeEnum.NUM_500_0046.getErrCode(), "多线程处理出问题", e);
            }
        }

        QueryResultSet queryResultSet = getQueryResultSet(executeContext, queryResults);

        return queryResultSet;
    }


    private QueryResultSet getQueryResultSet(ExecuteContext executeContext, List<QueryResult> queryResults) {
        QueryResultSet queryResultSet = QueryResultSet.withData(this.mainDatasource, queryResults);

        if (CollectionUtils.isNotEmpty(this.dataProcess)) {
            for (DataSourceProcessor dataSourceProcessor : dataProcess) {
                if (UiBotConstants.DATA_PROCESS_ACTIVE_POINT_EXECUTE_COMPLETED.equals(dataSourceProcessor.getActivePoint())) {
                    BuildPageDataProcessService buildPageDataProcessService = (BuildPageDataProcessService) SpringUtil.tryGetBean(dataSourceProcessor.getServiceName());
                    buildPageDataProcessService.handelPageData(executeContext, dataSourceProcessor, queryResultSet);
                }
            }
        }

        Integer appType = null != executeContext ? executeContext.getAppType() : null;
        // 等于 5 的时候，是 2.0 模型驱动
        if (null != appType && appType == 5) {
            // pageData 新增通用字段
            Map<String, Object> fieldMap = new HashMap<>();
            // 数据状态映射关系字段： 未修改not-changed   已修改changed   删除delete 新增数据new-data
            String key = Optional.ofNullable(executeContext.getDataStatus())
                    .map(DataStatus::getKey)
                    .orElse(null);
            // 如果是基础资料的双档作业，uibot 在请求进来的时候，就将 executeContext 的 editType 设置为 pageDefine 中的 editType
            // 这里加了一个判断是因为：双档作业在操作“复制”的时候，editType 是 copy，这个时候数据是前端传过来的，这个时候的数据状态应该是新增的 newData，而不是 not-changed
            boolean isCopy = "copy".equals(executeContext.getEditType());
            String value = Optional.ofNullable(executeContext.getDataStatus())
                    .map(ds -> isCopy ?  ds.getNewData() : ds.getNotChanged())
                    .orElse(null);
            if (StringUtils.isNotBlank(key) && StringUtils.isNotBlank(value)) {
                fieldMap.put(key, value);
            }
            appendCommonField4Two(queryResultSet.getPageData(), fieldMap);
        } else {
            // pageData 新增通用字段
            appendCommonField(queryResultSet.getPageData());
        }
        return queryResultSet;
    }

    /**
     * pageData 新增通用字段
     * @param pageData
     */
    private void appendCommonField(Map<String, Object> pageData) {
        if (MapUtils.isEmpty(pageData)) {
            return;
        }
        for (Object dataList : pageData.values()) {
            if (dataList instanceof List) {
                List rowDataList = (List) dataList;
                for (Object rowData : rowDataList) {
                    Map rowDataMap = (Map) rowData;
                    rowDataMap.putAll(COMMON_FIELD_MAP);
                }
            } else if(dataList instanceof Map)  {
                Map rowDataMap =  (Map)dataList;
                rowDataMap.putAll(COMMON_FIELD_MAP);
            }
        }
    }

    private static void appendCommonField4Two(Map<String, Object> pageData, Map<String, Object> fieldMap) {
        if (MapUtils.isEmpty(pageData)) {
            return;
        }

        // 若未传入字段 map，则使用默认字段
        Map<String, Object> effectiveFieldMap = MapUtils.isNotEmpty(fieldMap) ? fieldMap : COMMON_FIELD_MAP;

        for (Object data : pageData.values()) {
            processData(data, effectiveFieldMap);
        }
    }

    /**
     * 递归处理对象，向第一层 Map 添加公共字段。
     * - 如果为 Map，则添加字段；
     * - 如果为 List，则对其中的 Map 添加字段，并递归其 List 值；
     * - 内部嵌套 Map 不处理。
     *
     * @param data              当前处理的数据项，可能是 Map 或 List
     * @param effectiveFieldMap 已决定使用的字段 Map（非空）
     */
    private static void processData(Object data, Map<String, Object> effectiveFieldMap) {
        if (data instanceof Map) {
            Map<String, Object> mapData = (Map<String, Object>) data;
            mapData.putAll(effectiveFieldMap);

            for (Object value : mapData.values()) {
                if (value instanceof List) {
                    processData(value, effectiveFieldMap);
                }
                // 注意：如果 value 是 map，不处理（内部 map 不加字段）
            }
        } else if (data instanceof List) {
            for (Object item : (List<?>) data) {
                if (item instanceof Map) {
                    Map<String, Object> mapItem = (Map<String, Object>) item;
                    mapItem.putAll(effectiveFieldMap);

                    for (Object innerValue : mapItem.values()) {
                        if (innerValue instanceof List) {
                            processData(innerValue, effectiveFieldMap);
                        }
                        // 内部 map 不处理
                    }
                } else if (item instanceof List) {
                    // list 中嵌套 list
                    processData(item, effectiveFieldMap);
                }
            }
        }
    }

    @JsonIgnore
    public String getFirstActionId() {
        if (CollectionUtils.isEmpty(dataSourceList)) {
            return null;
        }
        return this.dataSourceList.get(0).getActionId();
    }

    @JsonIgnore
    public boolean isEmpty() {
        return CollectionUtils.isEmpty(dataSourceList);
    }

    @JsonIgnore
    public DataSourceBase getFirstDataQuery() {
        if (CollectionUtils.isEmpty(dataSourceList)) {
            return null;
        }
        return this.dataSourceList.get(0);
    }

    @JsonIgnore
    public QueryAction getFirstAction() {
        if (CollectionUtils.isEmpty(dataSourceList)) {
            return null;
        }
        return this.dataSourceList.get(0).getAction();
    }
}
