package com.digiwin.athena.atdm.action.impl;

import com.digiwin.athena.appcore.constant.LogConstant;
import com.digiwin.athena.appcore.domain.log.LogDto;
import com.digiwin.athena.appcore.exception.BusinessException;
import com.digiwin.athena.appcore.util.JsonUtils;
import com.digiwin.athena.appcore.util.MessageUtils;
import com.digiwin.athena.appcore.util.SpringUtil;
import com.digiwin.athena.atdm.ActionConstants;
import com.digiwin.athena.atdm.UiBotConstants;
import com.digiwin.athena.atdm.action.ActionExecutor;
import com.digiwin.athena.atdm.action.executor.DataSubmissionService;
import com.digiwin.athena.atdm.action.util.ActionUtils;
import com.digiwin.athena.atdm.action.util.EspParameterConverterUtils;
import com.digiwin.athena.atdm.constant.ErrorCodeEnum;
import com.digiwin.athena.atdm.datasource.ActionExecuteReq;
import com.digiwin.athena.atdm.datasource.domain.ActionParameterMapping;
import com.digiwin.athena.atdm.datasource.domain.ApiMetadata;
import com.digiwin.athena.atdm.datasource.domain.ExecuteResult;
import com.digiwin.athena.atdm.datasource.domain.MetadataField;
import com.digiwin.athena.atdm.datasource.domain.SubmitAction;
import com.digiwin.athena.atdm.datasource.domain.SubmitExecuteContext;
import com.digiwin.athena.atdm.thememap.CommonMetadataService;
import com.digiwin.athena.atdm.thememap.CommonThemeMapService;
import com.google.common.collect.Maps;
import com.google.common.eventbus.AsyncEventBus;
import com.jugg.agile.framework.core.util.reflect.bean.copy.JaBeanCopy;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.collections4.MapUtils;
import org.apache.commons.lang3.BooleanUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.InitializingBean;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.stream.Collectors;

@Slf4j
@Service
public class DataSubmissionServiceImpl implements InitializingBean, DataSubmissionService {
    @Autowired
    List<ActionExecutor> actionExecutorList;

    @Autowired
    private CommonThemeMapService commonThemeMapQueryService;

    private HashMap<String, ActionExecutor> actionExecutors = new HashMap<>();

    @Autowired
    private MessageUtils messageUtils;

    @Override
    public ExecuteResult submit(ActionExecuteReq actionExecuteReq) {
        SubmitAction action = actionExecuteReq.getAction();
        SubmitExecuteContext executeContext = actionExecuteReq.getSubmitExecuteContext();
        Map<String, Object> data = actionExecuteReq.getData();
        LogDto logDto = new LogDto("执行core action开始，actionId：" + action.getActionId(), executeContext.getTenantId() + LogConstant.TRACE_SEPARATOR + executeContext.getBacklogId());
        log.info(logDto.toString());
        ExecuteResult executeResult = this.executeCoreWapper(JaBeanCopy.cp(actionExecuteReq, ActionExecuteReq.class));
        LogDto logDtoEnd = new LogDto("执行core action结束，actionId：" + action.getActionId(), executeContext.getTenantId() + LogConstant.TRACE_SEPARATOR + executeContext.getBacklogId());
        log.info(logDtoEnd.toString());
        if (CollectionUtils.isNotEmpty(action.getAttachActions())) {
            //先执行所有非流程相关的action
            for (SubmitAction attachAction : action.getAttachActions()) {
                if (attachAction.isDispatchAction() || Boolean.TRUE.equals(attachAction.getExecuteAfterCheckCompleted())) {
                    continue;
                }
                executeContext.setParentAction(action);
                LogDto logDtoAttach = new LogDto("执行attach action开始，backlogId：" + executeContext.getBacklogId(), executeContext.getTenantId() + LogConstant.TRACE_SEPARATOR + executeContext.getBacklogId());
                log.info(logDtoAttach.toString());
                executeAttachAction(new ActionExecuteReq(executeContext, action, executeResult, attachAction, data, actionExecuteReq.getWorkItemIdToData()));
                LogDto logDtoAttachEnd = new LogDto("执行attach action结束，backlogId：" + executeContext.getBacklogId(), executeContext.getTenantId() + LogConstant.TRACE_SEPARATOR + executeContext.getBacklogId());
                log.info(logDtoAttachEnd.toString());
            }
        }
        return executeResult;
    }

    @Override
    public boolean isSupportSubmitMergeData(SubmitAction action) {
        String actionKey = ActionUtils.getActionKey(action);
        if (actionExecutors.containsKey(actionKey)) {
            return actionExecutors.get(actionKey).supportDataMerge();
        } else {
            throw BusinessException.create(ErrorCodeEnum.NUM_500_0024.getErrCode(), messageUtils.getMessage("exception.can.not.deal.action") + actionKey);
        }
    }

    @Override
    public ExecuteResult execute(SubmitExecuteContext executeContext, SubmitAction action, Map<String, Object> data) {
        ExecuteResult executeResult = this.executeCoreWapper(executeContext, null, action, data);
        if (CollectionUtils.isNotEmpty(action.getAttachActions())) {
            //需要先把bpm相关连的action提取出来
            SubmitAction bpmAction = null;
            Optional<SubmitAction> optionalAction = action.getAttachActions().stream().filter(attachAction -> attachAction.isDispatchAction()).findFirst();
            if (optionalAction.isPresent()) {
                bpmAction = optionalAction.get();
            }
            //先执行所有非流程相关的action
            for (SubmitAction attachAction : action.getAttachActions()) {
                if (attachAction.isDispatchAction()) {
                    continue;
                }
                executeAttachAction(executeContext, action, executeResult, attachAction, data);
            }

            executeResult.setCompleted(true);
            //如果已经处理完，且存在bpm action，则执行
            if (executeResult.isCompleted() && bpmAction != null) {
                this.executeAttachAction(executeContext, action, executeResult, bpmAction, data);
            }
        } else {
            executeResult.setCompleted(true);
        }

        return executeResult;
    }

    private ExecuteResult executeCore(SubmitExecuteContext executeContext, ExecuteResult parentExecuteResult, SubmitAction action, Map<String, Object> data) {
        return executeCore(new ActionExecuteReq(executeContext, parentExecuteResult, action, data));
    }

    private Map<String, Object> convertedParam(SubmitExecuteContext executeContext, SubmitAction action, ApiMetadata apiMetadata, Map<String, Object> data, Map<String, Object> paras) {
        Map<String, Object> convertedData = data;
        if (action.getActionParams() != null) {
            convertedData = EspParameterConverterUtils.convert(data, action.getActionParams(), action);
            /**
             * actionId有元数据定义，则根据元数据定义进一步解析convertedData。
             * 像出现频率最高的submit-data、dispatch等并没有元数据定义，若执行convertDataToMapByActionMetadata，则会打印获取不到元数据的异常日志，被运维提单改善日志。
             */
            if (apiMetadata != null) {
                Map<String, Object> dataMap = convertDataToMapByActionMetadata(executeContext, action, convertedData, apiMetadata);
                if (dataMap != null) {
                    convertedData = dataMap;
                }
            }
        }
        // 特殊类型处理
        handlingSpecialTypes(executeContext, action, convertedData);

        // 合并数据与常量
        if (MapUtils.isNotEmpty(paras)) {
            Map<String, Object> comData = EspParameterConverterUtils.combine(convertedData, paras);
            if (MapUtils.isNotEmpty(comData)) {
                convertedData = comData;
            }
        }
        return convertedData;
    }


    private ExecuteResult executeCore(ActionExecuteReq actionExecuteReq) {
        SubmitAction action = actionExecuteReq.getAction();
        Map<String, Object> data = actionExecuteReq.getData();
        SubmitExecuteContext executeContext = actionExecuteReq.getSubmitExecuteContext();
        ApiMetadata apiMetadata = actionExecuteReq.getApiMetadata();
        String actionKey = ActionUtils.getActionKey(action);
        if (!actionExecutors.containsKey(actionKey)) {
            throw BusinessException.create(ErrorCodeEnum.NUM_500_0023.getErrCode(), messageUtils.getMessage("exception.can.not.deal.action") + actionKey);
        }
        //如果上游没有传再查询一次，非合并的任务不会传
        if (null == apiMetadata
                && action.getActionParams() != null
                && actionExecutors.get(actionKey).hasActionMetadata()) {
            try {
                apiMetadata = SpringUtil.getBean(CommonMetadataService.class).getMetadata(executeContext.getLocale(), action.getActionId());
            } catch (Exception ignored) {
            }
        }
        Map<String, Object> convertedData = convertedParam(executeContext, action, apiMetadata, data, action.getParas());
        // 代表当前api已经被执行了，直接返回构造转换后的参数
        if (action.isExecuteAction()) {
            ExecuteResult ok = ExecuteResult.ok();
            ok.setRequestData(convertedData);
            return ok;
        }
        // 转换最小化拆分的
        Map<Long, Map<String, Object>> workItemIdToData = actionExecuteReq.getWorkItemIdToData();
        if (workItemIdToData != null) {
            Map<Long, Map<String, Object>> convertDataMap = Maps.newLinkedHashMapWithExpectedSize(workItemIdToData.size());
            Map<Long, Map<String, Object>> workItemToBpmParas = action.getWorkItemToBpmParas();

            for (Map.Entry<Long, Map<String, Object>> entry : workItemIdToData.entrySet()) {
                Map<String, Object> paras = action.getParas();
                if (workItemToBpmParas != null) {
                    paras = workItemToBpmParas.get(entry.getKey());
                }

                Map<String, Object> map = convertedParam(executeContext, action, apiMetadata, entry.getValue(), paras);
                convertDataMap.put(entry.getKey(), map);
            }
            actionExecuteReq.setWorkItemIdToData(convertDataMap);
        }
        LogDto logDto = new LogDto("执行Executor：" + actionKey + "开始，backlogId：" + executeContext.getBacklogId(), executeContext.getTenantId() + LogConstant.TRACE_SEPARATOR + executeContext.getBacklogId());
        log.info(logDto.toString());
        ActionExecutor actionExecutor = actionExecutors.get(actionKey);
        action.setSupportRetry(actionExecutor.supportRetry());
        actionExecuteReq.setData(convertedData);
        ExecuteResult executeResult = actionExecutor.execute(actionExecuteReq);
        LogDto logDtoEnd = new LogDto("执行Executor：" + actionKey + "结束，backlogId：" + executeContext.getBacklogId(), executeContext.getTenantId() + LogConstant.TRACE_SEPARATOR + executeContext.getBacklogId());
        log.info(logDtoEnd.toString());

        if (executeResult == null) {
            executeResult = ExecuteResult.ok();
        }
        //将提交data中的uuid返回给前端
        if (MapUtils.isNotEmpty(executeResult.getData())) {
            convertDataBindUuid(executeResult.getData(), data);
        }
        executeResult.setRequestData(convertedData);
        return executeResult;
    }

    private void handlingSpecialTypes(SubmitExecuteContext executeContext, SubmitAction action, Map<String, Object> convertedData) {
        if (CollectionUtils.isNotEmpty(action.getActionParams())) {
            for (ActionParameterMapping actionParam : action.getActionParams()) {
                if (!"TM_VARIABLE".equals(actionParam.getType())) {
                    continue;
                }
                Object value = commonThemeMapQueryService.queryVariable(actionParam.getValue(), executeContext);
                convertedData.put(actionParam.getName(), value);
            }
        }
    }


    @Override
    public void executeAttachAction(ActionExecuteReq actionExecuteReq) {
        SubmitExecuteContext executeContext = actionExecuteReq.getSubmitExecuteContext();
        ExecuteResult parentExecuteResult = actionExecuteReq.getParentExecuteResult();
        Map<String, Object> parentConvertedData = parentExecuteResult.getRequestData();
        Map<String, Object> parentResponseData = parentExecuteResult.getData();
        SubmitAction attachAction = actionExecuteReq.getAction();
        SubmitAction parentAction = actionExecuteReq.getParentAction();
        attachAction.setBusinessUnit(parentAction.getBusinessUnit());
        Map<String, Object> submitData = actionExecuteReq.getData();
        if (null != parentAction.getServiceId()) {
            attachAction.getServiceId().setProxyToken(parentAction.getServiceId().getProxyToken());
        }
        String dataFrom = "";
        if (attachAction.getExtendParas() != null && attachAction.getExtendParas().containsKey("dataSource")) {
            dataFrom = attachAction.getExtendParas().get("dataSource").toString();
        }
        Map<String, Object> convertedData = null;
        switch (dataFrom) {
            case "parent":
            case "parentRequest": {
                convertedData = parentConvertedData;
                break;
            }
            case "submit": {
                convertedData = submitData;
                break;
            }
            case "parentAndParentResponse": {
                convertedData = parentConvertedData;
                if (parentResponseData != null) {
                    for (Map.Entry<String, Object> stringListEntry : parentResponseData.entrySet()) {
                        convertedData.put(stringListEntry.getKey(), stringListEntry.getValue());
                    }
                }
                break;
            }
            case "submitWithParentResponse":
            case "parentResponse": {
                convertedData = submitData;
                if (parentResponseData != null) {
                    for (Map.Entry<String, Object> stringListEntry : parentResponseData.entrySet()) {
                        convertedData.put(stringListEntry.getKey(), stringListEntry.getValue());
                    }
                }
                break;
            }
            default: {
                convertedData = submitData;
            }
        }
        executeContext.setParentAction(parentAction);
        ExecuteResult attachActionExecuteResult = this.executeCoreWapper(new ActionExecuteReq(executeContext, parentExecuteResult, attachAction, convertedData, actionExecuteReq.getWorkItemIdToData()));

        if (CollectionUtils.isNotEmpty(attachAction.getAttachActions())) {
            for (SubmitAction attachAttachAction : attachAction.getAttachActions()) {
                if (UiBotConstants.ACTION_CATEGORY_BPM.equals(attachAction.getCategory()) || UiBotConstants.ACTION_CATEGORY_TASK_ENGINE.equals(attachAction.getCategory())) {
                    continue;
                }
                executeAttachAction(new ActionExecuteReq(executeContext, attachAction, attachActionExecuteResult, attachAttachAction, submitData, actionExecuteReq.getWorkItemIdToData()));
            }
        }
    }

    @Override
    public void executeAttachAction(SubmitExecuteContext executeContext, SubmitAction parentAction, ExecuteResult parentExecuteResult, SubmitAction attachAction, Map<String, Object> submitData) {
        Map<String, Object> parentConvertedData = parentExecuteResult.getRequestData();
        Map<String, Object> parentResponseData = parentExecuteResult.getData();
        attachAction.setBusinessUnit(parentAction.getBusinessUnit());
        if (null != parentAction.getServiceId()) {
            attachAction.getServiceId().setProxyToken(parentAction.getServiceId().getProxyToken());
        }
        String dataFrom = "";
        if (attachAction.getExtendParas() != null && attachAction.getExtendParas().containsKey("dataSource")) {
            dataFrom = attachAction.getExtendParas().get("dataSource").toString();
        }
        Map<String, Object> convertedData = null;
        switch (dataFrom) {
            case "parent":
            case "parentRequest": {
                convertedData = parentConvertedData;
                break;
            }
            case "submit": {
                convertedData = submitData;
                break;
            }
            case "parentAndParentResponse": {
                convertedData = parentConvertedData;
                if (parentResponseData != null) {
                    for (Map.Entry<String, Object> stringListEntry : parentResponseData.entrySet()) {
                        convertedData.put(stringListEntry.getKey(), stringListEntry.getValue());
                    }
                }
                break;
            }
            case "submitWithParentResponse":
            case "parentResponse": {
                convertedData = submitData;
                if (parentResponseData != null) {
                    for (Map.Entry<String, Object> stringListEntry : parentResponseData.entrySet()) {
                        convertedData.put(stringListEntry.getKey(), stringListEntry.getValue());
                    }
                }
                break;
            }
            default: {
                convertedData = submitData;
            }
        }
        executeContext.setParentAction(parentAction);
        ExecuteResult attachActionExecuteResult = this.executeCoreWapper(executeContext, parentExecuteResult, attachAction, convertedData);

        if (CollectionUtils.isNotEmpty(attachAction.getAttachActions())) {
            for (SubmitAction attachAttachAction : attachAction.getAttachActions()) {
                if (UiBotConstants.ACTION_CATEGORY_BPM.equals(attachAction.getCategory()) || UiBotConstants.ACTION_CATEGORY_TASK_ENGINE.equals(attachAction.getCategory())) {
                    continue;
                }
                executeAttachAction(executeContext, attachAction, attachActionExecuteResult, attachAttachAction, submitData);
            }
        }
    }

    @Override
    public void afterPropertiesSet() throws Exception {
        if (actionExecutors != null) {
            for (ActionExecutor actionExecutor : actionExecutorList) {
                this.actionExecutors.put(actionExecutor.supportKey(), actionExecutor);
            }
        }
    }

    /**
     * 根据submitAction 提交的元数据入参是对象还是数组，若对象时则转换data为对象
     *
     * @param executeContext
     * @param action
     * @param data
     */
    private Map<String, Object> convertDataToMapByActionMetadata(ApiMetadata apiMetadata, SubmitExecuteContext executeContext, SubmitAction action, Map<String, Object> data) {
        Map<String, Object> dataMap = null;
        try {
            //获取元数据 ，根据元数据上的入参决定数据是什么类型
            if (apiMetadata == null) {
                return null;
            }

            List<MetadataField> requestFields = new ArrayList<>();
            if (apiMetadata.getRequestFields() != null) {
                requestFields = apiMetadata.getRequestFields();
            }
            //元数据中，如果有invokeType，且原本action中没有invokeType值，则赋值给action
            if (StringUtils.isNotEmpty(apiMetadata.getInvokeType()) && StringUtils.isEmpty(action.getInvokeType())) {
                action.setInvokeType(apiMetadata.getInvokeType());
            }
            if (data == null || data.size() == 0) {
                return null;
            }
            List<String> typeList = Arrays.asList("ACTIVE_ROW", "GET_ACTION_RESPONSE");
            if (CollectionUtils.isNotEmpty(action.getActionParams())) {
                List<String> actionParaNames = action.getActionParams().stream().filter(para -> typeList.contains(para.getType())).map(item -> item.getName()).collect(Collectors.toList());
                //根据元数据来判断入参是对象还是数组
                if (CollectionUtils.isEmpty(actionParaNames) || actionParaNames.size() == 0) {
                    return null;
                }
                String conNames = actionParaNames.get(0);
                if (actionParaNames.get(0).contains(".")) {
                    conNames = actionParaNames.get(0).split("\\.")[0];
                }
                if (!requestIsArray(requestFields, conNames)) {
                    Map<String, Object> dataNew = new HashMap<>();
                    for (Map.Entry<String, Object> dataEntry : data.entrySet()) {
                        List dataList = JsonUtils.jsonToListObject(JsonUtils.objectToString(dataEntry.getValue()), Map.class);
                        if (dataList instanceof List && dataList.size() > 0) {
                            Map dataMapOne = JsonUtils.jsonToListObject(JsonUtils.objectToString(dataEntry.getValue()), Map.class).get(0);
                            dataNew.put(dataEntry.getKey(), dataMapOne);
                        }
                    }
                    if (MapUtils.isNotEmpty(dataNew)) {
                        dataMap = dataNew;
                    }
                } else {
                    Map<String, Object> dataNew = new HashMap<>();
                    for (Map.Entry<String, Object> dataEntry : data.entrySet()) {
                        if (dataEntry.getValue() instanceof Map) {
                            List<Object> o = new ArrayList<>();
                            o.add(dataEntry.getValue());
                            dataNew.put(dataEntry.getKey(), o);
                        }
                    }
                    if (MapUtils.isNotEmpty(dataNew)) {
                        dataMap = dataNew;
                    }

                }
            }
            return dataMap;
        } catch (Exception e) {
            return null;
        }
    }

    /**
     * 根据submitAction 提交的元数据入参是对象还是数组，若对象时则转换data为对象
     *
     * @param executeContext
     * @param action
     * @param data
     */
    private Map<String, Object> convertDataToMapByActionMetadata(SubmitExecuteContext executeContext, SubmitAction action, Map<String, Object> data, ApiMetadata apiMetadata) {
        try {
            if (apiMetadata == null) {
                apiMetadata = SpringUtil.getBean(CommonMetadataService.class).getMetadata(executeContext.getLocale(), action.getActionId());
            }
            return convertDataToMapByActionMetadata(apiMetadata, executeContext, action, data);
        } catch (Exception e) {
            return null;
        }
    }

    /**
     * 针对现在入参可能是对象，可能是数组的情况下，利用元数据上的requestFields来判断入参是什么
     *
     * @param requestFields
     * @param conNames
     * @return
     */
    private boolean requestIsArray(List<MetadataField> requestFields, String conNames) {
        if (CollectionUtils.isNotEmpty(requestFields)) {
            for (MetadataField metadataField : requestFields) {
                if (conNames.equals(metadataField.getName()) && "object".equals(metadataField.getDataType())) {
                    if (BooleanUtils.isTrue(metadataField.getArray())) {
                        return true;
                    } else {
                        return false;
                    }
                }
            }
        }
        return true;
    }


    /**
     * 处理uuid--将提交data中的uuid返回给前端
     *
     * @param convertedData
     * @param data
     */
    private static void convertDataBindUuid(Map<String, Object> convertedData, Map<String, Object> data) {
        if (MapUtils.isEmpty(convertedData)) {
            return;
        }
        if (MapUtils.isEmpty(data)) {
            return;
        }
        try {

            List uuidList = new ArrayList();
            for (Map.Entry<String, Object> dataEntry : data.entrySet()) {
                Object dataObj = dataEntry.getValue();
                if (dataObj instanceof List) {
                    List dataList = (List) dataObj;
                    for (Object rowData : dataList) {
                        Map rowDataMap = (Map) rowData;
                        if (rowDataMap.containsKey("uuid")) {
                            uuidList.add(rowDataMap.get("uuid"));
                        }
                    }
                }
            }
            if (CollectionUtils.isEmpty(uuidList)) {
                return;
            }
            for (Map.Entry<String, Object> dataEntry : convertedData.entrySet()) {
                Object convertedDataObj = dataEntry.getValue();
                if (convertedDataObj instanceof List) {
                    List convertedDataObjList = (List) convertedDataObj;
                    if (convertedDataObjList.size() != uuidList.size()) {
                        return;
                    }
                    int uuid = 0;
                    for (Object rowData : convertedDataObjList) {
                        Map rowDataMap = (Map) rowData;
                        if (!rowDataMap.containsKey("uuid")) {
                            rowDataMap.put("uuid", uuidList.get(uuid));
                        }
                        uuid++;
                    }
                }
            }
        } catch (Exception e) {
        }
    }

    public ExecuteResult executeCoreWapper(SubmitExecuteContext executeContext, ExecuteResult parentExecuteResult, SubmitAction action, Map<String, Object> data) {
        ExecuteResult executeResult = null;
        try {
            //只有重试的请求判断是否有执行结果
            if (Boolean.TRUE.equals(executeContext.getIsRetry()) && (Objects.nonNull(action.getExecuteResult()) && (Boolean.TRUE.equals(action.getExecuteResult().getExecuteState())))) {
                return action.getExecuteResult();
            }
            executeResult = this.executeCore(executeContext, parentExecuteResult, action, data);
            action.setExecuteResult(executeResult);
            return executeResult;
        } finally {
            action.setExecuted(true);
        }
    }

    @Override
    public ExecuteResult executeCoreWapper(ActionExecuteReq actionExecuteReq) {
        SubmitExecuteContext executeContext = actionExecuteReq.getSubmitExecuteContext();
        SubmitAction action = actionExecuteReq.getAction();
        ExecuteResult executeResult = null;
        try {
            //只有重试的请求判断是否有执行结果
            if (Boolean.TRUE.equals(executeContext.getIsRetry()) && (Objects.nonNull(action.getExecuteResult()) && (Boolean.TRUE.equals(action.getExecuteResult().getExecuteState())))) {
                return action.getExecuteResult();
            }
            executeResult = this.executeCore(actionExecuteReq);
            action.setExecuteResult(executeResult);
            return executeResult;
        } finally {
            action.setExecuted(true);
        }
    }
}
