package com.digiwin.athena.kmservice.service;

import cn.hutool.core.collection.CollectionUtil;
import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import com.alibaba.fastjson.JSONPath;
import com.digiwin.app.container.exceptions.DWBusinessException;
import com.digiwin.app.service.DWServiceContext;
import com.digiwin.athena.config.DataType;
import com.digiwin.athena.config.OpType;
import com.digiwin.athena.domain.common.Constants;
import com.digiwin.athena.domain.common.TenantObject;
import com.digiwin.athena.domain.core.Project;
import com.digiwin.athena.domain.core.Task;
import com.digiwin.athena.domain.core.TenantObjectAdaptation;
import com.digiwin.athena.dto.BaseCondition;
import com.digiwin.athena.dto.BasicQuery;
import com.digiwin.athena.dto.SortField;
import com.digiwin.athena.kmservice.cache.CacheConfig;
import com.digiwin.athena.kmservice.utils.*;
import com.digiwin.athena.mechanism.bo.HookBO;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.jayway.jsonpath.JsonPath;
import com.mongodb.client.result.DeleteResult;
import com.mongodb.client.result.UpdateResult;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.MapUtils;
import org.apache.commons.lang3.StringUtils;
import org.bson.types.ObjectId;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Sort;
import org.springframework.data.mongodb.core.MongoTemplate;
import org.springframework.data.mongodb.core.query.Criteria;
import org.springframework.data.mongodb.core.query.Query;
import org.springframework.data.mongodb.core.query.Update;
import org.springframework.data.redis.core.StringRedisTemplate;
import org.springframework.util.CollectionUtils;

import java.util.*;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;

@Slf4j
public abstract class DataPickService {
    @Autowired
    protected ObjectMapper objectMapper;
    @Autowired
    protected StringRedisTemplate stringRedisTemplate;
    @Autowired
    IAMUtils iamUtils;
    @Autowired
    EocUtils eocUtils;

    public static String KmRebuildDoubleWriteKey="DatamapRebuildDoubleWriteKey";

    public <T extends TenantObject> T findByCode(String code,Class<T> c) throws DWBusinessException {
        BasicQuery request = new BasicQuery();
        request.setQuery(new HashMap<>());
        request.getQuery().put("code",code);
        T t =findOne(request,c);
        return t;
    }

    public <T extends TenantObject> List<T> findByCodes(Collection<String> codes,Class<T> c) throws DWBusinessException {
        BasicQuery request = new BasicQuery();
        request.setQuery(new HashMap<>());
        request.getQuery().put("code",codes);
        List<T> ts =find(request,c);
        return ts;
    }
    public <T extends TenantObject> T findOne(BasicQuery request,Class<T> c) throws DWBusinessException {
        List<T> ts = find(request,c);
        if(!ts.isEmpty()){
            return ts.get(0);
        }
        return null;
    }
    public <T extends TenantObject> List<T> find(BasicQuery request,Class<T> c) throws DWBusinessException {
        return find(request,c,null,false);
    }

    public <T extends TenantObject> List<T> find(BasicQuery request,Class<T> c,String col,boolean systemOnly) throws DWBusinessException {
        List<T> ts = new ArrayList<>();
        Map<String, Object> eocInfo = request.getEocInfo();
        if(CollectionUtils.isEmpty(eocInfo)){
            eocInfo = ServiceUtils.getContext().getEocInfo();
        }
        if(!isTenantCol(c)){
            List<T> systems =null;
            Query systemQuery = buildSystemQuery(request,c);
            if(StringUtils.isNotEmpty(col)){
                systems =systemTemplate().find(systemQuery,c,col);
            }else{
                systems =systemTemplate().find(systemQuery,c);
            }
            ts.addAll(systems);
        }
        //todo 这里有个问题，租户级配置字段可能是不全的，这样会导致查不到，如果仅根据code来查没有此问题
        if(!isSystemCol(c) && !systemOnly){
            List<T> users =null;
            Query queryUser = buildTenantQuery(request,c, eocInfo);
            if(StringUtils.isNotEmpty(col)){
                users =tenantTemplate().find(queryUser,c,col);
            }else{
                users =tenantTemplate().find(queryUser,c,tenantCol(c));
            }
            sortTenantList(users);
            ts.addAll(users);
        }
        ts = MergeUtil.excludeSameCode(ts,c,eocInfo);
        return ts;
    }

    public <T extends TenantObject> long remove(BasicQuery request,Class<T> c) throws DWBusinessException {
        Query queryUser = buildTenantQuery(request,c,request.getEocInfo());
        DeleteResult remove =tenantTemplate().remove(queryUser,c,tenantCol(c));
        return remove.getDeletedCount();
    }
    public <T extends TenantObject> long update(BasicQuery request,Class<T> c) throws DWBusinessException {
        if(CollectionUtils.isEmpty(request.getData())){return 0;}
        Query queryUser = buildTenantQuery(request,c,request.getEocInfo());
        Update update = new Update();
        request.getData().forEach(update::set);
        UpdateResult updated =tenantTemplate().updateMulti(queryUser,update,tenantCol(c));
        return updated.getModifiedCount();
    }

    /*
    因为历史脏数据需要按照时间排序取最老的那条。存在一些数据时间最老但不是排在最前面所以也不能使用库中的默认顺序
     */
    private <T extends TenantObject> void sortTenantList(List<T> ts){
        try {
            if (CollectionUtils.isEmpty(ts) || ts.size() <= 1) {
                return ;
            }
            Collections.sort(ts, new Comparator<T>() {
                @Override
                public int compare(T o1, T o2) {
                    Long time1 = getTimeById(o1.getId());
                    Long time2 = getTimeById(o2.getId());
                    if(time1>time2){
                        return -1;
                    }else if(time1<time2){
                        return 1;
                    }
                    return 0;
                }
            });
        }catch (Exception e){e.printStackTrace();}
    }



    public static Long getTimeById(String id) {
        ObjectId objectId = new ObjectId(id);
        // 获取时间戳
        return objectId.getDate().getTime();
    }


    public Query buildSystemQuery(BasicQuery request,Class c) throws DWBusinessException {
        Query query = buildQuery(request,c);
        String tenantId = ServiceUtils.getTenantIdSilent();
        if(null!=tenantId){
            String version = tenantVersion();
            List<String> tenantIds = new ArrayList<>();
            tenantIds.add(tenantId);
            tenantIds.add(Constants.SYSTEM);
            query.addCriteria(Criteria.where("tenantId").in(tenantIds));
            query.addCriteria(Criteria.where("version").is(version));
        }
        return query;
    }

    public Query buildTenantQuery(BasicQuery request,Class c, Map<String, Object> eocInfo) throws DWBusinessException {
     //todo 临时屏蔽
        processOperateUnit(request, eocInfo);
        Query query = buildQuery(request,c);
        String tenantId = ServiceUtils.getTenantIdSilent();
        if(null!=tenantId){
            query.addCriteria(Criteria.where("tenantId").is(tenantId));
        }
        return query;
    }

    /**
     * V2版本查询当前组织树上的运营单元数据
     * @param request 请求
     * @param eocInfo 运营单元信息
     */
    private void processOperateUnit(BasicQuery request, Map<String, Object> eocInfo) throws DWBusinessException {
        if (iamUtils.isTenantOperationUnitV2()) {
            eocInfo = eocInfo == null ? new HashMap<>() : eocInfo;
            Object operateUnitId = JSONPath.eval(eocInfo, "$.operation_unit_v2.eoc_mapping_id");
            Map<String, Object> headers = DWServiceContext.getContext().getRequestHeader();
            String token = MapUtils.getString(headers, "token");
            if (operateUnitId != null) {
                String operationPath = eocUtils.getOperationUnitPath(token, operateUnitId.toString());
                if (StringUtils.isNotBlank(operationPath)) {
                    request.getQuery().put("eocInfo.operation_unit_v2.eoc_mapping_id", Arrays.asList(operationPath.split(":")));
                    eocInfo.put("operationPath", operationPath);
                } else {
                    // 没有找到运营单元，给个查不到的条件
                    request.getQuery().put("eocInfo", null);
                }
            } else {
                // 没有带id过来，直接需要查询最上层的运营单元
                String rootOperationUnitId = eocUtils.getRootOperationUnitId(token);
                if (StringUtils.isNotBlank(rootOperationUnitId)) {
                    request.getQuery().put("eocInfo.operation_unit_v2.eoc_mapping_id", rootOperationUnitId);
                    eocInfo.put("operationPath", rootOperationUnitId);
                } else {
                    // 没有id，给个查不到的条件
                    request.getQuery().put("eocInfo", null);
                }
            }
        }
    }

    public  Query buildQuery(BasicQuery request,Class c){
        Query query =new Query();
        if(null!=request.getQuery()){
            processApplicationParam(request.getQuery(),c);
            request.getQuery().forEach((k, v)->{
                if(v instanceof Collection){
                    Collection collection = (Collection) v;
                    query.addCriteria(Criteria.where(k).in(collection));
                }else if (v instanceof Criteria){
                    query.addCriteria(Criteria.where(k).elemMatch((Criteria)v));
                } else{
                    if("name".equalsIgnoreCase(k) && v instanceof String){
                        String sv = (String) v;
                        Criteria nameCriteria = new Criteria().orOperator(
                                Criteria.where("name").regex(sv),
                                Criteria.where("lang.name.zh_CN").regex(sv),
                                Criteria.where("lang.name.zh_TW").regex(sv),
                                Criteria.where("lang.name.en_US").regex(sv)
                        );
                        query.addCriteria(nameCriteria);
                    }else{
                        query.addCriteria(Criteria.where(k).is(v));
                    }
                }
            });
        }else if (null!=request.getCondition()){
            query.addCriteria(buildCriteria(request.getCondition()));
        }
        if(null!=request.getPage() && null!=request.getPageSize()){
            query.with(PageRequest.of(request.getPage(), request.getPageSize()));
        }
        if(CollectionUtil.isNotEmpty(request.getSortFields())){
            List<Sort.Order> orders = new ArrayList<>();
            for(SortField field: request.getSortFields()){
                Sort.Direction direction = "desc".equalsIgnoreCase(field.getDirection())?Sort.Direction.DESC:Sort.Direction.ASC;
                Sort.Order order = new Sort.Order(direction, field.getField());
                orders.add(order);
            }
            query.with(Sort.by(orders));
        }
        if(CollectionUtil.isNotEmpty(request.getReturnFields())){
            List<String> newReturnFields = new ArrayList<>(request.getReturnFields());
            newReturnFields.add("code");
            newReturnFields.add("tenantId");
            newReturnFields.add("version");
            request.setReturnFields(newReturnFields);
            for(String field : request.getReturnFields()){
                query.fields().include(field);
            }
        }
        return query;
    }


    public String tenantVersion() throws DWBusinessException {
        String localKey = "TenantVersion";
        String version = (String) ServiceUtils.getContext().getProfiles().get(localKey);
        if (null!=version){
            return version;
        }
        String tenantId = ServiceUtils.getTenantId();
        String cacheKey = CacheConfig.cachePrefixWord +tenantId+":"+localKey;
        version = stringRedisTemplate.opsForValue().get(cacheKey);
        if (null!=version){
            ServiceUtils.getContext().getProfiles().put(localKey,version);
            return version;
        }
        version= tenantVersion(tenantId);
        if (null == version) {
            version= Constants.prodVersion;
        }
        ServiceUtils.getContext().getProfiles().put(localKey,version);
        stringRedisTemplate.opsForValue().set(cacheKey,version,7200, TimeUnit.SECONDS);
        return version;
    }


    public  String tenantCol(Class c){
        String col = systemTemplate().getCollectionName(c);
//        if(!isTenantCol(c)){
//            col = col+"Tenant";
//        }
        return col;
    }

    public <T> T convert(Object obj,Class<T> c){
        T t = objectMapper.convertValue(obj,c);
        return t;
    }



    public <T extends TenantObject> void save(T t,Class<T> c){
        try {
            T toSave = tenantObjectToSave(t,c);
            saveTenantObject(toSave,c);
        } catch (DWBusinessException e) {
            throw new RuntimeException(e);
        }
    }


    public <T extends TenantObject> T tenantObjectToSave(T t, Class<T> c) throws DWBusinessException {
        if(null==t || null==t.getCode()){return null;}
        String tenantId = ServiceUtils.getTenantId();
        t.setId(null);
        t.setSourceLevel(null);
        t.setSourceId(null);
        t.setAthena_namespace(null);
        t.setTenantId(tenantId);
        if (!Constants.defaultTenantId.equalsIgnoreCase(tenantId)) {
            t.setLang(null);
        }
        Criteria criteria = Criteria.where("tenantId").is(tenantId).and("code").is(t.getCode());
        criteria = addEocInfoToCriteria(t.getEocInfo(), criteria);
        Query query = Query.query(criteria);
        T toSave = null;
        List<T> ts = tenantTemplate().find(query,c,tenantCol(c));
        sortTenantList(ts);
        if(!ts.isEmpty()){
            for(T tt: ts){
                if(StringUtils.isEmpty(tt.getPluginId())){
                    toSave = tt;
                }
            }
        }
        if(null==toSave){
            toSave = t;
        }else{
            MergeUtil.mergeObject(t,toSave);
        }

        return toSave;
    }

    public <T extends TenantObject> void saveTenantObject(T t,Class<T> c) {
        t.setUserEdit(true);
        t.setObjUpdateTime(System.currentTimeMillis());
        tenantTemplate().save(t,tenantCol(c));

    }


    public Criteria addEocInfoToCriteria(Map<String, Object> eocInfo , Criteria criteria){
        if (iamUtils.isTenantOperationUnitV2()) {
            Object eval = JSONPath.eval(eocInfo, "$.operation_unit_v2.eoc_mapping_id");
            criteria = new Criteria().andOperator(criteria, Criteria.where("eocInfo.operation_unit_v2.eoc_mapping_id").is(eval != null ? String.valueOf(eval) : null));
            return criteria;
        }
        String companyId = null;
        String siteId = null;
        String regionId = null;

        if (eocInfo != null) {
            companyId = MapUtils.getString(eocInfo, "eoc_company_id");
            siteId = MapUtils.getString(eocInfo, "eoc_site_id");
            regionId = MapUtils.getString(eocInfo, "eoc_region_id");
        }

        if (StringUtils.isNotEmpty(companyId)) {
            criteria = new Criteria().andOperator(criteria, Criteria.where("eocInfo.eoc_company_id").is(companyId));
        }else{
            criteria = new Criteria().andOperator(criteria, Criteria.where("eocInfo.eoc_company_id").is(null));
        }
        if (StringUtils.isNotEmpty(siteId)) {
            criteria = new Criteria().andOperator(criteria, Criteria.where("eocInfo.eoc_site_id").is(siteId));
        }else{
            criteria = new Criteria().andOperator(criteria, Criteria.where("eocInfo.eoc_site_id").is(null));
        }
        if (StringUtils.isNotEmpty(regionId)) {
            criteria = new Criteria().andOperator(criteria, Criteria.where("eocInfo.eoc_region_id").is(regionId));
        }else{
            criteria = new Criteria().andOperator(criteria, Criteria.where("eocInfo.eoc_region_id").is(null));
        }

        return criteria;
    }

    public void processApplicationParam(Map<String,Object> query, Class clazz){
        if(null==query || null==clazz){return;}
        String type =null;
        if(clazz.isAssignableFrom(Project.class)){
            type="task";
        } else if (clazz.isAssignableFrom(Task.class)) {
            type="activity";
        }else{
            return;
        }
        String appCode = (String) query.get("application");
        Object  primaryCode =  query.get("code");
        if(null!=appCode && null==primaryCode){
            List<String> codes = getCodeByTypeAndAppCode(type,appCode);
            if(!CollectionUtils.isEmpty(codes)){
                query.put("code",codes);
                query.remove("application");
            }else{
                //todo 保证他查不到数据,暂时是会退到原来逻辑，当application字段删除时使用下面该逻辑
                // request.getQuery().put("application","datanotfound");
            }
        }
    }








    public <T extends TenantObject> T findBetter(String code, Class<T> c) throws DWBusinessException {
        T t = findByCode(code, c);
        t = mergeWithAdaption(t, c);
        return t;
    }

    public <T extends TenantObject> T findBetter(BasicQuery request, Class<T> c) throws DWBusinessException {
        T t = findOne(request, c);
        t = mergeWithAdaption(t, c);
        return t;
    }

    public <T extends TenantObject> List<T> findBetters(BasicQuery request, Class<T> c) throws DWBusinessException {
        List<T> ts = find(request, c);
        ts = mergeWithAdaption(ts, c);
        return ts;
    }

    public <T extends TenantObject> T mergeWithAdaption(T obj, Class<T> c) throws DWBusinessException {
        T t = obj;
        if (null != t) {
            String tenantId = ServiceUtils.getTenantId();
            String table = tableTenantObjectAdaptation(c);
            Criteria criteria = Criteria.where("tenantId").is(tenantId).and("code").is(t.getCode());
            Query query = Query.query(criteria);
            List<TenantObjectAdaptation> adaptations = tenantTemplate().find(query, TenantObjectAdaptation.class, table);
            for (TenantObjectAdaptation each : adaptations) {
                t = mergeAdaption(each, t, c);
            }
        }
        return t;
    }

    public <T extends TenantObject> List<T> mergeWithAdaption(List<T> list, Class<T> c) throws DWBusinessException {
        List<T> ts = list;
        if (!CollectionUtils.isEmpty(ts)) {
            String tenantId = ServiceUtils.getTenantId();
            Map<String, T> tsMap = new HashMap<>();
            ts.forEach(t -> {
                tsMap.put(t.getCode(), t);
            });
            String table = tableTenantObjectAdaptation(c);
            Criteria criteria = Criteria.where("tenantId").is(tenantId).and("code").in(tsMap.keySet());
            Query query = Query.query(criteria);
            List<TenantObjectAdaptation> adaptations = tenantTemplate().find(query, TenantObjectAdaptation.class, table);
            for (TenantObjectAdaptation ta : adaptations) {
                T t = tsMap.get(ta.getCode());
                T t2 = mergeAdaption(ta, t, c);
                if (null != t2) {
                    tsMap.put(t.getCode(), t2);
                }
            }
            ts = new ArrayList<>();
            ts.addAll(tsMap.values());
        }
        return ts;
    }

    private <T extends TenantObject> T mergeAdaption(TenantObjectAdaptation adaptation, T to, Class<T> c) {
        T result = to;
        if (null == adaptation || null == to || null == adaptation.getPath()) {
            return result;
        }
        try {
            JSONObject jsonObjectV = JSON.parseObject(JSON.toJSONString(to));
            if (DataType.array.equalsIgnoreCase(adaptation.getDateType()) && OpType.add.equalsIgnoreCase(adaptation.getOp())) {
                if (null != adaptation.getValue()) {
                    JsonPath.parse(jsonObjectV).add(adaptation.getPath(), adaptation.getValue());
                }
            } else if (OpType.put.equalsIgnoreCase(adaptation.getOp())) {
                JsonPath.parse(jsonObjectV).put(adaptation.getPath(), adaptation.getAttribute(), adaptation.getValue());
            } else if ("comb".equalsIgnoreCase(adaptation.getOp())) {
                try {
                    Object read = JsonPath.read(jsonObjectV, adaptation.getPath());
                    JSONArray targetArray = JSONArray.parseArray(JSON.toJSONString(read));
                    if (targetArray.isEmpty()) {
                        targetArray.addAll(JSONArray.parseArray(JSON.toJSONString(adaptation.getValue())));
                    } else {
                        JSONArray sourceArray = JSONArray.parseArray(JSON.toJSONString(adaptation.getValue()));
                        for (int i = 0; i < sourceArray.size(); i++) {
                            boolean flag = true;
                            JSONObject sourceObject = sourceArray.getJSONObject(i);
                            String sourceEventSource = sourceObject.getString("eventSource");
                            for (int j = 0; j < targetArray.size(); j++) {
                                JSONObject targetObject = targetArray.getJSONObject(j);
                                String targetEventSource = targetObject.getString("eventSource");
                                if (sourceEventSource.equals(targetEventSource)) {
                                    HookBO sourceHookBO = JSONObject.parseObject(JSON.toJSONString(sourceObject), HookBO.class);
                                    HookBO targetHookBO = JSONObject.parseObject(JSON.toJSONString(targetObject), HookBO.class);
                                    MergeUtil.mergeObject(targetHookBO, sourceHookBO, HookBO.class, "^.*Hook", false);
                                    targetArray.set(i, targetHookBO);
                                    flag = false;
                                    break;
                                }
                            }
                            if (flag) {
                                targetArray.addAll(JSONArray.parseArray(JSON.toJSONString(adaptation.getValue())));
                            }
                        }

                    }
                    adaptation.setValue(targetArray);
                } catch (Exception e) {
                    log.warn("DataPickService.excludeSameCode 读取数据失败，e={}", e);
                }
                JsonPath.parse(jsonObjectV).set(adaptation.getPath(), adaptation.getValue());
            } else {
                JsonPath.parse(jsonObjectV).set(adaptation.getPath(), adaptation.getValue());
	            try {
                    // jsonObjectV有可能不携带adaptation.getPath()中的字段，使用set命令则会失败
                    // 此处就是在设置值之后判断是否有此字段，如果没有则用put命令添加字段
		            if(JsonPath.parse(jsonObjectV).read(adaptation.getPath())){
		                log.info("赋值正确！");
		            }
	            } catch (Exception e) {
                    JsonPath.parse(jsonObjectV).put("$",adaptation.getPath().replace("$.",""), adaptation.getValue());
	            }
            }
            result = JSON.parseObject(JSON.toJSONString(jsonObjectV), c);
        } catch (Exception e) {
            log.error(e.getMessage(), e);
        }
        return result;
    }


    public  static Criteria buildCriteria(BaseCondition condition){
        Criteria criteria = new Criteria();
        if(null==condition || null==condition.getOp()){return criteria;}
        if("and".equalsIgnoreCase(condition.getOp())){
            for(BaseCondition sub : condition.getConditions()){
                criteria.andOperator(buildCriteria(sub));
            }
        }else if ("or".equalsIgnoreCase(condition.getOp())){
            for(BaseCondition sub : condition.getConditions()){
                criteria.orOperator(buildCriteria(sub));
            }
        }else {
            criteria = new Criteria(condition.getKey());
            switch (condition.getOp()){
                case "eq":
                    criteria.is(condition.getValue());
                    break;
                case "ne":
                    criteria.ne(condition.getValue());
                    break;
                case "gt":
                    criteria.gt(condition.getValue());
                    break;
                case "gte":
                    criteria.gte(condition.getValue());
                    break;
                case "lt":
                    criteria.lt(condition.getValue());
                    break;
                case "lte":
                    criteria.lte(condition.getValue());
                    break;
                case "regex":
                    criteria.regex((String) condition.getValue());
                    break;
                case "in":
                    criteria.in(condition.getValue());
                    break;
                case "nin":
                    criteria.nin(condition.getValue());
                    break;
                case "isnull":
                    criteria.is(null);
                    break;
                case "notnull":
                    criteria.ne(null);
                    break;
                default:throw new IllegalArgumentException("not support op "+condition.getOp());
            }
        }
        return criteria;
    }

    /**
     * 设置v2租户运营单元
     * @param tenantObject 待设置v2运营单元的租户对象
     * @throws DWBusinessException 异常
     */
    public void setRootOperationUnitId(TenantObject tenantObject) throws DWBusinessException {
        if (!iamUtils.isTenantOperationUnitV2()) {
            return;
        }
        if (MapUtils.isEmpty(tenantObject.getEocInfo())) {
            tenantObject.setEocInfo(new HashMap<>());
        }
        Object eval = JSONPath.eval(tenantObject.getEocInfo(), "$.operation_unit_v2.eoc_mapping_id");
        if (eval != null) {
            return;
        }
        Map<String, Object> headers = DWServiceContext.getContext().getRequestHeader();
        String token = MapUtils.getString(headers, "token");
        String id = eocUtils.getRootOperationUnitId(token);
        if (StringUtils.isNotBlank(id)) {
            JSONObject operationUnitV2 = new JSONObject().fluentPut("eoc_mapping_id", id);
            tenantObject.getEocInfo().put("operation_unit_v2", operationUnitV2);
        }
    }

    public static String tableTenantObjectAdaptation(Class c) {
        return c.getSimpleName() + "Adaptation";
    }

    public abstract MongoTemplate systemTemplate();

    public abstract MongoTemplate tenantTemplate();

    public abstract String tenantVersion(String tenantId);
    public abstract List<String> getCodeByTypeAndAppCode(String type, String app);
    public abstract boolean isTenantCol(Class c);

    public abstract boolean isSystemCol(Class c);
}
