package com.digiwin.athena.kmservice.utils;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.digiwin.app.container.exceptions.DWBusinessException;
import com.digiwin.app.service.DWServiceContext;
import com.digiwin.athena.kmservice.common.Constants;
import org.apache.commons.collections.MapUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.http.HttpEntity;
import org.springframework.http.HttpHeaders;
import org.springframework.http.HttpMethod;
import org.springframework.http.ResponseEntity;
import org.springframework.stereotype.Component;
import org.springframework.web.client.RestTemplate;

import java.io.IOException;
import java.net.HttpURLConnection;
import java.net.URL;
import java.util.HashMap;
import java.util.Map;
import java.util.Optional;

/**
 * @ClassName IAMUtils
 * @Description TODO
 * @Author zhuangli
 * @Date 2020/9/7 15:17
 * @Version 1.0
 **/
@Component
public class IAMUtils {
    private static Logger LOGGER = LoggerFactory.getLogger(IAMUtils.class);

    /**
     * 1.iam环境变量默认配置null，如需使用该工具类，引入方需增加环境变量配置
     * 2.外部调用方如果使用ActionExecutionPathService服务也需增加环境变量配置
     */
    @Value("${iamUrl:null}")
    private String iamUrl;
    @Value("${iamApToken:null}")
    private String iamApToken;
    /**
     * 是否支持v2租户，默认开启
     */
    @Value("${supportTenantV2:true}")
    private Boolean supportTenantV2;
    @Autowired
    @Qualifier("commonRestTemplate")
    RestTemplate restTemplate;

    private static ThreadLocal<String> localeHolder = new ThreadLocal();

    private static final String DEFAULT_LOCALE = "zh_CN";

    private String getDefaultLocale(){
        if(StringUtils.endsWithIgnoreCase(iamUrl, ".cn")){
            return "zh_CN";
        }else{
            return "zh_TW";
        }
    }

    /**
     * @return
     * @Author zhuangli
     * @Description 注意:此方法会将locale加入threadlocale,调用后及时remove
     * @Date 16:45 2020/9/7
     * @Param
     **/
    public void setTenantLocalByToken(String token) {
        String apiMetadataUrl = iamUrl + "/api/iam/v2/tenant/metadata/language";
        String locale = null;
        try {
            URL restURL = new URL(apiMetadataUrl);

            HttpURLConnection conn = (HttpURLConnection) restURL.openConnection();
            conn.setRequestMethod("GET");
            conn.setRequestProperty("Content-Type", "application/json");
            conn.setRequestProperty("digi-middleware-auth-user", token);
            conn.setRequestProperty("digi-middleware-auth-app", iamApToken);
            conn.setDoOutput(true);

            JSONObject response = JSON.parseObject(conn.getInputStream(), JSONObject.class);

            locale = null == response.getString("data") || "".equals(response.getString("data"))
                    ? getDefaultLocale() : response.getString("data");
        } catch (Exception e) {
            LOGGER.info("http get metadata language error:{}, set default locale:{}",e.getMessage(), getDefaultLocale());
            localeHolder.set(getDefaultLocale());
        }
        localeHolder.set(locale);
    }

    public String getTenantLocal(){
        return localeHolder.get();
    }

    public String getTenantLocale(String token){
        String apiMetadataUrl = iamUrl + "/api/iam/v2/tenant/metadata/language";
        String locale = null;
        try {
            URL restURL = new URL(apiMetadataUrl);

            HttpURLConnection conn = (HttpURLConnection) restURL.openConnection();
            conn.setRequestMethod("GET");
            conn.setRequestProperty("Content-Type", "application/json");
            conn.setRequestProperty("digi-middleware-auth-user", token);
            conn.setRequestProperty("digi-middleware-auth-app", iamApToken);
            conn.setDoOutput(true);

            JSONObject response = JSON.parseObject(conn.getInputStream(), JSONObject.class);

            locale = StringUtils.isEmpty(response.getString("data")) ? getDefaultLocale() : response.getString("data");
            return locale;
        } catch (Exception e) {
            LOGGER.info("http get metadata language error:{}, set default locale:{}",e.getMessage(), getDefaultLocale());
            return getDefaultLocale();
        }
    }

    public Integer getTenantType(String token) throws IOException {
        String apiMetadataUrl = iamUrl + "/api/iam/v2/tenant/current";
        try {
            URL restURL = new URL(apiMetadataUrl);

            HttpURLConnection conn = (HttpURLConnection) restURL.openConnection();
            conn.setRequestMethod("GET");
            conn.setRequestProperty("Content-Type", "application/json");
            conn.setRequestProperty("digi-middleware-auth-user", token);
            conn.setRequestProperty("digi-middleware-auth-app", iamApToken);
            conn.setDoOutput(true);

            JSONObject response = JSON.parseObject(conn.getInputStream(), JSONObject.class);
            Integer type = response.getJSONObject("tenant").getInteger("enterpriseType");
            return type;
        } catch (Exception e) {
            LOGGER.error("http get tenant type error:{}, set default locale:{}",e.getMessage(), getDefaultLocale());
            throw e;
        }
    }

    public void removeLocale() {
        localeHolder.remove();
    }

    /**
     * 获取租户信息
     * @param token token
     * @return 租户信息
     */
    public JSONObject getTenantInfo(String token) throws DWBusinessException {
        String tenantInfoSimpleUrl = iamUrl + "/api/iam/v2/tenant/current/simple";
        HttpHeaders headers = new HttpHeaders();
        headers.add("Content-Type", "application/json");
        headers.add("digi-middleware-auth-user", token);
        headers.add("digi-middleware-auth-app", iamApToken);
        HttpEntity<Map<String, Object>> requestEntity = new HttpEntity<>(null, headers);
        ResponseEntity<JSONObject> resEntity = restTemplate.exchange(tenantInfoSimpleUrl, HttpMethod.GET, requestEntity, JSONObject.class);
        JSONObject body = resEntity.getBody();
        if (body == null) {
            LOGGER.error("查询租户信息异常，返回body为空");
            throw new DWBusinessException("查询租户信息异常");
        } else if (MapUtils.getInteger(body, "code") != 200) {
            throw new DWBusinessException(MapUtils.getString(body, "message"));
        }
        return body.getJSONObject("data");
    }

    /**
     * 查询当前租户运营单元版本
     * @return 查询当前租户运营单元版本
     * @throws DWBusinessException io异常
     */
    public String getTenantOperationUnitVersion(String token) throws DWBusinessException {
        if (StringUtils.isEmpty(token)) {
            return "v1";
        }
        JSONObject tenantInfo = null;
        tenantInfo = getTenantInfo(token);
        return Optional.ofNullable(tenantInfo).map(t -> MapUtils.getString(t, "version", "v1")).orElse("v1");
    }

    /**
     * 获取当前租户运营单元版本
     * @return 当前租户运营单元版本
     */
    public String getTenantOperationUnitVersion() throws DWBusinessException {
        if (DWServiceContext.getContext().getProfile() == null || MapUtils.isEmpty(DWServiceContext.getContext().getProfile())) {
            // 匿名方法没有profile，或者返回的是不可变map，重新初始化一下，
            DWServiceContext.getContext().setProfile(new HashMap<>());
        }
        if (!Boolean.TRUE.equals(supportTenantV2)) {
            DWServiceContext.getContext().getProfile().put(Constants.TENANT_OPERATION_UNIT_VERSION, "v1");
            return "v1";
        }
        if (!DWServiceContext.getContext().getProfile().containsKey(Constants.TENANT_OPERATION_UNIT_VERSION) || DWServiceContext.getContext().getProfile().get(Constants.TENANT_OPERATION_UNIT_VERSION) == null) {
            Map<String, Object> headers = DWServiceContext.getContext().getRequestHeader();
            DWServiceContext.getContext().getProfile().put(Constants.TENANT_OPERATION_UNIT_VERSION, getTenantOperationUnitVersion(MapUtils.getString(headers, "token")));
        }
        String tenantOperationUnitVersion = MapUtils.getString(DWServiceContext.getContext().getProfile(), Constants.TENANT_OPERATION_UNIT_VERSION);
        return Optional.ofNullable(tenantOperationUnitVersion).orElse("v1");
    }

    /**
     * 判断当前租户运营单元版本是否为v2
     * @return boolean
     */
    public boolean isTenantOperationUnitV2() {
        try {
            return "v2".equals(getTenantOperationUnitVersion());
        } catch (DWBusinessException e) {
            LOGGER.error("get tenant operation unit version error:", e);
            return false;
        }
    }

    /**
     * 判断当前租户运营单元版本是否为v2
     * @return boolean
     */
    public static boolean isTenantOperationUnitV2FromContext() {
        String version = Optional.ofNullable(DWServiceContext.getContext().getProfile()).map(p -> MapUtils.getString(p, Constants.TENANT_OPERATION_UNIT_VERSION)).orElse("v1");
        return  "v2".equals(version);
    }

}
