package com.digiwin.athena.kmservice.utils;

import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import com.digiwin.app.container.exceptions.DWBusinessException;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.MapUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.http.HttpEntity;
import org.springframework.http.HttpHeaders;
import org.springframework.http.HttpMethod;
import org.springframework.http.ResponseEntity;
import org.springframework.stereotype.Component;
import org.springframework.util.CollectionUtils;
import org.springframework.web.client.RestTemplate;

import java.io.IOException;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;

@Slf4j
@Component
public class EocUtils {
    @Value("${eocUrl:null}")
    private String eocUrl;

    @Value("${iamApToken:null}")
    private String iamApToken;

    @Autowired
    @Qualifier("commonRestTemplate")
    RestTemplate restTemplate;

    /**
     * 获取租户信息
     * @param token token
     * @return 租户信息
     */
    public JSONObject getOperationUnitTree(String token) throws DWBusinessException {
        String url = eocUrl + "/api/eoc/v2/org/tree";
        return requestEoc(token, url);
    }

    /**
     * 获取租户信息
     * @param token token
     * @return 租户信息
     */
    public String getRootOperationUnitId(String token) throws DWBusinessException {
        String rootOperationUnitId = MapUtils.getString(ServiceUtils.getContext().getEocInfo(), "rootOperationUnitId");
        if (StringUtils.isNotBlank(rootOperationUnitId)) {
            return rootOperationUnitId;
        }
        JSONObject operationUnitTree = getOperationUnitTree(token);
        if (MapUtils.isNotEmpty(operationUnitTree) && operationUnitTree.containsKey("id")) {
            String id = MapUtils.getString(operationUnitTree, "id");
            if (ServiceUtils.getContext().getEocInfo() == null) {
                ServiceUtils.getContext().setEocInfo(new HashMap<>());
            }
            ServiceUtils.getContext().getEocInfo().put("rootOperationUnitId", id);
            return id;
        } else {
            return null;
        }
    }

    /**
     * 获取租户信息
     * @param token token
     * @return 租户信息
     */
    public String getOperationUnitPath(String token, String id) throws DWBusinessException {
        if (StringUtils.isBlank(token) || StringUtils.isBlank(id)) {
            return null;
        }
        String operationPath = MapUtils.getString(ServiceUtils.getContext().getEocInfo(), "operationPath");
        if (StringUtils.isNotBlank(operationPath)) {
            return operationPath;
        }
        String url = eocUrl + "/api/eoc/v2/org/tree/node?type=1&id="+id;
        JSONObject data = requestEoc(token, url);
        if (data != null) {
            String uri = data.getString("uri");
            if (ServiceUtils.getContext().getEocInfo() == null) {
                ServiceUtils.getContext().setEocInfo(new HashMap<>());
            }
            ServiceUtils.getContext().getEocInfo().put("operationPath", uri);
            return uri;

        }
        return null;
    }

    /**
     * 获取租户信息
     * @param token token
     * @return 租户信息
     */
    public JSONObject requestEoc(String token, String url) throws DWBusinessException {
        HttpHeaders headers = new HttpHeaders();
        headers.add("Content-Type", "application/json");
        headers.add("digi-middleware-auth-user", token);
        headers.add("digi-middleware-auth-app", iamApToken);
        HttpEntity<Map<String, Object>> requestEntity = new HttpEntity<>(null, headers);
        ResponseEntity<JSONObject> resEntity = restTemplate.exchange(url, HttpMethod.GET, requestEntity, JSONObject.class);
        JSONObject body = resEntity.getBody();
        if (body == null) {
            log.error("查询租户组织树异常，返回body为空");
            throw new DWBusinessException("查询租户信息异常");
        } else if (MapUtils.getInteger(body, "code") != 200) {
            throw new DWBusinessException(MapUtils.getString(body, "message"));
        }
        return body.getJSONObject("data");
    }

    /**
     * 获取运营单元id集合
     * @param token token
     * @return 运营单元id集合
     * @throws IOException 获取异常
     */
    public List<String> getOperationUnitIdList(String token) throws DWBusinessException {
        JSONObject operationUnitTree = getOperationUnitTree(token);
        // 保证有序，第一个是根节点
        List<String> idList = new LinkedList<>();
        // 递归从operationUnitTree获取id
        extractIds(operationUnitTree, idList);
        return idList;
    }

    private void extractIds(JSONObject node, List<String> idList) {
        if (node == null) {
            return;
        }

        String id = node.getString("id");
        if (StringUtils.isNotBlank(id)) {
            idList.add(id);
        }

        JSONArray children = node.getJSONArray("children");
        if (!CollectionUtils.isEmpty(children)) {
            for (int i = 0; i < children.size(); i++) {
                extractIds(children.getJSONObject(i), idList);
            }
        }
    }
}
