package com.digiwin.athena.kmservice.neo4j;

import com.digiwin.athena.kmservice.neo4j.Cql;
import org.neo4j.driver.*;
import org.neo4j.driver.exceptions.TransientException;
import org.neo4j.driver.internal.value.*;
import org.neo4j.driver.util.Pair;


import java.util.*;

/**
 * @author liyuetao
 * @title: BaseNeo4jManager
 * @projectName athena_deployer_service
 * @description: TODO
 * @date 2022/9/1310:13
 */
public class Neo4jManager {

    private Driver driver;

    private static final Integer RETRY_TIMES = 4;

    public Neo4jManager(Driver driver) {
        this.driver = driver;
    }

    public void ExecuteNoQuery(String statement) {
        try (Session session = driver.session()) {
            session.run(statement);
        }
    }

    public void ExecuteNoQuery(String statement, Map<String, Object> params) {
        try (Session session = driver.session()) {
            session.run(statement, params);
        }
    }

    public List<Map<String, Object>> ExecuteQuery(String statement) {
        try (Session session = driver.session()) {
            Result result = session.run(statement);
            List<Map<String, Object>> mapObjects = new ArrayList<>();
            while (result.hasNext()) {
                Record record = result.next();
                List<Pair<String, Value>> fields = record.fields();
                Map<String, Object> recordAsMap = new HashMap<>();
                for (Pair<String, Value> field : fields) {
                    recordAsMap.put(field.key(), ParseNode(field.value()));
                }
                mapObjects.add(recordAsMap);
            }
            return mapObjects;
        } catch (Exception e) {
            throw e;
        }
    }

    public List<Map<String, Object>> ExecuteQuery(String statement, Map<String, Object> params) {
        try (Session session = driver.session()) {
            Result result = session.run(statement, params);
            List<Map<String, Object>> mapObjects = new ArrayList<>();
            while (result.hasNext()) {
                Record record = result.next();
                List<Pair<String, Value>> fields = record.fields();
                Map<String, Object> recordAsMap = new HashMap<>();
                for (Pair<String, Value> field : fields) {
                    recordAsMap.put(field.key(), ParseNode(field.value()));
                }
                mapObjects.add(recordAsMap);
            }
            return mapObjects;
        } catch (Exception e) {
            throw e;
        }
    }

    public void ExecuteTransactionNoQuery(List<Cql> statementsAndParams) {
        for (int i = 0; i < RETRY_TIMES; i++) {
            try (Session session = driver.session()) {
                try (Transaction tx = session.beginTransaction()) {
                    for (Cql cql : statementsAndParams) {
                        if (Objects.isNull(cql.getParams()))
                            tx.run(cql.getCql());
                        else
                            tx.run(cql.getCql(), cql.getParams());
                    }
                    tx.commit();
                }
                return;
            } catch (TransientException e) {
                String message = e.getMessage();
                if (message.startsWith("LockClient")) {
                    if (i < RETRY_TIMES - 1) {
                        try {
                            Thread.sleep(1000);
                        } catch (InterruptedException ex) {
                            Thread.currentThread().interrupt();
                            throw new RuntimeException(ex);
                        }
                    } else {
                        throw e;
                    }

                } else {
                    throw e;
                }
            }
        }
    }

    public void ExecuteOnCqlList(List<Cql> statementsAndParams) {
        try (Session session = driver.session()) {
            statementsAndParams.forEach(cql -> {
                session.run(cql.getCql(), cql.getParams());
            });
        }
    }

    private static Object ParseNode(Object value) {
        if (value instanceof NodeValue) {
            return ((Value) value).asNode();
        } else if (value instanceof StringValue) {
            return ((Value) value).asString();
        } else if (value instanceof BooleanValue) {
            return ((Value) value).asBoolean();
        } else if (value instanceof IntegerValue) {
            long val = ((IntegerValue) value).asLong();
            if (val > Integer.MAX_VALUE || val < Integer.MIN_VALUE) {
                return val;
            } else return ((Value) value).asInt();
        } else if (value instanceof FloatValue) {
            return ((Value) value).asFloat();
        } else if (value instanceof ListValue) {
            ArrayList<Object> parsedValue = new ArrayList<>();
            List<Object> valueAsList = ((Value) value).asList();
            for (Object element : valueAsList) {
                parsedValue.add(ParseNode(element));
            }
            return parsedValue;
        } else if (value instanceof MapValue){
            return ((MapValue) value).asMap();
        } else {
            return value;
        }
    }
}
