package com.digiwin.athena.kmservice.configuration;

import com.digiwin.app.eai.DWEAIProperties;
import com.digiwin.app.module.spring.SpringContextUtils;
import com.digiwin.athena.kmservice.support.KmRestTemplate;
import com.digiwin.athena.kmservice.utils.SpringAopUtils;
import com.digiwin.http.client.DWHttpClientProperties;
import com.digiwin.http.client.utils.DWRequestHeaderUtils;
import com.digiwin.loadbalance.util.HttpRouteUtils;
import org.apache.http.client.HttpClient;
import org.apache.http.client.config.RequestConfig;
import org.apache.http.config.Registry;
import org.apache.http.config.RegistryBuilder;
import org.apache.http.conn.routing.HttpRoutePlanner;
import org.apache.http.conn.socket.ConnectionSocketFactory;
import org.apache.http.conn.socket.PlainConnectionSocketFactory;
import org.apache.http.conn.ssl.SSLConnectionSocketFactory;
import org.apache.http.impl.client.HttpClientBuilder;
import org.apache.http.impl.client.HttpClients;
import org.apache.http.impl.conn.PoolingHttpClientConnectionManager;
import org.springframework.beans.factory.SmartInitializingSingleton;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.annotation.Primary;
import org.springframework.http.HttpHeaders;
import org.springframework.http.HttpRequest;
import org.springframework.http.client.ClientHttpRequestExecution;
import org.springframework.http.client.ClientHttpRequestInterceptor;
import org.springframework.http.client.ClientHttpResponse;
import org.springframework.http.client.HttpComponentsClientHttpRequestFactory;
import org.springframework.web.client.RestTemplate;

import java.io.IOException;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;

@Configuration
public class CommonRestTemplateConfiguration implements SmartInitializingSingleton {
    /**
     * 根据需要使用 HttpRouteUtils.DEFAULT_HTTPCLIENT_ROUTE_PLAN
     * 或者   HttpRouteUtils.ATTEMPT_HTTPCLIENT_ROUTE_PLAN （设置此值时 需设置 dwhttpClient.loadbalance.mode=auto    ）
     */
    @Autowired(required = false)
    @Qualifier(HttpRouteUtils.ATTEMPT_HTTPCLIENT_ROUTE_PLAN)
    HttpRoutePlanner httpRoutePlanner;


    @Primary
    @Bean
    public KmRestTemplate commonRestTemplate() {
        KmRestTemplate restTemplate = createTemplate();
        List<ClientHttpRequestInterceptor> interceptorList = new ArrayList<>();
        interceptorList.add(new AttachAppInfoInterceptor());
        restTemplate.setInterceptors(interceptorList);
        return restTemplate;


    }

    @Bean(name = "simpleRestTemplate")
    public RestTemplate simpleRestTemplate() {
        RestTemplate restTemplate = new RestTemplate();
        List<ClientHttpRequestInterceptor> interceptorList = new ArrayList<>();
        interceptorList.add(new AttachAppInfoInterceptor());
        restTemplate.setInterceptors(interceptorList);
        return restTemplate;
    }

    private KmRestTemplate createTemplate() {
        DWEAIProperties dweaiProperties = new DWEAIProperties();
        dweaiProperties.setRestfulServiceBasePath("");
        DWEAIProperties.setProperties(dweaiProperties);
        HttpClientBuilder httpClientBuilder = HttpClients.custom();
        if (null != httpRoutePlanner) {
            httpClientBuilder.setRoutePlanner(httpRoutePlanner);
        }
        Registry<ConnectionSocketFactory> registry = RegistryBuilder.<ConnectionSocketFactory>create()
                .register("http", PlainConnectionSocketFactory.getSocketFactory())
                .register("https", SSLConnectionSocketFactory.getSocketFactory())
                .build();
        PoolingHttpClientConnectionManager connectionManager = new PoolingHttpClientConnectionManager(registry);
        /** 设置整个连接池最大连接数 */
        connectionManager.setMaxTotal(1024);
        /** 路由是对maxTotal的细分 */
        connectionManager.setDefaultMaxPerRoute(256);
        /** 空闲存活时间 30s */
        connectionManager.setValidateAfterInactivity(10000);
        RequestConfig requestConfig = RequestConfig.custom()
                /** 返回数据的超时时间 */
                .setSocketTimeout(30000)
                /** 连接上服务器的超时时间 */
                .setConnectTimeout(5000)
                /** 从连接池中获取连接的超时时间 */
                .setConnectionRequestTimeout(5000)
                .build();
        httpClientBuilder.setDefaultRequestConfig(requestConfig);
        httpClientBuilder.setConnectionManager(connectionManager);
        HttpClient httpClient = httpClientBuilder.build();
        KmRestTemplate  restTemplate= new KmRestTemplate(new HttpComponentsClientHttpRequestFactory(httpClient));
        restTemplate.setConnectionManager(connectionManager);
        return restTemplate;
    }

    @Override
    public void afterSingletonsInstantiated() {
        if (SpringContextUtils.getContext() != null) {
            Map<String, RestTemplate> restTemplateMap = SpringContextUtils.getContext().getBeansOfType(RestTemplate.class);
            List<RestTemplate> beanList = restTemplateMap.isEmpty() ? new ArrayList<>() : new ArrayList<>(restTemplateMap.values());
            if (!beanList.isEmpty()) {
                for (RestTemplate restTemplate : beanList) {
                    List<ClientHttpRequestInterceptor> clientHttpRequestInterceptorList = SpringAopUtils.getSingletonTarget(restTemplate).getInterceptors();
                    boolean hasAttachAppInfoInterceptor = false;
                    for (ClientHttpRequestInterceptor interceptor : clientHttpRequestInterceptorList) {
                        if (interceptor instanceof AttachAppInfoInterceptor) {
                            hasAttachAppInfoInterceptor = true;
                            break;
                        }
                    }
                    if (!hasAttachAppInfoInterceptor) {
                        clientHttpRequestInterceptorList.add(0, new AttachAppInfoInterceptor());
                    }
                }
            }
        }
    }

    /**
     * 附加應用信息切片
     */
    public static class AttachAppInfoInterceptor implements ClientHttpRequestInterceptor {
        @Override
        public ClientHttpResponse intercept(HttpRequest request, byte[] body, ClientHttpRequestExecution execution) throws IOException {

            AttachAppInfoInterceptor.attachAppAuthInfo(request.getURI().getHost(), request);

            return execution.execute(request, body);
        }

        /**
         * 附加應用授權信息
         *
         * @param host    請求目標
         * @param request 請求
         */
        public static void attachAppAuthInfo(String host, HttpRequest request) {

            HttpHeaders httpHeaders = request.getHeaders();
            // 1.檢查是否存在 app token 信息
            if (!httpHeaders.containsKey(DWRequestHeaderUtils.HEADER_IAM_API_APP_TOKEN)) {

                // 當環境中設置了 app token 且沒有排除此 host 時, 加上 app token 信息
                String apToken = DWHttpClientProperties.getProperties().getIamApToken();
                //暂无需根据请求的host判断,只对指定的restTemplate附加该Interceptor
                //if (apToken != null && !apToken.isEmpty() && !DWHttpClientProperties.getProperties()
                // .getAttachAppAuthInfoProperties().isExclude(host)) {
                if (apToken != null && !apToken.isEmpty()) {
                    httpHeaders.add(DWRequestHeaderUtils.HEADER_IAM_API_APP_TOKEN, apToken);
                }
            }
            // 2.檢查是否存在 app secret 信息(未纳入,暂不启用)
            /*if (!httpHeaders.containsKey(DWRequestHeaderUtils.HEADER_IAM_API_APP_SECRET)) {

                // 當環境中設置了 app token 且沒有排除此 host 時, 加上 app secret 信息
                String apSecret = DWHttpClientProperties.getProperties().getIamApSecret();
                if (apSecret != null && !apSecret.isEmpty() && !DWHttpClientProperties.getProperties()
                .getAttachAppAuthInfoProperties().isExclude(host))
                    httpHeaders.add(DWRequestHeaderUtils.HEADER_IAM_API_APP_SECRET, apSecret);
            }*/
        }
    }


}
