package com.digiwin.athena.kmservice.action.execution;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import com.digiwin.app.container.exceptions.DWBusinessException;
import com.digiwin.app.container.exceptions.DWException;
import com.digiwin.athena.kg.monitorRule.TenantServiceConfig;
import com.digiwin.athena.kmservice.common.Constants;
import com.digiwin.athena.kmservice.povo.productOperation.ProductOperationResult;
import com.digiwin.athena.kmservice.service.KmTenantService;
import com.digiwin.athena.kmservice.utils.GsonUtils;
import com.digiwin.athena.kmservice.utils.I18nUtils;
import com.digiwin.athena.kmservice.utils.KmHttpUtil;
import com.google.gson.reflect.TypeToken;
import com.jugg.agile.framework.core.dapper.alarm.JaAlarmMessage;
import com.jugg.agile.framework.core.dapper.alarm.adapter.qywx.JaQyWxAlarm;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.http.HttpEntity;
import org.springframework.http.HttpHeaders;
import org.springframework.http.HttpMethod;
import org.springframework.http.ResponseEntity;
import org.springframework.stereotype.Service;
import org.springframework.util.ObjectUtils;
import org.springframework.web.client.RestClientException;
import org.springframework.web.client.RestTemplate;

import java.io.IOException;
import java.time.LocalDateTime;
import java.util.*;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;

@Service
public class ProductNameResolver {
    private static Logger LOGGER = LoggerFactory.getLogger(ProductNameResolver.class);
    @Autowired
    KmTenantService kmTenantService;
//    @Autowired
//    private CacheService cacheService;

    @Autowired
    @Qualifier("commonRestTemplate")
    RestTemplate restTemplate;

    // jugg告警
    private static final JaQyWxAlarm jaQyWxAlarm = new JaQyWxAlarm("Mdc");
    static {
        jaQyWxAlarm.setUrl(Constants.JA_ALARM_URI);
    }


    public String[] getPrioritizedProductList(String serviceName){

        TenantServiceConfig serviceConfig = kmTenantService.getTenantServiceConfigByServiceName(serviceName);


        return (serviceConfig == null || StringUtils.isEmpty(serviceConfig.getProductName())) ? null : StringUtils.split(serviceConfig.getProductName(), ",");
    }


    private String getMatchedSystemProduct(List<String> tenantProductList, String serviceName){
        if(CollectionUtils.isEmpty(tenantProductList)){
            return null;
        }
        String[] prioritizedProductList = this.getPrioritizedProductList(serviceName);
        if(prioritizedProductList == null){
            return null;
        }

        for(int i=0; i<prioritizedProductList.length; i++){
            if(tenantProductList.contains(prioritizedProductList[i])){
                return prioritizedProductList[i];
            }
        }

        return null;
    }

    public List<String> getCloudProductNames(String tenantId) throws Exception {
        LOGGER.info("getCloudProductNames  tenantId:{}", tenantId);
        List<String> productNameList = new ArrayList<>();
        try {
            JSONObject response = JSON.parseObject(JSON.toJSONString(tenantProductList(tenantId,null)), JSONObject.class);
            LOGGER.info("tenantProductOperationList response:{}", response);
            JSONArray gateways = response.getJSONObject("data").getJSONArray("gateway");
            boolean found = false;

            for (Object gateway : gateways) {
                JSONArray products = ((JSONObject)gateway).getJSONArray("product");
                for (Object product : products) {
    //                if (((JSONObject)product).getString("uid").equals("")) {
                    //保留获取到的所有产品信息，稳态可能会有多个
                    productNameList.add(((JSONObject)product).getString("name"));
    //                }
                }
            }
        } catch (IOException e) {
            LOGGER.error("getCloudProductNames error:{}", e);
        }
        return productNameList;
    }

    public List<String> getCloudProductNamesWithRetry(String tenantId, int maxRetry) throws Exception{
        List<String> cloudProductNames = null;
        try {
            cloudProductNames = this.getCloudProductNames(tenantId);
        } catch (Exception e) {
            // 告警埋点
            String message = JaAlarmMessage.getCommonMessage();
            message += "getCloudProductNamesWithRetry2,Retry=" + maxRetry;
            message += e.toString();
            jaQyWxAlarm.alarm(message);
            if(maxRetry > 0){
                TimeUnit.MILLISECONDS.sleep(50);
                LOGGER.info("getCloudProductNamesWithRetry error, begin retry max retry:{} , time:{}", maxRetry, LocalDateTime.now());
                cloudProductNames = this.getCloudProductNamesWithRetry(tenantId, --maxRetry);
            }else{
//                e.printStackTrace();
                LOGGER.error("getCloudProductNamesWithRetryError:{}", e);
                throw new DWException("P.MDC.500.9999", I18nUtils.getValue("knowledgegraph.knowledgegraphError") + e.getMessage(), e);
            }
        }

        return cloudProductNames;
    }

    public List<String> getCloudProductNamesAfterEocSet(String tenantId) throws Exception {
        LOGGER.info("getCloudProductNamesAfterEocSet  tenantId:{}", tenantId);
        List<String> productNameList = new ArrayList<>();
        try {
            JSONObject response = JSON.parseObject(JSON.toJSONString(tenantProductOperationList(tenantId)), JSONObject.class);
            LOGGER.info("tenantProductOperationList response:{}", response);
            JSONArray gateways = response.getJSONArray("prod_eoc_mapping");
            if(gateways == null){
                return productNameList;
            }
            Set<String> prodcutSet = new HashSet<>();
            for (Object gateway : gateways) {
                String product = ((JSONObject)gateway).getString("prod_name");
                if(prodcutSet.contains(product)){
                    continue;
                }
                productNameList.add(product);
                prodcutSet.add(product);
            }
        } catch (IOException e) {
            LOGGER.error("TenantProductOperationListError:{}", e);
        }
        return productNameList;
    }
    public Object tenantProductList(String tenantId,String serviceName) throws Exception {
        LOGGER.info("tenantProductList  tenantId:{}", tenantId);
        String apiMetadataUrl = ESPConfiguration.getEspMdcUrl() + "/restful/standard/mdc/TenantProductList/Get";

        HttpHeaders headers = new HttpHeaders();
        headers.add("Content-Type", "application/json");
        KmHttpUtil.requiredHeaders(headers);
        Map<String, Object> requestMap = new HashMap<>();
        requestMap.put("tenant_id", tenantId);
        if (StringUtils.isNotEmpty(serviceName)) {
            requestMap.put("api_name", serviceName);
        }
        HttpEntity<Map> requestEntity = new HttpEntity<>(requestMap, headers);
        ResponseEntity<Object> resEntity = restTemplate.exchange(apiMetadataUrl, HttpMethod.POST, requestEntity, Object.class);
        Object body = resEntity.getBody();
        LOGGER.info("tenantProductListResponse  tenantId:{}, response:{}", tenantId, body);
        return body;
    }

    /**
     * 批量获取MDC产品名称
     * @param tenantId      租户ID
     * @param serviceNames  服务名称集合
     * @return  Object
     * @throws Exception
     */
    public Object batchTenantProductList(String tenantId, List<String> serviceNames) throws Exception {
        if (StringUtils.isEmpty(tenantId) || CollectionUtils.isEmpty(serviceNames)) {
            return null;
        }
        LOGGER.info("batchTenantProductList tenantId:{}，serviceNames.size:{}", tenantId, serviceNames.size());
        String apiMetadataUrl = ESPConfiguration.getEspMdcUrl() + "/restful/standard/mdc/TenantApiProduct/batch/get";
        HttpHeaders headers = new HttpHeaders();
        headers.add("Content-Type", "application/json");
        KmHttpUtil.requiredHeaders(headers);
        // 请求入参
        Map<String, Object> requestMap = new HashMap<>();
        requestMap.put("tenantId", tenantId);
        if (CollectionUtils.isNotEmpty(serviceNames)) {
            requestMap.put("apiNames", serviceNames);
        }
        // http请求entity
        HttpEntity<Map<String, Object>> requestEntity = new HttpEntity<>(requestMap, headers);
        // http响应体
        ResponseEntity<Object> resEntity = restTemplate.exchange(apiMetadataUrl, HttpMethod.POST, requestEntity, Object.class);
        Object body = resEntity.getBody();
        LOGGER.info("batchTenantProductList Response tenantId:{}, response:{}", tenantId, body);
        return body;
    }

    public Object tenantProductOperationList(String tenantId) throws Exception {
        try {
            LOGGER.info("tenantProductOperationList  tenantId:{}", tenantId);
            String apiMetadataUrl = ESPConfiguration.getEspMdcUrl() + "/restful/standard/mdc/TenantProductOperationList/Get";

            HttpHeaders headers = new HttpHeaders();
            KmHttpUtil.requiredHeaders(headers);
            headers.add("Content-Type", "application/json");

            Map<String, Object> requestMap = new HashMap<>();
            requestMap.put("tenant_id", tenantId);
            HttpEntity<Map> requestEntity = new HttpEntity<>(requestMap, headers);
            ResponseEntity<Object> resEntity = restTemplate.exchange(apiMetadataUrl, HttpMethod.POST, requestEntity, Object.class);
            Object body = resEntity.getBody();
            LOGGER.info("tenantProductOperationListResponse  tenantId:{}, response:{}", tenantId, body);
            return body;
        } catch (Exception e) {
            // 告警埋点
            String message = JaAlarmMessage.getCommonMessage();
            message += "tenantProductOperationList,Retry=1";
            message += e.toString();
            jaQyWxAlarm.alarm(message);
            throw e;
        }
    }

    /**
     * 调用的是/restful/standard/mdc/TenantProductOperationList/Get接口，主要给交付设计器获取租户产品使用
     * @param tenantId
     * @param maxRetry
     * @return
     * @throws Exception
     *
     * 不加缓存
     */
//    @Cache(namespace = REDIS_KNOWLEDGE_GRAPH, key = "ProductNameAll$0",  ttlSecs = 86400L)
    public List<String> getCloudProductNamesWithRetryAfterEocSet(String tenantId, int maxRetry) throws Exception{
        List<String> cloudProductNames = null;
        try {
            cloudProductNames = this.getCloudProductNamesAfterEocSet(tenantId);
        } catch (Exception e) {
            // 告警埋点
            String message = JaAlarmMessage.getCommonMessage();
            message += "getCloudProductNamesWithRetryAfterEocSet2,Retry=" + maxRetry;
            message += e.toString();
            jaQyWxAlarm.alarm(message);
            if(maxRetry > 0){
                TimeUnit.MILLISECONDS.sleep(50);
                LOGGER.info("getCloudProductNamesWithRetryAfterEocSet error, begin retry max retry:{} , time:{}", maxRetry, LocalDateTime.now());
                cloudProductNames = this.getCloudProductNamesWithRetryAfterEocSet(tenantId, --maxRetry);
            }else{
//                e.printStackTrace();
                LOGGER.error("getCloudProductNamesWithRetryAfterEocSet:{}", e);
                throw new Exception(I18nUtils.getValue("knowledgegraph.knowledgegraphError"), e);
            }

        }

        return cloudProductNames;
    }

    /**
     * 批量获取产品名称
     * 控制批次数量，每批次200
     * @param tenantId      租户ID
     * @param serviceNames  服务名称列表
     * @param maxRetry      最大重试次数
     * @return  Map<String, List<Map<String, String>>>
     * @throws Exception
     */
    public Map<String, List<Map<String, String>>> batchGetCloudProductNamesWithRetry(String tenantId, List<String> serviceNames, int maxRetry) throws Exception {
        Map<String, List<Map<String, String>>> serviceProductMap = new HashMap<>();
        LOGGER.info("batchGetCloudProductNamesWithRetry serviceNames:{}, tenantId:{}, maxRetry:{}", JSON.toJSONString(serviceNames), tenantId, maxRetry);
        try {
            JSONObject response = JSON.parseObject(JSON.toJSONString(batchTenantProductList(tenantId, serviceNames)), JSONObject.class);
            LOGGER.info("batchGetCloudProductNamesWithRetry response:{}", response);
            JSONObject data = response.getJSONObject("data");
            if(data != null){
                JSONObject serviceProductMapJson = data.getJSONObject("serviceProductMap");
                serviceProductMap = GsonUtils.fromJson(serviceProductMapJson.toJSONString(), new TypeToken<Map<String, List<Map<String, String>>>>(){});
            }
        } catch (Exception e) {
            // 告警埋点
            String message = JaAlarmMessage.getCommonMessage();
            message += "batchGetCloudProductNamesWithRetry3,Retry=" + maxRetry;
            message += e.toString();
            jaQyWxAlarm.alarm(message);
            if(maxRetry > 0){
                TimeUnit.MILLISECONDS.sleep(1000);
                LOGGER.info("batchGetCloudProductNamesWithRetry error, begin retry max retry:{} , time:{}", maxRetry, LocalDateTime.now());
                serviceProductMap = this.batchGetCloudProductNamesWithRetry(tenantId, serviceNames, --maxRetry);
            }else{
                LOGGER.error("batchGetCloudProductNamesWithRetry:{}", e.toString());
                throw new DWBusinessException("P.KG.503.0002",I18nUtils.getValue("knowledgegraph.knowledgegraphError"), e);
            }
        }
        return serviceProductMap;
    }

    public List<String> getCloudProductNames(String serviceName, String tenantId) throws Exception {
        LOGGER.info("getCloudProductNames serviceName:{}, tenantId:{}", serviceName, tenantId);
        JSONObject response = JSON.parseObject(JSON.toJSONString(tenantProductList(tenantId,serviceName)), JSONObject.class);
        LOGGER.info("getCloudProductNamesReponse serviceName:{}, tenantId:{} response:{}", serviceName, tenantId, response);
        List<String> productNameList = new ArrayList<>();
        JSONObject data = response.getJSONObject("data");
        if(data == null){
            return productNameList;
        }
        JSONArray gateways = data.getJSONArray("gateway");
        if(gateways == null){
            return productNameList;
        }
        boolean found = false;
        for (Object gateway : gateways) {
            JSONArray products = ((JSONObject)gateway).getJSONArray("product");
            for (Object product : products) {
//                if (((JSONObject)product).getString("uid").equals("")) {
                    //保留获取到的所有产品信息，稳态可能会有多个
                    productNameList.add(((JSONObject)product).getString("name"));
//                }
            }
        }
        return productNameList;
    }


    public List<String> getCloudProductNamesWithRetry(String serviceName, String tenantId, int maxRetry) throws Exception{
        List<String> cloudProductNames = null;
        try {
            cloudProductNames = this.getCloudProductNames(serviceName, tenantId);
        } catch (Exception e) {
            // 告警埋点
            String message = JaAlarmMessage.getCommonMessage();
            message += "getCloudProductNamesWithRetry3,Retry=" + maxRetry;
            message += e.toString();
            jaQyWxAlarm.alarm(message);
            if(maxRetry > 0){
                TimeUnit.MILLISECONDS.sleep(1000);
                LOGGER.info("getCloudProductNamesWithRetry error, begin retry max retry:{} , time:{}", maxRetry, LocalDateTime.now());
                cloudProductNames = this.getCloudProductNamesWithRetry(serviceName, tenantId, --maxRetry);
            }else{
//                e.printStackTrace();
                LOGGER.error("getCloudProductNamesWithRetryError:{}", e);
                throw new DWBusinessException("P.KG.503.0002",I18nUtils.getValue("knowledgegraph.knowledgegraphError"), e);
            }
        }

        return cloudProductNames;
    }

    /**
     * 批量设置Map结构的Action产品名称
     *
     * @param data     map结构的action
     * @param tenantId 租户ID
     * @throws Exception
     */
    private void batchSetProductInfo(Map<String, Object> data, String tenantId) throws Exception {
        if (org.springframework.util.CollectionUtils.isEmpty(data)) {
            return;
        }
        // 定义Map类型的data平铺接收容器
        List<Map<String, Object>> actionMap = new ArrayList<>();
        // 递归Map嵌套类型并平铺为actionMap
        batchMapSetActionMap(data, tenantId, actionMap);
        // 批量设置Action产品名称
        batchSetDataActionProductInfo(actionMap, tenantId);
    }

    /**
     * 批量设置Map结构的Action产品名称
     *
     * @param data    list map结构的action
     * @param tenantId 租户ID
     * @throws Exception exception
     */
    public void batchSetProductInfoForList(List<Map<String, Object>> data, String tenantId) throws Exception {
        List<Map<String, Object>> actionMap = new ArrayList<>();
        batchListSetActionMap(data, tenantId, actionMap);
        batchSetDataActionProductInfo(actionMap, tenantId);
    }

    /**
     * 递归Map嵌套类型并平铺为actionMap
     *
     * @param data      Map数据源
     * @param tenantId  租户ID
     * @param actionMap 目标action集合
     * @throws Exception
     */
    private void batchMapSetActionMap(Map<String, Object> data, String tenantId, List<Map<String, Object>> actionMap) {
        if (org.springframework.util.CollectionUtils.isEmpty(data)) {
            return;
        }
        // map结构的数据加入actionMap
        actionMap.add(data);
        for (Map.Entry<String, Object> item : data.entrySet()) {
            Object value = item.getValue();
            if (value instanceof Map) {
                batchMapSetActionMap((Map) value, tenantId, actionMap);
            } else if (value instanceof List) {
                batchListSetActionMap((List) value, tenantId, actionMap);
            }
        }
    }

    /**
     * 递归List的Map嵌套类型并平铺为actionMap
     *
     * @param data      List数据源
     * @param tenantId  租户ID
     * @param actionMap 目标action集合
     * @throws Exception
     */
    private void batchListSetActionMap(List data, String tenantId, List<Map<String, Object>> actionMap) {
        if (org.springframework.util.CollectionUtils.isEmpty(data)) {
            return;
        }
        for (Object item : data) {
            if (item instanceof Map) {
                batchMapSetActionMap((Map) item, tenantId, actionMap);
            } else if (item instanceof List) {
                batchListSetActionMap((List) item, tenantId, actionMap);
            }
        }
    }

    /**
     * 批量设置Action产品名称
     *
     * @param actions  actions
     * @param tenantId 租户ID
     * @throws Exception
     */
    private void batchSetDataActionProductInfo(List<Map<String, Object>> actions, String tenantId) throws Exception {
        if (org.springframework.util.CollectionUtils.isEmpty(actions)) {
            return;
        }
        // 过滤出产品名称为空且是ESP类型的并且serviceName不为空
        actions = actions.stream()
                .filter(action -> (ObjectUtils.isEmpty(action.get("productName")))
                        && "ESP".equalsIgnoreCase(String.valueOf(action.get("type")))
                        && StringUtils.isNotEmpty(String.valueOf(action.get("serviceName"))))
                .collect(Collectors.toList());
        // 再次判空处理
        if (org.springframework.util.CollectionUtils.isEmpty(actions)) {
            return;
        }
        // 获取serviceName集合
        List<String> serviceNames = actions.stream().map(action -> String.valueOf(action.get("serviceName"))).distinct().collect(Collectors.toList());
        Map<String, String> serviceNameMap = getProductNameMap(tenantId, serviceNames);
        // 设置产品名称
        for (Map<String, Object> action : actions) {
            action.put("productName", serviceNameMap.get(String.valueOf(action.get("serviceName"))));
        }
    }

    /**
     * 批量获取产品名称Map
     * serviceName->productName键值对
     * @param tenantId      租户ID
     * @param serviceNames  去重后的服务名称列表
     * @return Map<String, String>
     */
    public Map<String, String> getProductNameMap(String tenantId, List<String> serviceNames) throws Exception {
        if (StringUtils.isEmpty(tenantId) || CollectionUtils.isEmpty(serviceNames)) {
            return Collections.emptyMap();
        }
        // 定义返回
        Map<String, String> serviceNameMap = new HashMap<>();
        // 定义重试次数
        final int maxRetry = 2;
        // 每200一个批次获取结果数据
        final int batchCount = 200;
        // 定义批次接收数据集
        Map<String, List<Map<String, String>>> serviceProductMap = new HashMap<>();
        if (serviceNames.size() > batchCount) {
            final int page = (serviceNames.size() / batchCount + (serviceNames.size() % batchCount == 0 ? 0 : 1));
            for (int i = 0; i < page; i++) {
                final int startJ = i * batchCount;
                final int endJ = Math.min((i + 1) * batchCount, serviceNames.size());
                serviceProductMap.putAll(this.batchGetCloudProductNamesWithRetry(tenantId, serviceNames.subList(startJ, endJ), maxRetry));
            }
        } else {
            // 大部分业务场景数据不超过200条，无需做集合截取处理
            serviceProductMap = this.batchGetCloudProductNamesWithRetry(tenantId, serviceNames, maxRetry);
        }
        // 遍历serviceProductMap并生成serviceNameMap
        for (Map.Entry<String, List<Map<String, String>>> entry : serviceProductMap.entrySet()) {
            String k = entry.getKey();
            List<Map<String, String>> v = entry.getValue();
            if (v.size() == 1) {
                serviceNameMap.put(k, v.get(0).get("name"));
            } else if (v.size() > 1) {
                List<String> cloudProductNames = v.stream().map(item -> item.get("name")).collect(Collectors.toList());
                //当获取到多个时，需要和数据库配置的租户产品信息做匹配
                TenantServiceConfig tenantServiceConfig = kmTenantService.findTenantServiceConfig(tenantId, k);
                if (tenantServiceConfig != null) {
                    //数据库有配置，则匹配记录
                    Optional<String> result = cloudProductNames.stream().filter(str -> str.equals(tenantServiceConfig.getProductName())).findFirst();
                    //匹配成功
                    if (result.isPresent()) {
                        serviceNameMap.put(k, result.get());
                        continue;
                    }
                }
                // 数据库未配置或者没有匹配成功继续下面匹配
                String matchedSystemProduct = this.getMatchedSystemProduct(cloudProductNames, k);
                if(StringUtils.isNotEmpty(matchedSystemProduct)){
                    serviceNameMap.put(k, matchedSystemProduct);
                }else{
                    throw new DWBusinessException("P.KG.500.0025",I18nUtils.getValue("knowledgegraph.multipleProduct", tenantId, k, cloudProductNames));
                }
            } else {
                // 没有获取到产品名称
                serviceNameMap.put(k, "");
                LOGGER.warn("serviceProductMap 存在产品名称为null的问题数据 tenantId：{}， serviceName：{}", tenantId, k);
            }
        }
        return serviceNameMap;
    }

    //禁用缓存  mdc已有缓存
    public String getProductName(String serviceName, String tenantId)  throws Exception{
        //从esp获取产品信息
        List<String> cloudProductNames = this.getCloudProductNamesWithRetry(serviceName, tenantId,2);
        if(CollectionUtils.isEmpty(cloudProductNames)){
            return "";
        }
        //当只有一个产品的时候也入库，避免后期api新增产品时，原先对应的产品错误
        if(cloudProductNames.size() == 1) {
            return cloudProductNames.get(0);
        }
        //当获取到多个时，需要和数据库配置的租户产品信息做匹配
        TenantServiceConfig tenantServiceConfig = kmTenantService.findTenantServiceConfig(tenantId, serviceName);

        String matchedSystemProduct = this.getMatchedSystemProduct(cloudProductNames, serviceName);
        if(tenantServiceConfig == null){ //若数据库没配置租户对应服务信息，取一条esp获取的信息保存，并返回???
            if(StringUtils.isEmpty(matchedSystemProduct)){
                throw new DWBusinessException("P.KG.500.0025",I18nUtils.getValue("knowledgegraph.multipleProduct",tenantId,serviceName,cloudProductNames));
            }else{
                return matchedSystemProduct;
            }
//            tenantServiceConfigDAO.addTenantServiceConfig(tenantId,serviceName,cloudProductNames.get(0));
//            return cloudProductNames.get(0);
        }
        //数据库有配置，则匹配记录
        Optional<String> result = cloudProductNames.stream().filter(str -> str.equals(tenantServiceConfig.getProductName())).findFirst();
        if(result.isPresent()){  //匹配成功，直接返回
            return result.get();
        }else{
            if(StringUtils.isEmpty(matchedSystemProduct)){
                throw new DWBusinessException("P.KG.500.0025",I18nUtils.getValue("knowledgegraph.multipleProduct",tenantId,serviceName,cloudProductNames));
            }else{
                return matchedSystemProduct;
            }
            //匹配失败，如何处置? （把数据库信息置为失效，然后保存一条esp获取的最新数据??）
//            tenantServiceConfigDAO.disableTenantService(tenantId,serviceName);
//            tenantServiceConfigDAO.addTenantServiceConfig(tenantId,serviceName,cloudProductNames.get(0));
//            return cloudProductNames.get(0);
        }
    }

    public void setProductNames(String serviceName, String tenantId, String productName) throws Exception{
        //当获取到多个时，需要和数据库配置的租户产品信息做匹配
        boolean changed = false;
        TenantServiceConfig tenantServiceConfig = kmTenantService.findTenantServiceConfig(tenantId, serviceName);
        if(tenantServiceConfig == null){
            changed = true;
            kmTenantService.addTenantServiceConfig(tenantId,serviceName,productName);
        }else{
            if(!StringUtils.equals(productName, tenantServiceConfig.getProductName())){
                changed = true;
                kmTenantService.removeTenantServiceConfig(tenantId,serviceName);
                kmTenantService.addTenantServiceConfig(tenantId,serviceName, productName);
            }
        }
        //删除产品信息缓存
        if(changed){
            String cacheKey = Constants.REDIS_KNOWLEDGE_GRAPH + ":SYSTEM:SYSTEM:zh_CN:ProductName$"+tenantId;
            //cacheService.deleteKeys(Collections.singletonList(cacheKey));
        }
    }

    public ProductOperationResult getProductEoc(String tenantId) throws Exception {
        LOGGER.info("getCloudProductNamesAfterEocSet  tenantId:{}", tenantId);
        JSONObject response = JSON.parseObject(JSON.toJSONString(tenantProductOperationList(tenantId)), JSONObject.class);
        LOGGER.info("tenantProductList response:{}", response);
        ProductOperationResult productOperationResult = JSON.parseObject(JSON.toJSONString(response), ProductOperationResult.class);
        return productOperationResult;
    }

    public void batchSetProductInfoForTaskView(String tenantId, Map<String, Object> taskView) {
        Map<String, Object> tmp = new HashMap<>();
        try {
            // todo 因为获取projectName的方法移动不过去，暂时放这里
            Map<String, Object> dataSources = (Map<String, Object>) taskView.get("dataSources");
            Map<String, Object> pages = (Map<String, Object>) taskView.get("pages");
            tmp.put("dataSources", dataSources);
            tmp.put("pages", pages);
            Object expectedDurationObj = taskView.get("expectedDuration");
            if (null != expectedDurationObj) {
                Map<String, Object> expectedDuration = null;
                if (expectedDurationObj instanceof Map) {
                    expectedDuration = (Map<String, Object>) expectedDurationObj;
                } else {
                    expectedDuration = JSON.parseObject(JSON.toJSONString(expectedDurationObj));
                    taskView.put("expectedDuration", expectedDuration);
                }
                Map<String, Object> dueDateRule = (Map<String, Object>) expectedDuration.get("dueDateRule");
                if (null != dueDateRule) {
                    Map<String, Object> getDateAction = (Map<String, Object>) dueDateRule.get("getDateAction");
                    tmp.put("getDateAction", getDateAction);
                }
            }
            batchSetProductInfo(tmp, tenantId);
        } catch (Exception e) {
            LOGGER.error("batchSetProductInfoForTaskView error", e);
        }
    }
}
