package com.digiwin.athena.knowledgegraph.clients;

import com.digiwin.app.container.exceptions.DWBusinessException;
import com.digiwin.athena.knowledgegraph.utils.AthenaUtils;
import com.digiwin.athena.knowledgegraph.utils.I18nUtils;
import com.digiwin.athena.knowledgegraph.utils.Utils;
import org.apache.commons.lang.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.core.ParameterizedTypeReference;
import org.springframework.http.*;
import org.springframework.stereotype.Service;
import org.springframework.web.client.RestTemplate;

import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.time.LocalDateTime;
import java.time.format.DateTimeFormatter;
import java.util.*;
import java.util.stream.Collectors;

@Service
public class CACUtils {

    @Value("${cacUrl}")
    private String cacUrl;

    @Autowired
    @Qualifier("cacRestTemplate")
    RestTemplate restTemplate;

    @Value("${iamApToken}")
    private String iamApToken;

    private static final String CAC_TENANT_GOODS = "/api/cac/v4/authorizations/tenants/%s/goods/%s";
    private static final String CAC_TENANT_USER = "/api/cac/v4/authorizations/current/user/apps";
    private static final String CAC_TENANT_USER_GOODS = "/api/cac/v4/counting/currenttenant/goods/%s";

//    @Autowired
//    RestTemplate restTemplate;

    public Boolean getAppAuth(String tenantId, String appId) throws DWBusinessException, ParseException {
        HttpHeaders headers = new HttpHeaders();
        Utils.requiredHeaders(headers);
        headers.setAccept(Arrays.asList(MediaType.APPLICATION_JSON));
        MediaType type = MediaType.parseMediaType("application/json; charset=UTF-8");
        headers.setContentType(type);
        HttpEntity<Map<String, String>> entity = new HttpEntity<>(headers);
        ResponseEntity<Map> result = restTemplate.exchange(String.format(cacUrl + CAC_TENANT_GOODS, tenantId, appId), HttpMethod.GET, entity, Map.class);
        if (result.getStatusCodeValue() == 200) {
            if (!result.getBody().isEmpty()) {
                Map re = result.getBody();
                SimpleDateFormat sdf = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss");
                Date effectiveTime = sdf.parse((String) re.get("effectiveTime"));
                Date expiredTime = sdf.parse((String) re.get("expiredTime"));
                Date currentTime = new Date();
                if (effectiveTime.before(currentTime) && currentTime.before(expiredTime)) {
                    return true;//当前时间在授权生效时间和过期时间 期间
                } else {
                    return false;
                }
            } else {
                return false;
            }
        } else {
            throw new DWBusinessException(I18nUtils.getValue("knowledgegraph.getAuthError"));
        }
    }

    // 获取应用有效时间
    public Map<String, String> getAppAuthDate(String tenantId, String appId) throws DWBusinessException, ParseException {
        Map<String, String> resultMap = new HashMap<>();
        HttpHeaders headers = new HttpHeaders();
        Utils.requiredHeaders(headers);
        headers.setAccept(Arrays.asList(MediaType.APPLICATION_JSON));
        MediaType type = MediaType.parseMediaType("application/json; charset=UTF-8");
        headers.setContentType(type);
        HttpEntity<Map<String, String>> entity = new HttpEntity<>(headers);
        ResponseEntity<Map> result = restTemplate.exchange(String.format(cacUrl + CAC_TENANT_GOODS, tenantId, appId), HttpMethod.GET, entity, Map.class);
        if (result.getStatusCodeValue() == 200) {
            if (null != result.getBody() && !result.getBody().isEmpty()) {
                Map re = result.getBody();
                String effectiveTime = re.get("effectiveTime").toString();
                String expiredTime = re.get("expiredTime").toString();
                String[] efTime = effectiveTime.split(" ");
                String[] exTime = expiredTime.split(" ");
                resultMap.put("effectiveTime", efTime[0]);
                resultMap.put("expiredTime", exTime[0]);
                return resultMap;
            } else {
                resultMap.put("effectiveTime", "");
                resultMap.put("expiredTime", "");
                return resultMap;
            }
        } else {
            throw new DWBusinessException(I18nUtils.getValue("knowledgegraph.getAuthError"));
        }
    }

    /**
     * 获取有权限的应用数据
     * 集成token需要传userId获取用户的授权应用code，不是集成token就不需要userId获取租户的授权应用
     * @return 返回应用code
     * @throws DWBusinessException
     */
    public List<String> getAppAuth() throws DWBusinessException {
        return getAppAuth(null);
    }

    /**
     * 根据用户id获取用户下有权限的应用
     * @param userId 用户id
     * @return
     * @throws DWBusinessException
     */
    public List<String> getAppAuth(String userId) throws DWBusinessException {
        HttpHeaders headers = new HttpHeaders();
        Utils.requiredHeaders(headers);
        headers.setContentType(MediaType.APPLICATION_JSON_UTF8);
        headers.set("digi-middleware-auth-user", AthenaUtils.getHeaderToken());
        headers.set("digi-middleware-auth-app", iamApToken);
        String url = cacUrl + CAC_TENANT_USER;
        if (StringUtils.isNotEmpty(userId)) {
            url = url + "?userId=" + userId;
        }

        ResponseEntity<Map<String, Object>> response = restTemplate.exchange(url, HttpMethod.GET,
                new HttpEntity<>(headers), new ParameterizedTypeReference<Map<String, Object>>() {});

        if (response.getStatusCode() == HttpStatus.OK) {
            Map<String, Object> body = response.getBody();
            List<Map<String, Object>> dataList = (List<Map<String, Object>>) body.get("data");

            if (dataList != null) {
                return dataList.stream().filter(e->this.isEffective(e)).map(data -> data.get("id").toString()).collect(Collectors.toList());
            }else {
                return Collections.emptyList();
            }
        }

        throw new DWBusinessException(I18nUtils.getValue("knowledgegraph.getAuthError"));
    }

    /**
     * 判断是否过期
     * @param map 源数据
     * @return 判断结果
     */
    private boolean isEffective(Map<String, Object> map){
        Object expiredTimeObj = map.get("expiredTime");
        if(expiredTimeObj != null){
            String expiredTime = String.valueOf(expiredTimeObj);
            DateTimeFormatter dtf2 = DateTimeFormatter.ofPattern("yyyy-MM-dd HH:mm:ss");
            LocalDateTime expiredLocalDateTime = LocalDateTime.parse(expiredTime, dtf2);
            return LocalDateTime.now().isBefore(expiredLocalDateTime);
        }
        return false;
    }
}
