package com.digiwin.athena.knowledgegraph.report;

import com.alibaba.fastjson.JSON;
import com.digiwin.app.container.exceptions.DWBusinessException;
import com.digiwin.app.service.DWServiceContext;
import com.digiwin.athena.kg.monitorRule.StandardPollingRule;
import com.digiwin.athena.kg.authority.AppPurchaseInfo;
import com.digiwin.athena.kg.report.hz.model.ReportMonitorRuleDTO;
import com.digiwin.athena.knowledgegraph.clients.CacService;
import com.digiwin.athena.knowledgegraph.domain.MonitorRule;
import com.digiwin.athena.knowledgegraph.domain.VariableDefinition;
import com.digiwin.athena.domain.core.app.ApplicationRelation;
import com.digiwin.athena.kg.monitorRule.DynamicCondition;
import com.digiwin.athena.repository.neo4j.MonitorRuleRepository;

import com.digiwin.athena.knowledgegraph.rule.MonitorRuleConfigService;
import com.digiwin.athena.kg.monitorRule.MonitorRuleProductConfig;
import com.digiwin.athena.kg.monitorRule.MonitorRuleTenantConfig;
import com.digiwin.athena.knowledgegraph.service.KgInnerService;
import com.digiwin.athena.knowledgegraph.service.impl.ActionService;
import com.digiwin.athena.knowledgegraph.service.impl.ProductService;
import com.digiwin.athena.knowledgegraph.service.inner.KgHelpService;
import com.digiwin.athena.knowledgegraph.service.model.MonitorRuleConfigDTO;
import com.digiwin.athena.knowledgegraph.set.*;
import com.digiwin.athena.knowledgegraph.utils.AthenaUtils;
import com.digiwin.athena.knowledgegraph.utils.TranslateUtils;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.data.mongodb.core.MongoTemplate;
import org.springframework.data.mongodb.core.query.Criteria;
import org.springframework.data.mongodb.core.query.Query;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import java.util.*;
import java.util.stream.Collectors;

@Service
@Slf4j
public class MonitorPartParser {

    @Autowired
    @Qualifier("knowledgegraphTenant")
    MongoTemplate mongoTemplate;

    @Autowired
    @Qualifier("knowledgegraphSystem")
    private MongoTemplate mongoTemplateSystem;

    @Autowired
    KgHelpService kgHelpService;

    @Autowired
    MonitorRuleRepository monitorRuleRepository;

    @Autowired
    KgInnerService kgInnerService;

    @Autowired
    CacService cacService;

    @Autowired
    TranslateUtils translateUtils;

    @Autowired
    MonitorRuleConfigService monitorRuleConfigService;

    @Autowired
    PartParserService partParserService;

    @Autowired
    ActionService actionService;

    @Autowired
    ProductService productService;

    public Object buildReportMonitorRule(ReportMonitorRuleDTO reportMonitorRuleDTO) throws DWBusinessException {
        if (null != reportMonitorRuleDTO) {
            String tenantId = AthenaUtils.getTenantId();
            String tenantVersion = kgInnerService.currentTenantVersion();
            SetsRelatedInfo setsRelatedInfo = new SetsRelatedInfo();
            // 组件id
            String pluginId = reportMonitorRuleDTO.getCode();
            // 应用code
            String appCode = reportMonitorRuleDTO.getAppCode();
            // 应用名称
            String appName = reportMonitorRuleDTO.getAppName();
            // 侦测id
            String ruleId = "MonitorRule_" + pluginId;
            // 侦测名称
            String ruleName = reportMonitorRuleDTO.getName() + "_侦测";
            // 更新  根据套件code查询是否存在，存在则删除后再插入数据
            deleteIfPresent(appCode,pluginId,ruleId,tenantVersion);
            //如果是删除操作，删除后则结束编译
            if (Objects.equals(reportMonitorRuleDTO.getOperate(),"delete")) {
                // 更新租户的侦测数据并且通知sd停用
                updateTenantMonitorRuleStatus(ruleId);
                return null;
            }
            // 新增侦测
            MonitorRule monitorRule = new MonitorRule();
            monitorRule.setId(ruleId);
            monitorRule.setName(ruleName);
            monitorRule.setVersion(tenantVersion);
            monitorRule.setProductName(reportMonitorRuleDTO.getProductName());
            saveMonitorRule(reportMonitorRuleDTO ,monitorRule);

            // 构建应用参数
            buildVariableDefinition(setsRelatedInfo,reportMonitorRuleDTO,ruleId,ruleName,tenantVersion);
            // 构建应用和参数的关系
            buildApplicationRelation(setsRelatedInfo,pluginId,appCode,appName,ruleId,ruleName,tenantVersion);

            saveReportRelatedInfo(setsRelatedInfo);

            // 保存租户数据并同步到SD
            // 查询租户创建侦测和租户的关系
            List<String> tenantIds = createMonitorRelation(appCode,ruleId) ;
            saveMonitorRuleByTenant(tenantIds,ruleId,reportMonitorRuleDTO.getProductName());

            // 设置原租户id
            Map<String, Object> profile = new HashMap<>();
            profile.put("tenantId", tenantId);
            DWServiceContext.getContext().setProfile(profile);
            return ruleId;
        }
        return null;
    }

    private void saveReportRelatedInfo(SetsRelatedInfo setsRelatedInfo) throws DWBusinessException {
        String setsRelatedInfoStr = translateUtils.translateContent(setsRelatedInfo);
        SetsRelatedInfo relatedInfo = JSON.parseObject(setsRelatedInfoStr, SetsRelatedInfo.class);
        mongoTemplateSystem.save(relatedInfo.getApplicationRelation());
        mongoTemplateSystem.save(relatedInfo.getVariableDefinition());
    }

    private void deleteIfPresent(String appCode,String pluginId,String ruleId,String tenantVersion) {
        if (StringUtils.isNotEmpty(pluginId)) {
            // 删除侦测
            deleteMonitorRule(pluginId,tenantVersion);
            // 删除参数与应用关系
            deleteApplicationRelation(appCode,ruleId,tenantVersion);
            // 删除应用参数
            deleteVariableDefinition(ruleId,tenantVersion);
        }
    }

    private void saveMonitorRule(ReportMonitorRuleDTO monitorRuleDTO , MonitorRule monitorRule) {
        // 存neo4j
        String jsonString = "CREATE (m:MonitorRule{code:$code,"
                + "autoRun:true,pluginId:$pluginId,"
                + "name:$name,"
                + "actionCommonId:$actionCommonId,"
                + "scene:$scene,";

        Map<String, Object> props = new HashMap<>();
//        props.put("actionType",monitorRuleDTO.getActionType());
        props.put("code",monitorRule.getId());
        props.put("pluginId",monitorRuleDTO.getCode());
        props.put("name",monitorRule.getName());
        props.put("actionCommonId", monitorRuleDTO.getActionId()); // 行动id
        props.put("scene",JSON.toJSONString(monitorRuleDTO.getScene()));
        if (monitorRuleDTO.getStandard_polling_rule() != null) {
            jsonString += "standardPollingRule:$standardPollingRule,";
            props.put("standardPollingRule", JSON.toJSONString(monitorRuleDTO.getStandard_polling_rule()));
        }
        jsonString += "type:'SCHEDULE',category:'SCHEDULE',actionType:'agileData',version:$version,athena_namespace:$athena_namespace})";
        props.put("version",monitorRule.getVersion());
        props.put("athena_namespace",monitorRuleDTO.getAppCode());
        kgHelpService.executeCyhper(jsonString,props);

        // 侦测存储到mongo
        MonitorRuleProductConfig monitorRuleProductConfig = new MonitorRuleProductConfig();
        monitorRuleProductConfig.setMonitorRuleId(monitorRule.getId());
        monitorRuleProductConfig.setProductName(monitorRule.getProductName());
        monitorRuleProductConfig.setVersion(monitorRule.getVersion());
        monitorRuleProductConfig.setPluginId(monitorRuleDTO.getCode());
        monitorRuleProductConfig.setIsPollingIds(Boolean.FALSE);
        monitorRuleProductConfig.setAthena_namespace(monitorRuleDTO.getAppCode());
        monitorRuleProductConfig.setActionParamDefine(monitorRuleDTO.getActionParamDefine());
        monitorRuleProductConfig.setRecast(monitorRuleDTO.getRecast());
        mongoTemplateSystem.save(monitorRuleProductConfig);
    }

    private List<String> createMonitorRelation(String appCode,String ruleId) {
        List<String> tenantIds = new ArrayList<>();
        List<AppPurchaseInfo> tenantIdDatas = cacService.getAllEffectiveTenantAndAppInfo(appCode).getList();
        if (!CollectionUtils.isEmpty(tenantIdDatas)) {
            tenantIds = tenantIdDatas.stream().map(AppPurchaseInfo::getTenantId).collect(Collectors.toList());
//            String createMonitorRelation = "MATCH (t:TenantEntity) " +
//                    "WHERE t.tenantId IN $tenantIds " + "MATCH (m:MonitorRule {code: $monitorRuleId}) MERGE (t)-[:USE]->(m)";
//            Map<String, Object> params = new HashMap<>();
//            params.put("tenantIds", tenantIds);
//            params.put("monitorRuleId", ruleId);
//            kgHelpService.executeCyhper(createMonitorRelation,params);
        }
        return tenantIds;
    }


    // 侦测保存后默认生效
    private void saveMonitorRuleByTenant(List<String> tenantIds,String ruleId,String productName) {
        if (CollectionUtils.isEmpty(tenantIds)) {
            return;
        }
        for (String id : tenantIds) {
            MonitorRuleConfigDTO config = new MonitorRuleConfigDTO();
            config.setMonitorRuleId(ruleId);
            config.setTenantId(id);
            config.setProductName(productName);
            config.setStatus(1);
            try {
                Map<String, Object> profile = new HashMap<>();
                profile.put("tenantId", config.getTenantId());//athenaPaasW AthenaWFPASS E10ATHENApass athenaTestW
                DWServiceContext.getContext().setProfile(profile);
                monitorRuleConfigService.post(config);
            } catch (Exception e) {
                log.error("buildReport call post method error : {}",e);
            }
        }
    }

    private void deleteMonitorRule(String pluginId,String tenantVersion) {
        Map<String,Object> param = new HashMap<>();
        param.put("pluginId",pluginId);
        param.put("version",tenantVersion);
        String deleteCyhper = "match(m:MonitorRule{pluginId:$pluginId,version:$version}) detach delete m";
        kgHelpService.executeCyhper(deleteCyhper,param);

        Query query = Query.query(Criteria.where("pluginId").is(pluginId).and("version").is(tenantVersion));
        mongoTemplateSystem.remove(query, MonitorRuleProductConfig.class);
    }

    // 构建侦测和应用之间的关系
    private void buildApplicationRelation(SetsRelatedInfo setsRelatedInfo, String pluginId,String appCode, String appName, String ruleId, String ruleName, String tenantVersion) {
        ApplicationRelation applicationRelation = new ApplicationRelation();
        applicationRelation.setAppCode(appCode);
        applicationRelation.setAppName(appName);
        applicationRelation.setCode(ruleId);
        applicationRelation.setType("monitorRule");
        applicationRelation.setName(ruleName);
        applicationRelation.setVersion(tenantVersion);
        applicationRelation.setApplication(appCode);
        applicationRelation.setAthena_namespace(appCode);
        applicationRelation.setPluginId(pluginId);
        setsRelatedInfo.setApplicationRelation(applicationRelation);
    }

    private void deleteApplicationRelation(String appCode,String ruleId,String tenantVersion) {
        Query query = new Query();
        query.addCriteria(Criteria.where("appCode").is(appCode).and("version").is(tenantVersion).and("code").is(ruleId));
        mongoTemplateSystem.remove(query, ApplicationRelation.class);
    }

    private void buildVariableDefinition(SetsRelatedInfo setsRelatedInfo,ReportMonitorRuleDTO reportMonitorRuleDTO, String ruleId, String ruleName,String tenantVersion) {
        StandardPollingRule rules = reportMonitorRuleDTO.getStandard_polling_rule();
        String timeType = getDefaultValue(rules);
        VariableDefinition variableDefinition = new VariableDefinition();
        variableDefinition.setName(ruleName);
        variableDefinition.setDescription(ruleName);
        variableDefinition.setIsExecute(Boolean.TRUE);
        variableDefinition.setCode(ruleId);
        variableDefinition.setForceOpen(Boolean.FALSE);
        variableDefinition.setIsOperation(Boolean.FALSE);
        variableDefinition.setDefaultValue(timeType);
        variableDefinition.setExtendData(ruleId);
        variableDefinition.setFormat("rule;yyyyMMdd HH:mm;switch");
        variableDefinition.setReadOnly(Boolean.FALSE);
        variableDefinition.setRequired(Boolean.TRUE);
        variableDefinition.setExecutable(1);
        variableDefinition.setSequence(1);
        variableDefinition.setDisplayType("PERIODIC");
        variableDefinition.setCategory("period");
        variableDefinition.setIsSimulation(Boolean.FALSE);
        variableDefinition.setVersion(tenantVersion);
        variableDefinition.setAthena_namespace(reportMonitorRuleDTO.getAppCode());
        variableDefinition.setPluginId(reportMonitorRuleDTO.getCode());

        DynamicCondition visibleCondition = new DynamicCondition();
        visibleCondition.setType("SINGLE");
        visibleCondition.setOp("IN");
        visibleCondition.setLeft("productName");
        visibleCondition.setLeftValueType("SYSTEM");
        visibleCondition.setRightValueType("CONSTANT");
        visibleCondition.setRight(reportMonitorRuleDTO.getProductName());
        variableDefinition.setVisibleCondition(visibleCondition);
        setsRelatedInfo.setVariableDefinition(variableDefinition);
    }

    private String getDefaultValue(StandardPollingRule rules) {
        String timeType = "";
        if (null != rules.getRepeat_type()) {
            if (rules.getRepeat_type() == 1) {
                timeType = "F";
            }
            if (rules.getRepeat_type() == 2) {
                timeType = "H";
            }
            if (rules.getRepeat_type() == 3) {
                timeType = "D";
            }
            timeType = timeType + rules.getFrequency();
        }else if (rules.getWeekly() != null) {
            Map<String,String> week = new HashMap<>();
            week.put("SU","7");
            week.put("MO","1");
            week.put("TU","2");
            week.put("WE","3");
            week.put("TH","4");
            week.put("FR","5");
            week.put("SA","6");
            timeType = "W" + week.get(rules.getByday());
        }else if (rules.getMonthly() != null) {
            timeType = "M" + rules.getBymonthday();
        }
        return timeType + ";" + rules.getStart_time().substring(0,16).replaceAll("-","") + ";" + Boolean.TRUE;
    }

    private void deleteVariableDefinition(String ruleId,String tenantVersion) {
        // 删除应用参数
        Query query = new Query();
        query.addCriteria(Criteria.where("code").is(ruleId).and("version").is(tenantVersion));
        mongoTemplateSystem.remove(query, VariableDefinition.class);
    }

    // 更新租户的侦测数据并且通知sd停用
    private void updateTenantMonitorRuleStatus(String ruleId) {
        Query query = new Query();
        Criteria criteria = Criteria.where("monitorRuleId").is(ruleId);
        query.addCriteria(criteria);
        List<MonitorRuleTenantConfig> monitorRuleTenantConfigList = mongoTemplate.find(query, MonitorRuleTenantConfig.class);
        try {
            if (!CollectionUtils.isEmpty(monitorRuleTenantConfigList)) {
                for (MonitorRuleTenantConfig monitorRuleTenantConfig : monitorRuleTenantConfigList) {
                    monitorRuleConfigService.putDisable(monitorRuleTenantConfig.getTenantId(),monitorRuleTenantConfig.getMonitorRuleId(),monitorRuleTenantConfig.getEocMap());
                }
            }
        } catch (Exception e) {
            log.error("Report  updateTenantMonitorRuleStatus error: {}",e);
        }
    }
}
