package com.digiwin.athena.knowledgegraph.service;

import com.digiwin.app.container.exceptions.DWBusinessException;
import com.digiwin.app.service.AllowAnonymous;
import com.digiwin.app.service.DWService;
import com.digiwin.athena.annotation.ScheduledCall;
import com.digiwin.athena.kg.activity.Activity;
import com.digiwin.athena.domain.core.app.Application;
import com.digiwin.athena.domain.core.app.ApplicationRelation;
import com.digiwin.athena.domain.core.app.FunctionCtrl;
import com.digiwin.athena.kg.domain.OpenWindowDefinition;
import com.digiwin.athena.kg.domain.ApplicationCustomizedConfig;
import com.digiwin.athena.knowledgegraph.dto.ApplicationRelationQueryDTO;
import com.digiwin.athena.knowledgegraph.po.AppRelationBatchPo;
import org.springframework.web.bind.annotation.RequestBody;

import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Set;

/**
 * 应用定义接口
 *
 * @javadoc
 */
public interface IAppService extends DWService {

    /**
     * 根据应用code和类型查询应用下关联关系
     *
     * @param type 组件类型
     * @param code 组件code
     * @return 应用下关联关系
     * @throws DWBusinessException
     */
    ApplicationRelation getApplicationByTypeAndCode(String type, String code) throws DWBusinessException;

    /**
     * 根据应用code和类型查询应用下关联关系
     *
     * @param type 组件类型
     * @param code 组件code
     * @param lang 语言
     * @return 应用下关联关系
     * @throws DWBusinessException
     */
    ApplicationRelation getApplicationByTypeAndCode(String type, String code, String lang) throws DWBusinessException;

    /**
     * 根据组件类型，组件code模糊查询应用下关联关系
     *
     * @param type        组件类型
     * @param codes       组件code集合
     * @param activityMap 任务信息
     * @return 应用下关联关系
     * @throws DWBusinessException
     */
    Map<String, List<ApplicationRelation>> getApplicationsMapByTypeAndCodes(String type, Collection<String> codes,
                                                                            Map<String, Activity> activityMap) throws DWBusinessException;

    /**
     * 根据组件类型，组件code模糊查询应用下关联关系
     *
     * @param type  组件类型
     * @param codes 组件code集合
     * @return 应用下关联关系
     * @throws DWBusinessException
     */
    Map<String, List<ApplicationRelation>> getApplicationsMapByTypeAndCodes(String type, Collection<String> codes) throws DWBusinessException;

    /**
     * 根据组件类型，组件code查询应用下关联关系
     *
     * @param type     组件类型
     * @param codeList 组件code集合
     * @return 应用下关联关系
     * @throws DWBusinessException
     */
    List<ApplicationRelation> getApplicationByTypeAndCode(String type, List<String> codeList) throws DWBusinessException;

    /**
     * 根据组件类型，组件code查询应用下关联关系
     *
     * @param type     组件类型
     * @param subType  二级组件类型
     * @param codeList 组件code集合
     * @return 应用下组件关系
     * @throws DWBusinessException
     */
    List<ApplicationRelation> getApplicationByTypeAndCode(String type, String subType, List<String> codeList) throws DWBusinessException;

    /**
     * 根据组件类型，组件code查询应用下组件关系
     *
     * @param type 组件类型
     * @param code 组件code集合
     * @return 应用下关联关系
     * @throws DWBusinessException
     */
    List<ApplicationRelation> getApplicationsByTypeAndCode(String type, String code) throws DWBusinessException;

    /**
     * 根据应用code集合查询应用
     *
     * @param codes 组件code集合
     * @return 应用
     * @throws DWBusinessException
     */
    List<Application> getApplicationsByCodes(Collection<String> codes) throws DWBusinessException;

    /**
     * 根据应用code查询应用
     *
     * @param code 组件code
     * @return 应用
     * @throws DWBusinessException
     */
    Application getApplicationByCode(String code) throws DWBusinessException;

    /**
     * 根据应用code和租户id查询应用
     *
     * @param code     组件code
     * @param tenantId 租户id
     * @return
     * @throws DWBusinessException
     */
    Application getApplicationByCode(String code, String tenantId) throws DWBusinessException;

    /**
     * 根据业务组件code查询应用扩展信息
     *
     * @param code 组件code
     * @return 应用扩展信息
     * @throws DWBusinessException
     */
    FunctionCtrl getApplicationFunctionCtrlByBizCode(String code) throws DWBusinessException;

    /**
     * 根据应用code查询应用扩展信息
     *
     * @param code 应用code
     * @return 应用扩展信息
     * @throws DWBusinessException
     */
    FunctionCtrl getApplicationFunctionCtrlByAppCode(String code) throws DWBusinessException;

    // TODO liugla 2024/10/22 11:31 注释

    /**
     * ApplicationRecommendation表的作用
     *
     * @return
     * @throws DWBusinessException
     */
    List<Map> getApplicationRecommendations() throws DWBusinessException;

    /**
     * 根据组件类型，组件code查询应用下关联关系，并根据应用code分组
     *
     * @param type  组件类型
     * @param codes 组件code
     * @param lang  语言
     * @return 应用下关联关系
     * @throws DWBusinessException
     */
    Map<String, List<ApplicationRelation>> getApplicationRelationsMapByTypeAndCodesAndLang(String type, Collection<String> codes, String lang) throws DWBusinessException;

    /**
     * 根据组件类型，组件code查询应用下关联关系集合
     *
     * @param type 组件类型
     * @param code 组件code
     * @param lang 语言
     * @return 应用下关联关系
     * @throws DWBusinessException
     */
    List<ApplicationRelation> getApplicationRelationsByTypeAndCodeAndLang(String type, String code, String lang) throws DWBusinessException;

    /**
     * 根据组件类型，组件code查询应用下关联关系
     *
     * @param type 组件类型
     * @param code 组件code
     * @param lang 语言
     * @return 应用下关联关系
     * @throws DWBusinessException
     */
    ApplicationRelation getApplicationByTypeAndCodeAndLang(String type, String code, String lang) throws DWBusinessException;

    /**
     * 根据组件code查询应用下关联关系
     *
     * @param codes     组件code
     * @param localeStr 语言
     * @return 应用下关联关系 {组件code, 应用下关联关系}
     * @throws DWBusinessException
     */
    Map<String, ApplicationRelation> getApplicationRelationsMapByApplicationAndLang(Collection<String> codes, String localeStr) throws DWBusinessException;

    /**
     * 根据组件code查询应用下关联关系
     *
     * @param code      组件code
     * @param localeStr 语言
     * @return 应用下关联关系
     * @throws DWBusinessException
     */
    ApplicationRelation getApplicationRelationByApplicationAndLang(String code, String localeStr) throws DWBusinessException;

    /**
     * 根据租户id获取应用许可密钥
     *
     * @param tenantId 租户id
     * @return 许可密钥加密串
     * @throws Exception
     */
    @AllowAnonymous
    Object getLicenseKey(String tenantId) throws Exception;

    /**
     * 根据租户id获取应用许可密钥
     *
     * @param tenantId    租户id
     * @param requestFrom 数据来源
     * @param key         校验许可的md值
     * @param timestamp   有效时间
     * @return 许可密钥加密串
     * @throws Exception
     */
    @AllowAnonymous
    Object getLicenseKey(String tenantId, String requestFrom, String key, long timestamp) throws Exception;

    /**
     * 设置当前用户许可
     *
     * @return 操作结果
     * @throws Exception
     */
    Object postSetLicenseKey() throws Exception;

    /**
     * 给指定用户设置许可
     *
     * @param tenantId 租户id
     * @return 操作结果
     * @throws Exception
     */
    @AllowAnonymous
    Object postSetLicenseKey(String tenantId) throws Exception;

    /**
     * 根据应用code获取应用
     *
     * @param code 应用code
     * @return 应用
     * @throws Exception
     */
    Object get(String code) throws Exception;

    /**
     * 根据应用code获取应用（内涵侦测规则和基础参数）
     *
     * @param applicationCodeList 应用code
     * @return 应用
     * @throws Exception
     */
    Object getList(Set<String> applicationCodeList) throws Exception;

    /**
     * 根据应用名称和标签查询应用集合
     *
     * @param applicationName 应用名称
     * @param tags            标签
     * @return 应用集合
     * @throws Exception
     */
    Object getByNameAndTags(String applicationName, List<String> tags) throws Exception;

    /**
     * 根据应用code返回应用下的机制，从应用关联关系表获取
     *
     * @param applicationCode 应用code
     * @return 应用下的机制
     * @throws Exception
     */
    Object getApplicationMechanismByApplicationCode(String applicationCode) throws Exception;

    /**
     * 批量查询应用机制集合
     *
     * @param applicationCodes 应用code集合
     * @return 应用下的机制
     * @throws Exception
     */
    Object postApplicationMechanismByApplicationCodes(List<String> applicationCodes) throws Exception;

    /**
     * 通过机制code获取应用下机制组件
     *
     * @param mechanismCode 机制code
     * @return 返回应用下机制
     * @throws Exception
     */
    Object getApplicationMechanismByMechanismCode(String mechanismCode) throws Exception;

    /**
     * 根据应用code获取应用
     *
     * @param code 应用code
     * @return 应用
     * @throws DWBusinessException
     */
    Application getApplicationInfo(String code) throws DWBusinessException;

    /**
     * 根据应用code从应用组件关系表获取任务
     *
     * @param appCode 应用code
     * @return 任务code
     * @throws DWBusinessException
     */
    List<String> getTaskCode(String appCode) throws DWBusinessException;

    /**
     * 根据应用code从应用组件关系表获取机制
     *
     * @param appCode 应用code
     * @return 机制code
     * @throws DWBusinessException
     */
    List<String> getMechanismCode(String appCode) throws DWBusinessException;

    /**
     * 根据应用code从应用组件关系表获取侦测
     *
     * @param appCode 应用code
     * @return 侦测code
     * @throws DWBusinessException
     */
    List<String> getMonitorRuleCode(String appCode) throws DWBusinessException;

    /**
     * 根据组件code和类型获取应用组件关系
     *
     * @param applicationRelationQueryDTO 查询参数，组件code和类型
     * @return 应用下关联关系
     * @throws DWBusinessException
     */
    List<ApplicationRelation> postApplicationRelationsByCodesAndType(@RequestBody ApplicationRelationQueryDTO applicationRelationQueryDTO) throws DWBusinessException;

    /**
     * 根据任务code获取组件应用关系
     *
     * @param code 任务code
     * @return 应用组件关系
     * @throws DWBusinessException
     */
    List<ApplicationRelation> getAppCodeByTaskCode(String code) throws DWBusinessException;

    /**
     * 根据机制code获取应用组件关系
     *
     * @param code 机制code
     * @return 应用组件关系
     * @throws DWBusinessException
     */
    List<ApplicationRelation> getAppCodeByMechanismCode(String code) throws DWBusinessException;

    /**
     * 根据侦测code查询应用组件关系，没有数据再查找Application中直接配置再paramDescription中的数据
     *
     * @param code 侦测code
     * @return 应用组件关系
     * @throws DWBusinessException
     */
    List<ApplicationRelation> getAppCodeByMonitorRuleCode(String code) throws DWBusinessException;

    /**
     * 根据应用code和租户版本获取应用下的组件关系
     *
     * @param appCodes      应用code
     * @param tenantVersion 租户版本
     * @return 返回应用组件关系
     * @throws DWBusinessException
     */
    List<ApplicationRelation> getApplicationRelation(List<String> appCodes, String tenantVersion) throws DWBusinessException;

    /**
     * 根据应用code和租户版本获取应用下的组件关系
     *
     * @param appCodes      应用code
     * @param codes         组件code
     * @return 返回应用组件关系
     * @throws DWBusinessException
     */
    List<ApplicationRelation> getApplicationRelation(List<String> appCodes, List<String> codes, String type, String subType) throws DWBusinessException;

    /**
     * 根据应用code获取应用下的组件关系
     *
     * @param appCodes 应用code
     * @return 应用组件关系
     * @throws DWBusinessException
     */
    List<ApplicationRelation> postApplicationRelation(List<String> appCodes) throws DWBusinessException;

    /**
     * 根据应用code获取流程（现已废弃）
     *
     * @param appCode 应用code
     * @return flowCode
     * @throws DWBusinessException
     */
    List<String> getFlowCode(String appCode) throws DWBusinessException;

    /**
     * 根据flow code获取应用组件关系数据（现已废弃）
     *
     * @param code 组件flow code
     * @return 应用组件关系数据
     * @throws DWBusinessException
     */
    List<ApplicationRelation> getAppCodeByFlowCode(String code) throws DWBusinessException;

    /**
     * 根据应用code获取参数变量，数据来源ApplicationRelation
     *
     * @param appCode 应用code
     * @return 返回参数变量
     * @throws DWBusinessException
     */
    List<String> getVariableCode(String appCode) throws DWBusinessException;

    /**
     *这个方法不要把@AllowAnonymous去除，租户初始化时获取系统有哪些应用code的方法，如果想要有版本信息，请另写方法
     */
    @AllowAnonymous
    @ScheduledCall
    List<String> getAllAppCode() throws DWBusinessException;

    /**
     * 获取非公共的应用
     *
     * @return 应用code
     * @throws DWBusinessException
     */
    List<String> getNotCommonAppCode() throws DWBusinessException;

    /**
     * 根据应用code获取侦测
     *
     * @param appCode 应用code
     * @return 侦测code
     * @throws DWBusinessException
     */
    List<String> getMonitorRulesRelatedToTask(String appCode) throws DWBusinessException;

    /**
     * 根据应用code获取数字员工数据（现数字员工业务已停止）
     *
     * @param codeList        应用code
     * @param enhanceCodeList 组件code
     * @return 返回数据
     * @throws DWBusinessException
     */
    Map<String, List> postQueryDigitalStaffByCode(Set<String> codeList, Set<String> enhanceCodeList) throws DWBusinessException;

    /**
     * （现数字员工业务已停止）
     *
     * @param code 入参
     * @return 结果
     * @throws DWBusinessException
     */
    List<Application> getEnhanceByAppCode(String code) throws DWBusinessException;

    /**
     * （现数字员工业务已停止）
     *
     * @param code 入参
     * @return 结果
     * @throws DWBusinessException
     */
    List<Application> getEnhanceByMechanismCode(String code) throws DWBusinessException;

    /**
     * 根据组件code和类型查询应用组件关系
     *
     * @param type 组件类型
     * @param code 组件code
     * @return 应用下关联关系
     * @throws Exception
     */
    List<ApplicationRelation> getAppCodeByTypeAndCode(String type, String code) throws Exception;

    /**
     * 根据应用code和组件类型获取范式数据
     *
     * @param type    组件类型
     * @param appCode 应用code
     * @return 应用下组件数据
     * @throws Exception
     */
    List<ApplicationRelation> getParadigmByTypeAndAppCode(String type, String appCode) throws Exception;

    /**
     * 通过参数的开窗条件列名称查找开窗信息
     *
     * @param openWin 开窗条件列名称
     * @return 开窗信息
     * @throws Exception
     */
    Map<String, Map<String, OpenWindowDefinition>> postOpenWinInfo(Map<String, Map<String, String>> openWin) throws Exception;

    /**
     * 根据应用获取应用code
     *
     * @param application 应用
     * @return 应用code
     * @throws Exception
     */
    String getAppCodeByApplication(String application) throws Exception;

    /**
     * 根据应用code和类型获取应用下关联关系
     *
     * @param type    组件类型
     * @param appCode 应用code
     * @return 返回应用组件关系
     * @throws Exception
     */
    @AllowAnonymous
    @ScheduledCall
    List<ApplicationRelation> getApplicationRelationByTypeAndAppCode(String type, String appCode) throws Exception;

    /**
     * 根据应用code，类型和租户id获取应用下关联关系
     *
     * @param type    组件类型
     * @param appCode 应用code
     * @param version 租户版本
     * @return 返回应用下关联关系
     * @throws Exception
     */
    List<ApplicationRelation> getApplicationRelationByTypeAndAppCodeAndVersion(String type, String appCode, String version) throws Exception;

    /**
     * 根据应用code，类型批量获取应用下关联关系
     *
     * @param type     组件类型
     * @param appCodes 应用code
     * @return 应用下关联关系
     * @throws Exception
     */
    List<ApplicationRelation> getApplicationRelationByTypeAndAppCode(String type, List<String> appCodes) throws Exception;

    /**
     * 根据应用code，类型批量获取应用下关联关系
     *
     * @param type     组件类型
     * @param appCodes 应用code
     * @return 返回应用下关联关系
     * @throws Exception
     */
    List<ApplicationRelation> postQueryApplicationRelationByTypeAndAppCode(String type, List<String> appCodes) throws Exception;

    /**
     * 根据应用code，组件类型和租户id获取应用组件关系
     *
     * @param tenantId 租户id
     * @param type     组件类型
     * @param appCode  应用code
     * @return 应用下租户关系
     */
    List<ApplicationRelation> getApplicationRelationByTypeAndAppCode(String tenantId, String type, String appCode) throws DWBusinessException;

    /**
     * 根据应用code + 组件code + 类型更新应用下关联关系
     *
     * @param request 参数请求
     * @return 操作的数据条数
     * @throws Exception
     */
    Object postAppRelationUpdate(AppRelationBatchPo request) throws Exception;

    /**
     * 根据类型获取应用组件关系
     *
     * @param type 组件类型
     * @return 应用组件关系
     */
    List<ApplicationRelation> getApplicationRelationByType(String type) throws Exception;
// TODO liugla 2024/10/22 14:40 ApplicationCustomizedConfig 表的作用
    /**
     * @return
     */
    List<ApplicationCustomizedConfig> postConfigByApplicationCustomized() throws DWBusinessException;

    /**
     * 查询应用客制插件配置-分组时加租户ID维度
     * @return
     */
    List<ApplicationCustomizedConfig> postConfigByGroupApplicationCustomized() throws DWBusinessException;

    /**
     * 获取所有的应用
     *
     * @return 应用
     */
    @AllowAnonymous
    List<Application> postAllApplications();

    /**
     * 根据组件code和类型获取应用组件关系
     *
     * @param tenantId 租户id
     * @param type     组件类型
     * @param code     组件code
     * @return 组件应用关系
     * @throws DWBusinessException
     */
    @AllowAnonymous
    Object postApplicationRelationByTypeAndCode(String tenantId, String type, String code) throws DWBusinessException;

    /**
     * 保存应用关联关系数据
     * @param request 保存数据
     * @return
     * @throws DWBusinessException
     */
    Object postApplicationRelationByDDSM(AppRelationBatchPo request) throws DWBusinessException;


    /**
     * 该方法给敏捷数据使用
     * 根据应用code列表查询应用名称信息
     *
     * @param codes 应用code列表
     * @param appType  应用类型
     */
    List<Map<String, String>> postQueryApplicationsNameByCodeList(List<String> codes, Integer appType) throws DWBusinessException;

    /**
     * 根据机制code获取应用
     *
     * @param mechanismCode 机制code
     * @return 应用
     * @throws DWBusinessException exception
     */
    Application getAppByMechanismCode(String mechanismCode) throws DWBusinessException;

    /**
     * 根据作业id去查询应用
     * @param activityId 作业id
     * @return 应用
     * @throws DWBusinessException exception
     */
    Application getAppByActivityId(String activityId) throws DWBusinessException;

    /**
     * 根据应用code列表查询应用信息
     *
     * @param codes 应用code列表
     * @return 应用信息
     */
    Object postQueryApplicationsByCodes(Set<String> codes);
}