package com.digiwin.athena.knowledgegraph.service.impl;

import cn.hutool.core.bean.BeanUtil;
import cn.hutool.core.collection.CollectionUtil;
import cn.hutool.core.map.MapUtil;
import com.alibaba.fastjson.JSONObject;
import com.digiwin.app.container.exceptions.DWBusinessException;
import com.digiwin.app.service.DWServiceContext;
import com.digiwin.athena.kmservice.locale.Lang;
import com.digiwin.athena.kg.activity.Activity;
import com.digiwin.athena.domain.core.app.Application;
import com.digiwin.athena.knowledgegraph.dao.ActivityDAO;
import com.digiwin.athena.repository.neo4j.Activity2Repository;
import com.digiwin.athena.knowledgegraph.service.DataMapService;
import com.digiwin.athena.knowledgegraph.service.IAgileInteractionService;
import com.digiwin.athena.knowledgegraph.service.KgInnerService;
import com.digiwin.athena.knowledgegraph.service.inner.DataPickService;
import com.digiwin.athena.knowledgegraph.service.model.ActivityTemplate;
import com.digiwin.athena.knowledgegraph.service.model.ActivityTemplateDTO;
import com.digiwin.athena.knowledgegraph.service.model.ActivityTemplateRel;
import com.digiwin.athena.knowledgegraph.service.model.QryTemplateListReq;
import com.digiwin.athena.knowledgegraph.synonym.utils.DateUtil;
import com.digiwin.athena.knowledgegraph.utils.AthenaUtils;
import com.digiwin.athena.knowledgegraph.utils.HttpUtil;
import com.digiwin.athena.knowledgegraph.utils.I18nUtils;
import com.digiwin.athena.knowledgegraph.utils.LanguageUtil;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.data.mongodb.core.MongoTemplate;
import org.springframework.data.mongodb.core.query.Criteria;
import org.springframework.data.mongodb.core.query.Query;
import org.springframework.data.mongodb.core.query.Update;
import org.springframework.stereotype.Service;
import org.springframework.web.context.request.RequestContextHolder;
import org.springframework.web.context.request.ServletRequestAttributes;

import javax.servlet.http.HttpServletRequest;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.util.*;
import java.util.stream.Collectors;

@Lang
@Slf4j
@Service
public class AgileInteractionServiceImpl implements IAgileInteractionService {

    public static final String COPY_CUSTOMIZE = "/abi/copy.rpt.customize";

    public static final String DELETE_CUSTOMIZE = "/abi/delete.customize";

    @Value("${abiUrl}")
    private String dapUri;

    @Autowired
    private KgInnerService kgInnerService;

    @Autowired
    @Qualifier("knowledgegraphTenant")
    private MongoTemplate mongoTemplateUser;

    @Autowired
    @Qualifier("knowledgegraphSystem")
    private MongoTemplate mongoTemplate;

    @Autowired
    private DataMapService dataMapService;

    @Autowired
    private Activity2Repository activityRepository;

    @Autowired
    private DataPickService dataPickService;

    @Autowired
    private ActivityDAO activityDAO;

    @Override
    public Object getTemplateList(String application, String activityId, String pattern, String targetId, String client) throws DWBusinessException, NoSuchFieldException, IllegalAccessException {

        // 获取租户版本号
        String tenantId = AthenaUtils.getTenantId();
        String tenantVersion = kgInnerService.getTenantVersion(tenantId);
        Application app = dataPickService.tenantAppRelation(application, tenantId);
        // 应用是否悬停
        boolean individualAll = Optional.ofNullable(app).map(Application::getIndividualAll).orElse(false);
        // 查询模板关系表
        Query systemRelQuery = Query.query(Criteria.where("application").is(application).and("activityId").is(activityId).and("pattern").is(pattern).and("targetId").is(targetId).and("client").is(client).and("version").is(tenantVersion).and("tenantId").is(individualAll ? tenantId : "SYSTEM"));
        List<ActivityTemplateRel> activityTemplateRelList = mongoTemplate.find(systemRelQuery, ActivityTemplateRel.class, "activityTemplateRel");
        if (CollectionUtil.isEmpty(activityTemplateRelList)) {
            return new ArrayList<>();
        }
        // 按钮绑定的模板
        List<String> codeList = activityTemplateRelList.stream().map(ActivityTemplateRel::getCode).collect(Collectors.toList());

        List<ActivityTemplate> activityTemplateList = new ArrayList<>();
        // 查询系统级数据
        Query systemQuery = Query.query(Criteria.where("application").is(application).and("activityId").is(activityId).and("pattern").is(pattern).and("code").in(codeList).and("client").is(client).and("version").is(tenantVersion).and("tenantId").is(individualAll ? tenantId : "SYSTEM"));
        List<ActivityTemplate> systemActivityTemplateList = mongoTemplate.find(systemQuery, ActivityTemplate.class, "activityTemplate");

        // 系统数据为空，则认为不存在有效数据
        if (CollectionUtil.isEmpty(systemActivityTemplateList)) {
            return new ArrayList<>();
        }
        activityTemplateList.addAll(systemActivityTemplateList);

        // 查询租户级启用数据（包含辅助的租户级数据 以及 系统级编辑的租户数据）
        Query tenantIdQuery = Query.query(Criteria.where("tenantId").is(tenantId).and("application").is(application).and("activityId").is(activityId).and("pattern").is(pattern).and("code").in(codeList).and("client").is(client).and("version").is(tenantVersion));
        List<ActivityTemplate> tenantActivityTemplateList = mongoTemplateUser.find(tenantIdQuery, ActivityTemplate.class, "activityTemplate");

        if(CollectionUtil.isNotEmpty(tenantActivityTemplateList)){
            // 过滤失效数据 (系统级数据已经删除，租户级数据同步被过滤)
            List<String> systemIds = systemActivityTemplateList.stream().map(ActivityTemplate::getUniqueId).collect(Collectors.toList());
            tenantActivityTemplateList = tenantActivityTemplateList.stream().filter(activityTemplate -> systemIds.contains(activityTemplate.getOriginId())).collect(Collectors.toList());
            activityTemplateList.addAll(tenantActivityTemplateList);
        }

        // 数据处理 数据合并，数据过滤
        handelActivityTemplate(activityTemplateList);

        // 过滤未启用数据
        activityTemplateList = activityTemplateList.stream().filter(activityTemplate -> activityTemplate.getStatus().equals(1)).collect(Collectors.toList());

        // actionParams
        List<ActivityTemplateDTO> activityTemplateDTOS = setActionParams(activityTemplateList, activityTemplateRelList);

        LanguageUtil.processLocaleLanguage(activityTemplateDTOS, AthenaUtils.getCurrentLocale());

        return activityTemplateDTOS;
    }

    private List<ActivityTemplateDTO> setActionParams(List<ActivityTemplate> activityTemplateList, List<ActivityTemplateRel> activityTemplateRelList) {

        if (CollectionUtil.isEmpty(activityTemplateList)) {
            return new ArrayList<>();
        }
        List<ActivityTemplateDTO> activityTemplateDTOS = new ArrayList<>();
        activityTemplateList.stream().forEach(activityTemplate -> {
            ActivityTemplateDTO activityTemplateDTO = new ActivityTemplateDTO();
            BeanUtil.copyProperties(activityTemplate, activityTemplateDTO);
            Optional<ActivityTemplateRel> templateRelOptional = activityTemplateRelList.stream().filter(activityTemplateRel -> activityTemplateRel.getCode().equals(activityTemplate.getCode())).findFirst();
            if (templateRelOptional.isPresent()) {
                activityTemplateDTO.setActionParams(templateRelOptional.get().getActionParams());
            }
            activityTemplateDTOS.add(activityTemplateDTO);
        });
        return activityTemplateDTOS;
    }

    @Override
    public Object postTemplateList(QryTemplateListReq qryTemplateListReq) throws Exception {

        // 获取租户版本号
        String tenantId = AthenaUtils.getTenantId();
        String tenantVersion = kgInnerService.getTenantVersion(tenantId);
        Application app = dataPickService.tenantAppRelation(qryTemplateListReq.getApplication(), tenantId);
        // 应用是否悬停
        boolean individualAll = Optional.ofNullable(app).map(Application::getIndividualAll).orElse(false);
        List<ActivityTemplate> activityTemplateList = new ArrayList<>();

        // 查询系统级数据
        List<ActivityTemplate> systemActivityTemplateList = mongoTemplate.find(Query.query(Criteria.where("application").is(qryTemplateListReq.getApplication()).and("pattern").is(qryTemplateListReq.getPattern()).and("client").is(qryTemplateListReq.getClient()).and("version").is(tenantVersion).and("tenantId").is(individualAll ? tenantId : "SYSTEM")), ActivityTemplate.class, "activityTemplate");
        if (CollectionUtil.isEmpty(systemActivityTemplateList)) {
            return activityTemplateList;
        }
        activityTemplateList.addAll(systemActivityTemplateList);

        // 查询租户级数据
        List<ActivityTemplate> tenantActivityTemplateList = mongoTemplateUser.find(Query.query(Criteria.where("tenantId").is(tenantId).and("application").is(qryTemplateListReq.getApplication()).and("pattern").is(qryTemplateListReq.getPattern()).and("client").is(qryTemplateListReq.getClient()).and("version").is(tenantVersion)), ActivityTemplate.class, "activityTemplate");

        if (CollectionUtil.isNotEmpty(tenantActivityTemplateList)) {
            // 过滤失效数据 (系统级数据已经删除，租户级数据同步被过滤)
            List<String> systemIds = systemActivityTemplateList.stream().map(ActivityTemplate::getUniqueId).collect(Collectors.toList());
            tenantActivityTemplateList = tenantActivityTemplateList.stream().filter(activityTemplate -> systemIds.contains(activityTemplate.getOriginId())).collect(Collectors.toList());
            activityTemplateList.addAll(tenantActivityTemplateList);
        }

        // SYSTEM_EDIT 类型数据处理
        handelActivityTemplate(activityTemplateList);

        // 根据条件过滤数据
        activityTemplateList = filterActivityTemplate(qryTemplateListReq, activityTemplateList);

        // 时间类型转换
        convertDateType(activityTemplateList);

        // 设置作业名称
        List<ActivityTemplateDTO> activityTemplateDTOS = hanleViewData(activityTemplateList);


        LanguageUtil.processLocaleLanguage(activityTemplateDTOS, AthenaUtils.getCurrentLocale());

        return activityTemplateDTOS;

    }

    private void convertDateType(List<ActivityTemplate> activityTemplate) {

        activityTemplate.stream().forEach(activityTemplateDTO -> {
            activityTemplateDTO.setCreateDate(DateUtil.getStringDate(new Date(Long.parseLong(activityTemplateDTO.getCreateDate()))));
            activityTemplateDTO.setEditDate(DateUtil.getStringDate(new Date(Long.parseLong(activityTemplateDTO.getEditDate()))));
        });
    }

    private List<ActivityTemplateDTO> hanleViewData(List<ActivityTemplate> activityTemplateList) throws Exception {

        if (CollectionUtil.isEmpty(activityTemplateList)) {
            return new ArrayList<>();
        }

        List<ActivityTemplateDTO> activityTemplateDTOS = BeanUtil.copyToList(activityTemplateList, ActivityTemplateDTO.class);
        // 设置作业的名称
        // 任务/项目
        List<ActivityTemplateDTO> taskActivity = activityTemplateDTOS.stream().filter(activityTemplate -> "BUSINESS_PERFORM".equals(activityTemplate.getPattern()) || "BUSINESS_CHARGE".equals(activityTemplate.getPattern())).collect(Collectors.toList());
        if (CollectionUtil.isNotEmpty(taskActivity)) {
            Set<String> taskCodes = taskActivity.stream().map(ActivityTemplateDTO::getActivityId).collect(Collectors.toSet());
            List<JSONObject> activityList = dataMapService.batchTasks(new ArrayList<>(taskCodes));
            Map<String, JSONObject> activityMap = new HashMap<>();
            for (JSONObject activity : activityList) {
                activityMap.put(activity.get("activityId").toString(), activity);
            }
            activityTemplateDTOS.stream().forEach(activityTemplate -> {
                JSONObject activity = activityMap.get(activityTemplate.getActivityId());
                activityTemplate.setActivityName(activity.getString("name"));
                Map<String, Map<String, String>> lang = activityTemplate.getLang();
                Map<String, Map<String, String>> activityLang = (Map)activity.get("lang");
                if (MapUtil.isNotEmpty(activityLang)) {
                    lang.put("activityName", activityLang.get("name"));
                }
                activityTemplate.setLang(lang);

            });
        }

        // 基础资料
        List<ActivityTemplateDTO> basicActivity = activityTemplateDTOS.stream().filter(activityTemplate -> "DATA_ENTRY_PERFORM".equals(activityTemplate.getPattern())).collect(Collectors.toList());
        if (CollectionUtil.isNotEmpty(basicActivity)) {
            Set<String> basicCodes = basicActivity.stream().map(ActivityTemplateDTO::getActivityId).collect(Collectors.toSet());
            List<Activity> activitys = getActivitysByCodes(new ArrayList<>(basicCodes));
            activityTemplateDTOS.stream().forEach(activityTemplate -> {
                Optional<Activity> activityOptional = activitys.stream().filter(activityConfig -> activityConfig.getCode().equals(activityTemplate.getActivityId())).findFirst();
                if (activityOptional.isPresent()) {
                    Activity activity = activityOptional.get();
                    activityTemplate.setActivityName(activity.getName());
                    Map<String, Map<String, String>> lang = activityTemplate.getLang();
                    Map<String, Map<String, String>> activityLang = activity.getLang();
                    if (MapUtil.isNotEmpty(activityLang)) {
                        lang.put("activityName", activityLang.get("name"));
                    }
                    activityTemplate.setLang(lang);
                }
            });
        }
        return activityTemplateDTOS;
    }

    private List<ActivityTemplate> filterActivityTemplate(QryTemplateListReq qryTemplateListReq, List<ActivityTemplate> activityTemplateList) {

        // 根据语系过滤
        if (CollectionUtil.isNotEmpty(qryTemplateListReq.getLocaleList())) {
            activityTemplateList = activityTemplateList.stream().filter(activityTemplate -> qryTemplateListReq.getLocaleList().contains(activityTemplate.getLocale())).collect(Collectors.toList());
        }

        // 根据状态过滤
        if (CollectionUtil.isNotEmpty(qryTemplateListReq.getStatusList())) {
            activityTemplateList = activityTemplateList.stream().filter(activityTemplate -> qryTemplateListReq.getStatusList().contains(activityTemplate.getStatus())).collect(Collectors.toList());
        }

        // 根据名称过滤
        if (StringUtils.isNotBlank(qryTemplateListReq.getName())) {
            String currentLocale = AthenaUtils.getCurrentLocale();
            activityTemplateList = activityTemplateList.stream().filter(activityTemplate -> String.valueOf(activityTemplate.getLang().get("name").get(currentLocale)).contains(qryTemplateListReq.getName())).collect(Collectors.toList());
        }

        // 数据排序   预设数据放前面，租户数据放后面， 根据更新时间排序
        if (CollectionUtil.isNotEmpty(activityTemplateList)) {
            activityTemplateList = activityTemplateList.stream().sorted(Comparator.comparing(ActivityTemplate::getEditDate).reversed()).collect(Collectors.toList());

        }


        // 数据排序   预设数据放前面，租户数据放后面， 根据更新时间排序
        List<ActivityTemplate> activityTemplates = new ArrayList<>();
        if (CollectionUtil.isNotEmpty(activityTemplateList)) {
            // 系统级数据
            List<ActivityTemplate> systemActivityTemplates = activityTemplateList.stream().filter(activityTemplate -> !"TENANT_COPY".equals(activityTemplate.getType())).sorted(Comparator.comparing(ActivityTemplate::getCreateDate)).collect(Collectors.toList());
            activityTemplates.addAll(systemActivityTemplates);

            // 租户级数据
            List<ActivityTemplate> tenantCopyActivityTemplates = activityTemplateList.stream().filter(activityTemplate -> "TENANT_COPY".equals(activityTemplate.getType())).sorted(Comparator.comparing(ActivityTemplate::getCreateDate)).collect(Collectors.toList());
            activityTemplates.addAll(tenantCopyActivityTemplates);
        }

        return activityTemplates;
    }


    /**
     * 数据合并，系统级编辑的数据，name 取原系统级数据
     * 数据过滤（过滤租户级的系统数据）
     *
     * @param activityTemplateList
     */
    private void handelActivityTemplate(List<ActivityTemplate> activityTemplateList) {

        // 数据合并，系统级编辑的数据，name 取原系统级数据
        activityTemplateList.stream().forEach(activityTemplate -> {
            if ("SYSTEM_EDIT".equals(activityTemplate.getType())) {
                Optional<ActivityTemplate> optional = activityTemplateList.stream().filter(template -> activityTemplate.getOriginId().equals(template.getUniqueId())).findFirst();
                if (optional.isPresent()) {
                    activityTemplate.setName(optional.get().getName());
                    activityTemplate.setLang(optional.get().getLang());
                    activityTemplate.setCreateDate(optional.get().getCreateDate());
                }
            }
        });

        // 数据过滤（过滤租户级的系统数据）
        List<ActivityTemplate> systemEdits = activityTemplateList.stream().filter(activityTemplate -> "SYSTEM_EDIT".equals(activityTemplate.getType())).collect(Collectors.toList());
        if (CollectionUtil.isNotEmpty(systemEdits)) {
            // 过滤掉多余的系统级数据
            List<String> ids = systemEdits.stream().map(ActivityTemplate::getOriginId).collect(Collectors.toList());
            activityTemplateList.removeIf(activityTemplate -> ids.contains(activityTemplate.getUniqueId()));
        }
    }

    @Override
    public Object postCopy(ActivityTemplate template) throws DWBusinessException {

        // 获取租户版本号
        String tenantId = AthenaUtils.getTenantId();
        String tenantVersion = kgInnerService.getTenantVersion(tenantId);

        ActivityTemplate activityTemplate;
        // 查询被复制的数据
        Query query = Query.query(Criteria.where("uniqueId").is(template.getUniqueId()).and("version").is(tenantVersion));
        if ("SYSTEM".equals(template.getType())) {
            // 复制的数据为系统级数据
            activityTemplate = dataPickService.findOneByCondition(Criteria.where("uniqueId").is(template.getUniqueId()), ActivityTemplate.class, "activityTemplate");
        } else {
            activityTemplate = mongoTemplateUser.findOne(query, ActivityTemplate.class, "activityTemplate");
        }
        if (null != activityTemplate) {
            // 请求DAP 获取新的模板
            JSONObject responseJson = copyCustomize(activityTemplate.getTemplateId(), template.getName(), activityTemplate.getCode(), activityTemplate.getApplication());
            if (responseJson == null) {
                return returnObj(-1, I18nUtils.getValue("knowledgegraph.copyStatementError"),null);
            }
            if (200 != responseJson.getInteger("status")) {
                return returnObj(responseJson.getInteger("status"),responseJson.getString("message"),null);
            }

            String resId = responseJson.getString("data");
            if (StringUtils.isNotEmpty(resId)) {
                // SourceId 只记录系统级的id
                activityTemplate.setOriginId("SYSTEM".equals(activityTemplate.getType()) ? activityTemplate.getUniqueId() : activityTemplate.getOriginId());
                activityTemplate.setUniqueId(generateUniqueId(tenantId, activityTemplate.getApplication(), activityTemplate.getActivityId(), activityTemplate.getPattern(),
                        activityTemplate.getCode(), activityTemplate.getClient(), resId));
                activityTemplate.setTenantId(tenantId);
                activityTemplate.setTemplateId(resId);
                activityTemplate.setType("TENANT_COPY");
                activityTemplate.setStatus(0);
                activityTemplate.setName(template.getName());
                activityTemplate.setLang(template.getLang());
                activityTemplate.setCreateDate(String.valueOf(System.currentTimeMillis()));
                activityTemplate.setCreateBy(AthenaUtils.getUserId());
                activityTemplate.setEditDate(String.valueOf(System.currentTimeMillis()));
                activityTemplate.setEditBy(AthenaUtils.getUserId());
                mongoTemplateUser.insert(activityTemplate, "activityTemplate");
            }
        }
        return returnObj(200,I18nUtils.getValue("knowledgegraph.success"),null);
    }

    @Override
    public Object postDelete(ActivityTemplate template) throws DWBusinessException {
        // 获取租户版本号
        String tenantId = AthenaUtils.getTenantId();
        String tenantVersion = kgInnerService.getTenantVersion(tenantId);
        // 查询被删除的数据
        Query query = Query.query(Criteria.where("uniqueId").is(template.getUniqueId()).and("version").is(tenantVersion));
        ActivityTemplate activityTemplate = mongoTemplateUser.findOne(query, ActivityTemplate.class, "activityTemplate");
        if (null != activityTemplate) {
            // 删除DAP 侧数据
            JSONObject responseJson = deleteCustomize(activityTemplate.getTemplateId(), activityTemplate.getCode(), activityTemplate.getApplication());
            if (responseJson == null) {
                return returnObj(-1,I18nUtils.getValue("knowledgegraph.deleteStatementError"),null);
            }
            if (200 != responseJson.getInteger("status")) {
                return returnObj(responseJson.getInteger("status"),responseJson.getString("message"),null);
            }

            // 删除租户数据
            mongoTemplateUser.remove(query, "activityTemplate");
        }
        return returnObj(200,I18nUtils.getValue("knowledgegraph.success"),null);
    }

    @Override
    public void postEdit(ActivityTemplate template) throws DWBusinessException {
        // 获取租户版本号
        String tenantId = AthenaUtils.getTenantId();
        String tenantVersion = kgInnerService.getTenantVersion(tenantId);

        // 查询被编辑的数据
        ActivityTemplate activityTemplate;
        Query query = Query.query(Criteria.where("uniqueId").is(template.getUniqueId()).and("version").is(tenantVersion));
        if ("SYSTEM".equals(template.getType())) {
            // 编辑的数据为系统级数据
            activityTemplate = dataPickService.findOneByCondition(Criteria.where("uniqueId").is(template.getUniqueId()), ActivityTemplate.class, "activityTemplate");
            // 系统级数据，新增一笔数据
            handelSystemEdit(activityTemplate, template);
        } else {
            activityTemplate = mongoTemplateUser.findOne(query, ActivityTemplate.class, "activityTemplate");

            // 租户级数据，正常更新数据
            handelTenantEdit(activityTemplate, template);
        }

    }



    public List<Activity> getActivitysByCodes(List<String> activityCode) throws DWBusinessException {
        String tenantId = AthenaUtils.getTenantId();
        String tenantVersion = kgInnerService.getTenantVersion(tenantId);
        return activityDAO.getReportActivitiesByCode(activityCode, tenantId, tenantVersion);
    }

    private void handelTenantEdit(ActivityTemplate activityTemplate, ActivityTemplate template) throws DWBusinessException {

        if (null == activityTemplate) {
            return;
        }

        Update update = new Update();
        if (null != template.getStatus()) {
            update.set("status", template.getStatus());
        }
        if (StringUtils.isNotBlank(template.getName())) {
            update.set("name", template.getName());
            update.set("lang", template.getLang());
        }
        update.set("editDate",String.valueOf(System.currentTimeMillis()));
        update.set("editBy", AthenaUtils.getUserId());
        Query query = Query.query(Criteria.where("uniqueId").is(template.getUniqueId()).and("version").is(activityTemplate.getVersion()));
        mongoTemplateUser.upsert(query, update, "activityTemplate");
    }

    private void handelSystemEdit(ActivityTemplate activityTemplate, ActivityTemplate template) throws DWBusinessException {

        if (null == activityTemplate) {
            return;
        }

        if (null != template.getStatus()) {
            activityTemplate.setStatus(template.getStatus());
        }
        if (StringUtils.isNotBlank(template.getName())) {
            activityTemplate.setName(template.getName());
            activityTemplate.setLang(template.getLang());
        }
        activityTemplate.setOriginId(activityTemplate.getUniqueId());
        activityTemplate.setUniqueId(generateUniqueId(AthenaUtils.getTenantId(), activityTemplate.getApplication(), activityTemplate.getActivityId(), activityTemplate.getPattern(), activityTemplate.getCode(), activityTemplate.getClient(), activityTemplate.getTemplateId()));
        activityTemplate.setTenantId(AthenaUtils.getTenantId());
        activityTemplate.setType("SYSTEM_EDIT");
        activityTemplate.setEditDate(String.valueOf(System.currentTimeMillis()));
        activityTemplate.setEditBy(AthenaUtils.getUserId());
        mongoTemplateUser.insert(activityTemplate, "activityTemplate");
    }


    private JSONObject deleteCustomize(String resId, String code, String appCode) {
        JSONObject paramBody = new JSONObject();
        paramBody.put("resid", resId);
        paramBody.put("program_id", code);
        paramBody.put("app_id", appCode);
        String url = dapUri + DELETE_CUSTOMIZE + "?@routerKey=" + getTenant();
        HttpServletRequest request = ((ServletRequestAttributes) RequestContextHolder.getRequestAttributes()).getRequest();
        url = addUrlParamFromHeader(request, url, "@ReportKey", "reportKey");
        String token = request.getHeader("token");
        Map<String, String> headers = new HashMap<>();
        headers.put("digi-middleware-auth-user", token);
        String response = HttpUtil.post(url,paramBody.toJSONString(),headers);
        log.info("response---"+response);
        return JSONObject.parseObject(response);
    }


    private JSONObject copyCustomize(String resId, String name, String code, String appCode) {
        JSONObject paramBody = new JSONObject();
        paramBody.put("resid", resId);
        paramBody.put("rptname", name);
        paramBody.put("program_id", code);
        paramBody.put("app_id", appCode);
        String url = dapUri + COPY_CUSTOMIZE + "?@routerKey=" + getTenant();
        HttpServletRequest request = ((ServletRequestAttributes) RequestContextHolder.getRequestAttributes()).getRequest();
        url = addUrlParamFromHeader(request, url, "@ReportKey", "reportKey");
        String token = request.getHeader("token");
        Map<String, String> headers = new HashMap<>();
        headers.put("digi-middleware-auth-user", token);
        String response = HttpUtil.post(url, paramBody.toJSONString(), headers);
        log.info("response---" + response);
        return JSONObject.parseObject(response);

    }

    @SuppressWarnings("findsecbugs:CRLF_INJECTION_LOGS")
    private String addUrlParamFromHeader(HttpServletRequest request, String url, String urlParamKey, String headerKey) {
        String value = request.getHeader(headerKey);
        log.info("获取请求头参数headerKey：{}，value：{}", headerKey, value); //NOSONAR
        if (StringUtils.isNotEmpty(value)) {
            url += "&" + urlParamKey + "=" + value;
        }
        log.info("获取请求头参数url：{}", url); //NOSONAR
        return url;
    }


    private String getTenant() {
        Map<String, Object> profile = DWServiceContext.getContext().getProfile();
        return (String) profile.getOrDefault("tenantId", "");
    }

    private String generateUniqueId(String tenantId, String application, String activityId, String pattern, String code, String client, String templateId) {
        String input = tenantId + application + activityId + pattern + code + client + templateId;  // 拼接字段
        try {
            MessageDigest md = MessageDigest.getInstance("SHA-256");  // 使用 SHA-256 哈希算法
            byte[] hash = md.digest(input.getBytes());

            // 将字节数组转为十六进制字符串
            StringBuilder hexString = new StringBuilder();
            for (byte b : hash) {
                hexString.append(String.format("%02x", b));  // 转为小写十六进制
            }
            return hexString.toString();
        } catch (NoSuchAlgorithmException e) {
            e.printStackTrace();
            return null;
        }
    }

    private Object returnObj(Integer code, String message, Object object) {
        Map<String, Object> result = new HashMap<>();
        result.put("code",code);
        result.put("message",message);
        result.put("list",object);
        return result;
    }
}
