package com.digiwin.athena.knowledgegraph.service.impl;

import com.digiwin.app.container.exceptions.DWBusinessException;
import com.digiwin.app.container.exceptions.DWException;
import com.digiwin.athena.kg.report.hz.model.dataCollectGroup.DelConfigDTO;
import com.digiwin.athena.kmservice.locale.Lang;
import com.digiwin.athena.kmservice.aspect.MyExceptionHandler;
import com.digiwin.athena.knowledgegraph.dto.report.dataBoard.DataBoardGroupConfig;
import com.digiwin.athena.knowledgegraph.service.IDataBoardGroupService;
import com.digiwin.athena.knowledgegraph.utils.AthenaUtils;
import io.seata.common.util.StringUtils;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.data.mongodb.core.BulkOperations;
import org.springframework.data.mongodb.core.MongoTemplate;
import org.springframework.data.mongodb.core.query.Criteria;
import org.springframework.data.mongodb.core.query.Query;
import org.springframework.data.mongodb.core.query.Update;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import java.util.List;
import java.util.Objects;
import java.util.UUID;

@Lang
@Service
@Slf4j
@MyExceptionHandler
public class DataBoardGroupService implements IDataBoardGroupService {

    @Autowired
    @Qualifier("knowledgegraphTenant")
    MongoTemplate mongoTemplateUser;

    @Autowired
    DataBoardService dataBoardService;

    @Autowired
    SceneService sceneService;

    public static final String FAIL_MSG = "fail";

    public static final String SUCCESS_MSG = "success";

    // 将最大收藏问题数定义为常量
    private static final int MAX_SAVED_QUESTIONS = 8;
    @Override
    public Object postAddDataBoardGroupConfig(List<DataBoardGroupConfig> configs) throws DWException {
        if (CollectionUtils.isEmpty(configs)) {
            return FAIL_MSG;
        }

        Query queryCount = queryCriteria(new DataBoardGroupConfig());
        long groupListCount = this.mongoTemplateUser.count(queryCount, DataBoardGroupConfig.class);
        if (groupListCount + configs.size() > MAX_SAVED_QUESTIONS) {
            String currentLocale = AthenaUtils.getCurrentLocale();
            if (Objects.equals("zh_TW", currentLocale)) {
                throw new DWException("P.KG.500.99999", "目前僅支持最多8個收藏夾，每個收藏夾最多收藏8個數據");
            }
            throw new DWException("P.KG.500.99999", "目前仅支持最多8个收藏夹，每个收藏夹最多收藏8个数据");
        }

        BulkOperations bulkOperations = mongoTemplateUser.bulkOps(BulkOperations.BulkMode.UNORDERED, DataBoardGroupConfig.class);
        for (DataBoardGroupConfig config : configs) {
            if (StringUtils.isEmpty(config.getGroupId())) {
                config.setGroupId("group_" + UUID.randomUUID().toString().replace("-", ""));
            }
            Query query = Query.query(Criteria
                    .where("tenantId").is(config.getTenantId())
                    .and("groupId").is(config.getGroupId())
                    .and("userId").is(config.getUserId()));

            Update update = new Update()
                    .set("groupValue", config.getGroupValue())
                    .set("userName", config.getUserName());

            bulkOperations.upsert(query, update);
        }

        bulkOperations.execute();
        return SUCCESS_MSG;
    }

    @Override
    public Object postRemoveDataBoardGroupConfig(DelConfigDTO delConfig) throws Exception {
        if (CollectionUtils.isEmpty(delConfig.getGroupIds())) {
            return FAIL_MSG;
        }
        dataBoardService.postRemoveDataBoardConfig(delConfig.getGroupIds());
        this.mongoTemplateUser.remove(Query.query(Criteria.where("groupId").in(delConfig.getGroupIds())), DataBoardGroupConfig.class);
        return SUCCESS_MSG;
    }

    @Override
    public Object getQueryDataBoardGroupConfig() throws DWBusinessException, NoSuchFieldException, IllegalAccessException, Exception {
        Query query = queryCriteria(new DataBoardGroupConfig());
        List<DataBoardGroupConfig> groupList = this.mongoTemplateUser.find(query, DataBoardGroupConfig.class);
        return sceneService.getTranslateContent(groupList);
    }

    private Query queryCriteria(DataBoardGroupConfig config) throws DWBusinessException {
        String userId = AthenaUtils.getUserId();
        String tenantId = AthenaUtils.getTenantId();

        Query query = Query.query(Criteria.where("tenantId").is(tenantId)
                .and("userId").is(userId));

        if (StringUtils.isNotEmpty(config.getGroupId())) {
            query.addCriteria(Criteria.where("groupId").is(config.getGroupId()));
        }

        if (StringUtils.isNotEmpty(config.getGroupValue())) {
            query.addCriteria(Criteria.where("groupValue").is(config.getGroupValue()));
        }
        return query;
    }

}
