package com.digiwin.athena.knowledgegraph.service.impl;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.alibaba.fastjson.TypeReference;
import com.digiwin.app.container.exceptions.DWException;
import com.digiwin.athena.kmservice.locale.Lang;
import com.digiwin.athena.kmservice.aspect.MyExceptionHandler;
import com.digiwin.athena.knowledgegraph.dto.report.dataCollect.DataCollectConfig;
import com.digiwin.athena.knowledgegraph.dto.report.dataCollect.DataCollectQuestions;
import com.digiwin.athena.kg.report.hz.model.dataCollect.QueryConfigDTO;
import com.digiwin.athena.kg.report.hz.model.dataCollectGroup.DataCollectGroupConfig;
import com.digiwin.athena.kg.report.hz.model.sence.ReportSceneDTO;
import com.digiwin.athena.knowledgegraph.dto.report.dataCollect.QueryCollectDTO;
import com.digiwin.athena.knowledgegraph.service.IDataCollectService;
import com.digiwin.athena.knowledgegraph.service.KgInnerService;
import com.digiwin.athena.knowledgegraph.utils.AthenaUtils;
import com.digiwin.athena.knowledgegraph.utils.JsonUtil;
import io.seata.common.util.StringUtils;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.data.mongodb.core.MongoTemplate;
import org.springframework.data.mongodb.core.query.Criteria;
import org.springframework.data.mongodb.core.query.Query;
import org.springframework.data.mongodb.core.query.Update;
import org.springframework.stereotype.Service;
import org.springframework.util.ObjectUtils;

import java.util.*;
import java.util.stream.Collectors;

@Lang
@Service
@Slf4j
@MyExceptionHandler
public class DataCollectService implements IDataCollectService {

    @Autowired
    @Qualifier("knowledgegraphTenant")
    MongoTemplate mongoTemplateUser;

    @Autowired
    SceneService sceneService;

    @Autowired
    KgInnerService kgInnerService;

    @Autowired
    DataCollectGroupService dataCollectGroupService;

    public static final String FAIL_MSG = "fail";

    public static final String SUCCESS_MSG = "success";

    // 将最大收藏问题数定义为常量
    private static final int MAX_SAVED_QUESTIONS = 8;


    @Override
    public String postAddDataCollectConfig(DataCollectConfig config) throws Exception {
        if (StringUtils.isEmpty(config.getTenantId()) || StringUtils.isEmpty(config.getUserId())
                || CollectionUtils.isEmpty(config.getDataCollectQuestions())
                || Objects.isNull(config.getDataCollectQuestions().get(0))) {
            return FAIL_MSG;
        }

        // 查场景
        String tenantVersion = kgInnerService.currentTenantVersion();
        DataCollectQuestions collectQuestion = config.getDataCollectQuestions().get(0);
        List<String> metricIdList = CollectionUtils.isEmpty(collectQuestion.getMetricIdList()) ? new ArrayList<>() : collectQuestion.getMetricIdList();
        metricIdList.addAll(CollectionUtils.isEmpty(collectQuestion.getDatasetIdList()) ? new ArrayList<>() : collectQuestion.getDatasetIdList());
        List<ReportSceneDTO> sceneMapsData = sceneService.querySceneMapsData(
                CollectionUtils.isEmpty(metricIdList) ?
                        Collections.singletonList(collectQuestion.getSceneCode()) : metricIdList
                , tenantVersion);
        if (CollectionUtils.isEmpty(sceneMapsData)) {
            return FAIL_MSG;
        }
        // 设置appCode
        collectQuestion.setAppCode(sceneMapsData.get(0).getAppCode());

        List<DataCollectConfig> dataCollectConfigs;
        if (StringUtils.isNotEmpty(config.getAsaCode())) {
            dataCollectConfigs = getQueryDataCollectConfig(config.getAsaCode());
        } else {
            dataCollectConfigs = getQueryDataCollectConfig();
        }

        // 当groupId为空，且groupValue不为空，则自动查询创建分配groupId
        if (StringUtils.isEmpty(collectQuestion.getGroupId()) && StringUtils.isNotEmpty(collectQuestion.getGroupValue())) {
            assignGroupIdToCollectQuestion(config, collectQuestion);
        }

        // 对应的租户和用户没有数据就新增
        if (CollectionUtils.isEmpty(dataCollectConfigs)) {
            config.setBoardId("board_" + UUID.randomUUID().toString().replace("-", ""));
            collectQuestion.setQuestionId("question_" + UUID.randomUUID().toString().replace("-", ""));
            config.setDataCollectQuestions(Collections.singletonList(collectQuestion));
            this.mongoTemplateUser.save(config);
        } else {
            // 默认单个看板
            List<DataCollectQuestions> dataCollectQuestions = dataCollectConfigs.get(0).getDataCollectQuestions();

            processDataCollectQuestions(config, dataCollectQuestions);

            // 因为question多语言的问题，固需要把入参和数据库中已存在的做多语言处理
            Object dBDataObject = sceneService.getTranslateContent(dataCollectQuestions);
            List<DataCollectQuestions> dBDataList = JSONObject.parseObject(JsonUtil.toJsonString(dBDataObject), new TypeReference<List<DataCollectQuestions>>() {
            });

            Object requestObject = sceneService.getTranslateContent(collectQuestion);
            DataCollectQuestions requestQuestion = JsonUtil.toObject(JsonUtil.toJsonString(requestObject), DataCollectQuestions.class);

            // 数据集处理
            if(requestQuestion != null && CollectionUtils.isNotEmpty(requestQuestion.getDatasetIdList())) {
                String questionId = requestQuestion.getQuestionId();
                if (StringUtils.isEmpty(questionId)) {
                    requestQuestion.setQuestionId("question_" + UUID.randomUUID().toString().replace("-", ""));
                    dBDataList.add(0, requestQuestion);
                    updateDataCollectConfig(dataCollectConfigs.get(0).getBoardId(), config, dBDataList);
                }else {
                    for(DataCollectQuestions dBDataQuestion : dBDataList) {
                        if(Objects.equals(dBDataQuestion.getQuestionId(), questionId)) {
                            dBDataQuestion.setGroupId(StringUtils.isEmpty(requestQuestion.getGroupId()) ? dBDataQuestion.getGroupId() : requestQuestion.getGroupId());
                            dBDataQuestion.setGroupValue(StringUtils.isEmpty(requestQuestion.getGroupValue()) ? dBDataQuestion.getGroupValue() : requestQuestion.getGroupValue());
                            dBDataQuestion.setGroupType(StringUtils.isEmpty(requestQuestion.getGroupType()) ? dBDataQuestion.getGroupType() : requestQuestion.getGroupType());
                            if (Objects.nonNull(requestQuestion.getSolutionStep())) {
                                dBDataQuestion.setSolutionStep(requestQuestion.getSolutionStep());
                                break;
                            }
                        }
                    }
                    updateDataCollectConfig(dataCollectConfigs.get(0).getBoardId(), config, dBDataList);
                }
            }else{
                // 判断数据列表是否包含该问题
                if (dBDataList.contains(requestQuestion)) {
                    // 移除已经存在的数据,处理不同分组有相同收藏的问题
                    dBDataList.removeIf(data ->
                            Objects.equals(data.getSceneCode(), requestQuestion.getSceneCode()) &&
                                    Objects.equals(data.getMetricIdList(), requestQuestion.getMetricIdList()) &&
                                    Objects.equals(data.getQuestion(), requestQuestion.getQuestion()) &&
                                    Objects.equals(data.getAppCode(), requestQuestion.getAppCode())
                    );

                    if (requestQuestion != null && StringUtils.isEmpty(requestQuestion.getQuestionId())) {
                        requestQuestion.setQuestionId("question_" + UUID.randomUUID().toString().replace("-", ""));
                    }
                    // 插入到列表头部
                    dBDataList.add(0, requestQuestion);
                    updateDataCollectConfig(dataCollectConfigs.get(0).getBoardId(), config, dBDataList);
                } else if (!dataCollectQuestions.contains(collectQuestion)) {
                    // 默认后续数据都插入头部，按收藏顺序倒序显示
                    if (StringUtils.isEmpty(collectQuestion.getQuestionId())) {
                        collectQuestion.setQuestionId("question_" + UUID.randomUUID().toString().replace("-", ""));
                    }
                    dataCollectQuestions.add(0, collectQuestion);
                    updateDataCollectConfig(dataCollectConfigs.get(0).getBoardId(), config, dataCollectQuestions);
                }
            }

        }
        return SUCCESS_MSG;
    }

    private DataCollectGroupConfig getDataCollectGroupConfig(DataCollectConfig config, DataCollectQuestions collectQuestion) {
        DataCollectGroupConfig groupConfig = new DataCollectGroupConfig();
        groupConfig.setTenantId(config.getTenantId());
        groupConfig.setUserId(config.getUserId());
        groupConfig.setUserName(config.getUserName());
        groupConfig.setGroupValue("");
        groupConfig.setType(collectQuestion.getGroupType());
        groupConfig.setAsaCode(config.getAsaCode());
        return groupConfig;
    }

    @Override
    public List<DataCollectConfig> getQueryDataCollectConfig() throws Exception {
        String userId = AthenaUtils.getUserId();
        String tenantId = AthenaUtils.getTenantId();
        Query query = Query.query(Criteria.where("tenantId").is(tenantId)
                .and("userId").is(userId).orOperator(Criteria.where("asaCode").is(null), Criteria.where("asaCode").is("")));

        return this.mongoTemplateUser.find(query, DataCollectConfig.class);
    }

    @Override
    public List<DataCollectConfig> getQueryDataCollectConfig(String asaCode) throws Exception {
        String userId = AthenaUtils.getUserId();
        String tenantId = AthenaUtils.getTenantId();
        Query query = Query.query(Criteria.where("tenantId").is(tenantId)
                .and("userId").is(userId).and("asaCode").is(asaCode));

        return this.mongoTemplateUser.find(query, DataCollectConfig.class);
    }

    @Override
    public Object postQueryDataCollectConfig(QueryConfigDTO queryConfig) throws Exception {
        String asaCode = queryConfig.getAsaCode();
        List<DataCollectConfig> dataCollectConfigs = StringUtils.isNotEmpty(asaCode)
                ? this.getQueryDataCollectConfig(asaCode)
                : this.getQueryDataCollectConfig();

        if (StringUtils.isEmpty(queryConfig.getGroupId()) || null == dataCollectConfigs || dataCollectConfigs.isEmpty()) {
            return dataCollectConfigs;
        }

        List<DataCollectQuestions> collectConfigs = dataCollectConfigs.get(0).getDataCollectQuestions();
        if (null != collectConfigs) {
            List<DataCollectQuestions> filteredList = collectConfigs.stream()
                    .filter(collectConfig -> Objects.equals(queryConfig.getGroupId(), collectConfig.getGroupId()))
                    .collect(Collectors.toList());
            dataCollectConfigs.get(0).setDataCollectQuestions(filteredList);
        }
        return sceneService.getTranslateContent(dataCollectConfigs);
    }

    @Override
    public String postRemoveDataCollectConfig(DataCollectConfig config) throws Exception {
        List<DataCollectQuestions> dataCollectQuestion = config.getDataCollectQuestions();
        if (CollectionUtils.isEmpty(dataCollectQuestion)) {
            return SUCCESS_MSG;
        }

        String questionId =dataCollectQuestion.get(0).getQuestionId();
        if (!StringUtils.isEmpty(questionId)) {
            DataCollectConfig configs = mongoTemplateUser.findOne(Query.query(Criteria.where("dataCollectQuestions.questionId").is(questionId)), DataCollectConfig.class);
            if (Objects.nonNull(configs)) {
                List<DataCollectQuestions> questions = configs.getDataCollectQuestions();
                if (!CollectionUtils.isEmpty(questions)) {
                    questions.removeIf(dataCollectQues-> Objects.equals(questionId, dataCollectQues.getQuestionId()));
                    updateDataCollectConfig(configs.getBoardId(), config, questions);
                    return SUCCESS_MSG;
                }
            }
        }

        List<DataCollectConfig> dataCollectConfigs;
        if (StringUtils.isEmpty(config.getAsaCode())) {
            dataCollectConfigs = getQueryDataCollectConfig();
        } else {
            dataCollectConfigs = getQueryDataCollectConfig(config.getAsaCode());
        }

        if (CollectionUtils.isEmpty(dataCollectConfigs)) {
            return FAIL_MSG;
        }

        // 默认单个看板
        List<DataCollectQuestions> dataCollectQuestions = dataCollectConfigs.get(0).getDataCollectQuestions();
        // 默认一个个问题单独删除
        if (CollectionUtils.isNotEmpty(dataCollectQuestions)) {
            dataCollectQuestions.removeAll(config.getDataCollectQuestions());
        }

        updateDataCollectConfig(dataCollectConfigs.get(0).getBoardId(), config, dataCollectQuestions);
        return SUCCESS_MSG;
    }

    private void processDataCollectQuestions(DataCollectConfig config, List<DataCollectQuestions> dataCollectQuestions) throws Exception {
        // 默认单个看板
        String currentLocale = AthenaUtils.getCurrentLocale();
        DataCollectQuestions collectQuestion = config.getDataCollectQuestions().get(0);
        if (CollectionUtils.isEmpty(collectQuestion.getDatasetIdList()) && CollectionUtils.isNotEmpty(dataCollectQuestions) && dataCollectQuestions.contains(collectQuestion)) {
            int index = dataCollectQuestions.indexOf(collectQuestion);
            if ((StringUtils.isNotEmpty(collectQuestion.getGroupId()) &&
                    !Objects.equals(collectQuestion.getGroupId(), dataCollectQuestions.get(index).getGroupId()))) {
                collectQuestion.setSolutionStep(ObjectUtils.isEmpty(collectQuestion.getSolutionStep()) ? dataCollectQuestions.get(index).getSolutionStep() : collectQuestion.getSolutionStep());
                dataCollectQuestions.set(index, collectQuestion);
            } else {
                if (Objects.equals("zh_TW", currentLocale)) {
                    throw new DWException("P.KG.500.99990", "已收藏，請勿重複添加");
                }
                throw new DWException("P.KG.500.99990", "已收藏，请勿重复添加");
            }
        }


        // 创建一个map用于存储每个groupId的个数
        Map<String, Integer> groupCountMap = new HashMap<>();

        // 遍历现有的问题列表，统计每个groupId的个数
        for (DataCollectQuestions dataCollectQuestion1 : dataCollectQuestions) {
            if (StringUtils.isNotBlank(dataCollectQuestion1.getGroupId())) {
                groupCountMap.put(dataCollectQuestion1.getGroupId(), groupCountMap.getOrDefault(dataCollectQuestion1.getGroupId(), 0) + 1);
            }
        }

        // 获取当前collectQuestion的groupId的个数
        int currentGroupCount = groupCountMap.getOrDefault(collectQuestion.getGroupId(), 0);
        if (CollectionUtils.isNotEmpty(dataCollectQuestions)
                && currentGroupCount >= MAX_SAVED_QUESTIONS
                && (!dataCollectQuestions.contains(collectQuestion) || CollectionUtils.isNotEmpty(collectQuestion.getDatasetIdList())) &&  StringUtils.isNotBlank(collectQuestion.getGroupValue())) {

            if (Objects.equals("zh_TW", currentLocale)) {
                throw new DWException("P.KG.500.99999", "當前看板最多收藏8個問題");
            }
            throw new DWException("P.KG.500.99999", "当前看板最多收藏8个问题");
        }
    }

    public void updateDataCollectConfig(String boardId, DataCollectConfig config, List<DataCollectQuestions> dataCollectQuestions) throws Exception {
        // 默认单个看板
        Query query = Query.query(Criteria.where("tenantId").is(config.getTenantId())
                .and("boardId").is(boardId)
                .and("userId").is(config.getUserId()));
        Update update = new Update();
        update.set("dataCollectQuestions", dataCollectQuestions);
        if (StringUtils.isNotEmpty(config.getAgileDataMessageType())) {
            update.set("agileDataMessageType", config.getAgileDataMessageType());
        }
        this.mongoTemplateUser.updateMulti(query, update, DataCollectConfig.class);
    }


    public void postRemoveDataCollectConfig(String asaCode, List<String> groupIds) throws Exception {
        String userId = AthenaUtils.getUserId();
        String tenantId = AthenaUtils.getTenantId();
        Query query = Query.query(Criteria.where("tenantId").is(tenantId)
                .and("userId").is(userId));

        if (StringUtils.isNotEmpty(asaCode)) {
            query.addCriteria(Criteria.where("asaCode").is(asaCode));
        } else {
            Criteria criteria = new Criteria().orOperator(
                    Criteria.where("asaCode").is(""),
                    Criteria.where("asaCode").is(null));
            query.addCriteria(criteria);
        }

        List<DataCollectConfig> dataCollectConfigs = this.mongoTemplateUser.find(query, DataCollectConfig.class);

        if (CollectionUtils.isEmpty(dataCollectConfigs)) {
            return;
        }

        // 默认单个看板
        List<DataCollectQuestions> dataCollectQuestions = dataCollectConfigs.get(0).getDataCollectQuestions();
        // 默认一个个问题单独删除
        if (CollectionUtils.isEmpty(dataCollectQuestions)) {
            return;
        }

//        dataCollectQuestions.removeIf(dataCollectQuestion -> Objects.equals(groupId,dataCollectQuestion.getGroupId()));
        dataCollectQuestions.removeIf(dataCollectQuestion -> groupIds.contains(dataCollectQuestion.getGroupId()));
        updateDataCollectConfig(dataCollectConfigs.get(0).getBoardId(), dataCollectConfigs.get(0), dataCollectQuestions);
    }

    public void assignGroupIdToCollectQuestion(DataCollectConfig config, DataCollectQuestions collectQuestion) throws Exception {
        DataCollectGroupConfig groupConfig = getDataCollectGroupConfig(config, collectQuestion);
        Object dataCollectGroupList = dataCollectGroupService.postQueryDataCollectGroupConfig(groupConfig);
        Map<String, Object> dataCollectGroupMap = (Map<String, Object>) dataCollectGroupList;
        Object groupData = new Object();
        if (!ObjectUtils.isEmpty(dataCollectGroupMap)) {
            groupData = dataCollectGroupMap.get("groupData");
        }
        List<DataCollectGroupConfig> dataCollectGroupConfigs = Collections.emptyList();
        if (!ObjectUtils.isEmpty(groupData)) {
            dataCollectGroupConfigs = JSON.parseArray(groupData.toString(), DataCollectGroupConfig.class);
        }
        List<DataCollectGroupConfig> filteredList = dataCollectGroupConfigs.stream()
                .filter(dataCollectGroupConfig -> Objects.equals(collectQuestion.getGroupValue(), dataCollectGroupConfig.getGroupValue()))
                .collect(Collectors.toList());

        if (CollectionUtils.isNotEmpty(filteredList)) {
            collectQuestion.setGroupId(filteredList.get(0).getGroupId());
        } else {
            collectQuestion.setGroupId("group_" + UUID.randomUUID().toString().replace("-", ""));
            groupConfig.setOrderId(CollectionUtils.isEmpty(dataCollectGroupConfigs) ? 1 : dataCollectGroupConfigs.get(dataCollectGroupConfigs.size() - 1).getOrderId() + 1);
            groupConfig.setGroupId(collectQuestion.getGroupId());
            groupConfig.setGroupValue(collectQuestion.getGroupValue());
            dataCollectGroupService.postAddDataCollectGroupConfig(Collections.singletonList(groupConfig));
        }
    }

    @Override
    public Object postQueryCollectStep(String questionId) {
        if (StringUtils.isEmpty(questionId)) {
            return null;
        }
        Query query = Query.query(Criteria.where("dataCollectQuestions.questionId").is(questionId));

        DataCollectConfig dataCollectConfigs = this.mongoTemplateUser.findOne(query, DataCollectConfig.class);
        if (Objects.isNull(dataCollectConfigs)) {
            return null;
        }
        DataCollectQuestions data = dataCollectConfigs.getDataCollectQuestions().stream()
                .filter(dataCollectQuestion -> Objects.equals(questionId, dataCollectQuestion.getQuestionId()))
                .findFirst().get();
        if (ObjectUtils.isEmpty(data)) {
            return null;
        }

        Map<String, Object> result = new HashMap<>();
        result.put("solutionStep", data.getSolutionStep());
        result.put("groupId", data.getGroupId());
        result.put("groupValue", data.getGroupValue());
        result.put("groupType", data.getGroupType());
        result.put("questionId", data.getQuestionId());
        result.put("question", data.getQuestion());
        result.put("sceneCode", data.getSceneCode());
        result.put("appCode", data.getAppCode());
        result.put("metricIdList", data.getMetricIdList());
        result.put("showType", data.getShowType());
        result.put("snapshotId", data.getSnapshotId());
        result.put("datasetIdList", data.getDatasetIdList());
        result.put("boardId", dataCollectConfigs.getBoardId());
        return result;
    }

    @Override
    public Object postUpdateCollectByConfig(QueryCollectDTO config) throws Exception {
        if (ObjectUtils.isEmpty(config.getSolutionStep())) {
            return SUCCESS_MSG;
        }
        DataCollectConfig dataCollectConfig = new DataCollectConfig();
        dataCollectConfig.setTenantId(AthenaUtils.getTenantId());
        dataCollectConfig.setUserId(AthenaUtils.getUserId());
        dataCollectConfig.setBoardId(config.getBoardId());

        Query query = Query.query(Criteria.where("boardId").is(config.getBoardId()));
        DataCollectConfig dataCollectConfigs = this.mongoTemplateUser.findOne(query, DataCollectConfig.class);
        if (Objects.isNull(dataCollectConfigs)) {
            return FAIL_MSG;
        }
        dataCollectConfigs.getDataCollectQuestions().forEach(dataCollectQuestion -> {
            if (Objects.equals(dataCollectQuestion.getSceneCode(), config.getSceneCode())
                    && Objects.equals(dataCollectQuestion.getMetricIdList(), config.getMetricIdList())
                    && Objects.equals(dataCollectQuestion.getDatasetIdList(), config.getDatasetList())
                    && Objects.equals(dataCollectQuestion.getGroupId(), config.getGroupId())){
                if (StringUtils.isEmpty(dataCollectQuestion.getQuestionId())) {
                    dataCollectQuestion.setQuestionId("question_" + UUID.randomUUID().toString().replace("-", ""));
                }
                dataCollectQuestion.setSolutionStep(config.getSolutionStep());
            }
        });
        updateDataCollectConfig(config.getBoardId(), dataCollectConfig, dataCollectConfigs.getDataCollectQuestions());
        return SUCCESS_MSG;
    }
}