package com.digiwin.athena.knowledgegraph.service.impl;

import cn.hutool.core.bean.BeanUtil;
import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.digiwin.app.container.exceptions.DWBusinessException;
import com.digiwin.app.resource.DWResourceBundleUtils;
import com.digiwin.athena.kmservice.locale.Lang;
import com.digiwin.athena.kmservice.action.execution.model.ActionTypeEnum;
import com.digiwin.athena.kmservice.aspect.MyExceptionHandler;
import com.digiwin.athena.kg.action.Action;
import com.digiwin.athena.kg.action.ActionLabel;
import com.digiwin.athena.domain.core.app.Application;
import com.digiwin.athena.knowledgegraph.dto.report.ReportSceneDTOs;
import com.digiwin.athena.repository.neo4j.ActionRepository;
import com.digiwin.athena.kg.report.hz.model.sence.BusinessTypeDTO;
import com.digiwin.athena.kg.report.hz.model.sence.ReportSceneDTO;
import com.digiwin.athena.knowledgegraph.service.IMetricService;
import com.digiwin.athena.knowledgegraph.service.KgInnerService;
import com.digiwin.athena.knowledgegraph.service.inner.DataPickService;
import com.digiwin.athena.knowledgegraph.utils.AthenaUtils;
import com.digiwin.athena.knowledgegraph.utils.LanguageUtil;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.data.mongodb.core.MongoTemplate;
import org.springframework.data.mongodb.core.query.Criteria;
import org.springframework.data.mongodb.core.query.Query;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import java.util.*;
import java.util.function.Function;
import java.util.stream.Collectors;

@Lang
@Service
@Slf4j
@MyExceptionHandler
public class MetricService implements IMetricService {
    @Autowired
    @Qualifier("knowledgegraphSystem")
    MongoTemplate mongoTemplate;

    @Autowired
    KgInnerService kgInnerService;

    @Autowired
    ActionRepository actionRepository;

    @Autowired
    SceneService sceneService;

    @Autowired
    @Qualifier("knowledgegraphTenant")
    MongoTemplate mongoTemplateUser;

    @Autowired
    private DataPickService dataPickService;

    @Override
    public Object postMetricDataSource(String appCode, String version)  throws Exception {
        if (StringUtils.isBlank(appCode) || StringUtils.isBlank(version)) {
            throw new IllegalArgumentException("appCode and version cannot be null or empty!");
        }
        Map<String,Object> metricDataSource = new HashMap<>();
        // 根据appCode和version查询应用的数据来源
        Application application = dataPickService.findOneByCondition(Criteria.where("code").is(appCode), Application.class, "application");
        if (application != null && StringUtils.isNotEmpty(application.getSource())) {
            metricDataSource.put("dataSource",application.getSource());
            metricDataSource.put("appCode",appCode);
            metricDataSource.put("version",version);
        }
        return metricDataSource;
    }

    @Override
    public Object postMetricDataSource(List<String> indicatorIds) throws DWBusinessException {
        if (CollectionUtils.isEmpty(indicatorIds)) {
            throw new DWBusinessException("indicatorIds cannot be null or empty!");
        }

        String tenantVersion = kgInnerService.currentTenantVersion();
        // 查询数据集
        List<ReportSceneDTOs> dataSetList = mongoTemplate.find(Query.query(Criteria.where("version").is(tenantVersion).and("code").in(indicatorIds)), ReportSceneDTOs.class, "agiledatainquiry_data_set");
        if (CollectionUtils.isEmpty(dataSetList)) {
            // 查询场景，类型是指标
            List<ReportSceneDTO> scenes = sceneService.querySceneMapsData(indicatorIds, tenantVersion);
            if (CollectionUtils.isEmpty(scenes)) {
                return Collections.emptyList();
            }
            dataSetList.addAll(scenes.stream().map(scene -> {
                ReportSceneDTOs reportSceneDTOs = new ReportSceneDTOs();
                BeanUtil.copyProperties(scene, reportSceneDTOs);
                return reportSceneDTOs;
            }).collect(Collectors.toList()));
        }

        // 创建两个Set来存储结果
        List<String> appCodes = new ArrayList<>();
        Map<String,ReportSceneDTOs> metricMap = new HashMap<>();

        // 遍历列表
        for (ReportSceneDTOs scene : dataSetList) {
            // 添加appCode到集合中
            appCodes.add(scene.getAppCode());
            metricMap.put(scene.getCode(),scene);
        }

        return getMetricDataSource(metricMap,appCodes,tenantVersion);
    }

    private List<Map<String,Object>> getMetricDataSource(Map<String,ReportSceneDTOs> metricMap,
                                                         List<String> appCodes,String version) throws DWBusinessException {
        List<Map<String,Object>> metricDataSource = new ArrayList<>();
        // 根据appCode和version查询应用的数据来源
        List<Application> applications = dataPickService.find(Criteria.where("code").in(appCodes), Application.class, "application");
        if (!CollectionUtils.isEmpty(applications)) {
            Map<String, String> applicationCodeToSource = applications.stream()
                    .collect(Collectors.toMap(
                            Application::getCode, // keyMapper: 提取key的函数
                            app -> Optional.ofNullable(app.getSource()).orElse("")
                    ));

            for (Map.Entry<String, ReportSceneDTOs> entry : metricMap.entrySet()) {
                ReportSceneDTOs scene = entry.getValue();
                Map<String,Object> dataSource = new HashMap<>();
                dataSource.put("appCode", scene.getAppCode());
                dataSource.put("version", version);
                dataSource.put("indicatorId", scene.getCode());
                dataSource.put("businessType", getBusinessType(scene));
                dataSource.put("productLine",scene.getProductLine());
                dataSource.put("classification",scene.getClassification());
                if (StringUtils.isNotEmpty(scene.getAppCode()) && applicationCodeToSource.containsKey(scene.getAppCode())) {
                    dataSource.put("dataSource", applicationCodeToSource.get(scene.getAppCode()));
                } else {
                    dataSource.put("dataSource", "");
                }
                dataSource.put("fieldSchema", scene.getFieldSchema());
                dataSource.put("measures", scene.getMeasures());
                dataSource.put("dimensions", scene.getDimensions());
                metricDataSource.add(dataSource);
            }
        }
        return metricDataSource;
    }

    private void handleMetricBusinessType(List<Map<String, Object>> result,
                                          List<String> businessTypes, String tenantVersion) throws DWBusinessException {
        String tenantId = AthenaUtils.getTenantId();
        // 获取租户库数据
        List<BusinessTypeDTO> businessTypeDataList = mongoTemplateUser.find(
                Query.query(Criteria.where("code").in(businessTypes)
                        .and("tenantId").is(tenantId)), BusinessTypeDTO.class,"agiledata_business_type");

        if (CollectionUtils.isEmpty(businessTypeDataList)) {
            return;
        }

        // 转换为 Map
        Map<String, BusinessTypeDTO> businessTypeDataMap = businessTypeDataList.stream()
                .collect(Collectors.toMap(BusinessTypeDTO::getCode,
                        Function.identity()));

        // 设置业务类型
        result.forEach(data -> {
            String businessTypeCode = ((BusinessTypeDTO) data.get("businessType")).getCode();
            if (businessTypeDataMap.containsKey(businessTypeCode)) {
                data.put("businessType", businessTypeDataMap.get(businessTypeCode));
            }
        });
    }

    @Override
    public Object postMetricQuerySchema(List<String> indicatorIds) throws DWBusinessException, NoSuchFieldException, IllegalAccessException {
        String tenantVersion = kgInnerService.currentTenantVersion();
        String tenantId = AthenaUtils.getTenantId();

        Map<String, List<Map<String,Object>>> indicatorIdMap = new HashMap<>(); // 用于存储indicatorId和对应的querySchema列表

        for (String indicatorId : indicatorIds) {
            Deque<String> stack = new ArrayDeque<>(Collections.singletonList(indicatorId)); // 使用一个栈来模拟递归
            List<Action> metricActionList = getAllMetricAction(stack, tenantId, tenantVersion);

            // 处理多语言
            LanguageUtil.processLocaleLanguage(metricActionList, DWResourceBundleUtils.getCurrentLocale().toString());
            List<Map<String,Object>> actionInfoList = metricActionList.stream()
                    .filter(Objects::nonNull)
                    .map(action -> {
                        Map<String,Object> info = new HashMap<>();
                        info.put("actionId",action.getActionId());
                        info.put("actionName",action.getActionName());
                        info.put("querySchema",action.getQuerySchema());
                        info.put("response",JSON.parseObject(action.getResponse_object()));
                        info.put("productLine",StringUtils.isEmpty(action.getProductLine()) ? "ERP" : action.getProductLine());
                        return info;
                    })
                    .collect(Collectors.toList());

            indicatorIdMap.put(indicatorId,actionInfoList);
        }
        return indicatorIdMap;
    }

    @Override
    public Object postMetricBusinessByCode(String tenantId, String appCode, List<String> codes) throws DWBusinessException {
        String tenantVersion = kgInnerService.currentTenantVersion();
        // 先查询应用级数据
        List<BusinessTypeDTO> businessTypeSysDataList = mongoTemplate.find(
                Query.query(Criteria.where("code").in(codes)
                        .and("appCode").is(appCode).and("version").is(tenantVersion)), BusinessTypeDTO.class,
                "agiledata_business_type");

        // 查询租户库数据
        List<BusinessTypeDTO> businessTypeTenantDataList = mongoTemplateUser.find(
                Query.query(Criteria.where("code").in(codes)
                        .and("appCode").is(appCode)
                        .and("source").is("custom")  // 只查询自定义数据
                        .and("tenantId").is(tenantId)), BusinessTypeDTO.class,
                "agiledata_business_type");

        // 使用Map存储businessTypeTenantDataList的数据，code作为key
        Map<String, BusinessTypeDTO> tenantMap = businessTypeTenantDataList.stream()
                .collect(Collectors.toMap(BusinessTypeDTO::getCode, dto -> dto));

        // 将businessTypeSysDataList中的数据和businessTypeTenantDataList进行合并
        List<BusinessTypeDTO> result = businessTypeSysDataList.stream()
                .map(dto -> tenantMap.getOrDefault(dto.getCode(), dto))
                .collect(Collectors.toList());

        Map<String, BusinessTypeDTO> resultBusinessMap = result.stream()
                .collect(Collectors.toMap(BusinessTypeDTO::getCode,
                        Function.identity(),(existing, replacement) -> existing));
        codes.forEach(code -> resultBusinessMap.putIfAbsent(code, tenantMap.get(code)));
        return resultBusinessMap;
    }

    private List<Action> getAllMetricAction(Deque<String> stack, String tenantId,
                                                String tenantVersion) throws DWBusinessException {
        List<Action> metricActionList = new ArrayList<>();
        Set<String> processedIndicatorIds = new HashSet<>(); // 用于存储已处理的indicatorIds

        // 迭代处理栈中的元素
        while (!stack.isEmpty()) {
            String currentIndicatorId = stack.pop(); // 出栈
            if (!processedIndicatorIds.contains(currentIndicatorId)) {
                processedIndicatorIds.add(currentIndicatorId);
                List<ActionLabel> metricActions = dataPickService.filterByIndividual(actionRepository.getActionAndLabelsByMetricId(Collections.singletonList(currentIndicatorId), tenantId, tenantVersion), tenantId,
                        actionLabel -> actionLabel.getAction().getNameSpace());
                for (ActionLabel actionLabel : metricActions) {
                    if (Objects.isNull(actionLabel.getAction())) {
                        continue;
                    }

                    if (Objects.equals(ActionTypeEnum.METRIC.toString(), actionLabel.getAction().getType())) {
                        String metricId = actionLabel.getAction().getQuoteMetricCode();
                        stack.push(metricId); // 将子指标id入栈
                    } else if (Objects.equals(ActionTypeEnum.BMD_GENERAL.toString(), actionLabel.getAction().getType())) {
                        metricActionList.add(actionLabel.getAction());
                    }
                }
            }
        }
        return metricActionList;
    }

    public List<Object> getBusinessType(ReportSceneDTOs scene) {
        List<Object> businessType = scene.getBusinessType();
        if (!CollectionUtils.isEmpty(businessType)) {
            return businessType;
        }
        businessType = new ArrayList<>();
        extractBusinessType(scene.getFieldSchema(), businessType,  "name");
        extractBusinessType(scene.getMeasures(), businessType,"data_name");
        return businessType;
    }


    private void extractBusinessType(List<Object> list, List<Object> businessType,String key) {
        if (CollectionUtils.isEmpty(list)) {
            return;
        }
        List<Object> dataList = list.stream()
                .filter(item -> item instanceof Map || item instanceof JSONObject)
                .map(item -> {
                    if (item instanceof JSONObject) {
                        return (JSONObject) item;
                    } else {
                        return new JSONObject((Map<String, Object>) item);
                    }
                })
                .filter(jsonObject -> jsonObject.containsKey("businessType"))
                .map(jsonObject -> {
                    JSONObject businessTypeConfig = jsonObject.getJSONObject("businessType");
                    businessTypeConfig.put("key", jsonObject.get(key));
                    return businessTypeConfig;
                })
                .collect(Collectors.toList());
        businessType.addAll(dataList);
    }

}