package com.digiwin.athena.knowledgegraph.service.impl;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import com.digiwin.app.common.DWApplicationConfigUtils;
import com.digiwin.app.container.exceptions.DWBusinessException;
import com.digiwin.app.container.exceptions.DWException;
import com.digiwin.athena.domain.component.bo.ComponentStructuredListType;
import com.digiwin.athena.kmservice.locale.Lang;
import com.digiwin.athena.knowledgegraph.component.ComponentManager;
import com.digiwin.athena.knowledgegraph.constant.ComponentConstants;
import com.digiwin.athena.knowledgegraph.constant.Constants;
import com.digiwin.athena.knowledgegraph.domain.ComponentStructuredList;
import com.digiwin.athena.knowledgegraph.domain.TenantAppRelation;
import com.digiwin.athena.knowledgegraph.domain.TenantEntity;
import com.digiwin.athena.domain.core.app.ApplicationRelation;
import com.digiwin.athena.kg.domain.OpenWindowDefinition;
import com.digiwin.athena.kg.domain.FieldOpenWindowSetting;
import com.digiwin.athena.knowledgegraph.domain.mechanism.MechanismTenantParadigm;
import com.digiwin.athena.knowledgegraph.domain.mechanism.Paradigm;
import com.digiwin.athena.knowledgegraph.domain.mechanism.ParadigmCondition;
import com.digiwin.athena.knowledgegraph.domain.mechanism.ParadigmConditionConfig;
import com.digiwin.athena.knowledgegraph.dto.AfsDTO;
import com.digiwin.athena.knowledgegraph.dto.ParadigmDto;
import com.digiwin.athena.knowledgegraph.service.*;
import com.digiwin.athena.knowledgegraph.service.IParadigmService;
import com.digiwin.athena.knowledgegraph.service.KgInnerService;
import com.digiwin.athena.knowledgegraph.service.RedisLock;
import com.digiwin.athena.knowledgegraph.service.inner.DataPickService;
import com.digiwin.athena.knowledgegraph.utils.AthenaUtils;
import com.digiwin.athena.knowledgegraph.utils.HttpUtil;
import com.digiwin.athena.knowledgegraph.vo.HandleParadigmListVo;
import com.digiwin.athena.knowledgegraph.vo.ParadigmConditionVO;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.beanutils.BeanUtils;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang.StringEscapeUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.data.mongodb.core.MongoTemplate;
import org.springframework.data.mongodb.core.query.Criteria;
import org.springframework.data.mongodb.core.query.Query;
import org.springframework.data.mongodb.core.query.Update;
import org.springframework.stereotype.Service;

import java.lang.reflect.InvocationTargetException;
import java.time.LocalDate;
import java.time.format.DateTimeFormatter;
import java.util.*;
import java.util.function.Function;
import java.util.stream.Collectors;

@Lang
@Service
@Slf4j
public class ParadigmService implements IParadigmService {

    @Autowired
    @Qualifier("knowledgegraphSystem")
    private MongoTemplate mongoTemplateSystem;

    @Autowired
    @Qualifier("knowledgegraphTenant")
    private MongoTemplate mongoTemplateUser;

    @Autowired
    private TenantService tenantService;

    @Autowired
    private ComponentManager componentManager;

    @Autowired
    RedisLock redisLock;

    @Autowired
    KgInnerService kgInnerService;

    @Autowired
    IMechanismService iMechanismService;

    @Autowired
    IAppService iAppService;

    @Autowired
    private DataPickService dataPickService;

    @Override
    public Object getParadigmCondition(String code) throws DWException {
        String tenantId = AthenaUtils.getTenantId();
        String tenantVersion = tenantService.getTenantVersion(tenantId);

        Query query = new Query();
        query.addCriteria(Criteria.where("paradigmCode").is(code).and("version").is(tenantVersion));

        ParadigmConditionConfig paradigmConditionConfig = this.mongoTemplateSystem.findOne(query, ParadigmConditionConfig.class, "paradigmConditionConfig");

        if(paradigmConditionConfig == null){
            return null;
        }



        Query query2 = new Query();
        query2.addCriteria(Criteria.where("paradigmCode").is(code).and("tenantId").is(tenantId));
        ParadigmCondition paradigmCondition = mongoTemplateUser.findOne(query2, ParadigmCondition.class, "paradigmCondition");

        ParadigmConditionVO paradigmConditionVO = new ParadigmConditionVO();
        try {
            BeanUtils.copyProperties(paradigmConditionVO, paradigmConditionConfig);

            List<FieldOpenWindowSetting> fieldOpenWindowMap = paradigmConditionConfig.getFieldOpenWindowMap();
            if(!CollectionUtils.isEmpty(fieldOpenWindowMap)){
                Map<String, OpenWindowDefinition> fieldOpenWindowDefinitions = new HashMap<>();
                fieldOpenWindowMap.forEach(e->{
                    JSONObject jsonObject = JSON.parseObject(JSON.toJSONString(e));
                    String column = jsonObject.getString("column");
                    String openWindowKey = jsonObject.getString("openWindowKey");

                    Query query3 = new Query();
                    query3.addCriteria(Criteria.where("key").is(openWindowKey).and("version").is(tenantVersion));
                    OpenWindowDefinition openWindowDefinition = mongoTemplateSystem.findOne(query3, OpenWindowDefinition.class, "openWindowDefinition");
                    if(openWindowDefinition != null){
                        fieldOpenWindowDefinitions.put(column, openWindowDefinition);
                    }
                });

                paradigmConditionVO.setFieldOpenWindowDefinitions(fieldOpenWindowDefinitions);
            }

            if(paradigmCondition != null){
                paradigmConditionVO.setConditions(paradigmCondition.getConditions());
            }
        } catch (IllegalAccessException | InvocationTargetException e) {
            log.error(e.getMessage(), e);
        }

        return paradigmConditionVO;
    }

    @Override
    public Object postSaveParadigmCondition(ParadigmConditionVO paradigmCondition) throws DWException {
        String tenantId = AthenaUtils.getTenantId();
        Query query2 = new Query();
        query2.addCriteria(Criteria.where("paradigmCode").is(paradigmCondition.getParadigmCode()).and("tenantId").is(tenantId));

        Update update = new Update();
        update.set("conditions", paradigmCondition.getConditions());

        mongoTemplateUser.upsert(query2, update, "paradigmCondition");
        return null;
    }

    /**
     * 新增范式数据
     * @param paradigmDto 新增的范式数据
     */
    @Override
    public Boolean postSaveParadigm(ParadigmDto paradigmDto) throws DWException {
         //先删除后新增
         mongoTemplateSystem.remove(Query.query(Criteria.where("code").is(
                 paradigmDto.getCode())), Paradigm.class);
         //新增范式
         Paradigm paradigm = new Paradigm();
         org.springframework.beans.BeanUtils.copyProperties(paradigmDto, paradigm);
         //仅支持发版
         paradigm.setVersion("1.0");
         mongoTemplateSystem.insert(paradigm, "paradigm");

        return true;
    }


    /**
     * 删除范式数据
     * @param code 范式编码
     */
    @Override
    public Boolean postDeleteParadigm(String code) throws DWException {
        Query query = new Query();
        query.addCriteria(Criteria.where("code").is(code));

        mongoTemplateSystem.remove(query, "paradigm");
        return true;
    }

    @Override
    public Object getParadigmConditionByMechanismCode(String  mechanismCode) throws DWException{
        String tenantId = AthenaUtils.getTenantId();
        String version = kgInnerService.getTenantVersion(tenantId);
        ComponentStructuredList mechanismComponentStructuredList =
                this.componentManager.findMechanismComponentStructuredList(mechanismCode, version, tenantId);
        if(mechanismComponentStructuredList != null){

            String paradigmCode = mechanismComponentStructuredList.getParadigm();

            String tenantVersion = tenantService.getTenantVersion(tenantId);

            Query query = new Query();
            query.addCriteria(Criteria.where("paradigmCode").is(paradigmCode).and("version").is(tenantVersion));

            ParadigmConditionConfig paradigmConditionConfig = this.mongoTemplateSystem.findOne(query, ParadigmConditionConfig.class, "paradigmConditionConfig");

            if(paradigmConditionConfig == null){
                return null;
            }

            Query query2 = new Query();
            query2.addCriteria(Criteria.where("paradigmCode").is(paradigmCode).and("tenantId").is(tenantId));
            ParadigmCondition paradigmCondition = mongoTemplateUser.findOne(query2, ParadigmCondition.class, "paradigmCondition");

            if(paradigmCondition != null && paradigmCondition.getConditions() != null){
                Object conditions = paradigmCondition.getConditions();
                Map<String, Object> result = new HashMap<>();
                result.put("tenantId", tenantId);
                result.put("paradigmCode", paradigmCode);
                result.put("fieldMapsTo", paradigmConditionConfig.getFieldMapsTo());

                if(conditions instanceof Collection){
                    Map<String, Object> andCondition = new HashMap<>();
                    andCondition.put("type", "AND_GROUP");
                    andCondition.put("items", conditions);
                    result.put("conditions", andCondition);
                }else{
                    result.put("conditions", conditions);
                }

                return result;
            }

        }
        return null;
    }

    @Override
    public Object postHandleParadigmInit() {
        String paradigmInitSwitch = DWApplicationConfigUtils.getProperty(Constants.PARADIGM_INIT_SWITCH);
        log.info("paradigmInitSwitch:{}", paradigmInitSwitch);
        if(StringUtils.equalsIgnoreCase(paradigmInitSwitch, "false")){ //当开关关闭时，不进行校验，直接返回
            return null;
        }

        // 分布式锁key增加redis规范appId前缀
        String kgTenantParadigmInitLockKey = ComponentConstants.REDIS_KNOWLEDGE_GRAPH + ":tenantParadigmInitLockKey";
        String tenantParadigmInitLockKey = null;
        try {
            tenantParadigmInitLockKey = redisLock.tryLock(kgTenantParadigmInitLockKey, 5 * 60 * 1000);
            if (tenantParadigmInitLockKey != null) {
                log.info("=======拿到租户范式初始化锁{}:{}=======", kgTenantParadigmInitLockKey, tenantParadigmInitLockKey);
                log.info("===========开始TenantParadigmInit过程==========");

                //1.查询与范式有关联的应用
                List<ApplicationRelation> relations = iAppService.getApplicationRelationByType("paradigm");
                if (CollectionUtils.isEmpty(relations)) {
                    log.info("hasNoParadigmRelation...");
                    return null;
                }
                List<String> appCodeList = relations.stream().map(ApplicationRelation::getAppCode).distinct().collect(Collectors.toList());
                //2.对应应用的租户
                List<TenantAppRelation> tenantAppRelationByAppCode = dataPickService.getTenantAppRelationByAppCode(appCodeList);
                if (CollectionUtils.isEmpty(tenantAppRelationByAppCode)) {
                    log.info("noTenantHasParadigmApplication...");
                    return null;
                }

                //查询所有的范式
                List<Paradigm> allParadigms = mongoTemplateSystem.findAll(Paradigm.class);
                if (CollectionUtils.isEmpty(allParadigms)) {
                    log.info("not found paradigm...");
                    return null;
                }

                Map<String, Paradigm> collect = allParadigms.stream().collect(Collectors.toMap(Paradigm::getCode, Function.identity(), (a, b) -> a));
                Map<String, List<Paradigm>> appAndParadigmMap = relations.stream().collect(Collectors.toMap(ApplicationRelation::getAppCode,
                        p -> {
                            List<Paradigm> paradigmList = new ArrayList<>();
                            Paradigm paradigm = collect.get(p.getCode());
                            if (paradigm != null) {
                                paradigmList.add(paradigm);
                            }
                            return paradigmList;
                        },
                        (List<Paradigm> value1, List<Paradigm> value2) -> {
                            value1.addAll(value2);
                            return value1;
                        }
                ));


                Set<String> hasInitTenants = new HashSet<>();
                //3.获取租户的客服代号
                for (TenantAppRelation tenantAppInfoDTO : tenantAppRelationByAppCode) {
                    log.info("handle tenant:{} , appCode:{}", tenantAppInfoDTO.getTenantId(), tenantAppInfoDTO.getAppCode());
                    //TODO  改为tenantId+appCode ？
                    if (hasInitTenants.contains(tenantAppInfoDTO.getTenantId() + tenantAppInfoDTO.getAppCode())) {
                        continue;
                    }
                    TenantEntity tenant = tenantService.getTenant(tenantAppInfoDTO.getTenantId());
                    if (StringUtils.isEmpty(tenant.getCustomerServiceCode())) {
                        log.info("ParadigmInit tenantId:{} customerServiceCode is null", tenantAppInfoDTO.getTenantId());
                        continue;
                    }

                    try {
                        //4.调用接口获取订单信息
                        Map<String, Object> request = new HashMap<>();
                        request.put("AFS01", tenant.getCustomerServiceCode());
//                        request.put("AFS01", "72002181");
                        String reqUrl = DWApplicationConfigUtils.getProperty("tiptopUrl") + "/api/Athena/AthenaFS";
                        String result = HttpUtil.post(reqUrl, JSON.toJSONString(request));

                        JSONObject jsonObject = JSON.parseObject(result);
                        String domain = jsonObject.getString("domain");
                        String code = jsonObject.getString("code");
                        if (!StringUtils.equals(code, "0")) {  // 0表示调用接口正常
                            log.error("ParadigmInit tenantId:{} getOrderInfoError:{}", tenantAppInfoDTO.getTenantId(), domain);
                            continue;
                        }
                        //去除转义
                        String s = StringEscapeUtils.unescapeJava(domain);
                        List<AfsDTO> afsDTOS = JSONArray.parseArray(s, AfsDTO.class);
                        if (CollectionUtils.isEmpty(afsDTOS)) {
                            continue;
                        }
                        //5.匹配范式信息
                        Map<String, List<String>> appParadigmMap = new HashMap();
                        //先按订单时间倒序排序
                        List<AfsDTO> sortedAfsDTOS = afsDTOS.stream().sorted(Comparator.comparing(AfsDTO::getAFS04).reversed()).collect(Collectors.toList());
                        List<String> paradigmCodeList = new ArrayList<>();
                        String beforeOrderTime = null;
                        for (AfsDTO afsDTO : sortedAfsDTOS) {
                            //只关注最新日期的订单
                            if(StringUtils.isNotEmpty(beforeOrderTime) && !StringUtils.equals(beforeOrderTime, afsDTO.getAFS04())){
                                break;
                            }
                            beforeOrderTime = afsDTO.getAFS04();

                            //检查订单时间与当前时间的时间间隔，超过一定时间则不进行匹配？
                            LocalDate orderDate = LocalDate.parse(afsDTO.getAFS04(), DateTimeFormatter.ofPattern("yyyyMMdd"));
                            if (orderDate.plusDays(30).isBefore(LocalDate.now())) {
                                break;
                            }


//                            afsDTO.setAFS06("****_****(定點一站組裝)_****");
                            List<String> paradigmFromAFS06 = this.getParadigmFromAFS06(afsDTO.getAFS06());
                            if (CollectionUtils.isEmpty(paradigmFromAFS06)) {
                                continue;
                            }

                            List<String> paradigmCodes = this.matchParadigmByName(appAndParadigmMap, paradigmFromAFS06, tenantAppInfoDTO.getAppCode());
                            if (!CollectionUtils.isEmpty(paradigmCodes)) { //如果找到，就不用再找了
                                paradigmCodeList.addAll(paradigmCodes);
                            }


                        }
                        log.info("paradigmCodeList:{}", paradigmCodeList);
                        if (!CollectionUtils.isEmpty(paradigmCodeList)) {
                            paradigmCodeList = paradigmCodeList.stream().distinct().collect(Collectors.toList());
                            //6.新增或者更新范式
                            List<String> mechanismByAppCodeAndParadigms = this.getMechanismByAppCodeAndParadigms(tenantAppInfoDTO.getAppCode(), paradigmCodeList);
                            MechanismTenantParadigm mechanismTenantParadigm = this.getMechanismTenantParadigm(tenantAppInfoDTO.getAppCode(), tenantAppInfoDTO.getTenantId());
                            if (mechanismTenantParadigm == null) {
                                this.componentManager.InitializeTenantApplicationComponentList(tenantAppInfoDTO.getAppCode(), tenantAppInfoDTO.getTenantId(), paradigmCodeList, mechanismByAppCodeAndParadigms);
                            } else {
                                //如果当前范式和已存在范式一致，跳过，否则更新
                                List<String> existsParadigms = this.getExistsParadigms(mechanismTenantParadigm
                                );
                                boolean subCollection = CollectionUtils.isSubCollection(paradigmCodeList, existsParadigms);
                                if (!subCollection) {
                                    this.componentManager.UpdateTenantApplicationComponentListByMechanism(tenantAppInfoDTO.getAppCode(), tenantAppInfoDTO.getTenantId(),
                                            paradigmCodeList, mechanismByAppCodeAndParadigms, null);
                                }

                            }
                        }


                        hasInitTenants.add(tenantAppInfoDTO.getTenantId() + tenantAppInfoDTO.getAppCode());
                    } catch (Exception e) {
                        log.error("ParadigmInit tenantId:{} error:{}", tenantAppInfoDTO.getTenantId(), e.getMessage());
                    }

                }
            }else{
                log.info("=======没有拿到租户范式初始化的锁{}，应用的其他实例正在处理=======", kgTenantParadigmInitLockKey);
            }
        }catch (Exception e) {
            log.error("tenantParadigmInit error:{}", e.getMessage());
        } finally {
            if(tenantParadigmInitLockKey != null){
                log.info("释放锁{}：{}", kgTenantParadigmInitLockKey, tenantParadigmInitLockKey);
                redisLock.unlock(kgTenantParadigmInitLockKey, tenantParadigmInitLockKey);
            }
        }

        return null;
    }

    @Override
    public Object getParadigmsByAppCode(String appCode) throws DWException {
        String tenantId = AthenaUtils.getTenantId();
        Query query = new Query(Criteria.where("tenantId").is(tenantId).and("code").is(appCode));
        MechanismTenantParadigm one = this.mongoTemplateUser.findOne(query, MechanismTenantParadigm.class);
        if(one == null){
            return null;
        }

        if(CollectionUtils.isNotEmpty(one.getParadigms())){
            return one.getParadigms();
        }

        if(CollectionUtils.isNotEmpty(one.getValue())){
            List<ComponentStructuredList> componentStructuredLists = dataPickService.find(Criteria.where("mechanismCode").in(one.getValue()).and("appCode").is(appCode), ComponentStructuredList.class, "componentStructuredList");
            if(CollectionUtils.isNotEmpty(componentStructuredLists)){
                List<String> collect = componentStructuredLists.stream().map(ComponentStructuredList::getParadigm).distinct().collect(Collectors.toList());
                return collect;
            }
        }
        return null;
    }

    private  List<String> getExistsParadigms(MechanismTenantParadigm mechanismTenantParadigm) throws DWBusinessException {
        if(CollectionUtils.isEmpty(mechanismTenantParadigm.getParadigms())){
            List<String> value = mechanismTenantParadigm.getValue();
            Criteria criteria = Criteria.where(ComponentManager.MECHANISM_CODE).in(value)
                    .and(ComponentManager.TYPE).is(ComponentStructuredListType.Mechanism);
            List<ComponentStructuredList> result = dataPickService.find(criteria, ComponentStructuredList.class, "componentStructuredList");
            return result == null ? new ArrayList<>() : result.stream().map(ComponentStructuredList::getParadigm).distinct().collect(Collectors.toList());
        }else{
            return mechanismTenantParadigm.getParadigms();
        }
    }

    public List<String> getMechanismByAppCodeAndParadigms(String appCode, List<String> paradigmList) throws DWBusinessException {
        Criteria criteria = Criteria.where(ComponentManager.APP_CODE).is(appCode)
                .and(ComponentManager.PARADIGM).in(paradigmList)
                .and(ComponentManager.TYPE).is(ComponentStructuredListType.Mechanism);
        List<ComponentStructuredList> result = dataPickService.find(criteria, ComponentStructuredList.class, "componentStructuredList");
        if(CollectionUtils.isEmpty(result)){
            return new ArrayList<>();
        }else{
            return result.stream().map(ComponentStructuredList::getMechanismCode).distinct().collect(Collectors.toList());
        }

    }

    public List<String> getMechanismByParadigmCode(String ParadigmCode) throws DWBusinessException {
        Criteria criteria = Criteria.where(ComponentManager.PARADIGM).is(ParadigmCode)
                .and(ComponentManager.TYPE).is(ComponentStructuredListType.Mechanism);
        List<ComponentStructuredList> result = dataPickService.find(criteria, ComponentStructuredList.class, "componentStructuredList");
        if (CollectionUtils.isEmpty(result)) {
            return new ArrayList<>();
        } else {
            return result.stream().map(ComponentStructuredList::getMechanismCode).distinct().collect(Collectors.toList());
        }

    }


    private MechanismTenantParadigm getMechanismTenantParadigm(String appCode, String tenantId){
        Query query = new Query();
        Criteria criteria = new Criteria().andOperator(Criteria.where("code").is(appCode),
                Criteria.where("tenantId").is(tenantId));
        query.addCriteria(criteria);
        MechanismTenantParadigm mechanismTenantParadigm = this.mongoTemplateUser.findOne(query, MechanismTenantParadigm.class, "mechanismTenantParadigm");
        return mechanismTenantParadigm;
    }

    private List<String> matchParadigmByName( Map<String, List<Paradigm>> appAndParadigmMap, List<String> toMatchNameList, String appCode){

        List<Paradigm> paradigms = appAndParadigmMap.get(appCode);
        if(CollectionUtils.isEmpty(paradigms)){
            return null;
        }

        List<String> result = new ArrayList<>();
        for(String name : toMatchNameList){
            for(Paradigm paradigm : paradigms){
                Map<String, Map<String, String>> lang = paradigm.getLang();
                if(lang == null || lang.get("name") == null){
                    if(StringUtils.equals(paradigm.getName(), name)){
                        result.add(paradigm.getCode());
                    }
                }else{
                    Map<String, String> nameLang = lang.get("name");
                        String zh_tw = nameLang.get("zh_TW");
                        String zh_cn = nameLang.get("zh_CN");
                        if(StringUtils.equals(zh_tw, name) || StringUtils.equals(zh_cn, name)){
                            result.add(paradigm.getCode());
                        }
                }
            }
        }

        return result;

//        if(result.size() == toMatchNameList.size()){
//            return result;
//        }else{
//            return null;
//        }
    }

    /**
     * 格式为：  ****_****(范式1/范式2)_****
     * @param asf06
     * @return
     */
    private List<String> getParadigmFromAFS06(String asf06){
        if(StringUtils.isEmpty(asf06)){
            return null;
        }
        String[] s = StringUtils.split(asf06, "_");
        if(s.length != 3){
            return null;
        }
        String s1 = s[1];
        if(StringUtils.contains(s1, "(") && StringUtils.contains(s1, ")")){
            String paradigmInfo = s1.substring(s1.indexOf("(")+1, s1.indexOf(")"));
            String[] split = StringUtils.split(paradigmInfo, "/");
            return Arrays.asList(split);
        }

        return null;
    }

    public static void main(String[] args) {
        ParadigmService paradigmService = new ParadigmService();
        List<String> paradigmFromAFS06 = paradigmService.getParadigmFromAFS06("****_****(范式1/范式2)_****");
        System.out.println(paradigmFromAFS06);
    }


    @Override
    public HandleParadigmListVo postHandleParadigmList(String appCode, List<String> mechanismCodes,String tenantId) {

        HandleParadigmListVo handleParadigmListVo = new HandleParadigmListVo();

        try {
            //查询租户和应用下范式和机制的权限数据返回机制code集合
            List<String> checkedList  = (List<String>) iMechanismService.getMechanismTenantParadigm(appCode, tenantId);
            handleParadigmListVo.setCheckedList(checkedList);

            //查询应用code下的范式集合数据
            List<ApplicationRelation> paradigmByTypeAndAppCode = iAppService.getParadigmByTypeAndAppCode(ComponentConstants.PARADIGM, appCode);

            if(CollectionUtils.isNotEmpty(paradigmByTypeAndAppCode)){
                //拿到范式code集合
                List<String> paradigmCodeList = paradigmByTypeAndAppCode.stream().map(ApplicationRelation::getCode).collect(Collectors.toList());

                //根据范式集合code查询范式集合数据
                List<Paradigm> paradigmList = iMechanismService.postParadigmNameAndCode(paradigmCodeList);
                handleParadigmListVo.setParadigmList(paradigmList);
            }

            //根据机制code集合 去组件清单表查询对应的范式code，分组逻辑：key机制code，value：范式code
            Map<String, String> mechanismParadigmMap = iMechanismService.postParadigmCode(mechanismCodes);
            handleParadigmListVo.setMechanismParadigmMap(mechanismParadigmMap);

        } catch (Exception e) {
            log.error("Route==>ParadigmService method=getHandleParadigmList error:",e);
        }

        return handleParadigmListVo;
    }


    @Override
    public List<Map> postHandleParadigmShare(List<Map> list) throws DWException {
        String version = kgInnerService.getTenantVersion(AthenaUtils.getTenantId());
        list.forEach(x->{

            //范式配置  share：共享范式 independence：独立范式
            x.put("paradigmConfig", ComponentConstants.PARADIGM_CONFIG_INDEPENDENCE);
            //范式code
            String code = x.get("code") + "";

            Query query = new Query();
            Criteria criteria = Criteria.where(ComponentConstants.PARADIGM).in(code)
                    .and(ComponentConstants.VERSION).is(version);
            List<ComponentStructuredList> result = null;
            try {
                result = dataPickService.find(criteria, ComponentStructuredList.class, "componentStructuredList");
            } catch (DWBusinessException e) {
                log.error("查询共享范式组件清单异常:",e);
            }

            if(CollectionUtils.isNotEmpty(result)){
                //对appCode进行去重，拿到集合数据
                List<String> appCodeList = result.stream().map(ComponentStructuredList::getAppCode).distinct().collect(Collectors.toList());

                //集合大于1 说明有多个应用，代码是共享范式,todo 该逻辑主要是兼容扩展信息范式的数据，后面扩展信息范式删除之后，该逻辑可以删除
                if(appCodeList.size()>1){
                    x.put("paradigmConfig", ComponentConstants.PARADIGM_CONFIG_SHARE);
                //如果只有一条数据，去范式表查询范式的application=commonParadigm，说明是共享范式
                } else if (appCodeList.size()==1) {
                    Paradigm paradigm = mongoTemplateSystem.findOne(Query.query(Criteria.where("code").is(
                            code).and("version").is(version)), Paradigm.class);
                    if(Objects.nonNull(paradigm) && ComponentConstants.PARADIGM_COMMON_APPLICATION.equals(paradigm.getApplication())){
                        x.put("paradigmConfig", ComponentConstants.PARADIGM_CONFIG_SHARE);
                    }
                }
            }
        });

        return list;
    }
}
