package com.digiwin.athena.knowledgegraph.service.impl;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.serializer.SerializerFeature;
import com.digiwin.app.container.exceptions.DWBusinessException;
import com.digiwin.app.service.DWServiceContext;
import com.digiwin.athena.kmservice.locale.Lang;
import com.digiwin.athena.knowledgegraph.clients.CacService;
import com.digiwin.athena.domain.core.app.ApplicationRelation;
import com.digiwin.athena.knowledgegraph.domain.targets.*;
import com.digiwin.athena.knowledgegraph.domain.topic.Topic;
import com.digiwin.athena.knowledgegraph.dto.*;
import com.digiwin.athena.knowledgegraph.service.ITargetService;
import com.digiwin.athena.knowledgegraph.utils.AthenaUtils;
import com.digiwin.athena.knowledgegraph.clients.ESPUtils;
import com.digiwin.athena.knowledgegraph.utils.I18nUtils;
import com.digiwin.athena.knowledgegraph.utils.LanguageUtil;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.data.mongodb.core.MongoTemplate;
import org.springframework.data.mongodb.core.query.Criteria;
import org.springframework.data.mongodb.core.query.Query;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import java.util.*;
import java.util.function.Function;
import java.util.stream.Collectors;

import static com.digiwin.athena.knowledgegraph.utils.LanguageConst.*;

@Lang
@Service
@Slf4j
public class TargetService implements ITargetService {

    @Autowired
    @Qualifier("knowledgegraphSystem")
    MongoTemplate mongoTemplate;
    @Autowired
    @Qualifier("knowledgegraphTenant")
    MongoTemplate mongoTemplateUser;
    @Autowired
    CacService cacService;
    @Autowired
    AppService appService;
    @Autowired
    TopicService topicService;
    @Autowired
    ParadigmService paradigmService;
    @Autowired
    ESPUtils espUtils;

    @Override
    public Object getForest() throws DWBusinessException {
        //获取租户下配置的所有指标树
        String tenantId = AthenaUtils.getTenantId();
        Query query = new Query();
        query.addCriteria(Criteria.where("tenantId").is(tenantId));
        List<TargetTree> targetForest = mongoTemplateUser.find(query, TargetTree.class);
        if (!CollectionUtils.isEmpty(targetForest)) {
            //查找跟节点指标实体并赋值
            List<String> rootTargetIds = targetForest.stream().map(t -> t.getRoot().getId()).distinct().collect(Collectors.toList());
            Query queryTargets = new Query();
            queryTargets.addCriteria(Criteria.where("id").in(rootTargetIds).and("tenantId").is(tenantId));
            List<Target> targetRootList = mongoTemplateUser.find(queryTargets, Target.class);
            renderTargetToForest(targetForest, targetRootList);
        }
        return targetForest;
    }

    private void renderTargetToForest(List<TargetTree> targetForest, List<Target> targetRootList) {
        //将指标明细值赋值到树的跟节点
        String userLocale = AthenaUtils.getCurrentLocale();
        Map<String, Target> targetMap = targetRootList.stream().collect(Collectors.toMap(Target::getId, Function.identity()));
        targetForest.forEach(tree -> {
            String localeName = targetMap.get(tree.getRoot().getId()).getLang().get("name").get(userLocale);
            tree.getRoot().setName(StringUtils.isEmpty(localeName) ? targetMap.get(tree.getRoot().getId()).getName() : localeName);
            tree.getRoot().setEnabled(targetMap.get(tree.getRoot().getId()).getEnabled());
        });
    }

    @Override
    public Object getMechanismTargets(String mechanismId) throws DWBusinessException {
        String tenantId = AthenaUtils.getTenantId();
        Query query = new Query();
        query.addCriteria(Criteria.where("mechanismCode").is(mechanismId).and("tenantId").is(tenantId));
        List<MechanismTargetRelation> mechanismTargetRelations = mongoTemplateUser.find(query, MechanismTargetRelation.class);
        List<String> targetIds = mechanismTargetRelations.stream().map(MechanismTargetRelation::getTargetId).collect(Collectors.toList());
        if (CollectionUtils.isEmpty(targetIds)) {
            return Collections.EMPTY_LIST;
        } else {
            Query queryTargets = new Query();
            queryTargets.addCriteria(Criteria.where("id").in(targetIds).and("tenantId").is(tenantId));
            return mongoTemplateUser.find(queryTargets, Target.class);
        }
    }

    @Override
    public Object postBuildTree(TargetTreeForAppDto targetTree) throws Exception {
        //转换为本地指标树和指标结构保存到本地
        List<Target> targets = transformToTargets(targetTree);
        saveTargetsToLocal(targets);
        TargetTree localTargetTree = buildTargetTreeByAppTargets(targetTree);

        //转换为tbb指标树和指标请求结构,调用tbb接口
        String token = DWServiceContext.getContext().getToken();
        //保存指标值到tbb
        postTBBTargets(targetTree.getTenantId(), targets, token);
        //保存指标树到tbb
        //根据根节点是否为空决定是否是多颗指标树
        if (isEmptyRoot(localTargetTree)) {
            List<TargetTree> targetForest = buildForestByNoRootTargetTree(localTargetTree);
            //新建更新
            upsert(targetForest.get(0));
            //循环也只有一棵树
            for (TargetTree item : targetForest) {
                Map<String, Object> targetTreeResponse = postTBBTargetTree(TBBTargetTreeOperationConst.UPSERT, item, token);
            }
        } else {
            upsert(localTargetTree);
            Map<String, Object> targetTreeResponse = postTBBTargetTree(TBBTargetTreeOperationConst.UPSERT, localTargetTree, token);
        }
        return null;
    }

    private void upsert(TargetTree targetTree) throws DWBusinessException {
        Query query = new Query();
        query.addCriteria(Criteria.where("treeId").is(targetTree.getTreeId()).and("tenantId").is(targetTree.getTenantId()));
        TargetTree targetTreeInDB = mongoTemplateUser.findOne(query, TargetTree.class);
        if (StringUtils.isEmpty(targetTree.getTreeId())) {
            //create
            targetTree.setTreeId(UUID.randomUUID().toString());
            mongoTemplateUser.save(targetTree);
        } else if (null != targetTreeInDB) {
            //update
            mongoTemplateUser.save(targetTree);
        } else {
            //更新的树在库中不存在,抛异常
            throw new DWBusinessException("tree not found");
        }
    }

    private List<TargetTree> buildForestByNoRootTargetTree(TargetTree targetTree) {
        List<TargetNode> newRoots = targetTree.getRoot().getChildren();
        List<TargetTree> forest = new LinkedList<>();
        newRoots.forEach(item -> {
            TargetTree newTargetTree = TargetTree.builder()
                    .tenantId(targetTree.getTenantId())
                    .root(item)
                    .treeName(targetTree.getTreeName())
                    //只有一个树根,暂不支持多棵树
                    .treeId(targetTree.getTreeId()).build();
            forest.add(newTargetTree);
        });
        return forest;
    }

    private boolean isEmptyRoot(TargetTree targetTree) {
        return StringUtils.isEmpty(targetTree.getRoot().getId());
    }

    private Map<String, Object> postTBBTargets(String tenantId,List<Target> targets, String token) throws Exception {
        TBBTargetsRequest tbbTargetRequest = transformToTBBTargets(targets);
        tbbTargetRequest.setTenantId(tenantId);
        return espUtils.execute(token, tenantId, "indicator.import", tbbTargetRequest);
    }

    public static void main(String[] args) {
        TBBTarget tbbTarget = new TBBTarget();
        System.out.println(JSON.toJSONString(tbbTarget, SerializerFeature.WriteMapNullValue));
    }

    private Map<String, Object> postTBBTargetTree(Integer operation, TargetTree targetTree, String token) throws Exception {
        TBBTargetTree tbbTargetTreeRequest = transformToTBBTargetTree(operation, targetTree);
        return espUtils.execute(token, targetTree.getTenantId(), "indicator.tree.import", tbbTargetTreeRequest);
    }

    private TBBTargetTree transformToTBBTargetTree(Integer operation, TargetTree targetTree) {
        List<TBBTreeNode> rootChildren = transformToTBBTreeNode(targetTree.getRoot().getChildren());
        TBBTargetTree tbbTargetTree = TBBTargetTree.builder()
                .treeId(targetTree.getTreeId())
                .tenantId(targetTree.getTenantId())
                .operation(operation)
                .IndicatorKey(targetTree.getRoot().getId())
                .nodeName(targetTree.getTreeName())
                .direction("vertical")
                .formatter(new TBBTreeNodeFormatter())
                .children(rootChildren).build();
        return tbbTargetTree;
    }

    private List<TBBTreeNode> transformToTBBTreeNode(List<TargetNode> targetNodes) {
        List<TBBTreeNode> tbbTreeNodes = new LinkedList<>();
        if (!CollectionUtils.isEmpty(targetNodes)) {
            targetNodes.forEach(targetNode -> {
                TBBTreeNode tbbTreeNode = new TBBTreeNode();
                tbbTreeNode.setNodeName(targetNode.getName());
                tbbTreeNode.setDirection("vertical");
                tbbTreeNode.setIndicatorKey(targetNode.getId());
                tbbTreeNode.setChildren(transformToTBBTreeNode(targetNode.getChildren()));
                tbbTreeNodes.add(tbbTreeNode);
            });
        }
        return tbbTreeNodes;
    }

    private TBBTargetsRequest transformToTBBTargets(List<Target> targets) throws DWBusinessException {
        TBBTargetsRequest tbbTargetsRequest = new TBBTargetsRequest();
        List<TBBTarget> tbbTargets = new LinkedList<>();
        for (Target target : targets) {
            Map<String, Map<String, String>> langMap = target.getLang();
            if (CollectionUtils.isEmpty(langMap)) {
                throw new DWBusinessException(I18nUtils.getValue("lang.required"));
            }
            //构建基础target
            TBBTarget tbbTarget = TBBTarget.builder()
                    .IndicatorKey(target.getId())
                    .Athena_lang("zh-cn")
                    .Period(target.getPeriodKey())
                    .IndicatorName(target.getName())
                    //.IndicatorId("TODO")
                    .IndicatorValue(Double.valueOf(StringUtils.isEmpty(target.getValue()) ? "0" : target.getValue()))
                    .TrendType(Integer.valueOf(StringUtils.isEmpty(target.getTendency()) ? "0" : target.getTendency()))
                    .Unit(target.getUnit())
                    .PeriodKind(target.getPeriod())
                    .Cycle(target.getPeriod())
                    .System(null)
                    .HighAlert(Double.valueOf(StringUtils.isEmpty(target.getTarget()) ? "0" : target.getTarget()))
                    .LowAlert(Double.valueOf(StringUtils.isEmpty(target.getWarningStage()) ? "0" : target.getWarningStage()))
                    .TargetValue(Double.valueOf(StringUtils.isEmpty(target.getTarget()) ? "0" : target.getTarget()))
                    .MakeBy(CollectionUtils.isEmpty(target.getPersonInCharge()) ? "" : target.getPersonInCharge().get(0))
                    .build();
            //构建target多语言
            Map<String, String> nameLang = langMap.get("name");
            if (CollectionUtils.isEmpty(nameLang)) {
                throw new DWBusinessException(I18nUtils.getValue("lang.required"));
            }
            for (Map.Entry<String, String> entry : nameLang.entrySet()) {
                String k = entry.getKey();
                String v = entry.getValue();
                TBBTarget langTarget = new TBBTarget();
                switch (k) {
                    case zh_CN:
                        BeanUtils.copyProperties(tbbTarget, langTarget);
                        langTarget.setAthena_lang(TBB_zh_CN);
                        break;
                    case zh_TW:
                        BeanUtils.copyProperties(tbbTarget, langTarget);
                        langTarget.setAthena_lang(TBB_zh_TW);
                        break;
                    case en_US:
                        BeanUtils.copyProperties(tbbTarget, langTarget);
                        langTarget.setAthena_lang(TBB_en_US);
                        break;
                    default:
                        throw new DWBusinessException(I18nUtils.getValue("unsupported.lang"));
                }
                langTarget.setIndicatorName(v);
                tbbTargets.add(langTarget);
            }
        }
        tbbTargetsRequest.setIndicator_detail(tbbTargets);
        return tbbTargetsRequest;
    }

    private List<Target> transformToTargets(TargetTreeForAppDto targetTreeForAppDto) {
        List<TargetForAppNodeInfoDto> nodeInfos = targetTreeForAppDto.getNodeInfos();
        List<Target> targets = new LinkedList<>();
        if (!CollectionUtils.isEmpty(nodeInfos)) {
            nodeInfos.forEach(nodeInfo -> {
                Target target = Target.builder()
                        .tenantId(targetTreeForAppDto.getTenantId())
                        .id(nodeInfo.getId())
                        .name(nodeInfo.getName())
                        .tendency(nodeInfo.getTendency())
                        .period(nodeInfo.getPeriod())
                        .periodKey(nodeInfo.getPeriodKey())
                        .unit(nodeInfo.getUnit())
                        .precision(nodeInfo.getPrecision())
                        .description(nodeInfo.getDescription())
                        .target(nodeInfo.getTarget())
                        .warningStage(nodeInfo.getWarningStage())
                        .personInCharge(nodeInfo.getPersonInCharge())
                        .enabled(nodeInfo.getEnabled())
                        .code(nodeInfo.getCode())
                        .type(nodeInfo.getType())
                        .value(nodeInfo.getValue())
                        .build();
                target.setLang(nodeInfo.getLang());
                targets.add(target);
            });
        }
        return targets;
    }

    private void saveTargetsToLocal(List<Target> targets) {
        //TODO 是否允许新增指标?
        targets.forEach(target -> {
            Query query = new Query();
            query.addCriteria(Criteria.where("tenantId").is(target.getTenantId())
                    .and("id").is(target.getId()));
                    //periodKey不作为指标树结构的判定字段
                    //.and("periodKey").is(target.getPeriodKey()));
            Target targetInDB = mongoTemplateUser.findOne(query, Target.class);
            if (null == targetInDB) {
                //新增
                mongoTemplateUser.insert(target);
            } else {
                //更新
                String objectId = targetInDB.getObjectId();
                BeanUtils.copyProperties(target, targetInDB);
                targetInDB.setObjectId(objectId);
                mongoTemplateUser.save(targetInDB);
            }
        });
    }

    private TargetTree buildTargetTreeByAppTargets(TargetTreeForAppDto targetTreeForAppDto) throws DWBusinessException {
        String tenantId = targetTreeForAppDto.getTenantId();
        TargetTree targetTree = TargetTree.builder()
                .treeId(targetTreeForAppDto.getTreeId())
                .treeName(targetTreeForAppDto.getTreeName())
                .tenantId(targetTreeForAppDto.getTenantId())
                .build();
        //多语言字段名应一致,否则要做转换
        targetTree.setLang(targetTreeForAppDto.getLang());
        List<TargetForAppNodeInfoDto> nodeInfos = targetTreeForAppDto.getNodeInfos();
        //根据层次构建指标树
        List<TargetForAppNodeInfoDto> l0Dto = nodeInfos.stream().filter(
                item -> item.getType().equals(TargetTypeConst.TOPIC)
                        && StringUtils.isEmpty(item.getParentNodeId())).collect(Collectors.toList());
        List<TargetForAppNodeInfoDto> l1Dto = nodeInfos.stream().filter(
                item -> item.getType().equals(TargetTypeConst.APPLICATION)
                        && StringUtils.isEmpty(item.getParentNodeId())).collect(Collectors.toList());
        List<TargetForAppNodeInfoDto> l2Dto = nodeInfos.stream().filter(
                item -> item.getType().equals(TargetTypeConst.PARADIGM)
                        && StringUtils.isEmpty(item.getParentNodeId())).collect(Collectors.toList());
        List<TargetForAppNodeInfoDto> l3Dto = nodeInfos.stream().filter(
                item -> item.getType().equals(TargetTypeConst.MECHANISM)
                        && StringUtils.isEmpty(item.getParentNodeId())).collect(Collectors.toList());
        List<TargetForAppNodeInfoDto> l4Dto = nodeInfos.stream().filter(
                item -> item.getType().equals(TargetTypeConst.BUSINESS) || item.getType().equals(TargetTypeConst.USER_TRACK)).collect(Collectors.toList());
        List<TargetForAppNodeInfoDto> customizedDto = nodeInfos.stream().filter(
                item -> !item.getType().equals(TargetTypeConst.BUSINESS) && !item.getType().equals(TargetTypeConst.USER_TRACK) && !StringUtils.isEmpty(item.getParentNodeId())).collect(Collectors.toList());
        Map<String, List<TargetForAppNodeInfoDto>> levelDtoMap = new HashMap<>();
        levelDtoMap.put(TreeLevel.LEVEL0, l0Dto);
        levelDtoMap.put(TreeLevel.LEVEL1, l1Dto);
        levelDtoMap.put(TreeLevel.LEVEL2, l2Dto);
        levelDtoMap.put(TreeLevel.LEVEL3, l3Dto);
        levelDtoMap.put(TreeLevel.LEVEL4, l4Dto);
        levelDtoMap.put(TreeLevel.CUSTOMIZED, customizedDto);
        Map<String, List<TargetForAppNodeInfoDto>> groupedChildrenParentMap = customizedDto.stream().collect(Collectors.groupingBy(TargetForAppNodeInfoDto::getParentNodeId));
        //需求变更,议题可以不指定
        /*if (l0Dto.size() != 1) {
            throw new DWBusinessException(I18nUtils.getValue("target.nol1"));
        }*/
        //如未指定l0层,则按应用进行构建指标树,为了保持逻辑的一致,虚拟一个空的l0层议题
        Topic currentTopic;
        if (CollectionUtils.isEmpty(l0Dto)) {
            currentTopic = new Topic();
        } else {
            //获取租户所有topic
            Set<Topic> topicSet = getTopicsByTenantId(targetTreeForAppDto.getTenantId());
            if (CollectionUtils.isEmpty(topicSet)) {
                throw new DWBusinessException(I18nUtils.getValue("target.noTopic"));
            }
            currentTopic = topicSet.stream().filter(item -> item.getCode().equals(l0Dto.get(0).getCode())).collect(Collectors.toList()).get(0);
            if (null == currentTopic) {
                throw new DWBusinessException(I18nUtils.getValue("target.noCurrentTopic"));
            }
        }
        String topicCode = currentTopic.getCode();

        //向下构建l1
        Set<String> currentTopicAppCode;
        if (CollectionUtils.isEmpty(l0Dto)) {
            if (l1Dto.size() != 1) {
                throw new DWBusinessException("仅支持一个应用节点作为根");
            }
            currentTopicAppCode = new HashSet<>(l1Dto.stream().map(TargetForAppNodeInfoDto::getCode).collect(Collectors.toList()));
        } else {
            currentTopicAppCode = new HashSet<>(currentTopic.getRelatedApps());
        }
        //构建l0
        TargetNode root = TargetNode.builder().id(CollectionUtils.isEmpty(l0Dto) ? null : l0Dto.get(0).getId()).name(targetTreeForAppDto.getTreeName()).enabled(true).build();
        root.setLang(CollectionUtils.isEmpty(l0Dto) ? null : l0Dto.get(0).getLang());
        root.setNodeId(CollectionUtils.isEmpty(l0Dto) ? null : l0Dto.get(0).getNodeId());
        root.setName(CollectionUtils.isEmpty(l0Dto) ? null : l0Dto.get(0).getName());

        targetTree.setRoot(root);
        buildDown(root, TreeLevel.LEVEL1, levelDtoMap, currentTopicAppCode, tenantId, groupedChildrenParentMap);
        if (!CollectionUtils.isEmpty(groupedChildrenParentMap.get(root.getNodeId()))) {
            //向下构建自定义树
            buildDown(root, TreeLevel.CUSTOMIZED, levelDtoMap, null, tenantId, groupedChildrenParentMap);
        }

        /*if (CollectionUtils.isEmpty(currentTopicAppCode)) {
            throw new DWBusinessException(I18nUtils.getValue("target.noAppOnCurrentTopic"));
        }
        for (TargetForAppNodeInfoDto dto : l1Dto) {
            //根据应用list查询范式列表,组装应用和范式关系map
            Set<String> paradigmSet = new HashSet<>();
            //只有当前议题包含的应用才会被加入指标树
            if (currentTopicAppCode.contains(dto.getCode())) {
                //构建l1层(应用层)
                TargetNode l1TargetNode = TargetNode.builder().id(dto.getCode()).enabled(true).build();
                root.addChildren(l1TargetNode);
                //向下构建l2↓
                //应用机制关系
                List<ApplicationRelation> appParadigmRelations = appService.getApplicationRelationByTypeAndAppCode(tenantId, "paradigm", dto.getCode());
                if (!CollectionUtils.isEmpty(appParadigmRelations)) {
                    List<String> paradigmList = appParadigmRelations.stream().map(ApplicationRelation::getCode).collect(Collectors.toList());
                    paradigmSet.addAll(paradigmList);
                }
                for (TargetForAppNodeInfoDto paradigm : l2Dto) {//依次校验是否存在于购买的应用中
                    if (!paradigmSet.contains(paradigm.getCode())) {
                        //log.error(I18nUtils.getValue("target.noParadigmInL2") + ":" + paradigm.getCode());
                        continue;
                    }
                    //构建l2层
                    TargetNode l2TargetNode = TargetNode.builder().id(dto.getCode()).enabled(true).build();
                    l1TargetNode.addChildren(l2TargetNode);
                    //向下构建l3↓
                    //查询范式包含的机制
                    List<String> paradigmMechanisms = paradigmService.getMechanismByParadigmCode(tenantId, dto.getCode());
                    //当前范式下是否包含该机制
                    for (TargetForAppNodeInfoDto mechanismDto : l3Dto) {
                        if (!paradigmMechanisms.contains(mechanismDto.getCode())) {
                            //log.error(I18nUtils.getValue("target.noMechanismInL3") + ":" + paradigm.getCode());
                            continue;
                        }
                        //将当前机制挂到当前范式
                        TargetNode l3TargetNode = TargetNode.builder().id(mechanismDto.getCode()).enabled(true).build();
                        l2TargetNode.addChildren(l3TargetNode);
                        //向下构建l4↓
                        for (TargetForAppNodeInfoDto customDto : l4Dto) {
                            if (customDto.getParentNodeId().equals(mechanismDto.getNodeId())) {
                                TargetNode l4TargetNode = TargetNode.builder().id(customDto.getCode()).enabled(true).build();
                                l3TargetNode.addChildren(l4TargetNode);
                            }
                        }
                    }
                }
            }
        }*/
        return targetTree;
    }

    private void buildDown(TargetNode targetNode, String level, Map<String, List<TargetForAppNodeInfoDto>> levelDtoMap, Set<String> nextRelatedCodes, String tenantId, Map<String, List<TargetForAppNodeInfoDto>> groupedChildrenParentMap) throws DWBusinessException {
        switch (level) {
            case TreeLevel.LEVEL1:
                if (null == nextRelatedCodes || nextRelatedCodes.isEmpty()) {
                    throw new DWBusinessException(I18nUtils.getValue("target.noAppOnCurrentTopic"));
                }
                List<String> tenantApp = cacService.getAuthorizationsApplication(tenantId);
                List<TargetForAppNodeInfoDto> l1Dtos = levelDtoMap.get(TreeLevel.LEVEL1);
                for (TargetForAppNodeInfoDto dto : l1Dtos) {
                    if (!tenantApp.contains(dto.getCode())) {
                        throw new DWBusinessException(I18nUtils.getValue("app.expired"));
                    }
                    //根据应用list查询范式列表,组装应用和范式关系map
                    Set<String> paradigmSet = new HashSet<>();
                    //应用机制关系
                    List<ApplicationRelation> appParadigmRelations = appService.getApplicationRelationByTypeAndAppCode(tenantId, "paradigm", dto.getCode());
                    if (!CollectionUtils.isEmpty(appParadigmRelations)) {
                        List<String> paradigmList = appParadigmRelations.stream().map(ApplicationRelation::getCode).collect(Collectors.toList());
                        paradigmSet.addAll(paradigmList);
                    }
                    //只有当前议题包含的应用才会被加入指标树
                    if (nextRelatedCodes.contains(dto.getCode())) {
                        //构建l1层(应用层)
                        TargetNode l1TargetNode = TargetNode.builder().id(dto.getId()).name(dto.getName()).enabled(true).build();
                        l1TargetNode.setLang(dto.getLang());
                        targetNode.addChildren(l1TargetNode);
                        //向下构建l2
                        buildDown(l1TargetNode, TreeLevel.LEVEL2, levelDtoMap, paradigmSet, tenantId, groupedChildrenParentMap);
                        if (!CollectionUtils.isEmpty(groupedChildrenParentMap.get(dto.getNodeId()))) {
                            //下层依赖该nodeId
                            l1TargetNode.setNodeId(dto.getNodeId());
                            //向下构建自定义树
                            buildDown(l1TargetNode, TreeLevel.CUSTOMIZED, levelDtoMap, null, tenantId, groupedChildrenParentMap);
                        }
                    }

                }
                break;
            case TreeLevel.LEVEL2:
                List<TargetForAppNodeInfoDto> l2Dtos = levelDtoMap.get(TreeLevel.LEVEL2);
                for (TargetForAppNodeInfoDto paradigm : l2Dtos) {//依次校验是否存在于购买的应用中
                    if (nextRelatedCodes == null || !nextRelatedCodes.contains(paradigm.getCode())) {
                        continue;
                    }
                    //查询范式包含的机制
                    Set<String> paradigmMechanisms = new HashSet(paradigmService.getMechanismByParadigmCode(paradigm.getCode()));
                    //构建l2层
                    TargetNode l2TargetNode = TargetNode.builder().id(paradigm.getId()).name(paradigm.getName()).enabled(true).build();
                    l2TargetNode.setLang(paradigm.getLang());
                    targetNode.addChildren(l2TargetNode);
                    buildDown(l2TargetNode, TreeLevel.LEVEL3, levelDtoMap, paradigmMechanisms, tenantId, groupedChildrenParentMap);
                    if (!CollectionUtils.isEmpty(groupedChildrenParentMap.get(paradigm.getNodeId()))) {
                        //下层依赖该nodeId
                        l2TargetNode.setNodeId(paradigm.getNodeId());
                        //向下构建自定义树
                        buildDown(l2TargetNode, TreeLevel.CUSTOMIZED, levelDtoMap, null, tenantId, groupedChildrenParentMap);
                    }
                }
                break;
            case TreeLevel.LEVEL3:
                List<TargetForAppNodeInfoDto> l3Dtos = levelDtoMap.get(TreeLevel.LEVEL3);
                for (TargetForAppNodeInfoDto mechanismDto : l3Dtos) {
                    if (nextRelatedCodes == null || !nextRelatedCodes.contains(mechanismDto.getCode())) {
                        continue;
                    }
                    //将当前机制挂到当前范式
                    TargetNode l3TargetNode = TargetNode.builder().id(mechanismDto.getId()).name(mechanismDto.getName()).enabled(true).build();
                    l3TargetNode.setLang(mechanismDto.getLang());
                    targetNode.addChildren(l3TargetNode);
                    Set singleEleSet = new HashSet();
                    singleEleSet.add(mechanismDto.getNodeId());
                    buildDown(l3TargetNode, TreeLevel.LEVEL4, levelDtoMap, singleEleSet, tenantId, groupedChildrenParentMap);
                    if (!CollectionUtils.isEmpty(groupedChildrenParentMap.get(mechanismDto.getNodeId()))) {
                        //下层依赖该nodeId
                        l3TargetNode.setNodeId(mechanismDto.getNodeId());
                        //向下构建自定义树
                        buildDown(l3TargetNode, TreeLevel.CUSTOMIZED, levelDtoMap, null, tenantId, groupedChildrenParentMap);
                    }
                }
                break;
            case TreeLevel.LEVEL4:
                List<TargetForAppNodeInfoDto> l4Dtos = levelDtoMap.get(TreeLevel.LEVEL4);
                String parentNodeId = "";
                if (nextRelatedCodes == null) {
                    break;
                }
                for (String nextRelatedCode : nextRelatedCodes) {
                    parentNodeId = nextRelatedCode;
                }
                for (TargetForAppNodeInfoDto customDto : l4Dtos) {
                    if (customDto.getParentNodeId().equals(parentNodeId)) {
                        TargetNode l4TargetNode = TargetNode.builder().id(customDto.getId()).name(customDto.getName()).enabled(true).build();
                        l4TargetNode.setLang(customDto.getLang());
                        targetNode.addChildren(l4TargetNode);
                    }
                }
                break;
            case TreeLevel.CUSTOMIZED:
                List<TargetForAppNodeInfoDto> childrenDtos = groupedChildrenParentMap.get(targetNode.getNodeId());
                if (!CollectionUtils.isEmpty(childrenDtos)) {
                    for (TargetForAppNodeInfoDto childrenDto : childrenDtos) {
                        TargetNode childTargetNode = TargetNode.builder()
                                .id(childrenDto.getId())
                                .nodeId(childrenDto.getNodeId())
                                .name(childrenDto.getName())
                                .enabled(true).build();
                        targetNode.setLang(childrenDto.getLang());
                        targetNode.addChildren(childTargetNode);
                        buildDown(childTargetNode, TreeLevel.CUSTOMIZED, levelDtoMap, null, tenantId, groupedChildrenParentMap);
                    }
                }
            default:
                break;
        }

    }

    private Set<Topic> getTopicsByTenantId(String tenantId) throws DWBusinessException {
        //根据tenantId获取租户购买的应用
        //通过cac获取授权应用,包含非athena应用
        List<String> tenantApp = cacService.getAuthorizationsApplication(tenantId);
        //根据athena应用进行过滤
        List<String> allAthenaApp = appService.getAllAppCode();
        //根据athenaApp进行过滤
        tenantApp = tenantApp.stream().filter(item -> allAthenaApp.contains(item)).collect(Collectors.toList());
        //根据应用集合获取所有topic
        List<Topic> allTopics = (List<Topic>) topicService.getAllTopics();
        List<String> finalTenantApp = tenantApp;
        Set<Topic> tenantAllTopics = allTopics.stream().filter(item -> !Collections.disjoint(item.getRelatedApps(), finalTenantApp)).collect(Collectors.toSet());
        return tenantAllTopics;
    }

    @Override
    public Object postAppendLeaf(AppendLeafDto appendLeafDto) throws Exception {
        String token = DWServiceContext.getContext().getToken();
        String rootId = appendLeafDto.getRootId();
        String parentId = appendLeafDto.getParentId();
        List<String> targetId = appendLeafDto.getTargetId();
        String tenantId = AthenaUtils.getTenantId();
        Query query = new Query();
        query.addCriteria(Criteria.where("tenantId").is(tenantId).and("id").in(targetId));
        List<Target> targets = mongoTemplateUser.find(query, Target.class);
        //判断是否是根节点,如果parentId为空或null则为根节点新增树
        if (StringUtils.isEmpty(rootId)) {
            //根节点新增树
            List<TargetTree> targetTrees = buildTreesFromRoot(targets, tenantId);
            mongoTemplateUser.insertAll(targetTrees);
            for (TargetTree targetTree : targetTrees) {
                //同步指标树到TBB
                postTBBTargetTree(TBBTargetTreeOperationConst.UPSERT, targetTree, token);
            }
            return targetTrees;
        } else {
            //查找到当前树
            if (StringUtils.isEmpty(parentId)) {
                throw new DWBusinessException("缺少parentId参数");
            }
            //从根节点遍历找到对应节点
            Query queryTree = new Query();
            queryTree.addCriteria(Criteria.where("tenantId").is(tenantId).and("treeId").is(rootId));
            TargetTree targetTree = mongoTemplateUser.findOne(queryTree, TargetTree.class);
            //修改树结构
            targetTree = appendChildren(targetTree, parentId, targets);
            //将修改后的树重新保存到db
            mongoTemplateUser.save(targetTree);
            //同步指标树到TBB
            postTBBTargetTree(TBBTargetTreeOperationConst.UPSERT, targetTree, token);
            return null;
        }
    }

    private TargetTree appendChildren(TargetTree targetTree, String parentId, List<Target> targets) {
        String userLocale = AthenaUtils.getCurrentLocale();
        TargetNode parentNode = findNode(targetTree, parentId);
        targets.forEach(target -> {
            TargetNode targetNode = new TargetNode();
            targetNode.setId(target.getId());
            String localeName = target.getLang().get("name").get(userLocale);
            targetNode.setName(StringUtils.isEmpty(localeName) ? target.getName() : localeName);
            parentNode.addChildren(targetNode);
        });
        return targetTree;
    }

    private TargetNode findNode(TargetTree targetTree, String parentId) {
        Queue<TargetNode> queue = new LinkedList<>();
        queue.offer(targetTree.getRoot());
        while (!queue.isEmpty()) {
            TargetNode targetNode = queue.poll();
            if (targetNode.getId().equals(parentId)) {
                return targetNode;
            } else {
                if (!CollectionUtils.isEmpty(targetNode.getChildren())) {
                    for (TargetNode targetChild : targetNode.getChildren()) {
                        queue.offer(targetChild);
                    }
                }
            }
        }
        return null;
    }

    private List<TargetTree> buildTreesFromRoot(List<Target> targets, String tenantId) {
        List<TargetTree> targetTrees = new LinkedList<>();
        String userLocale = AthenaUtils.getCurrentLocale();
        targets.forEach(target -> {
            TargetTree targetTree = new TargetTree();
            targetTree.setTenantId(tenantId);
            targetTree.setTreeId(UUID.randomUUID().toString());
            TargetNode targetNode = new TargetNode();
            targetNode.setId(target.getId());
            String localeName = target.getLang().get("name").get(userLocale);
            targetNode.setName(StringUtils.isEmpty(localeName) ? target.getName() : localeName);
            targetTree.setRoot(targetNode);
            targetTrees.add(targetTree);
        });
        return targetTrees;
    }

    @Override
    public Object getTargetPool() throws DWBusinessException {
        String tenantId = AthenaUtils.getTenantId();
        String locale = AthenaUtils.getCurrentLocale();
        Query query = new Query();
        query.addCriteria(Criteria.where("tenantId").is(tenantId));
        List<Target> targetPool = mongoTemplateUser.find(query, Target.class);
        if (!CollectionUtils.isEmpty(targetPool)) {
            targetPool.forEach(item -> {
                try {
                    LanguageUtil.processLocaleLanguage(item, locale);
                } catch (Exception e) {
                    log.warn("多语言解析异常", e);
                }
            });
        }
        return targetPool;
    }

    @Override
    public Object getTargetTree(String rootId) throws DWBusinessException {
        String tenantId = AthenaUtils.getTenantId();
        String locale = AthenaUtils.getCurrentLocale();
        Query query = new Query();
        query.addCriteria(Criteria.where("tenantId").is(tenantId).and("treeId").is(rootId));
        TargetTree targetTree = mongoTemplateUser.findOne(query, TargetTree.class);
        //获取树下所有targetId集合
        Set<String> flatTargetIds = getFlatTargetIdFromTree(targetTree);
        Query queryTargets = new Query();
        queryTargets.addCriteria(Criteria.where("id").in(flatTargetIds).and("tenantId").is(tenantId));
        List<Target> treeTargets = mongoTemplateUser.find(queryTargets, Target.class);
        Map<String, Target> treeTargetMap = new HashMap<>();
        if (!CollectionUtils.isEmpty(treeTargets)) {
            treeTargetMap = treeTargets.stream().collect(Collectors.toMap(Target::getId, Function.identity()));
        }
        TargetTreeDto targetTreeDto = renderRichTargetTree(targetTree, treeTargetMap, locale);
        return targetTreeDto;
    }

    private TargetTreeDto renderRichTargetTree(TargetTree targetTree, Map<String, Target> treeTargetMap, String locale) {
        TargetTreeDto targetTreeDto = new TargetTreeDto();
        targetTreeDto.setTreeId(targetTree.getTreeId());
        targetTreeDto.setObjectId(targetTree.getObjectId());
        targetTreeDto.setTenantId(targetTree.getTenantId());
        TargetNodeDto targetNodeDto = new TargetNodeDto();
        targetTreeDto.setRoot(targetNodeDto);
        renderRecursively(targetNodeDto, targetTree.getRoot(), treeTargetMap, locale);
        return targetTreeDto;
    }

    private void renderRecursively(TargetNodeDto targetNodeDto, TargetNode node, Map<String, Target> treeTargetMap, String locale) {
        targetNodeDto.setId(node.getId());
        Target target = treeTargetMap.get(node.getId());
        if (null != target) {
            targetNodeDto.setObjectId(target.getObjectId());
            targetNodeDto.setTarget(target.getTarget());
            String localeDescription = target.getLang().getOrDefault("description", new HashMap<>()).getOrDefault(locale, target.getDescription());
            targetNodeDto.setDescription(localeDescription);
            targetNodeDto.setTenantId(target.getTenantId());
            targetNodeDto.setPeriod(target.getPeriod());
            targetNodeDto.setTendency(target.getTendency());
            targetNodeDto.setTendencyName(target.getTendencyName());
            String localeUnit = target.getLang().getOrDefault("unit", new HashMap<>()).getOrDefault(locale, target.getDescription());
            targetNodeDto.setUnit(localeUnit);
            targetNodeDto.setPrecision(target.getPrecision());
            targetNodeDto.setWarningStage(target.getWarningStage());
            targetNodeDto.setLang(target.getLang());
            String localeName = target.getLang().getOrDefault("name", new HashMap<>()).getOrDefault(locale, target.getDescription());
            targetNodeDto.setName(localeName);
            targetNodeDto.setPersonInCharge(target.getPersonInCharge());
            targetNodeDto.setEnabled(target.getEnabled());
        }
        List<TargetNode> children = node.getChildren();
        if (!CollectionUtils.isEmpty(children)) {
            Iterator<TargetNode> iterator = children.iterator();
            while (iterator.hasNext()) {
                TargetNode nodeChild = iterator.next();
                TargetNodeDto targetNodeDtoChild = new TargetNodeDto();
                targetNodeDto.addChildren(targetNodeDtoChild);
                renderRecursively(targetNodeDtoChild, nodeChild, treeTargetMap, locale);
            }
        }
    }

    private Set<String> getFlatTargetIdFromTree(TargetTree targetTree) {
        if (null == targetTree || null == targetTree.getRoot() || StringUtils.isEmpty(targetTree.getRoot().getId())) {
            return Collections.EMPTY_SET;
        }
        Set<String> flatTargetIds = new HashSet<>();
        Queue<TargetNode> queue = new LinkedList<>();
        queue.offer(targetTree.getRoot());
        while (!queue.isEmpty()) {
            TargetNode targetNode = queue.poll();
            flatTargetIds.add(targetNode.getId());
            if (!CollectionUtils.isEmpty(targetNode.getChildren())) {
                for (TargetNode targetChild : targetNode.getChildren()) {
                    queue.offer(targetChild);
                }
            }
        }
        return flatTargetIds;
    }

    private TargetTree getTree(String treeId, String tenantId) {
        Query query = new Query();
        query.addCriteria(Criteria.where("tenantId").is(tenantId).and("treeId").is(treeId));
        TargetTree targetTree = mongoTemplateUser.findOne(query, TargetTree.class);
        return targetTree;
    }

    @Override
    public Object postRemoveNode(RemoveNodeDto removeNodeDto) throws Exception {
        String token = DWServiceContext.getContext().getToken();
        String rootId = removeNodeDto.getRootId();
        String targetId = removeNodeDto.getTargetId();
        String tenantId = AthenaUtils.getTenantId();
        if (StringUtils.isEmpty(rootId) || StringUtils.isEmpty(targetId)) {
            throw new DWBusinessException("rootId和targetId不能为空");
        }
        TargetTree targetTree = getTree(rootId, tenantId);
        if (null == targetTree) {
            throw new DWBusinessException("要删除的树不存在");
        }
        //判断是否根节点
        if (isRoot(targetTree, targetId)) {
            mongoTemplateUser.remove(targetTree);
            //提交tbb
            postTBBTargetTree(TBBTargetTreeOperationConst.DELETE, targetTree, token);
        } else {
            removeFromTree(targetTree, targetId);
            mongoTemplateUser.save(targetTree);
            //提交tbb
            postTBBTargetTree(TBBTargetTreeOperationConst.UPSERT, targetTree, token);
        }
        return null;
    }

    private boolean isRoot(TargetTree targetTree, String targetId) {
        return targetTree.getRoot().getId().equals(targetId);
    }

    private void removeFromTree(TargetTree targetTree, String targetId) {
        TargetNode root = targetTree.getRoot();
        //遍历树,记录父节点和要找的节点,root一定不是要找的targetId,同样parent也不会是
        traverseAndRemove(root, targetId, Flag.getInstance(false));
    }

    private void traverseAndRemove(TargetNode parent, String targetId, Flag findFlag) {
        List<TargetNode> children = parent.getChildren();
        if (!CollectionUtils.isEmpty(children)) {
            Iterator<TargetNode> iterator = children.iterator();
            while (iterator.hasNext() && !findFlag.getFlag()) {
                TargetNode child = iterator.next();
                if (child.getId().equals(targetId)) {
                    //找到节点,停止遍历
                    iterator.remove();
                    findFlag.setFlag(true);
                } else {
                    //深度优先
                    traverseAndRemove(child, targetId, findFlag);
                }
            }
        }
    }

    @Override
    public Object getTarget(String targetId) throws DWBusinessException, NoSuchFieldException, IllegalAccessException {
        String tenantId = AthenaUtils.getTenantId();
        Query query = new Query();
        query.addCriteria(Criteria.where("tenantId").is(tenantId).and("id").is(targetId));
        Target target = mongoTemplateUser.findOne(query, Target.class);
        LanguageUtil.processLocaleLanguage(target, AthenaUtils.getCurrentLocale());
        return target;
    }

    @Override
    public Object postTarget(Target target) throws DWBusinessException {
        String tenantId = AthenaUtils.getTenantId();
        Query query = new Query();
        query.addCriteria(Criteria.where("tenantId").is(tenantId).and("id").is(target.getId()));
        Target targetInDB = mongoTemplateUser.findOne(query, Target.class);
        targetInDB.setTarget(target.getTarget());
        targetInDB.setWarningStage(target.getWarningStage());
        targetInDB.setPersonInCharge(target.getPersonInCharge());
        targetInDB.setEnabled(target.getEnabled());
        mongoTemplateUser.save(targetInDB);
        return null;
    }

}
