package com.digiwin.athena.knowledgegraph.service.impl;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.digiwin.app.container.exceptions.DWBusinessException;
import com.digiwin.athena.kmservice.locale.Lang;
import com.digiwin.athena.knowledgegraph.constant.Constants;
import com.digiwin.athena.knowledgegraph.domain.topic.Topic;
import com.digiwin.athena.knowledgegraph.dto.topic.TopicDto;
import com.digiwin.athena.knowledgegraph.service.ITopicService;
import com.digiwin.athena.knowledgegraph.utils.AthenaUtils;
import com.digiwin.athena.knowledgegraph.utils.TranslateUtils;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.data.domain.Sort;
import org.springframework.data.mongodb.core.MongoTemplate;
import org.springframework.data.mongodb.core.query.Criteria;
import org.springframework.data.mongodb.core.query.Query;
import org.springframework.stereotype.Service;

import java.util.*;

@Lang
@Service
@Slf4j
public class TopicService implements ITopicService {

    @Autowired
    @Qualifier("knowledgegraphSystem")
    MongoTemplate mongoTemplate;

    @Autowired
    TenantService tenantService;

    @Autowired
    TranslateUtils translateUtils;

    @Override
    public Topic getTopic(String code) throws DWBusinessException {
        String tenantId = AthenaUtils.getTenantId();
        String tenantVersion = tenantService.getTenantVersion(tenantId);
        Query query = new Query();
        query.addCriteria(Criteria.where("code").is(code)
                .and("status").is(Constants.TOPIC_EFFECTIVE_STATUS)
                .and("version").is(tenantVersion));
        return mongoTemplate.findOne(query, Topic.class, "topic");
    }

    @Override
    public List<Topic> getAllTopics() throws DWBusinessException {
        String tenantId = AthenaUtils.getTenantId();
        String tenantVersion = tenantService.getTenantVersion(tenantId);
        Query query = new Query();
        query.addCriteria(Criteria.where("status").is(Constants.TOPIC_EFFECTIVE_STATUS)
                .and("version").is(tenantVersion));
        query.with(Sort.by(Sort.Order.desc("editDate")));
        return mongoTemplate.find(query, Topic.class, "topic");
    }

    @Override
    public boolean postSaveTopic(TopicDto dto) throws DWBusinessException {
        log.info("Route==>TopicService method=postSaveTopic dto={}", JSON.toJSONString(dto));

        String tenantId = AthenaUtils.getTenantId();
        String tenantVersion = tenantService.getTenantVersion(tenantId);


        Query query = new Query();
        query.addCriteria(Criteria.where("code").is(dto.getCode())
                .and("version").is(tenantVersion));
        Topic topic = mongoTemplate.findOne(query, Topic.class, "topic");
        //tips:修改的数据如果是被删除了，也要恢复数据
        if(Objects.nonNull(topic)){
            topic.setLang(dto.getLang());
            topic.setDescription(dto.getDescription());
            topic.setName(dto.getName());
            topic.setRelatedApps(dto.getRelatedApps());
            topic.setEditDate(new Date());
            try {
                topic.setEditBy(AthenaUtils.getUserId());
            } catch (Exception e) {
                log.error("Route==>TopicService method=updateTopic getUserId is null");
            }
            //新增/修改都要加上默认值
            topic.setStatus(Constants.TOPIC_EFFECTIVE_STATUS);
            topic.setApplication(Constants.TOPIC_APPLICATION);
            mongoTemplate.save(topic,"topic");
        }else {
            //新增议题方法
            addTopicMethod(dto, tenantVersion);
        }
        return true;
    }

    /**
     * 新增议题方法
     * @param dto 入参
     * @param tenantVersion 租户版本
     */
    private void addTopicMethod(TopicDto dto, String tenantVersion) {
        Topic topic = new Topic();
        BeanUtils.copyProperties(dto, topic);

        topic.setLang(dto.getLang());
        topic.setDescription(dto.getDescription());
        topic.setCreateDate(new Date());
        topic.setEditDate(new Date());
        topic.setVersion(tenantVersion);
        try {
            topic.setCreateBy(AthenaUtils.getTenantId());
        } catch (Exception e) {
            log.error("Route==>TopicService method=addTopicMethod getUserId is null");
        }
        topic.setStatus(Constants.TOPIC_EFFECTIVE_STATUS);
        topic.setApplication(Constants.TOPIC_APPLICATION);
        mongoTemplate.insert(topic,"topic");
    }

    @Override
    public boolean postDeleteTopic(String code) throws DWBusinessException {
        log.info("Route==>TopicService method=postDeleteTopic code={}", code);

        Topic topic = this.getTopic(code);
        if(Objects.nonNull(topic)){
            topic.setStatus(Constants.TOPIC_INVALID_STATUS);
            topic.setEditDate(new Date());
            mongoTemplate.save(topic,"topic");
        }
        return true;
    }
}
