package com.digiwin.athena.knowledgegraph.support;

import com.digiwin.athena.kmservice.utils.ServiceUtils;
import com.digiwin.athena.knowledgegraph.domain2.base.AbsEntityDict;
import com.digiwin.athena.kg.action.Abstraction;
import com.digiwin.athena.knowledgegraph.domain2.base.Dimension;
import com.digiwin.athena.knowledgegraph.repoSystem.AbsEntityDictRepository;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.data.mongodb.core.MongoTemplate;
import org.springframework.data.mongodb.core.query.Criteria;
import org.springframework.data.mongodb.core.query.Query;
import org.springframework.stereotype.Component;
import org.springframework.util.Assert;

import java.lang.reflect.Field;
import java.util.Collection;
import java.util.List;
import java.util.Map;

/**
 根据产品/租户信息替换抽象部分.
 抽象部分只能以叶子节点存在
 缺点:对于一次处理,可能要多次查询mongo,有时间消耗.这个可通过缓存来解决这个问题
 **/
@Slf4j
@Component
public class AbstractionHandler {

    @Autowired
    AbsEntityDictRepository absRepo;

    @Autowired
    @Qualifier("knowledgegraphSystem")
    MongoTemplate mongoTemplate;

    public void dealWithAbstraction(Object obj, Dimension dimension){
        if(obj==null || ServiceUtils.isPrimitive(obj.getClass()) || Dimension.isNull(dimension)){
            return;
        }
        Class c = obj.getClass();
        if(obj instanceof Collection){
            Collection collection = (Collection) obj;
            for (Object o :collection){
                dealWithAbstraction(o,dimension);
            }
        }else if(obj.getClass().isArray()){
            Object[] arr = (Object[]) obj;
            for (Object o :arr){
                dealWithAbstraction(o,dimension);
            }
        }else if(obj instanceof Map){
            Map map = (Map) obj;
            for( Object o : map.values()){
                dealWithAbstraction(o,dimension);
            }
        }else {
            //复合类型
            List<Field> fields = ServiceUtils.getFields(obj.getClass());
            for(Field f :fields){
                f.setAccessible(true);
                try {
                    Object o = f.get(obj);
                    dealWithAbstraction(o,dimension);
                } catch (IllegalAccessException e) {
                    log.error(e.getMessage(), e);
                }
            }

            if(obj instanceof Abstraction){
                Abstraction abs = (Abstraction) obj;
                innerDo(abs,dimension);
            }
        }


    }


    public void innerDo(Abstraction abs,Dimension dimension){
        if(null == abs.getDimension() || null == abs.getKey() || 0 == abs.getDimension()){
            return;
        }
        AbsEntityDict dict = getDict(abs,dimension);
        if(null!=dict && null!=dict.getAbstraction()){
            cloneAbs(abs,dict.getAbstraction());
        }
    }


    private AbsEntityDict getDict(Abstraction abs,Dimension dimension){
        AbsEntityDict result = null;

        Query query = new Query();
        query.addCriteria(Criteria.where("category").is(abs.getClass().getSimpleName()));
        query.addCriteria(Criteria.where("key").is(abs.getKey()));
        query.addCriteria(Criteria.where("status").is(1));

        if((Abstraction.tenant & abs.getDimension()) > 0 ){
            Assert.notNull(dimension.getTenant(),"租户id不能位空!");
            query.addCriteria(Criteria.where("tenant").is(dimension.getTenant()));
        }
        if((Abstraction.product & abs.getDimension()) > 0 ){
            Assert.notNull(dimension.getProduct(),"产品不能位空!");
            query.addCriteria(Criteria.where("product").is(dimension.getProduct()));
        }
        if((Abstraction.unit & abs.getDimension()) > 0 ){
            Assert.notEmpty(dimension.getUnit(),"运营单元不能位空!");
            query.addCriteria(Criteria.where("unit").is(dimension.getUnit()));
        }

        result = mongoTemplate.findOne(query,AbsEntityDict.class);

        return result;
    }


    private static void cloneAbs(Object abs,Object absNew){
        List<Field> fields = ServiceUtils.getFields(abs.getClass());
        for(Field f :fields){
            f.setAccessible(true);
            try {
                Object newValue = f.get(absNew);
                if(null!=newValue){
                    f.set(abs,newValue);
                }
            } catch (IllegalAccessException e) {
                log.error(e.getMessage(), e);
            }
        }
    }



}
