package com.digiwin.athena.knowledgegraph.synonym.controller.impl;

import com.digiwin.athena.knowledgegraph.enums.NewSourceType;
import com.digiwin.athena.knowledgegraph.service.KgInnerService;
import com.digiwin.athena.knowledgegraph.synonym.controller.IBusinessTypeService;
import com.digiwin.athena.knowledgegraph.synonym.dao.BusinessTypeDao;
import com.digiwin.athena.knowledgegraph.synonym.domain.BusinessType;
import com.digiwin.athena.knowledgegraph.synonym.domain.po.DeleteVo;
import com.digiwin.athena.knowledgegraph.synonym.domain.po.ResultBean;
import com.digiwin.athena.knowledgegraph.synonym.exception.BusinessException;
import com.digiwin.athena.knowledgegraph.utils.AthenaUtils;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import javax.annotation.Resource;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import java.util.Map;
import java.util.function.Function;
import java.util.stream.Collectors;

/**
 * @Author: zjz
 * @CreateTime: 2024-03-25  17:30
 * @Description:
 * @Version: 1.0
 */
@Slf4j
@Service
public class BusinessTypeServiceImpl implements IBusinessTypeService {

    @Resource
    private BusinessTypeDao businessTypeDao;

    @Resource
    private KgInnerService kgInnerService;

    @Override
    public ResultBean<Object> update(BusinessType businessType) {
        try {
            businessTypeDao.update(businessType);
        }catch (BusinessException businessException){
            return ResultBean.fail(1,businessException.getMessage());
        }
        return ResultBean.success("");
    }

    @Override
    public ResultBean<Object> getSystemOne(String appCode, String code) {
        try {
            String tenantId = AthenaUtils.getTenantId();
            String tenantVersion = kgInnerService.getTenantVersion(tenantId);
            return ResultBean.success(businessTypeDao.querySystemBusinessType(appCode, code,tenantVersion));
        }catch (Exception exception){
            return ResultBean.fail(1,exception.getMessage());
        }
    }

    @Override
    public ResultBean<Object> getPage( String appCode, String tid, int pageNum, int pageSize) {
        try {
            return ResultBean.success(businessTypeDao.getPage(appCode, tid,pageNum,pageSize));
        }catch (BusinessException businessException){
            return ResultBean.fail(1,businessException.getMessage());
        }
    }

    @Override
    public ResultBean<Object> syncData(DeleteVo deleteVo) {
        try {
            this.syncValueData(deleteVo.getTid(), deleteVo.getAppCode());
            return ResultBean.success("");
        }catch (BusinessException businessException){
            return ResultBean.fail(1,businessException.getMessage());
        }
    }

    /**
     * 从system库同步业务类型
     */
    private void syncValueData(String tid, String appCode){
        String tenantVersion = kgInnerService.getTenantVersion(tid);
        // system源数据
        List<BusinessType> systemBusinessTypes = businessTypeDao.querySystemBusinessTypeByVersion(appCode, tenantVersion);
        // 业务数据
        List<BusinessType> businessTypes = businessTypeDao.findList(appCode, tid);
        if(CollectionUtils.isEmpty(systemBusinessTypes)){
            // 如果system库无数据，则清空业务数据
            DeleteVo deleteVo = new DeleteVo();
            deleteVo.setAppCode(appCode);
            deleteVo.setTid(tid);
            businessTypeDao.batchDelete(deleteVo);
            log.info("无默认业务类型，删除业务类型，同步结束 tid={}, appCode={}", tid, appCode);
            return;
        }
        List<BusinessType> insertList = new ArrayList<>();
        Map<String, BusinessType> businessTypeMap = businessTypes.stream()
                .collect(Collectors.toMap(BusinessType::getCode, Function.identity(), (a, b)->a));
        for(BusinessType businessType:systemBusinessTypes){
            BusinessType localBusinessType = businessTypeMap.get(businessType.getCode());
            if(null == localBusinessType || NewSourceType.SYSTEM.getCode().equals(localBusinessType.getSource())){
                BusinessType insertBusinessType = new BusinessType();
                insertBusinessType.setSource(NewSourceType.SYSTEM.getCode());
                insertBusinessType.setTenantId(tid);
                insertBusinessType.setEditTime(new Date(businessType.getEditDate()));
                insertBusinessType.setCode(businessType.getCode());
                insertBusinessType.setName(businessType.getName());
                insertBusinessType.setAppCode(businessType.getAppCode());
                insertBusinessType.setShowType(businessType.getShowType());
                insertBusinessType.setLang(businessType.getLang());
                insertBusinessType.setDescription(businessType.getDescription());
                insertBusinessType.setEditBy(businessType.getEditBy());
                insertBusinessType.setDecimal(businessType.getDecimal());
                insertBusinessType.setDecimalRule(businessType.getDecimalRule());
                insertBusinessType.setDataUnit(businessType.getDataUnit());
                if(null != localBusinessType && NewSourceType.SYSTEM.getCode().equals(localBusinessType.getSource())){
                    insertBusinessType.setEditBy(localBusinessType.getEditBy());
                    insertBusinessType.setEditTime(localBusinessType.getEditTime());
                }
                insertList.add(insertBusinessType);
                continue;
            }
            insertList.add(localBusinessType);
        }

        // 先删后增
        DeleteVo deleteVo = new DeleteVo();
        deleteVo.setAppCode(appCode);
        deleteVo.setTid(tid);
        businessTypeDao.batchDelete(deleteVo);
        businessTypeDao.batchInsert(insertList);
    }

}
