package com.digiwin.athena.knowledgegraph.synonym.dao;

import com.digiwin.athena.knowledgegraph.synonym.domain.EntityTypeConfig;
import com.digiwin.athena.knowledgegraph.synonym.domain.LocalAppInfo;
import com.digiwin.athena.knowledgegraph.synonym.domain.po.PageSelectVo;
import com.digiwin.athena.knowledgegraph.synonym.domain.po.PageVo;
import com.digiwin.athena.knowledgegraph.synonym.exception.BusinessException;
import org.apache.commons.lang3.StringUtils;
import org.bson.Document;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.data.mongodb.core.BulkOperations;
import org.springframework.data.mongodb.core.MongoTemplate;
import org.springframework.data.mongodb.core.query.Criteria;
import org.springframework.data.mongodb.core.query.Query;
import org.springframework.data.mongodb.core.query.Update;
import org.springframework.stereotype.Repository;
import org.springframework.util.CollectionUtils;

import java.util.List;
import java.util.Map;
import java.util.function.Function;
import java.util.stream.Collectors;

/**
 * @Author: wangao
 * @CreateTime: 2024-06-06  16:55
 * @Description: TODO
 * @Version: 1.0
 */
@Repository
public class SyncAppInfoDao {
    @Autowired
    @Qualifier("knowledgegraphTenant")
    private MongoTemplate mongoTemplate;
    @Autowired
    @Qualifier("knowledgegraphSystem")
    private MongoTemplate systemMongoTemplate;
    public void insertLocalList(List<LocalAppInfo> localList) {
        mongoTemplate.insertAll(localList);
    }
    public PageVo getLocalListPage(PageSelectVo page) {
        if (StringUtils.isEmpty(page.getTenantId())){
            throw new BusinessException("租户id不能为空");
        }
        Query query = new Query();
        Criteria criteria = new Criteria();
        criteria.andOperator(Criteria.where("tenantId").is(page.getTenantId()));
        query.addCriteria(criteria);
        long totalCount = mongoTemplate.count(query, LocalAppInfo.class);
        int pageNum = page.getPage();
        int pageSize = page.getPageSize();
        if (pageNum==0&&pageSize==0){
            List<LocalAppInfo> localAppInfos = mongoTemplate.find(query, LocalAppInfo.class);
            return new PageVo((int) totalCount,localAppInfos,page.getPage(),pageSize,1);
        }else {
            Pageable pageable = PageRequest.of(pageNum, pageSize);
            List<PageSelectVo> pageSelectVos = this.mongoTemplate.find(query.with(pageable), PageSelectVo.class);
            int totalPage = (int) (totalCount % pageSize == 0 ? totalCount / pageSize : totalCount / pageSize + 1);
            return new PageVo((int) totalCount,pageSelectVos,page.getPage(),pageSize,totalPage);
        }
    }
    public List<LocalAppInfo> getLocalList(String tenantId) {
        Query query = new Query();
        Criteria criteria = new Criteria();
        criteria.andOperator(Criteria.where("tenantId").is(tenantId));
        query.addCriteria(criteria);
        List<LocalAppInfo> localAppInfos = mongoTemplate.find(query, LocalAppInfo.class);
        return localAppInfos;
    }

    public List<LocalAppInfo> getAll() {
        List<LocalAppInfo> localAppInfos = mongoTemplate.findAll(LocalAppInfo.class);
        return localAppInfos;
    }

    public void updateLocal(LocalAppInfo localAppInfo) {
        if (StringUtils.isEmpty(localAppInfo.getObjectId())){
            throw new BusinessException("当前数据无id"+localAppInfo.toString());
        }
        mongoTemplate.save(localAppInfo);
    }

    public void deleteLocal(List<String> list) {
        if (CollectionUtils.isEmpty(list)){
            return;
        }
        Query query = new Query(Criteria.where("objectId").in(list));
        this.mongoTemplate.remove(query,LocalAppInfo.class);
    }

    public void saveAll(List<LocalAppInfo> localAppInfos) {
        // 去重并过滤空 objectId
        Map<String, LocalAppInfo> uniqueInfos = localAppInfos.stream()
                .filter(info -> info.getObjectId() != null)
                .collect(Collectors.toMap(
                        LocalAppInfo::getObjectId,
                        Function.identity(),
                        (existing, replacement) -> existing // 保留现有记录
                ));

        // 批量替换（存在则更新，不存在则插入）
        BulkOperations bulkOps = mongoTemplate.bulkOps(BulkOperations.BulkMode.ORDERED, LocalAppInfo.class);
        uniqueInfos.values().forEach(info -> {
            // 使用 _id 作为查询条件（因为 objectId 被映射为 _id）
            Query query = new Query(Criteria.where("_id").is(info.getObjectId()));

            Document doc = new Document();
            mongoTemplate.getConverter().write(info, doc);

            // 关键修复：移除 _id 字段避免更新冲突
            doc.remove("_id");

            // 创建更新操作，排除 _id 字段
            Update update = new Update();
            doc.forEach((key, value) -> update.set(key, value));

            bulkOps.upsert(query, update);
        });
        bulkOps.execute();
    }


    public void deleteEntityConfig(String appName, String appType) {
        Criteria criteria = new Criteria();
        criteria.andOperator(Criteria.where("appName").is(appName).and("appType").is(appType));
        Query query = new Query(criteria);
        systemMongoTemplate.remove(query, EntityTypeConfig.class);
    }

    public void saveConfigsAll(List<EntityTypeConfig> configs) {
        List<String> collect = configs.stream().map(EntityTypeConfig::getId).collect(Collectors.toList());
        Query query = new Query(Criteria.where("id").in(collect));
        systemMongoTemplate.remove(query,EntityTypeConfig.class);
        systemMongoTemplate.insertAll(configs);
    }

    public List<EntityTypeConfig> getEntityConfigs(String appName, String appType) {
        Criteria criteria = new Criteria();
        criteria.andOperator(Criteria.where("appName").is(appName).and("appType").is(appType));
        Query query = new Query(criteria);
        return systemMongoTemplate.find(query, EntityTypeConfig.class);
    }

    public LocalAppInfo find(String id) {
        return mongoTemplate.findById(id, LocalAppInfo.class);
    }

    public void updateEntityTypeConfig(EntityTypeConfig entityTypeConfig) {
        if (StringUtils.isEmpty(entityTypeConfig.getId())){
            throw new BusinessException("当前数据无id"+entityTypeConfig.toString());
        }
        systemMongoTemplate.save(entityTypeConfig);
    }
    public void dataMigration(){
        long count = systemMongoTemplate.count(new Query(), LocalAppInfo.class);
        if (count > 0){
            List<LocalAppInfo> appThresholdConfigs = systemMongoTemplate.find(new Query(), LocalAppInfo.class);
            mongoTemplate.insertAll(appThresholdConfigs);
        }
    }
}
