package com.digiwin.athena.knowledgegraph.clients;

import com.digiwin.app.container.exceptions.DWBusinessException;
import com.digiwin.app.service.DWFile;
import com.digiwin.athena.knowledgegraph.po.FileInfo;
import com.digiwin.athena.knowledgegraph.po.MidWareResponse;
import com.digiwin.athena.knowledgegraph.utils.I18nUtils;
import com.digiwin.http.client.utils.DWRequestHeaderUtils;
import lombok.Data;
import lombok.extern.slf4j.Slf4j;
import net.sf.jmimemagic.Magic;
import net.sf.jmimemagic.MagicMatch;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.core.io.ByteArrayResource;
import org.springframework.http.*;
import org.springframework.stereotype.Service;
import org.springframework.util.LinkedMultiValueMap;
import org.springframework.util.MultiValueMap;
import org.springframework.web.client.RestTemplate;

import javax.servlet.ServletOutputStream;
import javax.servlet.http.HttpServletResponse;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

@Service
@Slf4j
@Data
public class DMCUtils {

    @Value("${dmcUrl}")
    private String dmcUrl;
    @Value("${iamApToken}")
    private String appToken;
    @Value("${dmcUserName}")
    private String dmcUserName;
    @Value("${dmcPwdhash}")
    private String dmcPwdhash;
    @Value("${dmcDirId}")
    private String dmcDirId;
    @Value("${dmcBucket}")
    private String dmcBucket;

    @Autowired
    @Qualifier("dmcRestTemplate")
    private RestTemplate dmcRestTemplate;

    public String getDmcToken() throws DWBusinessException{
        HttpHeaders header = new HttpHeaders();
        header.setContentType(MediaType.APPLICATION_JSON);
        Map<String, String> reqMap = new HashMap<>();
        reqMap.put("username", dmcUserName);
        reqMap.put("pwdhash", dmcPwdhash);

        String url = dmcUrl + "/api/dmc/v1/auth/login";

        HttpEntity<MultiValueMap<String, String>> httpEntity = new HttpEntity(reqMap, header);
        ResponseEntity<Map> resMap = dmcRestTemplate.postForEntity(url, httpEntity, Map.class);
        String token = (String) resMap.getBody().get("userToken");
        if (StringUtils.isEmpty(token)) {
            throw new DWBusinessException(I18nUtils.getValue("knowledgegraph.getDmcTokenError"));
        }
        log.info("dmcToken:{}", token);
        return token;
    }

    public String upload(DWFile dwFile, FileInfo fileInfo) throws Exception {

        MultiValueMap<String, Object> multiValueMap = new LinkedMultiValueMap<>();
        HttpHeaders header = new HttpHeaders();
        header.setContentType(MediaType.MULTIPART_FORM_DATA);
        header.add(DWRequestHeaderUtils.HEADER_IAM_API_USER_TOKEN, this.getDmcToken());

        HttpHeaders fileHeader = new HttpHeaders();

        fileHeader.setContentType(MediaType.parseMediaType(this.getMediaType(dwFile.getFileByteArray())));
        fileHeader.setContentDispositionFormData("file", dwFile.getFileName());

        HttpHeaders fileInfoHeader = new HttpHeaders();
        fileInfoHeader.setContentType(MediaType.APPLICATION_JSON);

        try {
            HttpEntity<ByteArrayResource> fileEntity = new HttpEntity<>(new ByteArrayResource(dwFile.getFileByteArray()),
                    fileHeader);

            fileInfo.setTenantId("default");
            fileInfo.setDirectoryId(dmcDirId);
            HttpEntity<FileInfo> fileInfoEntity = new HttpEntity<>(fileInfo, fileInfoHeader);
            multiValueMap.add("file", fileEntity);
            multiValueMap.add("fileInfo", fileInfoEntity);

            String url = dmcUrl + "/api/dmc/v2/file/Athena/upload";

            HttpEntity<MultiValueMap<String, Object>> httpEntity = new HttpEntity<>(multiValueMap, header);
            ResponseEntity<MidWareResponse> postForEntity = dmcRestTemplate.postForEntity(url, httpEntity, MidWareResponse.class);
            MidWareResponse<Map> midWareResponse = postForEntity.getBody();
            if (midWareResponse.getSuccess()) {
                return (String) midWareResponse.getData().get("id");
            } else {
                log.error("调用上传api失败");
                throw new DWBusinessException(I18nUtils.getValue("knowledgegraph.invokeApiError"));
            }
        } catch (Exception e) {
            log.error("fileUploadError:{}", e);
            throw new DWBusinessException("", e);
        }

    }

    /**
     * @Author zhuangli
     * @Description 用于下载应用自定义导入模板
     * @Date 10:31 2022/2/10
     * @Param
     * @return
     **/
    public void downLoadTemplate(String fileId, HttpServletResponse response) throws DWBusinessException {
        if (org.springframework.util.StringUtils.isEmpty(fileId)) {
            throw new DWBusinessException(I18nUtils.getValue("knowledgegraph.wordNotBuild"));
        }
        response.setContentType("application/vnd.openxmlformats-officedocument.spreadsheetml.sheet");
        //设置文件名
        response.setCharacterEncoding("utf-8");
        response.addHeader("Content-Disposition", "attachment;filename=template.xlsx");
        response.addHeader("Access-Control-Expose-Headers", "Content-Disposition");
        byte[] inputBytes = getErrorTableInputBytes(fileId);
        try (ServletOutputStream out = response.getOutputStream()) {
            out.write(inputBytes, 0, inputBytes.length);
            out.flush();
        } catch (Exception e) {
            log.error("下载模板失败", e);
        }
    }

    private byte[] readInputStream(InputStream fis) throws IOException {
        ByteArrayOutputStream outStream = new ByteArrayOutputStream();
        byte[] buffer = new byte[1024];
        int len;
        while( (len=fis.read(buffer)) != -1 ){
            outStream.write(buffer, 0, len);
        }
        //fis.close();此处一定不能关闭流
        return outStream.toByteArray();
    }

    private byte[] getErrorTableInputBytes(String fileId) throws DWBusinessException {
        HttpHeaders headers = new HttpHeaders();
        headers.add(DWRequestHeaderUtils.HEADER_IAM_API_USER_TOKEN, this.getDmcToken());
        headers.add(DWRequestHeaderUtils.HEADER_IAM_API_APP_TOKEN, appToken);
        List list = new ArrayList<>();
        list.add(MediaType.parseMediaType("application/vnd.openxmlformats-officedocument.spreadsheetml.sheet"));
        headers.setAccept(list);
        String url = dmcUrl + "/api/dmc/v2/file/" + this.getDmcBucket()+ "/download/" + fileId;
        ResponseEntity<byte[]> restRes = dmcRestTemplate.exchange(
                url,
                HttpMethod.GET,
                new HttpEntity<byte[]>(headers),
                byte[].class);
        return restRes.getBody();
    }

    public String getMediaType(byte[] fileByte){
        String mimeType = null;
        try {
            MagicMatch magicMatch = Magic.getMagicMatch(fileByte);
            mimeType = magicMatch.getMimeType();
        } catch (Exception e) {
            log.error(e.getMessage(), e);
        }
//        Tika tika = new Tika();
//        String detect = tika.detect(fileByte);
        log.info("mediaType:{}", mimeType);
        return mimeType;
    }
}
