package com.digiwin.athena.knowledgegraph.controller;

import com.digiwin.app.service.AllowAnonymous;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Controller;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.RequestMapping;

import javax.servlet.http.HttpServletResponse;
import java.io.*;
import java.lang.reflect.Method;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.time.LocalDateTime;
import java.time.format.DateTimeFormatter;
import java.util.Enumeration;
import java.util.jar.JarEntry;
import java.util.jar.JarFile;
import java.util.stream.Stream;
import java.util.zip.ZipEntry;
import java.util.zip.ZipOutputStream;

/**
 * @author cjbi
 */
@Controller
@Slf4j
@RequestMapping("/knowledgegraph/jacoco")
public class JacocoController {

    @Value("${jacoco.workingDir:/knowledgegraph_backend}")
    private String workingDir = "";

    @Value("${jacoco.jarDir:/jacoco}")
    private String jacocoJarDir;

    private void dumpJacocoExec() {
        try {
            Class<?> rt = Class.forName("org.jacoco.agent.rt.RT");
            Method getAgent = rt.getMethod("getAgent");
            Method dump = getAgent.getReturnType().getMethod("dump", boolean.class);
            Object agent = getAgent.invoke(null);
            //  if true, the current execution data is cleared
            dump.invoke(agent, false);
        } catch (ClassNotFoundException e) {
            log.error("Jacoco not enabled", e);
            throw new RuntimeException("Jacoco not enabled");
        } catch (Exception e) {
            log.error("Dump exec failed", e);
            throw new RuntimeException("Dump exec failed");
        }
    }

    /**
     * 将目录下所有文件打包为 ZIP，写入 outputStream。
     */
    public void zipDirectory(String name, Path sourceDir, OutputStream outputStream) throws IOException {
        try (ZipOutputStream zipOut = new ZipOutputStream(outputStream);
             Stream<Path> walkStream = Files.walk(sourceDir)) { // 同时关闭流
            walkStream.forEach(path -> {
                Path rel = sourceDir.relativize(path);
                String entryName = (name + "/" + rel.toString().replace(File.separatorChar, '/')
                        + (Files.isDirectory(path) ? "/" : "")).replace("//", "/");
                try {
                    zipOut.putNextEntry(new ZipEntry(entryName));
                    if (Files.isRegularFile(path)) {
                        Files.copy(path, zipOut);
                    }
                    zipOut.closeEntry();
                } catch (IOException e) {
                    log.error("An error occurred in packaging file : {}", path, e);
                }
            });
        }
    }

    private void extractSourceJar(String workingDir) {
        File sourceDirectory = new File(workingDir, "src/main/java");
        if (!sourceDirectory.exists()) {
            try {
                extractSourceJar(workingDir + "/lib/knowledgegraph-1.0.1-sources.jar", workingDir + "/src/main/java");
            } catch (IOException e) {
                throw new RuntimeException(e);
            }
        }
    }

    public void extractSourceJar(String jarPath, String destDir) throws IOException {
        JarFile jarFile = new JarFile(jarPath);
        Enumeration<JarEntry> entries = jarFile.entries();

        while (entries.hasMoreElements()) {
            JarEntry entry = entries.nextElement();
            File entryFile = new File(destDir, entry.getName());

            if (entry.isDirectory()) {
                entryFile.mkdirs();
            } else {
                // 确保父目录存在
                entryFile.getParentFile().mkdirs();

                try (InputStream in = jarFile.getInputStream(entry);
                     OutputStream out = new FileOutputStream(entryFile)) {

                    byte[] buffer = new byte[4096];
                    int bytesRead;

                    while ((bytesRead = in.read(buffer)) != -1) {
                        out.write(buffer, 0, bytesRead);
                    }
                }
            }
        }

        jarFile.close();
        log.info("Decompression completed: " + destDir);
    }

    private void execCmd(String coverageReportDir) throws InterruptedException, IOException {
        ProcessBuilder processBuilder = new ProcessBuilder()
                .command("java", "-jar",
                        jacocoJarDir + "/jacococli.jar", "report", workingDir + "/jacoco.exec",
                        "--classfiles", workingDir + "/lib/knowledgegraph-1.0.1.jar",
                        "--sourcefiles", workingDir + "/src/main/java",
                        "--html", workingDir + coverageReportDir)
                .redirectErrorStream(true);
        Process process = processBuilder.start();

        // 读取命令输出
        BufferedReader reader = new BufferedReader(
                new InputStreamReader(process.getInputStream()));
        String line;
        // 记录每行输出
        while ((line = reader.readLine()) != null) {
            log.info("JaCoCo CLI Output: {}", line);
        }
        int exitCode = process.waitFor();
        if (exitCode == 0) {
            log.info("HTML report generation successfully!");
        } else {
            log.error("Report generation failed. exitCode: {}", exitCode);
        }
    }

    /**
     * 获取Jacoco exec 文件
     *
     * @return
     */
    @GetMapping("/jacoco.exec")
    @AllowAnonymous
    public void getJacocoExec(HttpServletResponse response) throws IOException {
        dumpJacocoExec();
        // 设置响应头
        try (OutputStream os = response.getOutputStream()) {
            File file = new File(workingDir + "/jacoco.exec");
            // 写入你的数据
            os.write(Files.readAllBytes(file.toPath()));
            os.flush();
        }
    }

    /**
     * 获取Jacoco HTML报告
     *
     * @param response
     * @throws Exception
     */
    @GetMapping("/jacoco-report.zip")
    @AllowAnonymous
    public void getReport(HttpServletResponse response) throws Exception {
        dumpJacocoExec();
        extractSourceJar(workingDir);
        String coverageReportDir = "/coveragereport";
        execCmd(coverageReportDir);
        String filename = "knowledgegraph-jacoco-report_" + LocalDateTime.now().format(DateTimeFormatter.ofPattern("yyyyMMddHHmmss")) + ".zip";
        // 设置响应头
        response.setContentType("application/octet-stream");
        response.setHeader("Content-Disposition", "attachment; filename=\"" + filename + "\"");
        try (OutputStream os = response.getOutputStream()) {
            // 写入你的数据
            zipDirectory(filename, Paths.get(workingDir + coverageReportDir), os);
            os.flush();
        }
    }

}
