package com.digiwin.athena.knowledgegraph.dao;

import com.digiwin.athena.kg.activity.Activity;
import com.digiwin.athena.kg.activity.ActivityBaseInfoVO;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.data.mongodb.core.MongoTemplate;
import org.springframework.data.mongodb.core.aggregation.Aggregation;
import org.springframework.data.mongodb.core.aggregation.AggregationResults;
import org.springframework.data.mongodb.core.aggregation.GroupOperation;
import org.springframework.data.mongodb.core.aggregation.ProjectionOperation;
import org.springframework.data.mongodb.core.query.Criteria;
import org.springframework.data.mongodb.core.query.Query;
import org.springframework.stereotype.Repository;

import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

@Repository
public class ActivityDAO {

    @Autowired
    @Qualifier("knowledgegraphSystem")
    private MongoTemplate mongoTemplateSystem;

    /**
     * 查询所有ABi报表code
     * @return List<String>
     */
    public List<String> getAllABiReportCodes() {
        Criteria criteria = Criteria.where("category").is("ABI-STATEMENT");
        List<Activity> result = mongoTemplateSystem.find(Query.query(criteria), Activity.class, "activityConfigsBasic");
        return result.stream().map(Activity::getCode).distinct().collect(Collectors.toList());
    }

    /**
     * 根据code和版本查询报表
     * @param activityCode      activityCode
     * @param tenantVersion     租户版本
     * @param tenantId          租户ID
     * @return  List<Activity>
     */
    public List<Activity> getActivityByCodeAndVersion(String activityCode, String tenantVersion, String tenantId) {
        Criteria criteria = Criteria.where("code").is(activityCode)
                .and("version").is(tenantVersion)
                .andOperator(getCriteriaByTenantId(tenantId, true));
        return mongoTemplateSystem.find(Query.query(criteria), Activity.class, "activityConfigsBasic");
    }

    /**
     * 根据租户ID查询公共数据报表
     * @param tenantId    租户ID
     * @return
     */
    public List<Activity> getCommonBaseData(String tenantId) {
        Criteria criteria = Criteria.where("nameSpace").is("common")
                .andOperator(getCriteriaByTenantId(tenantId, true));
        return mongoTemplateSystem.find(Query.query(criteria), Activity.class, "activityConfigsBasic");
    }

    /**
     * 根据activityIdList查询签核列表
     * @param activityIdList    activityIdList
     * @param tenantId          租户ID
     * @param tenantVersion     租户版本
     * @return  List<ActivityBaseInfoVO>
     */
    public List<ActivityBaseInfoVO> getApproveActivityListByActivityIds(List<String> activityIdList, String tenantId, String tenantVersion) {
        Criteria criteria = Criteria.where("code").in(activityIdList)
                .and("version").is(tenantVersion)
                .and("category").is("APPROVAL")
                .andOperator(getCriteriaByTenantId(tenantId, false));
        return mongoTemplateSystem.find(Query.query(criteria), ActivityBaseInfoVO.class, "activityConfigsBasic");
    }

    /**
     * 根据pattern查询报表
     * @param tenantId        租户ID
     * @param pattern         模式
     * @param tenantVersion   租户版本
     * @param codes
     * @return  List<Activity>
     */
    public List<Activity> getActivitiesByPattern(String tenantId, String pattern, String tenantVersion,List<String> codes) {
        Criteria criteria = Criteria.where("code").in(codes)
                .and("version").is(tenantVersion)
                .and("pattern").is(pattern)
                .andOperator(getCriteriaByTenantId(tenantId, true));
        return mongoTemplateSystem.find(Query.query(criteria), Activity.class, "activityConfigsBasic");
    }

    /**
     * 根据code集合查询报表
     * @param reportCodes     报表code列表
     * @param tenantId        租户ID
     * @param tenantVersion   租户版本
     * @return  List<Activity>
     */
    public List<Activity> getReportActivitiesByCode(List<String> reportCodes, String tenantId, String tenantVersion) {
        Criteria criteria = Criteria.where("code").in(reportCodes)
                .and("version").is(tenantVersion)
                .andOperator(getCriteriaByTenantId(tenantId, true));
        return mongoTemplateSystem.find(Query.query(criteria), Activity.class, "activityConfigsBasic");
    }

    /**
     * 根据code查询报表不含租户ID查询条件
     * @param bizCode  报表code
     * @return
     */
    public Activity findActivityNoTenantId(String bizCode) {
        Criteria criteria = Criteria.where("code").is(bizCode)
                .andOperator(new Criteria()
                        .orOperator(Criteria.where("pattern").nin("DATA_ENTRY", "STATEMENT")
                                , Criteria.where("pattern").is(null)
                        )
                );
        return mongoTemplateSystem.findOne(Query.query(criteria), Activity.class, "activityConfigsBasic");
    }

    /**
     * 根据name/pattern/condition/tenantId/tenantVersion查询报表
     * @param pattern       模式
     * @param condition     条件
     * @param tenantId      租户ID
     * @param tenantVersion 租户版本
     * @return  List<Activity>
     */
    public List<Activity> getActivitiesByPatternCondition(String pattern, String condition, String tenantId, String tenantVersion) {
        // 构建名称匹配条件
        Criteria nameCriteria = new Criteria().orOperator(
                Criteria.where("name").regex(condition),
                Criteria.where("lang.name.zh_TW").regex(condition),
                Criteria.where("lang.name.en_US").regex(condition),
                Criteria.where("lang.name.zh_CN").regex(condition)
        );
        // 构建完整查询
        Criteria criteria = new Criteria().andOperator(
                Criteria.where("version").is(tenantVersion),
                Criteria.where("pattern").is(pattern),
                nameCriteria,
                getCriteriaByTenantId(tenantId, true)
        );
        return mongoTemplateSystem.find(Query.query(criteria), Activity.class, "activityConfigsBasic");
    }

    /**
     * 根据nameSpace查询报表
     * @param nameSpace   名称空间
     * @return  List<Activity>
     */
    public List<Activity> getDataEntryByNameSpace(String nameSpace) {
        Criteria criteria = Criteria.where("nameSpace").is(nameSpace)
                .and("pattern").is("DATA_ENTRY");
        return mongoTemplateSystem.find(Query.query(criteria), Activity.class, "activityConfigsBasic");
    }

    /**
     * 获取全量的任务和活动
     * @return  List<Map>
     */
    public List<Map> getAllActivity() {
        // 使用 $group 来去重，选择需要的字段作为 _id
        GroupOperation groupOperation = Aggregation.group("code", "name", "nameSpace");

        // 使用 $project 来提取需要的字段（从 _id 中提取）
        ProjectionOperation projectOperation = Aggregation.project()
                .and("$_id.code").as("code")
                .and("$_id.name").as("name")
                .and("$_id.nameSpace").as("nameSpace");

        // 构建聚合管道
        Aggregation aggregation = Aggregation.newAggregation(
                groupOperation,
                projectOperation
        );

        // 执行聚合查询
        AggregationResults<Map> results = mongoTemplateSystem.aggregate(aggregation, "activityConfigsBasic", Map.class);

        return results.getMappedResults();
    }

    /**
     * 根据code查询报表
     * @param code   报表code
     * @return  Activity
     */
    public Activity findOneByCode(String code) {
        Criteria criteria = Criteria.where("code").is(code);
        return mongoTemplateSystem.findOne(Query.query(criteria), Activity.class, "activityConfigsBasic");
    }

    /**
     * 根据租户ID和是否包含子租户查询条件
     * @param tenantId          租户ID
     * @param inclusionTenant   是否包含子租户
     * @return  Criteria
     */
    public Criteria getCriteriaByTenantId(String tenantId, boolean inclusionTenant) {
        Criteria criteria = Criteria.where("tenantId").in(tenantId, "SYSTEM", null);
        if (inclusionTenant) {
            criteria = new Criteria().orOperator(criteria, Criteria.where("inclusionTenant").is(tenantId));
        }
        return criteria;
    }
}
