package com.digiwin.athena.knowledgegraph.service;

import com.digiwin.app.container.exceptions.DWBusinessException;
import com.digiwin.app.service.AllowAnonymous;
import com.digiwin.app.service.DWService;
import com.digiwin.athena.knowledgegraph.domain.common.OperationUnit;
import com.digiwin.athena.kg.monitorRule.MonitorNoticeVO;
import com.digiwin.athena.kg.monitorRule.EocMapDTO;
import com.digiwin.athena.knowledgegraph.service.model.MonitorRuleConfigDTO;
import com.digiwin.athena.knowledgegraph.service.model.MonitorRuleSyncResult;
import com.digiwin.athena.knowledgegraph.vo.MonitorEocChangeVO;

import java.util.List;

/**
 * 侦测规则配置服务
 *
 * @javadoc
 */
public interface IMonitorRuleConfigService extends DWService {

    /**
     * 保存侦测规则配置
     *
     * @param config 侦测规则配置
     * @return
     * @throws Exception
     */
    @AllowAnonymous
    Object put(MonitorRuleConfigDTO config) throws Exception;

    /**
     * 删除侦测规则配置
     *
     * @param tenantId 租户ID
     * @param ruleId   规则ID
     * @return
     * @throws Exception
     */
    @AllowAnonymous
    Object delete(String tenantId, String ruleId) throws Exception;

    /**
     * 更新侦测规则配置
     *
     * @param config 侦测规则配置
     * @return
     * @throws Exception
     */
    Object post(MonitorRuleConfigDTO config) throws Exception;

    /**
     * 生效侦测规则配置
     *
     * @param tenantId 租户ID
     * @param ruleId   规则ID
     * @return
     * @throws Exception
     */
    @AllowAnonymous
    Object putEnable(String tenantId, String ruleId) throws Exception;

    /**
     * 失效侦测规则配置
     *
     * @param tenantId 租户ID
     * @param ruleId   规则ID
     * @return
     * @throws Exception
     */
    @AllowAnonymous
    Object putDisable(String tenantId, String ruleId) throws Exception;

    /**
     * 生效侦测规则配置
     *
     * @param tenantId 租户ID
     * @param ruleId   规则ID
     * @param eocMap   运营单元
     * @return
     * @throws Exception
     */
    @AllowAnonymous
    Object putEnable(String tenantId, String ruleId, EocMapDTO eocMap) throws Exception;

    /**
     * 失效侦测规则配置
     *
     * @param tenantId 租户ID
     * @param ruleId   规则ID
     * @param eocMap   运营单元
     * @return
     * @throws Exception
     */
    @AllowAnonymous
    Object putDisable(String tenantId, String ruleId, EocMapDTO eocMap) throws Exception;

    /**
     * 根据机制code查询侦测规则code
     *
     * @param mechanismCode 机制code
     * @return
     * @throws Exception
     */
    @AllowAnonymous
    Object getRuleCodeByMechanism(String mechanismCode) throws DWBusinessException;

    /**
     * 利用mechanismCode在neo4j中查询机制包含的ruleId，利用ruleId和product从monogdb的MonitorRuleProductConfig文档里查询侦测与对应产品的配置信息
     * 如果配置信息里的运营单元和operationUnit都为空或者相同，则保留ruldId
     *
     * @param mechanismCode 机制code
     * @param operationUnit 运营单元
     * @param product       产品
     * @return
     * @throws DWBusinessException
     */
    Object postRuleCodeByMechanismWithEoc(String mechanismCode, OperationUnit operationUnit, String product) throws DWBusinessException;

    /**
     * 根据ruleId查询侦测规则配置
     *
     * @param tenantId 租户ID
     * @param ruleId   规则ID
     * @return
     * @throws Exception
     */
    @AllowAnonymous
    Object get(String tenantId, String ruleId) throws Exception;

    /**
     * 根据ruleId查询侦测规则配置
     *
     * @param tenantId 租户ID
     * @param ruleIds  规则ID
     * @return
     * @throws Exception
     */
    @AllowAnonymous
    Object postEocChangeFromTenant(String tenantId, List<String> ruleIds) throws Exception;

    /**
     * 同步EOC变更信息
     *
     * @param monitorEocChange 监测EOC变更信息
     * @return
     * @throws Exception
     */
    @AllowAnonymous
    Object postEocChange(MonitorEocChangeVO monitorEocChange) throws Exception;

    /**
     * 根据ruleId查询产品名称
     *
     * @param ruleId 规则ID
     * @return
     * @throws Exception
     */
    Object getProductNameByMonitorRule(String ruleId) throws Exception;

    /**
     * 发送通知
     * @param config 侦测规则配置
     * @return
     * @throws Exception
     */
    Object postNotice(MonitorRuleConfigDTO config) throws Exception;

    /**
     * 根据条件删除侦测规则配置
     * @param config 侦测规则配置
     * @return
     * @throws Exception
     */
    @AllowAnonymous
    Object delete(MonitorRuleConfigDTO config) throws Exception;

    /**
     * 根据ruleId查询是否有正在执行的操作
     * @param ruleId 规则ID
     * @return
     * @throws Exception
     */
    Object getCheckExecutingAction(String ruleId) throws Exception;

    /**
     * 根据ruleId查询侦测规则配置
     *
     * @param tenantId      租户ID
     * @param ruleId        规则ID
     * @param operationUnit 运营单元
     * @param product       产品
     * @return
     * @throws Exception
     */
    @AllowAnonymous
    Object postFindMonitorRuleConfig(String tenantId, String ruleId, OperationUnit operationUnit, String product) throws Exception;

    /**
     * 根据ruleIdList查询侦测规则配置
     *
     * @param tenantId   租户ID
     * @param ruleIdList 规则ID列表
     * @return
     * @throws Exception
     */
    Object postFindMonitorRuleConfig(String tenantId, List<String> ruleIdList, OperationUnit operationUnit, String product) throws Exception;

    /**
     * 根据ruleIdList查询产品名称
     *
     * @param ruleIdList 规则ID列表
     * @return
     * @throws Exception
     */
    Object getProductNamesByMonitorRuleList(List<String> ruleIdList) throws Exception;

    /**
     * 接收侦测规则同步结果
     *
     * @param syncResult 同步结果
     * @return
     * @throws Exception
     */
    @AllowAnonymous
    Object postReceiveMonitorRuleSyncResult(MonitorRuleSyncResult syncResult) throws Exception;

    /**
     * 根据租户ID查询侦测规则code
     *
     * @return
     * @throws Exception
     */
    Object getCodeByTenant() throws Exception;

    /**
     * 侦测规则配置变更时发送通知
     *
     * @param noticeInfo 通知信息
     * @return
     * @throws Exception
     */
    @AllowAnonymous
    Object postNoticeWhenMonitorRuleConfigChange(MonitorNoticeVO noticeInfo) throws Exception;
}
