package com.digiwin.athena.knowledgegraph.service.impl;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import com.digiwin.app.container.exceptions.DWBusinessException;
import com.digiwin.athena.kg.authority.*;
import com.digiwin.athena.kg.authority.app.AuthorityAction;
import com.digiwin.athena.kmservice.locale.Lang;
import com.digiwin.athena.knowledgegraph.clients.CacService;
import com.digiwin.athena.knowledgegraph.clients.GMCService;
import com.digiwin.athena.domain.component.bo.ComponentBO;
import com.digiwin.athena.domain.component.bo.TenantAppComponentListBO;
import com.digiwin.athena.knowledgegraph.component.dao.TenantAppComponentListDAO;
import com.digiwin.athena.knowledgegraph.dao.ActivityDAO;
import com.digiwin.athena.knowledgegraph.domain.Task;
import com.digiwin.athena.kg.activity.Activity;
import com.digiwin.athena.repository.neo4j.Activity2Repository;
import com.digiwin.athena.knowledgegraph.service.DataMapService;
import com.digiwin.athena.knowledgegraph.service.IAuthorityService;
import com.digiwin.athena.knowledgegraph.utils.I18nUtils;
import com.digiwin.athena.knowledgegraph.utils.TranslateUtils;
import com.digiwin.athena.kg.authority.app.AppDetailInfo;
import com.luhuiguo.chinese.ChineseUtils;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.data.mongodb.core.MongoTemplate;
import org.springframework.data.mongodb.core.query.Criteria;
import org.springframework.data.mongodb.core.query.Query;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;
import org.springframework.util.ObjectUtils;

import java.security.SecureRandom;
import java.util.*;
import java.util.stream.Collectors;

@Lang
@Service
@Slf4j
public class AuthorityService implements IAuthorityService {

    private static final String REPORT_CODE = "report";
    private static final String REPORT_NAME = "报表";
    private static final String BASIC_DATA_ENTRY_CODE = "basicDataEntry";
    private static final String BASIC_DATA_ENTRY_NAME = "业务数据录入";
    private static final String START_PROJECT_CODE = "startProject";
    private static final String START_PROJECT_NAME = "发起项目";
    @Autowired
    private GMCService gmcService;
    @Autowired
    private CacService cacService;
    @Autowired
    private IamService iamService;

    @Autowired
    @Qualifier("knowledgegraphSystem")
    MongoTemplate mongoTemplate;
    @Autowired
    DataMapService dataMapService;
    @Autowired
    Activity2Repository activity2Repository;

    @Autowired
    @Qualifier("knowledgegraphTenant")
    MongoTemplate mongoTemplateUser;

    @Autowired
    TranslateUtils translateUtils;

    @Autowired
    private TenantAppComponentListDAO tenantAppComponentListDAO;

    @Autowired
    private ActivityDAO activityDAO;

    @Override
    public Object postStartProjectAuthorityByAppCode(String appCode) throws Exception {
        Query query = Query.query(Criteria.where("authorityPrefix").is(appCode + ":" + START_PROJECT_CODE));
        List<Task> taskList = mongoTemplate.find(query, Task.class,"taskConfigs");
        try{
            Map<String,Object> query1 = new HashMap<>();
            query1.put("authorityPrefix",appCode + ":" + START_PROJECT_CODE);
            List datamapTasks = dataMapService.getProjectsByAuthorityPrefix(appCode + ":" + START_PROJECT_CODE);
            if(!CollectionUtils.isEmpty(datamapTasks)){
                datamapTasks.forEach(task->{
                    Task dmtask = JSON.parseObject(JSON.toJSONString(task),Task.class);
                    taskList.add(dmtask);
                });
            }
        }catch(Exception e0){}

        if(CollectionUtils.isEmpty(taskList)){
            throw new DWBusinessException(I18nUtils.getValue("knowledgegraph.projectNotFound"));
        }
        AuthorityConfigVO configVO = this.generateToStartProjectAuthorityConfig(appCode, null, taskList);
        Object result = this.postGenerateAuthority(configVO);
        return result;
    }


    @Override
    public Object postStatementAuthorityByAppCode(String appCode) throws Exception {
        Query query = Query.query(Criteria.where("authorityPrefix").is(appCode + ":" + REPORT_CODE));
        List<Activity> defaultActivityList = mongoTemplate.find(query, Activity.class,"activityConfigs");
        if(CollectionUtils.isEmpty(defaultActivityList)){
            throw new DWBusinessException(I18nUtils.getValue("knowledgegraph.statementNotFound"));
        }
        AuthorityConfigVO configVO = this.generateToAuthorityConfig(appCode, null, REPORT_CODE, REPORT_NAME, defaultActivityList);
        Object result = this.postGenerateAuthority(configVO);
        return result;
    }

    @Override
    public Object postStatementAuthorityByCode(String code) throws Exception {
        Query query = Query.query(Criteria.where("code").is(code));
        Activity defaultActivity = mongoTemplate.findOne(query, Activity.class,"activityConfigs");
        if(StringUtils.isEmpty(defaultActivity.getAuthorityPrefix())){
            throw new DWBusinessException(I18nUtils.getValue("knowledgegraph.statementNotConfig"));
        }
        String[] split = defaultActivity.getAuthorityPrefix().split(":");
        if(split.length != 2 || !"report".equals(split[1])){
            throw new DWBusinessException(I18nUtils.getValue("knowledgegraph.statementConfigError"));
        }
        List<Activity> defaultActivityList = new ArrayList<>();
        defaultActivityList.add(defaultActivity);

        AuthorityConfigVO configVO = this.generateToAuthorityConfig(split[0], null, split[1], REPORT_NAME, defaultActivityList);
        Object result = this.postGenerateAuthority(configVO);
        return result;
    }

    @Override
    public Object postGenerateBasicDataAuthorityByAppCode(String appCode) throws Exception{
        Query query = Query.query(Criteria.where("authorityPrefix").is(appCode+":" + BASIC_DATA_ENTRY_CODE));
        List<Activity> defaultActivityList = mongoTemplate.find(query, Activity.class,"activityConfigs");
        if(CollectionUtils.isEmpty(defaultActivityList)){
            throw new DWBusinessException(I18nUtils.getValue("knowledgegraph.baseDataNotFound"));
        }
        AuthorityConfigVO configVO = this.generateToAuthorityConfig(appCode, null, BASIC_DATA_ENTRY_CODE, BASIC_DATA_ENTRY_NAME, defaultActivityList);
        Object result = this.postGenerateAuthority(configVO);
        return result;
    }


    @Override
    public Object postGenerateBasicDataAuthorityByCode(String code) throws Exception{
        Query query = Query.query(Criteria.where("code").is(code));
        Activity defaultActivity = mongoTemplate.findOne(query, Activity.class,"activityConfigs");
        if(StringUtils.isEmpty(defaultActivity.getAuthorityPrefix())){
            throw new DWBusinessException(I18nUtils.getValue("knowledgegraph.baseDataNotConfig"));
        }
        String[] split = defaultActivity.getAuthorityPrefix().split(":");
        if(split.length != 2 || !"basicDataEntry".equals(split[1])){
            throw new DWBusinessException(I18nUtils.getValue("knowledgegraph.baseDataConfigError"));
        }
        List<Activity> defaultActivityList = new ArrayList<>();
        defaultActivityList.add(defaultActivity);

        AuthorityConfigVO configVO = this.generateToAuthorityConfig(split[0], null, split[1], BASIC_DATA_ENTRY_NAME, defaultActivityList);
        Object result = this.postGenerateAuthority(configVO);
        return result;
    }

    private AuthorityConfigVO generateToReportAuthorityConfig(String appCode, String appName,  List<Activity> activityList){
        AuthorityConfigVO configVO = new AuthorityConfigVO();

        AppVO app = new AppVO();
        app.setId(appCode);
        app.setName(StringUtils.isEmpty(appName) ? appCode : appName);
        configVO.setApp(app);

        configVO.setType("UPDATE");

        List<ModuleVO> moduleVOList = new ArrayList<>();
        ModuleVO moduleVO = new ModuleVO();
        moduleVO.setId(REPORT_CODE);
        moduleVO.setName(REPORT_NAME);
        moduleVOList.add(moduleVO);
        configVO.setModules(moduleVOList);

        List<ActionVO> actionVOList = new ArrayList<>();
        for(Activity activity : activityList) {
            ActionVO actionVO = new ActionVO();
            actionVO.setId(activity.getCode());
            actionVO.setModuleId(REPORT_CODE);
            actionVO.setName(activity.getName());

            List<ConditionVO> conditionVOList = new ArrayList<>();

            ConditionVO conditionVO = new ConditionVO();
            conditionVO.setKey("enable");
            conditionVO.setName("报表启用");
            conditionVO.setTypeParameter(this.getTypeParameter());
            conditionVO.setDefaultValue(this.getTypeParameter().get(0));

            conditionVOList.add(conditionVO);

            if(isABIReport(activity.getCode())){
                ConditionVO conditionVO2 = new ConditionVO();
                conditionVO2.setKey("design_report");
                conditionVO2.setName("设计报表");
                conditionVO2.setTypeParameter(this.getTypeParameter());
                conditionVO2.setDefaultValue(this.getTypeParameter().get(0));
                conditionVOList.add(conditionVO2);
            }

            actionVO.setCondition(conditionVOList);

            actionVOList.add(actionVO);
        }
        configVO.setActions(actionVOList);
        return  configVO;
    }

    /**
     * 判断是否为易信报表
     */
    private boolean isABIReport(String activityCode){
        //category 为ABI-STATEMENT为易信报表
        Activity oneByCode = this.activityDAO.findOneByCode(activityCode);
        if(oneByCode != null && StringUtils.equals(oneByCode.getCategory(), "ABI-STATEMENT")){
            return true;
        }
        return false;
    }

    private AuthorityConfigVO generateToStartProjectAuthorityConfig(String appCode, String appName,  List<Task> taskList){
        AuthorityConfigVO configVO = new AuthorityConfigVO();

        AppVO app = new AppVO();
        app.setId(appCode);
        app.setName(StringUtils.isEmpty(appName) ? appCode : appName);
        configVO.setApp(app);

        configVO.setType("UPDATE");

        List<ModuleVO> moduleVOList = new ArrayList<>();
        ModuleVO moduleVO = new ModuleVO();
        moduleVO.setId(START_PROJECT_CODE);
        moduleVO.setName(START_PROJECT_NAME);
        moduleVOList.add(moduleVO);
        configVO.setModules(moduleVOList);

        List<ActionVO> actionVOList = new ArrayList<>();
        for(Task task : taskList) {
            ActionVO actionVO = new ActionVO();
            actionVO.setId(task.getCode());
            actionVO.setModuleId(START_PROJECT_CODE);
            actionVO.setName(task.getName());

            List<ConditionVO> conditionVOList = new ArrayList<>();

            ConditionVO conditionVO = new ConditionVO();
            conditionVO.setKey("enable");
            conditionVO.setName("项目启用");
            conditionVO.setTypeParameter(this.getTypeParameter());
            conditionVO.setDefaultValue(this.getTypeParameter().get(0));

            conditionVOList.add(conditionVO);
            actionVO.setCondition(conditionVOList);

            actionVOList.add(actionVO);
        }
        configVO.setActions(actionVOList);
        return  configVO;
    }

    private AuthorityConfigVO generateToBasicDataEntryAuthorityConfig(String appCode, String appName,  List<Activity> activityList){
        AuthorityConfigVO configVO = new AuthorityConfigVO();
        AppVO app = new AppVO();
        app.setId(appCode);
        app.setName(StringUtils.isEmpty(appName) ? appCode : appName);
        configVO.setApp(app);

        configVO.setType("UPDATE");

        List<ModuleVO> moduleVOList = new ArrayList<>();
        ModuleVO moduleVO = new ModuleVO();
        moduleVO.setId(BASIC_DATA_ENTRY_CODE);
        moduleVO.setName(BASIC_DATA_ENTRY_NAME);
        moduleVOList.add(moduleVO);
        configVO.setModules(moduleVOList);

        List<ActionVO> actionVOList = new ArrayList<>();
        for(Activity activity : activityList) {
            ActionVO actionVO = new ActionVO();
            actionVO.setId(activity.getCode());
            actionVO.setModuleId(BASIC_DATA_ENTRY_CODE);
            actionVO.setName(activity.getName());

            List<ConditionVO> conditionVOList = new ArrayList<>();

            ConditionVO conditionVO = new ConditionVO();
            conditionVO.setKey("enable");
            conditionVO.setName("启用");
            conditionVO.setTypeParameter(this.getTypeParameter());
            conditionVO.setDefaultValue(this.getTypeParameter().get(0));
            conditionVOList.add(conditionVO);


            Map<String, ConditionVO> conditionVOMap = new HashMap<>();
            Map<String, Object> pages = activity.getPages();
            if(pages != null){
                JSONObject pagesJson = new JSONObject(pages);
                JSONObject basicDataJson = pagesJson.getJSONObject("basic-data");
                if(basicDataJson != null){
                    JSONArray operationsArray = basicDataJson.getJSONArray("operations");
                    if (operationsArray != null) {
                        for (int i = 0; i < operationsArray.size(); i++) {
                            JSONObject operJson = operationsArray.getJSONObject(i);
                            if (StringUtils.isNotEmpty(operJson.getString("type"))) {
                                this.setConditionVo(operJson.getString("type"), operJson.getString("title"), conditionVOMap);
                            }
                        }
                    }

                    JSONArray submitActionsArray = basicDataJson.getJSONArray("submitActions");
                    if (submitActionsArray != null) {
                        for (int j = 0; j < submitActionsArray.size(); j++) {
                            JSONObject submitActionsObject = submitActionsArray.getJSONObject(j);
                            if (StringUtils.isNotEmpty(submitActionsObject.getString("actionType"))) {
                                this.setConditionVo(submitActionsObject.getString("actionType"), submitActionsObject.getString("title"), conditionVOMap);
                            }
                        }
                    }
                }
            }

            if(conditionVOMap != null && conditionVOMap.size() > 0){
                Set<Map.Entry<String, ConditionVO>> entries = conditionVOMap.entrySet();
                Iterator<Map.Entry<String, ConditionVO>> iterator = entries.iterator();
                while(iterator.hasNext()){
                    Map.Entry<String, ConditionVO> next = iterator.next();
                    conditionVOList.add(next.getValue());
                }
            }

            actionVO.setCondition(conditionVOList);

            actionVOList.add(actionVO);
        }
        configVO.setActions(actionVOList);
        return  configVO;
    }

    private void setConditionVo(String key, String title, Map<String, ConditionVO> conditionVOMap){
        if(conditionVOMap.containsKey(key)){
            ConditionVO conditionVO = conditionVOMap.get(key);
            conditionVO.setName(conditionVO.getName() + "," + title);
            conditionVOMap.put(key, conditionVO);
        }else{
            ConditionVO conditionVO = new ConditionVO();
            conditionVO.setKey(key);
            conditionVO.setName(title);
            conditionVO.setTypeParameter(this.getTypeParameter());
            conditionVO.setDefaultValue(this.getTypeParameter().get(0));
            conditionVOMap.put(key, conditionVO);
        }
    }


    private AuthorityConfigVO generateToAuthorityConfig(String appCode, String appName, String moduleCode, String moduleName, List<Activity> activityList){
        if(BASIC_DATA_ENTRY_CODE.equals(moduleCode)){
            return this.generateToBasicDataEntryAuthorityConfig(appCode, appName, activityList);
        }else if(REPORT_CODE.equals(moduleCode)){
            return this.generateToReportAuthorityConfig(appCode, appName, activityList);
        }
        return null;
//        AuthorityConfigVO configVO = new AuthorityConfigVO();
//        configVO.getApp().setId(appCode);
//        configVO.getApp().setName(StringUtils.isEmpty(appName) ? appCode : appName);
//
//        configVO.setType("UPDATE");
//
//        List<ModuleVO> moduleVOList = new ArrayList<>();
//        ModuleVO moduleVO = new ModuleVO();
//        moduleVO.setId(moduleCode);
//        moduleVO.setName(moduleName);
//        moduleVOList.add(moduleVO);
//        configVO.setModules(moduleVOList);
//
//        List<ActionVO> actionVOList = new ArrayList<>();
//        for(Activity activity : activityList) {
//            ActionVO actionVO = new ActionVO();
//            actionVO.setId(activity.getCode());
//            actionVO.setModuleId(moduleCode);
//            actionVO.setName(activity.getName());
//
//            List<ConditionVO> conditionVOList = new ArrayList<>();
//            Map<String, Object> pages = activity.getPages();
//            if(pages == null){
//                continue;
//            }
//            JSONObject pagesJson = new JSONObject(pages);
//            JSONObject basicDataJson = pagesJson.getJSONObject("basic-data");
//            if(basicDataJson == null){
//                continue;
//            }
//            JSONArray operationsArray = basicDataJson.getJSONArray("operations");
//            if (operationsArray != null) {
//                for (int i = 0; i < operationsArray.size(); i++) {
//                    JSONObject operJson = operationsArray.getJSONObject(i);
//                    if (StringUtils.isNotEmpty(operJson.getString("key"))) {
//                        ConditionVO conditionVO = new ConditionVO();
//                        conditionVO.setKey(operJson.getString("key"));
//                        conditionVO.setName(operJson.getString("title"));
//                        conditionVO.setTypeParameter(this.getTypeParameter());
//                        conditionVO.setDefaultValue(this.getTypeParameter().get(0));
//                        conditionVOList.add(conditionVO);
//                    }
//                }
//            }
//
//            JSONArray submitActionsArray = basicDataJson.getJSONArray("submitActions");
//            if (submitActionsArray != null) {
//                for (int j = 0; j < submitActionsArray.size(); j++) {
//                    JSONObject submitActionsObject = submitActionsArray.getJSONObject(j);
//                    if (StringUtils.isNotEmpty(submitActionsObject.getString("key"))) {
//                        ConditionVO conditionVO = new ConditionVO();
//                        conditionVO.setKey(submitActionsObject.getString("key"));
//                        conditionVO.setName(submitActionsObject.getString("title"));
//                        conditionVO.setTypeParameter(this.getTypeParameter());
//                        conditionVO.setDefaultValue(this.getTypeParameter().get(0));
//                        conditionVOList.add(conditionVO);
//                    }
//                }
//            }
//            actionVO.setCondition(conditionVOList);
//
//            actionVOList.add(actionVO);
//        }
//        configVO.setActions(actionVOList);
//        return  configVO;
    }

//    private AuthorityConfigVO generateToAuthorityConfig(String appCode, String appName, String moduleCode, String moduleName, Activity activity){
//        AuthorityConfigVO configVO = new AuthorityConfigVO();
//        configVO.getApp().setId(appCode);
//        configVO.getApp().setName(StringUtils.isEmpty(appName) ? appCode : appName);
//
//        configVO.setType("UPDATE");
//
//        List<ModuleVO> moduleVOList = new ArrayList<>();
//        ModuleVO moduleVO = new ModuleVO();
//        moduleVO.setId(moduleCode);
//        moduleVO.setName(moduleName);
//        moduleVOList.add(moduleVO);
//        configVO.setModules(moduleVOList);
//
//        List<ActionVO> actionVOList = new ArrayList<>();
//        ActionVO actionVO = new ActionVO();
//        actionVO.setId(activity.getCode());
//        actionVO.setModuleId(moduleCode);
//        actionVO.setName(activity.getName());
//
//        List<ConditionVO> conditionVOList = new ArrayList<>();
//        Map<String, Object> pages = activity.getPages();
//        JSONObject pagesJson = new JSONObject(pages);
//        JSONObject basicDataJson = pagesJson.getJSONObject("basic-data");
//        JSONArray operationsArray = basicDataJson.getJSONArray("operations");
//        if(operationsArray != null){
//            for(int i = 0; i<operationsArray.size(); i++){
//                JSONObject operJson = operationsArray.getJSONObject(i);
//                if(StringUtils.isNotEmpty(operJson.getString("key"))){
//                    ConditionVO conditionVO = new ConditionVO();
//                    conditionVO.setKey(operJson.getString("key"));
//                    conditionVO.setName(operJson.getString("title"));
//                    conditionVO.setTypeParameter(this.getTypeParameter());
//                    conditionVO.setDefaultValue(this.getTypeParameter().get(0));
//                    conditionVOList.add(conditionVO);
//                }
//            }
//        }
//
//        JSONArray submitActionsArray = basicDataJson.getJSONArray("submitActions");
//        if(submitActionsArray != null){
//            for(int j=0; j< submitActionsArray.size(); j++){
//                JSONObject submitActionsObject = submitActionsArray.getJSONObject(j);
//                if(StringUtils.isNotEmpty(submitActionsObject.getString("key"))){
//                    ConditionVO conditionVO = new ConditionVO();
//                    conditionVO.setKey(submitActionsObject.getString("key"));
//                    conditionVO.setName(submitActionsObject.getString("title"));
//                    conditionVO.setTypeParameter(this.getTypeParameter());
//                    conditionVO.setDefaultValue(this.getTypeParameter().get(0));
//                    conditionVOList.add(conditionVO);
//                }
//            }
//        }
//        actionVO.setCondition(conditionVOList);
//
//        actionVOList.add(actionVO);
//        configVO.setActions(actionVOList);
//        return  configVO;
//    }


    private List<String> getTypeParameter(){
        List<String> result = new ArrayList<>();
        result.add("allow");
        result.add("forbidden");
        return result;
    }


    @Override
    public Object postGenerateAuthority(AuthorityConfigVO config) throws Exception {
        log.info("GenerateAuthority config:{}", JSON.toJSONString(config));
        this.check(config);

        Object cloudGood = this.gmcService.getCloudGood(config.getApp().getId());
        if(cloudGood == null){
            throw new DWBusinessException(I18nUtils.getValue("knowledgegraph.goodsNotFound") + config.getApp().getId());
        }
        JSONObject goodObj = JSON.parseObject(JSON.toJSONString(cloudGood));
        log.info("getCloudGood:{}", goodObj);
        JSONObject modules = goodObj.getJSONObject("modules");
        if(modules == null || (StringUtils.isEmpty(modules.getString("appToken")) && StringUtils.isEmpty(config.getAppToken()))){
            throw new DWBusinessException(I18nUtils.getValue("knowledgegraph.appTokenMiss"));
        }

        this.upsertAuthorityConfig(goodObj, config);
        log.info("toUpdateGoodsInfo:{}", goodObj);

        Object updateCloudGoodModulesRequestObj = this.getUpdateCloudGoodModulesRequestObj(goodObj);
        Object updateCloudGoodModulesResultObj = this.gmcService.updateCloudGoodModules(updateCloudGoodModulesRequestObj);
        JSONObject updateCloudGoodModulesResult = JSON.parseObject(JSON.toJSONString(updateCloudGoodModulesResultObj));
        if(!updateCloudGoodModulesResult.getBoolean("success")){
            throw new DWBusinessException(I18nUtils.getValue("knowledgegraph.goodsUpdateFail",config.getApp().getId()));
        }

        //modify by srj 以下流程直接通过updateCloudGoodModules一步完成，省去上下架的步骤
//        Object goodOffResultObj = this.gmcService.goodOff(config.getApp().getId());
//        JSONObject goodOffResult = JSON.parseObject(JSON.toJSONString(goodOffResultObj));
//        if(!goodOffResult.getBoolean("success")){
//            throw new DWBusinessException("商品("+ config.getApp().getId() +")下架失败");
//        }
//
//        Object updateResult = this.gmcService.updateCloudGood(goodObj);
//        log.info("cloudGoodUpdateResult:{}", updateResult);
//        JSONObject updateResultJson = JSON.parseObject(JSON.toJSONString(updateResult));
//        if(!StringUtils.equals(updateResultJson.getString("code"), "200")){
//            throw new DWBusinessException("更新商品信息（updateCloudGood）失败,result：" + updateResult);
//        }
//
//        Object multiLanguageReq = this.generateMultiLanguageRequest(goodObj);
//        Object multiLanguageResponse = this.gmcService.multiLanguage(multiLanguageReq);
//        JSONObject multiLanguageResponseObj = JSON.parseObject(JSON.toJSONString(multiLanguageResponse));
//        if(!multiLanguageResponseObj.getBoolean("success")){
//            throw new DWBusinessException("商品("+ config.getApp().getId() +")更新zh-CN多语言失败");
//        }
//
//        Object multiLanguageReqTW = this.generateMultiLanguageRequestTW(goodObj);
//        Object multiLanguageResponseTW = this.gmcService.multiLanguage(multiLanguageReqTW);
//        JSONObject multiLanguageResponseObjTW = JSON.parseObject(JSON.toJSONString(multiLanguageResponseTW));
//        if(!multiLanguageResponseObjTW.getBoolean("success")){
//            throw new DWBusinessException("商品("+ config.getApp().getId() +")更新zh-TW多语言失败");
//        }
//
//        Object goodOnResultObj = this.gmcService.applyGoodOn(config.getApp().getId());
//        JSONObject goodOnResult = JSON.parseObject(JSON.toJSONString(goodOnResultObj));
//        if(!goodOnResult.getBoolean("success")){
//            throw new DWBusinessException("商品("+ config.getApp().getId() +")申请上架失败");
//        }

        return "success";
    }

    private Object getUpdateCloudGoodModulesRequestObj(JSONObject goodObj) {
        Map<String, Object> result = new HashMap<>();
        result.put("code" , goodObj.getString("code"));

        List<Object> multiLanguageResourceList = new ArrayList<>();
        multiLanguageResourceList.add(this.generateMultiLanguageRequest(goodObj));
        multiLanguageResourceList.add(this.generateMultiLanguageRequestTW(goodObj));
        result.put("multiLanguageResource" , multiLanguageResourceList);

        result.put("modules" , goodObj.getJSONObject("modules"));

        return result;
    }

    private Object generateMultiLanguageRequest(JSONObject goodObj){
        Map<String, Object> result = new HashMap<>();
        result.put("dataSid", goodObj.getString("id"));
        result.put("language", "zh-CN");
        List<Map<String, String>> dataList = new ArrayList<>();
        Map<String, String> goodsNameMap = new HashMap<>();
        goodsNameMap.put("fieldName", "goodsName");
        goodsNameMap.put("content", ChineseUtils.toSimplified(goodObj.getString("displayName")));

        Map<String, String> modulesMap = new HashMap<>();
        modulesMap.put("fieldName", "modules");
        modulesMap.put("content", ChineseUtils.toSimplified(JSON.toJSONString(goodObj.getJSONObject("modules"))));

        Map<String, String> customunitMap = new HashMap<>();
        customunitMap.put("fieldName", "customunit");
        customunitMap.put("content", ChineseUtils.toSimplified(goodObj.getString("customUnit")));

        dataList.add(goodsNameMap);
        dataList.add(modulesMap);
        dataList.add(customunitMap);

        result.put("data", dataList);

        return result;
    }

    private Object generateMultiLanguageRequestTW(JSONObject goodObj){
        Map<String, Object> result = new HashMap<>();
        result.put("dataSid", goodObj.getString("id"));
        result.put("language", "zh-TW");
        List<Map<String, String>> dataList = new ArrayList<>();
        Map<String, String> goodsNameMap = new HashMap<>();
        goodsNameMap.put("fieldName", "goodsName");
//        goodsNameMap.put("content", ChineseUtils.toTraditional(goodObj.getString("displayName")));
        goodsNameMap.put("content", translateUtils.translateContentToTw(goodObj.getString("displayName")));

        Map<String, String> modulesMap = new HashMap<>();
        modulesMap.put("fieldName", "modules");
//        modulesMap.put("content", ChineseUtils.toTraditional(JSON.toJSONString(goodObj.getJSONObject("modules"))));
        modulesMap.put("content", translateUtils.translateContentToTw(goodObj.getJSONObject("modules")));

        Map<String, String> customunitMap = new HashMap<>();
        customunitMap.put("fieldName", "customunit");
//        customunitMap.put("content", ChineseUtils.toTraditional(goodObj.getString("customUnit")));
        customunitMap.put("content", translateUtils.translateContentToTw(goodObj.getString("customUnit")));

        dataList.add(goodsNameMap);
        dataList.add(modulesMap);
        dataList.add(customunitMap);

        result.put("data", dataList);

        return result;
    }

    private void upsertAuthorityConfig(JSONObject sourceObj,  AuthorityConfigVO config){
        this.makeUpConfig(config);
        JSONObject modules = sourceObj.getJSONObject("modules");
        if(modules == null){
            JSONObject configObj = JSON.parseObject(JSON.toJSONString(config));
            configObj.remove("type");
            sourceObj.put("modules", configObj);
        }else{
           if("ADD".equals(config.getType())) {
               modules.put("modules", JSONArray.parseArray(JSON.toJSONString(config.getModules())));
               modules.put("actions", JSONArray.parseArray(JSON.toJSONString(config.getActions())));
           }else if("UPDATE".equals(config.getType())){
               JSONArray modulesArray = modules.getJSONArray("modules");
               Map<String, List<ModuleVO>> collect = config.getModules().stream().collect(Collectors.groupingBy(e -> e.getId()));
               List<ModuleVO> moduleToAddList = new ArrayList<>();
               for(int i = 0 ;i<modulesArray.size(); i++){
                   JSONObject jsonObject = modulesArray.getJSONObject(i);
                   String moduleId = jsonObject.getString("id");
                   if (!CollectionUtils.isEmpty(collect.get(moduleId))) {
                       jsonObject.put("name",collect.get(moduleId).get(0).getName());
                   }
                   collect.remove(moduleId);
               }
               for(Map.Entry<String, List<ModuleVO>> entry : collect.entrySet()){
                   moduleToAddList.addAll(entry.getValue());
               }
               if(!CollectionUtils.isEmpty(moduleToAddList)){
                   modulesArray.addAll(moduleToAddList);
               }
               //处理action
               JSONArray actionsArray = modules.getJSONArray("actions");
               Map<String, List<ActionVO>> collect1 = config.getActions().stream().collect(Collectors.groupingBy(e -> e.getId() + "###" + e.getModuleId()));
               List<ActionVO> actionsToAddList = new ArrayList<>();
               for(int i = 0 ;i<actionsArray.size(); i++){
                   JSONObject jsonObject = actionsArray.getJSONObject(i);
                   String actionBelongModuleId = jsonObject.getString("moduleId");
                   String actionId = jsonObject.getString("id");
                   List<ActionVO> removeAction = collect1.remove(actionId + "###" + actionBelongModuleId);
                   if(!CollectionUtils.isEmpty(removeAction)){
                       ActionVO actionVo = removeAction.get(0);
                       jsonObject.put("name",actionVo.getName());
                       JSONArray conditionArray = jsonObject.getJSONArray("condition");
                       List<ConditionVO> conditionsToAddList = new ArrayList<>();
                       Map<String, List<ConditionVO>> collect2 = removeAction.get(0).getCondition().stream().collect(Collectors.groupingBy(e -> e.getKey()));
                       for(int j=0; j<conditionArray.size(); j++){
                           JSONObject conditionObj = conditionArray.getJSONObject(j);
                           String conditionKey = conditionObj.getString("key");
                           List<ConditionVO> removeCondition = collect2.remove(conditionKey);
                           if(!CollectionUtils.isEmpty(removeCondition)){
                               ConditionVO conditionVO = removeCondition.get(0);
                               conditionObj.put("name", conditionVO.getName());
                               conditionObj.put("type", conditionVO.getType());
                               conditionObj.put("typeParameter", conditionVO.getTypeParameter());
                               conditionObj.put("parameter", conditionVO.getParameter());
                               conditionObj.put("defaultValue", conditionVO.getDefaultValue());
                           }
                       }
                       for(Map.Entry<String, List<ConditionVO>> entry : collect2.entrySet()){
                           conditionsToAddList.addAll(entry.getValue());
                       }
                       if(!CollectionUtils.isEmpty(conditionsToAddList)){
                        conditionArray.addAll(conditionsToAddList);
                       }
                   }
               }
               for(Map.Entry<String, List<ActionVO>> entry : collect1.entrySet()){
                   actionsToAddList.addAll(entry.getValue());
               }
               if(!CollectionUtils.isEmpty(actionsToAddList)){
                   actionsArray.addAll(actionsToAddList);
               }
           }else if("DELETE_MODULE".equals(config.getType())){ //删除整个模组级模组下的信息，包含action和action下的condition
               this.deleteModule(modules, config);
           }else if("DELETE_ACTION".equals(config.getType())){ //删除指定Aciton的信息 包含action下的condition
               this.deleteAction(modules, config);
           }else if("DELETE_CONDITION".equals(config.getType())){ //删除对应condition的信息
               this.deleteCondition(modules, config);
           }

        }

    }

    private void deleteModule(JSONObject modules, AuthorityConfigVO config) {
        List<ModuleVO> configModules = config.getModules();
        if(CollectionUtils.isEmpty(configModules)){
            return;
        }
        Set<String> configModuleIdSet = configModules.stream().filter(e->StringUtils.isNotEmpty(e.getId())).map(ModuleVO::getId).collect(Collectors.toSet());
        if(CollectionUtils.isEmpty(configModuleIdSet)){
            return;
        }

        JSONArray modulesArray = modules.getJSONArray("modules");
        if(!ObjectUtils.isEmpty(modulesArray)){
            Iterator<Object> iterator = modulesArray.iterator();
            while(iterator.hasNext()){
                Object next = iterator.next();
                JSONObject jsonObject = JSON.parseObject(JSON.toJSONString(next));
                if(jsonObject != null && configModuleIdSet.contains(jsonObject.getString("id"))){
                    iterator.remove();
                }
            }
        }

        JSONArray actionsArray = modules.getJSONArray("actions");
        if(!ObjectUtils.isEmpty(actionsArray)){
            Iterator<Object> iterator = actionsArray.iterator();
            while(iterator.hasNext()){
                Object next = iterator.next();
                JSONObject jsonObject = JSON.parseObject(JSON.toJSONString(next));
                if(jsonObject != null && configModuleIdSet.contains(jsonObject.getString("moduleId"))){
                    iterator.remove();
                }
            }
        }

    }

    private void deleteAction(JSONObject modules, AuthorityConfigVO config) {
        List<ActionVO> configActions = config.getActions();
        if(CollectionUtils.isEmpty(configActions)){
            return;
        }
        Set<String> configModuleAndActionIdSet = configActions.stream().filter(e->StringUtils.isNotEmpty(e.getId()) && StringUtils.isNotEmpty(e.getModuleId()))
                .map(e->e.getModuleId() + "###" + e.getId()).collect(Collectors.toSet());
        if(CollectionUtils.isEmpty(configModuleAndActionIdSet)){
            return;
        }

        JSONArray actionsArray = modules.getJSONArray("actions");
        if(!ObjectUtils.isEmpty(actionsArray)){
            Iterator<Object> iterator = actionsArray.iterator();
            while(iterator.hasNext()){
                Object next = iterator.next();
                JSONObject jsonObject = JSON.parseObject(JSON.toJSONString(next));
                if(jsonObject != null && configModuleAndActionIdSet.contains(jsonObject.getString("moduleId") + "###" + jsonObject.getString("id"))){
                    iterator.remove();
                }
            }
        }
    }

    private void deleteCondition(JSONObject modules, AuthorityConfigVO config) {
        List<ActionVO> configActions = config.getActions();
        if(CollectionUtils.isEmpty(configActions)){
            return;
        }
        Map<String, List<ActionVO>> collect1 = configActions.stream().collect(Collectors.groupingBy(e -> e.getModuleId() + "###" + e.getId()));
        if(CollectionUtils.isEmpty(collect1)){
            return;
        }

        JSONArray actionsArray = modules.getJSONArray("actions");
        if(!ObjectUtils.isEmpty(actionsArray)){
            for(int i= 0; i<actionsArray.size(); i++){
                JSONObject jsonObject = actionsArray.getJSONObject(i);
                String key = jsonObject.getString("moduleId") + "###" + jsonObject.getString("id");
                List<ActionVO> actionVOList = collect1.get(key);
                if(CollectionUtils.isEmpty(actionVOList)){
                    continue;
                }
                Set<String> collect = actionVOList.get(0).getCondition().stream()
                        .filter(e -> StringUtils.isNotEmpty(e.getKey())).map(ConditionVO::getKey).collect(Collectors.toSet());
                JSONArray conditionArray = jsonObject.getJSONArray("condition");
                Iterator<Object> iterator = conditionArray.iterator();
                while(iterator.hasNext()){
                    Object next = iterator.next();
                    JSONObject jsonObject1 = JSON.parseObject(JSON.toJSONString(next));
                    if(jsonObject1 != null && collect.contains(jsonObject1.getString("key"))){
                        iterator.remove();
                    }
                }
            }
        }
    }

    private void makeUpConfig(AuthorityConfigVO config){
        //补充app中信息
        String id = config.getApp().getId();
        config.getApp().setCategoryId(id);
        config.getApp().setIdFirst(id);

        //补充condition中信息
        for(ActionVO actionVO : config.getActions()){
            String name = actionVO.getName();
            String actionVOId = actionVO.getId();
            for(ConditionVO conditionVO : actionVO.getCondition()){
                conditionVO.setSid(this.getSid());
                conditionVO.setType("single");
                conditionVO.setParameter(JSON.toJSONString(conditionVO.getTypeParameter()));
                conditionVO.setActionName(name);
                conditionVO.setActionId(actionVOId);
            }
        }
    }

    private static String getSid() {
        StringBuilder sb = new StringBuilder();
        //产生16位的强随机数
        Random rd = new SecureRandom();
        for (int i = 0; i < 16; i++) {
            sb.append(rd.nextInt(10));
        }
        return sb.toString();
    }

    private void check(AuthorityConfigVO config) throws Exception{
        if(config == null){
            throw new DWBusinessException(I18nUtils.getValue("knowledgegraph.authConfigMiss"));
        }

        AppVO app = config.getApp();
        if(app == null || StringUtils.isEmpty(app.getId()) || StringUtils.isEmpty(app.getName())){
            throw new DWBusinessException(I18nUtils.getValue("knowledgegraph.appInfoMiss"));
        }

        if(!"ADD".equals(config.getType()) &&  !"UPDATE".equals(config.getType())
                &&  !"DELETE_MODULE".equals(config.getType()) &&  !"DELETE_ACTION".equals(config.getType()) &&  !"DELETE_CONDITION".equals(config.getType()) ){
            throw new DWBusinessException(I18nUtils.getValue("knowledgegraph.typeError"));
        }

        if("ADD".equals(config.getType()) || "UPDATE".equals(config.getType())) {
            List<ModuleVO> modules = config.getModules();
            if (CollectionUtils.isEmpty(modules)) {
                throw new DWBusinessException(I18nUtils.getValue("knowledgegraph.modulesEmpty"));
            }
            for (ModuleVO e : modules) {
                if (StringUtils.isEmpty(e.getId()) || StringUtils.isEmpty(e.getName())) {
                    throw new DWBusinessException(I18nUtils.getValue("knowledgegraph.moduleInfoMiss"));
                }
            }

            List<ActionVO> actions = config.getActions();
            if (CollectionUtils.isEmpty(actions)) {
                throw new DWBusinessException(I18nUtils.getValue("knowledgegraph.actionsEmpty"));
            }
            for (ActionVO actionVO : actions) {
                if (StringUtils.isEmpty(actionVO.getId()) || StringUtils.isEmpty(actionVO.getName())) {
                    throw new DWBusinessException(I18nUtils.getValue("knowledgegraph.actionInfoMiss"));
                }
                if (StringUtils.isEmpty(actionVO.getModuleId())) {
                    throw new DWBusinessException(I18nUtils.getValue("knowledgegraph.actionModuleEmpty"));
                }
                if (CollectionUtils.isEmpty(actionVO.getCondition())) {
                    throw new DWBusinessException(I18nUtils.getValue("knowledgegraph.actionConditionEmpty"));
                }
                for (ConditionVO conditionVO : actionVO.getCondition()) {
//                if(StringUtils.isEmpty(conditionVO.getActionId())){
//                    throw new DWBusinessException("condition中actionId信息为空");
//                }
                    if (StringUtils.isEmpty(conditionVO.getKey())) {
                        throw new DWBusinessException(I18nUtils.getValue("knowledgegraph.conditionKeyEmpty"));
                    }
                    if (StringUtils.isEmpty(conditionVO.getName())) {
                        throw new DWBusinessException(I18nUtils.getValue("knowledgegraph.conditionNameEmpty"));
                    }
                    if (StringUtils.isEmpty(conditionVO.getDefaultValue())) {
                        throw new DWBusinessException(I18nUtils.getValue("knowledgegraph.conditionDefaultValueEmpty"));
                    }
                    if (CollectionUtils.isEmpty(conditionVO.getTypeParameter())) {
                        throw new DWBusinessException(I18nUtils.getValue("knowledgegraph.conditionTypeParameterEmpty"));
                    }
                    if (!conditionVO.getTypeParameter().contains(conditionVO.getDefaultValue())) {
                        throw new DWBusinessException(I18nUtils.getValue("knowledgegraph.conditionNotExist"));
                    }
                }

            }
        }

    }

    /**
     * 给应用的租户下所有的用户进行授权
     * @param config
     * @return
     * @throws Exception
     */
    @Override
    public Object postGrantAuthorityToUser(AuthorityConfigVO config) throws Exception {
        log.info("====GrantAuthorityToUser config:{}", JSON.toJSONString(config));
        this.grantAuthorityCheck(config);
        List<String> needToGrantAuthorityInfoList = this.getNeedToGrantAuthorityInfo(config);
        final String appCode = config.getApp().getId();
        AppAuthorizationsResponse allEffectiveTenantAndAppInfo = this.cacService.getAllEffectiveTenantAndAppInfo(appCode);
        log.info("通过cac查询还没失效的应用：{} 租户信息共：{} 条", config.getApp().getId(), allEffectiveTenantAndAppInfo.getTotal());
        new Thread(()->{
            for(AppPurchaseInfo appPurchaseInfo : allEffectiveTenantAndAppInfo.getList()){
                //判断是否限定租户，对限定的租户进行授权
                if(!CollectionUtils.isEmpty(config.getLimitedTenantIdList()) && !config.getLimitedTenantIdList().contains(appPurchaseInfo.getTenantId())){
                    continue;
                }
                //判断是否是需要排除的租户
                if(!CollectionUtils.isEmpty(config.getExcludedTenantIdList()) && config.getExcludedTenantIdList().contains(appPurchaseInfo.getTenantId())){
                    continue;
                }
                String tenantToken = null;
                List<AuthorityAction> needToGrandAuthorityActionInfo = null;
                List<SimpleUser> userList = null;
                try {
                    log.info("开始处理租户：{}", appPurchaseInfo.getTenantId());
                    tenantToken = this.iamService.getTenantToken(appPurchaseInfo.getTenantId());
                    AppDetailInfo appDetailInfo = this.iamService.appDetailInfo(appCode, tenantToken);
                    needToGrandAuthorityActionInfo = this.getNeedToGrandAuthorityActionInfo(appDetailInfo, needToGrantAuthorityInfoList);
                    if(CollectionUtils.isEmpty(needToGrandAuthorityActionInfo)){
                        continue;
                    }
                    userList = this.iamService.getUser(tenantToken);
                } catch (Exception e) {
                    log.error("===调用iam接口异常,租户：{},异常信息{}", appPurchaseInfo.getTenantId(),e.toString());
                }
                if(CollectionUtils.isEmpty(userList)){
                    continue;
                }
                for(SimpleUser simpleUser :userList){
                    for(AuthorityAction action : needToGrandAuthorityActionInfo){
                        try {
                            Object policyActionResult = this.iamService.policyAction("user", simpleUser.getSid(), action.getAppSid(), true, action.getSid(), tenantToken);
                            log.info("policyAction=> tenantId:{}, user:{},action:{}, result:{}",appPurchaseInfo.getTenantId(), simpleUser.getName(),action.getId(), policyActionResult);
                        } catch (Exception e) {
                            log.error(e.toString());
                        }
                    }
                }
            }
            log.info("====GrantAuthorityToUser finish====");
        }).start();
        return "已正常启动执行自动赋权操作";
    }


    private List<AuthorityAction> getNeedToGrandAuthorityActionInfo(AppDetailInfo appDetailInfo, List<String> needToGrantAuthorityInfoList){
        if(appDetailInfo == null || CollectionUtils.isEmpty(appDetailInfo.getActions()) || CollectionUtils.isEmpty(needToGrantAuthorityInfoList)){
            return null;
        }

        List<AuthorityAction> collect = appDetailInfo.getActions().stream()
                .map(action -> needToGrantAuthorityInfoList.stream()
                        .filter(e -> StringUtils.equals(action.getModuleId(), StringUtils.substringBefore(e, ":"))
                                && StringUtils.equals(action.getId(), StringUtils.substringAfter(e, ":")))
                        .findFirst().map(m -> action).orElse(null))
                .filter(Objects::nonNull).collect(Collectors.toList());
        return collect;
    }

    /**
     * 获取需要进行授权的行为信息，list值格式为 moduleId:Actionid
     * @param config
     * @return
     */
    private List<String> getNeedToGrantAuthorityInfo(AuthorityConfigVO config){
        List<String> collect = config.getActions().stream().map(e -> e.getModuleId() + ":" + e.getId()).collect(Collectors.toList());
        return collect;
    }

    private void grantAuthorityCheck(AuthorityConfigVO config) throws Exception{
        if(config == null){
            throw new DWBusinessException(I18nUtils.getValue("knowledgegraph.authConfigMiss"));
        }

        AppVO app = config.getApp();
        if(app == null || StringUtils.isEmpty(app.getId())){
            throw new DWBusinessException(I18nUtils.getValue("knowledgegraph.appIdMiss"));
        }

        List<ActionVO> actions = config.getActions();
        if(CollectionUtils.isEmpty(actions)){
            throw new DWBusinessException(I18nUtils.getValue("knowledgegraph.actionsEmpty"));
        }
        for(ActionVO actionVO : actions){
            if(StringUtils.isEmpty(actionVO.getId()) /*|| StringUtils.isEmpty(actionVO.getName())*/){
                throw new DWBusinessException(I18nUtils.getValue("knowledgegraph.actionIdEmpty"));
            }
            if(StringUtils.isEmpty(actionVO.getModuleId())){
                throw new DWBusinessException(I18nUtils.getValue("knowledgegraph.actionModuleIdEmpty"));
            }

        }

    }

    @Override
    public Object postGrantGlobalMessageAuthorityToUser(AuthorityConfigVO config) throws Exception{
        //查询机制变量 globalMessagePageType 设置过值的租户，并为这些租户进行授权
        Query query = new Query();
        query.addCriteria(Criteria.where("variableId").is("globalMessagePageType"));
        List<Map> mechanismVariable = this.mongoTemplateUser.find(query, Map.class, "mechanismVariable");
        if(CollectionUtils.isEmpty(mechanismVariable)){
            return "没找到租户";
        }
        List<String> tenantIdList = mechanismVariable.stream().map(e -> String.valueOf(e.get("tenantId"))).distinct().collect(Collectors.toList());
        config.setLimitedTenantIdList(tenantIdList);
        return this.postGrantAuthorityToUser(config);
    }


    @Override
    public Object postTenantBasicDataByAppCodeAndId(AuthorityConfigVO config) throws Exception {
        List<ActionVO> newActions = new ArrayList<>();
        List<ActionVO> actions = config.getActions();
        Map<String, Object> filterDataMap = new HashMap<>();

        // 查询租户的组件数据
        TenantAppComponentListBO tenantAppComponentListBo = tenantAppComponentListDAO
                .findTenantAppComponentList(config.getAppId(), config.getTenantId());

        // 获取租户组件的code集合
        List<String> componentCodeList = tenantAppComponentListBo == null ?
                Collections.emptyList() :
                tenantAppComponentListBo.getComponents()
                        .stream()
                        .map(ComponentBO::getCode)
                        .collect(Collectors.toList());

        // 筛选actions，回传需要被排除的行为
        if (!CollectionUtils.isEmpty(componentCodeList) && !CollectionUtils.isEmpty(actions)) {
            newActions = config.getActions()
                    .stream()
                    .filter(action -> !componentCodeList.contains(action.getId()))
                    .collect(Collectors.toList());
        }

        filterDataMap.put("actions", newActions);
        filterDataMap.put("modules", Collections.emptyList());
        return filterDataMap;
    }

}
