package com.digiwin.athena.knowledgegraph.service.impl;

import com.alibaba.fastjson.JSONObject;
import com.digiwin.app.container.exceptions.DWBusinessException;
import com.digiwin.athena.kmservice.locale.Lang;
import com.digiwin.athena.domain.core.app.ApplicationRelation;
import com.digiwin.athena.knowledgegraph.domain.decisionmodel.*;
import com.digiwin.athena.knowledgegraph.service.IDecisionService;
import com.digiwin.athena.knowledgegraph.service.inner.DataPickService;
import com.digiwin.athena.knowledgegraph.utils.AthenaUtils;
import com.digiwin.athena.knowledgegraph.utils.LanguageUtil;
import com.digiwin.athena.knowledgegraph.utils.TranslateUtils;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections4.MapUtils;
import org.bson.types.ObjectId;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.data.mongodb.core.MongoTemplate;
import org.springframework.data.mongodb.core.query.Criteria;
import org.springframework.data.mongodb.core.query.Query;
import org.springframework.data.mongodb.core.query.Update;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import java.util.*;
import java.util.function.Function;
import java.util.stream.Collectors;

@Lang
@Service
@Slf4j
public class DecisionService implements IDecisionService {

    @Autowired
    @Qualifier("knowledgegraphSystem")
    MongoTemplate mongoTemplate;
    @Autowired
    @Qualifier("knowledgegraphTenant")
    MongoTemplate mongoTemplateUser;
    @Autowired
    TenantService tenantService;
    @Autowired
    AppService appService;
    @Autowired
    TranslateUtils translateUtils;
    @Autowired
    private DataPickService dataPickService;

    @Override
    public Object postDecisionSchemesBatchByCodes(List<String> evlModelCodes) throws Exception {
        return null;
    }

    @Override
    public Object getDecisionModelsByApp(String appCode) throws Exception {
        String tenantId = AthenaUtils.getTenantId();
        String version = tenantService.getTenantVersion(tenantId);
        String locale = AthenaUtils.getCurrentLocale();
        List<ApplicationRelation> applicationRelations = appService.getApplicationRelationByTypeAndAppCodeAndVersion("decisionModel", appCode, version);
        if (CollectionUtils.isEmpty(applicationRelations)) {
            return Collections.EMPTY_LIST;
        }
        List<String> modelCode = applicationRelations.stream().map(ApplicationRelation::getCode).collect(Collectors.toList());
        List<DecisionModel> decisionModels = dataPickService.find(Criteria.where("code").in(modelCode), DecisionModel.class, "decisionModel");
        if (CollectionUtils.isEmpty(decisionModels)) {
            return Collections.EMPTY_LIST;
        }
        List<Map<String, String>> result = new LinkedList<>();
        decisionModels.forEach(item -> {
            try {
                LanguageUtil.processLocaleLanguage(item, locale);
            } catch (IllegalAccessException | NoSuchFieldException e) {
                log.error(e.getMessage(), e);
            }
            Map<String, String> codeName = new HashMap<>();
            codeName.put("code", item.getCode());
            codeName.put("name", item.getName());
            result.add(codeName);
        });
        return result;
    }

    private DecisionModel findByCode(String evlModelCode) throws DWBusinessException {
        return dataPickService.findOneByCondition(Criteria.where("code").is(evlModelCode), DecisionModel.class, "decisionModel");
    }

    /**
     * 提供给SD运行时获取模型数据调用
     * 返回的是开发平台变更同步修改后的数据
     * @param evlModelCode
     * @return
     * @throws Exception
     */
    @Override
    public Object getDecisionSchemesByCode(String evlModelCode) throws Exception {
        DecisionModel decisionModel = this.getDecisionModel(evlModelCode);
        if (Objects.isNull(decisionModel)) {
            return null;
        }
        // 同步开发平台评价方案的变更(决策因子，数据源，数据源名称)
        syncDecisionScheme(decisionModel);
        // 同步开发平台决策方案变更(新增，删除决策方案)
        syncDecisionMakingScheme(decisionModel);
        log.info("after sync decisionMode:{}", decisionModel);
        return decisionModel;
    }

    /**
     * 40762-辅助决策-DCP应用需求
     * 新增方法专为交付设计器调用
     * 根据决策模型code获取其下所有的系统级和租户级方案
     * @param evlModelCode
     * @return
     * @throws Exception
     */
    @Override
    public Object getDecisionSchemesByCodeForWeb(String evlModelCode) throws Exception {
        return this.getDecisionModel(evlModelCode);
    }

    /**
     * 同步开发平台评价方案的变更(决策因子删除，数据源删除，数据源名称变更)
     * @param decisionModel
     */
    private void syncDecisionScheme(DecisionModel decisionModel) {
        if (Objects.isNull(decisionModel)) {
            return;
        }
        if (CollectionUtils.isEmpty(decisionModel.getSchemes())) {
            // 租户级评价方案为空，不做处理
            return;
        }
        //  租户级评价方案
        List<DecisionScheme> schemes = decisionModel.getSchemes();
        // 系统级决策因子
        List<Factor> sysFactors = decisionModel.getFactors();
        if (CollectionUtils.isEmpty(sysFactors)) {
            // 系统级决策因子为空，同步租户方案的决策因子为空
            schemes.forEach(scheme -> scheme.setFactors(Collections.EMPTY_LIST));
            return;
        }
        Map<String, Map<String, Api>> sysFactorApiMap = sysFactors.stream()
                .collect(Collectors.toMap(
                        Factor::getCode,
                        factor -> factor.getApis().stream()
                                .collect(Collectors.toMap(Api::getUid, api -> api))
                ));
        List<String> sysFactorCode = sysFactors.stream().map(Factor::getCode).collect(Collectors.toList());
        // 循环租户级评价方案，同步开发平台变更
        for (DecisionScheme scheme : schemes) {
            // 租户级决策因子
            List<SchemeFactor> tenantFactors = scheme.getFactors();
            if (CollectionUtils.isEmpty(tenantFactors)) {
                // 租户级评价方案的决策因子为空，不做处理
                continue;
            }
            // 同步删除决策因子
            tenantFactors = tenantFactors.stream().filter(tenantFactor -> sysFactorCode.contains(tenantFactor.getCode())).collect(Collectors.toList());
            scheme.setFactors(tenantFactors);
            // 循环决策因子，同步删除数据源,同步数据源名称
            for (SchemeFactor tenantFactor : tenantFactors) {
                String tenantFactorCode = tenantFactor.getCode();
                // 系统级数据源
                Map<String, Api> sysApiMap = sysFactorApiMap.get(tenantFactorCode);
                if (MapUtils.isEmpty(sysApiMap)) {
                    // 系统级数据源为空，同步将租户数据源置为空
                    tenantFactor.setApis(Collections.EMPTY_LIST);
                    continue;
                }
                // 租户数据源
                List<Api> tenantApis = tenantFactor.getApis();
                if (CollectionUtils.isEmpty(tenantApis)) {
                    // 租户数据源为空
                    continue;
                }
                // 同步删除数据源
                tenantApis = tenantApis.stream().filter(api -> sysApiMap.containsKey(api.getUid())).collect(Collectors.toList());
                // 同步数据源名称
                tenantApis.forEach(api -> {
                    Api sysApi = sysApiMap.get(api.getUid());
                    if (Objects.nonNull(sysApi)) {
                        api.setName(sysApi.getName());
                        api.setLang(sysApi.getLang());
                    }
                });
                tenantFactor.setApis(tenantApis);
            }
        }
    }

    /**
     * 同步开发平台决策方案变更(新增，删除决策方案)
     * @param decisionModel
     */
    private void syncDecisionMakingScheme(DecisionModel decisionModel) {
        if (Objects.isNull(decisionModel)) {
            return;
        }
        // 系统级和租户级 决策方案都为空，不做处理
        if (CollectionUtils.isEmpty(decisionModel.getDecisionScheme()) && CollectionUtils.isEmpty(decisionModel.getOriginDecisionScheme())) {
            return;
        }
        // 系统级决策方案
        List<DecisionMakingScheme> sysDecisionScheme = decisionModel.getOriginDecisionScheme();
        if (CollectionUtils.isEmpty(sysDecisionScheme)) {
            // 系统级为空，租户级直接设置为空
            decisionModel.setDecisionScheme(Collections.EMPTY_LIST);
            return;
        }
        // 租户级决策方案
        List<DecisionMakingScheme> tenantDecisionScheme = decisionModel.getDecisionScheme();
        if (CollectionUtils.isEmpty(tenantDecisionScheme)) {
            // 租户级为空，塞入系统级数据
            decisionModel.setDecisionScheme(sysDecisionScheme);
            return;
        }
        // 系统级和租户级 决策方案都不为空,直接复制系统级数据，然后回写不需要同步的字段
        List<DecisionMakingScheme> newTenantDecisionScheme = new ArrayList<>(sysDecisionScheme.size());
        sysDecisionScheme.forEach(item -> {
            DecisionMakingScheme makingScheme = new DecisionMakingScheme();
            BeanUtils.copyProperties(item, makingScheme);
            newTenantDecisionScheme.add(makingScheme);
        });
        Map<String, DecisionMakingScheme> tenantDecisionSchemeMap = tenantDecisionScheme.stream().collect(Collectors.toMap(DecisionMakingScheme::getCode, Function.identity(), (a, b) -> b));
        for (DecisionMakingScheme decisionMakingScheme : newTenantDecisionScheme) {
            DecisionMakingScheme makingScheme = tenantDecisionSchemeMap.get(decisionMakingScheme.getCode());
            if (Objects.nonNull(makingScheme)) {
                decisionMakingScheme.setIfRecommend(makingScheme.getIfRecommend());
                decisionMakingScheme.setMode(makingScheme.getMode());
                decisionMakingScheme.setCondition(makingScheme.getCondition());
                decisionMakingScheme.setShowFields(makingScheme.getShowFields());
                decisionMakingScheme.setEvlToDecisionMakingRouting(makingScheme.getEvlToDecisionMakingRouting());
                //decisionMakingScheme.set_mergeRule(makingScheme.get_mergeRule());
            }
        }
        decisionModel.setDecisionScheme(newTenantDecisionScheme);
    }

    /**
     * 抽取公共代码
     * @param evlModelCode
     * @return
     * @throws Exception
     */
    private DecisionModel getDecisionModel(String evlModelCode) throws Exception  {
        String tenantId = AthenaUtils.getTenantId();
        String version = tenantService.getTenantVersion(tenantId);
        String locale = AthenaUtils.getCurrentLocale();
        //根据决策模型key获取决策方案
        //step 1:查询系统及决策方案 knowledgegraphSystem->DecisionModel
        DecisionModel decisionModel = findByCode(evlModelCode);
        if (null == decisionModel) {
            return null;
        }
        //step 2:查询租户级决策方案 knowledgegraph->decisionScheme
        Query queryUser = new Query();
        Criteria criteriaUser = Criteria.where("version").is(version)
                .and("evlModelCode").is(evlModelCode).and("tenantId").is(tenantId);
        queryUser.addCriteria(criteriaUser);
        List<DecisionScheme> decisionSchemes = mongoTemplateUser.find(queryUser, DecisionScheme.class);
        transformApiPrecondition(decisionSchemes);
        //step 3:查询模型是否禁用
        Query queryTenantModel = new Query();
        Criteria criteriaTenantModel = Criteria.where("code").is(evlModelCode)
                .and("tenantId").is(tenantId)
                .and("version").is(version);
        queryTenantModel.addCriteria(criteriaTenantModel);
        TenantDecisionModel tenantDecisionModel = mongoTemplateUser.findOne(queryTenantModel, TenantDecisionModel.class);
        //mergeDecisionScheme
        mergeDecisionModel(decisionModel, tenantDecisionModel);
        //若租户级决策方案功能关闭则跳过该决策模型(返回给SD判断)
        copyFromDecisionModel(decisionModel, decisionSchemes);
        decisionModel.setSchemes(decisionSchemes);
        if (CollectionUtils.isEmpty(decisionSchemes)) {
            decisionModel.setEnabled(false);
            decisionModel.setBanSwitch(false);
            LanguageUtil.processLocaleLanguage(decisionModel, locale);
            return decisionModel;
        }
        if (null == tenantDecisionModel) {
            if (CollectionUtils.isEmpty(decisionSchemes)) {
                decisionModel.setEnabled(false);
            } else {
                decisionModel.setEnabled(true);
            }
            decisionModel.setBanSwitch(false);
            decisionModel.setUseAI(false);
        } else {
            // 默认为true
            decisionModel.setEnabled(Optional.ofNullable(tenantDecisionModel.getEnabled()).orElse(true));
            decisionModel.setBanSwitch(Optional.ofNullable(tenantDecisionModel.getEnabled()).orElse(true));
            decisionModel.setUseAI(Optional.ofNullable(tenantDecisionModel.getUseAI()).orElse(false));
        }
        LanguageUtil.processLocaleLanguage(decisionModel, locale);
        log.info("decisionMode:{}", decisionModel);
        return decisionModel;
    }

    private void copyFromDecisionModel(DecisionModel decisionModel, List<DecisionScheme> decisionSchemes) {
        Map<String, Map<String, Api>> apiMap = decisionModel.getFactors().stream()
                .collect(Collectors.toMap(
                        Factor::getCode,
                        factor -> factor.getApis().stream()
                                .collect(Collectors.toMap(Api::getUid, api -> api))
                ));
        if (!CollectionUtils.isEmpty(decisionSchemes)) {
            decisionSchemes.forEach(decisionScheme -> {
                List<SchemeFactor> schemeFactors = decisionScheme.getFactors();
                if (schemeFactors != null) {
                    schemeFactors.forEach(schemeFactor -> {
                        String factorCode = schemeFactor.getCode();
                        Map<String, Api> factorApis = apiMap.get(factorCode);
                        if (factorApis != null) {
                            List<Api> apis = schemeFactor.getApis();
                            if (apis != null) {
                                apis.forEach(api -> {
                                    String apiUid = api.getUid();
                                    Api matchingApi = factorApis.get(apiUid);
                                    if (matchingApi != null) {
                                        // Set accuracy value from matchingApi to the corresponding api in DecisionScheme
                                        api.setAccuracy(matchingApi.getAccuracy());
                                    }
                                });
                            }
                        }
                    });
                }
            });

        }
    }

    private void mergeDecisionModel(DecisionModel decisionModel, TenantDecisionModel tenantDecisionModel) {
        // 塞入系统级决策方案
        decisionModel.setOriginDecisionScheme(decisionModel.getDecisionScheme());
        // 系统级和租户级决策方案都存在的场景
        if (null != tenantDecisionModel && !CollectionUtils.isEmpty(tenantDecisionModel.getDecisionScheme()) && !CollectionUtils.isEmpty(decisionModel.getDecisionScheme())) {
            List<DecisionMakingScheme> sysDecisionScheme = decisionModel.getDecisionScheme();
            Map<String, DecisionMakingScheme> sysDecisionSchemeMap = sysDecisionScheme.stream().collect(Collectors.toMap(DecisionMakingScheme::getCode, Function.identity()));

            List<DecisionMakingScheme> tenantDecisionScheme = tenantDecisionModel.getDecisionScheme();
            tenantDecisionScheme.forEach(item -> {
                DecisionMakingScheme sysScheme = sysDecisionSchemeMap.get(item.getCode());
                if (null != sysScheme) {
                    item.setDecisionCode(sysScheme.getDecisionCode());
                    item.setName(sysScheme.getName());
                    item.setShowFields(sysScheme.getShowFields());
                    item.setLang(sysScheme.getLang());
                }
            });
            decisionModel.setDecisionScheme(tenantDecisionScheme);
        }
        // 系统级决策方案不存在，租户级决策方案存在的场景
        if (null != tenantDecisionModel && !CollectionUtils.isEmpty(tenantDecisionModel.getDecisionScheme()) && CollectionUtils.isEmpty(decisionModel.getDecisionScheme())) {
            // 塞入租户级决策方案
            decisionModel.setDecisionScheme(tenantDecisionModel.getDecisionScheme());
        }
    }

    private void transformApiPrecondition(List<DecisionScheme> decisionSchemes) {
        if (!CollectionUtils.isEmpty(decisionSchemes)) {
            decisionSchemes.forEach(decisionScheme -> {
                List<SchemeFactor> factors = decisionScheme.getFactors();
                if (!CollectionUtils.isEmpty(factors)) {
                    factors.forEach(schemeFactor -> {
                        List<Api> apis = schemeFactor.getApis();
                        if (!CollectionUtils.isEmpty(apis)) {
                            apis.forEach(api -> {
                                ApiPreCondition apiPreCondition = api.getPrecondition();
                                String expression = transformApiExpression(apiPreCondition);
                                if (null != apiPreCondition) {
                                    apiPreCondition.setExpressionTransformed(expression);
                                    api.setPrecondition(apiPreCondition);
                                }
                            });
                        }
                    });
                }
            });
        }
    }

    private String transformApiExpression(ApiPreCondition apiPreCondition) {
        if (null == apiPreCondition) {
            return null;
        }
        switch (apiPreCondition.getConditionType()) {
            case "expression":
                return apiPreCondition.getExpression();
            case "visualization":
                return transformVisualizationToExpression(apiPreCondition.getVisualization());
            default:
                return null;
        }
    }

    private static String transformVisualizationToExpression(ApiVisualizationExpression visualization) {
        StringBuilder expressionBuilder = new StringBuilder();
        boolean bothNotEmpty = !CollectionUtils.isEmpty(visualization.getAnd()) && !CollectionUtils.isEmpty(visualization.getOr());
        if (visualization != null) {
            if (!CollectionUtils.isEmpty(visualization.getAnd())) {
                if (bothNotEmpty) {
                    expressionBuilder.append("(");
                }
                List<ApiRestrict> andList = visualization.getAnd();
                for (int i = 0; i < andList.size(); i++) {
                    if (i > 0) {
                        expressionBuilder.append(" && ");
                    }
                    ApiRestrict element = andList.get(i);
                    expressionBuilder.append("($(")
                            .append(element.getSourceId())
                            .append(")")
                            .append(element.getOperator())
                            .append(" ")
                            .append(element.getRestrictType().equals("dataSource") ? "$(" + element.getValue() + ")" : element.getValue())
                            .append(")");
                }
                if (bothNotEmpty) {
                    expressionBuilder.append(")");
                }
            }

            if (!CollectionUtils.isEmpty(visualization.getOr())) {
                if (!CollectionUtils.isEmpty(visualization.getAnd())) {
                    expressionBuilder.append(" && (");
                }
                if (bothNotEmpty) {
                    expressionBuilder.append("(");
                }
                List<ApiRestrict> orList = visualization.getOr();
                for (int i = 0; i < orList.size(); i++) {
                    if (i > 0) {
                        expressionBuilder.append(" || ");
                    }
                    ApiRestrict element = orList.get(i);
                    expressionBuilder.append("($(")
                            .append(element.getSourceId())
                            .append(")")
                            .append(element.getOperator())
                            .append(" ")
                            .append(element.getRestrictType().equals("dataSource") ? "$(" + element.getValue() + ")" : element.getValue())
                            .append(")");
                }
                if (bothNotEmpty) {
                    expressionBuilder.append(")");
                }
            }
        }
        return expressionBuilder.toString();
    }

    @Override
    public Object postDecisionModel(DecisionModel decisionModel) throws Exception {
        DecisionModel byCode = findByCode(decisionModel.getCode());
        // 如果做了个案并且悬停
        decisionModel.setTenantId(byCode.getTenantId());
        decisionModel.setSourceLevel(byCode.getSourceLevel());
        decisionModel.setVersion(byCode.getVersion());
        mongoTemplate.save(decisionModel);
        return null;
    }

    @Override
    public Object postDecisionScheme(DecisionScheme decisionScheme) throws Exception {
        String tenantId = AthenaUtils.getTenantId();
        String version = tenantService.getTenantVersion(tenantId);
        decisionScheme.setTenantId(tenantId);
        decisionScheme.setVersion(version);
        translateDecisionSchemeName(decisionScheme);
        mongoTemplateUser.save(decisionScheme);
        return decisionScheme;
    }

    @Override
    public Object postSetRecommend(DecisionScheme decisionScheme) throws Exception {
        String tenantId = AthenaUtils.getTenantId();
        String version = tenantService.getTenantVersion(tenantId);
        decisionScheme.setTenantId(tenantId);
        decisionScheme.setVersion(version);
        decisionScheme.setIfRecommend(true);
        mongoTemplateUser.save(decisionScheme);

        //将当前模型下其他方案ifRecommend设为false
        Query query = new Query();
        Criteria criteria = Criteria.where("evlModelCode").is(decisionScheme.getEvlModelCode())
                .and("tenantId").is(tenantId)
                .and("version").is(version).and("_id").ne(new ObjectId(decisionScheme.getId()));
        query.addCriteria(criteria);
        Update update = new Update();
        update.set("ifRecommend", false);
        mongoTemplateUser.updateMulti(query, update, DecisionScheme.class);
        /*List<DecisionScheme> decisionSchemes = mongoTemplateUser.find(query, DecisionScheme.class);
        decisionSchemes = decisionSchemes.stream().filter(item -> !item.getId().equals(decisionScheme.getId())).collect(Collectors.toList());
        if (!CollectionUtils.isEmpty(decisionSchemes)) {
            decisionSchemes.forEach(item -> item.setIfRecommend(false));
            mongoTemplateUser.save(decisionSchemes);
        }*/
        return decisionScheme;
    }

    @Override
    public Object postDecisionSchemeName(DecisionSchemeNameVO decisionSchemeNameVO) throws Exception {
        String tenantId = AthenaUtils.getTenantId();
        String version = tenantService.getTenantVersion(tenantId);
        String name = decisionSchemeNameVO.getName();
        String evlModelCode = decisionSchemeNameVO.getEvlModelCode();
        String evlSchemeCode = decisionSchemeNameVO.getEvlSchemeCode();
        Query query = new Query();
        Criteria criteria = Criteria.where("evlModelCode").is(evlModelCode)
                .and("code").is(evlSchemeCode)
                .and("tenantId").is(tenantId)
                .and("version").is(version);
        query.addCriteria(criteria);
        DecisionScheme decisionScheme = mongoTemplateUser.findOne(query, DecisionScheme.class);
        if (null == decisionScheme) {
            throw new DWBusinessException("no decisionScheme found");
        }
        decisionScheme.setName(name);
        translateDecisionSchemeName(decisionScheme);
        mongoTemplateUser.save(decisionScheme);
        return decisionScheme;
    }

    @Override
    public Object postDecisionMakingScheme(DecisionMakingSchemeVO decisionMakingSchemeVO) throws Exception {
        String tenantId = AthenaUtils.getTenantId();
        String version = tenantService.getTenantVersion(tenantId);
        DecisionModel decisionModel = findByCode(decisionMakingSchemeVO.getEvlModelCode());
        if (null == decisionModel) {
            throw new DWBusinessException("no decisionModel found");
        }
        if (!decisionModel.getUseDecisionScheme()) {
            throw new DWBusinessException("decisionModel not allowed to set decisionMakingScheme");
        }
        //比对系统级decisionMakingScheme配置
        List<DecisionMakingScheme> sysDecisionMakingScheme = decisionModel.getDecisionScheme();
        List<DecisionMakingScheme> tenantDecisionMakingScheme = decisionMakingSchemeVO.getDecisionScheme();
        if (!compareTenantSystem(sysDecisionMakingScheme, tenantDecisionMakingScheme)) {
            throw new DWBusinessException("tenant decisionMakingScheme does not match scheme");
        }
        TenantDecisionModel tenantDecisionModel = findTenantDecisionModel(decisionMakingSchemeVO.getEvlModelCode());
        if (null == tenantDecisionModel) {
            //没有就new一个
            tenantDecisionModel = new TenantDecisionModel();
            tenantDecisionModel.setTenantId(tenantId);
            tenantDecisionModel.setVersion(version);
            tenantDecisionModel.setCode(decisionMakingSchemeVO.getEvlModelCode());
        }
        tenantDecisionModel.setDecisionScheme(tenantDecisionMakingScheme);
        mongoTemplateUser.save(tenantDecisionModel);
        return tenantDecisionModel;
    }

    private boolean compareTenantSystem(List<DecisionMakingScheme> sysDecisionMakingScheme, List<DecisionMakingScheme> tenantDecisionMakingScheme) {
        if (CollectionUtils.isEmpty(sysDecisionMakingScheme) || CollectionUtils.isEmpty(tenantDecisionMakingScheme)) {
            return false;
        }
        Set codeSet = sysDecisionMakingScheme.stream().map(DecisionMakingScheme::getCode).collect(Collectors.toSet());
        for (int i = 0; i < tenantDecisionMakingScheme.size(); i++) {
            if (codeSet.contains(tenantDecisionMakingScheme.get(i).getCode())) {
                codeSet.remove(tenantDecisionMakingScheme.get(i).getCode());
            } else {
                return false;
            }
        }
        if (CollectionUtils.isEmpty(codeSet)) {
            return true;
        }
        return false;
    }

    @Override
    public Object postBanModel(String evlModelCode, Boolean setOrBan) throws Exception {
        String tenantId = AthenaUtils.getTenantId();
        String version = tenantService.getTenantVersion(tenantId);
        TenantDecisionModel decisionModel = findTenantDecisionModel(evlModelCode);
        if (null == decisionModel) {
            decisionModel = new TenantDecisionModel();
            decisionModel.setTenantId(tenantId);
            decisionModel.setVersion(version);
        }
        decisionModel.setEnabled(setOrBan);
        decisionModel.setCode(evlModelCode);
        mongoTemplateUser.save(decisionModel);
        return decisionModel;
    }

    private TenantDecisionModel findTenantDecisionModel(String evlModeCode) throws DWBusinessException {
        String tenantId = AthenaUtils.getTenantId();
        String version = tenantService.getTenantVersion(tenantId);
        Query query = new Query();
        Criteria criteria = Criteria.where("code").is(evlModeCode)
                .and("tenantId").is(tenantId)
                .and("version").is(version);
        query.addCriteria(criteria);
        TenantDecisionModel decisionModel = mongoTemplateUser.findOne(query, TenantDecisionModel.class);
        return decisionModel;
    }

    @Override
    public Object deleteDecisionScheme(String id) throws Exception {
        DecisionScheme decisionScheme = mongoTemplateUser.findById(id, DecisionScheme.class);
        mongoTemplateUser.remove(decisionScheme);
        return decisionScheme;
    }

    @Override
    public Object postSwitchAIAssistant(String evlModelCode, Boolean useAI) throws Exception {
        String tenantId = AthenaUtils.getTenantId();
        String version = tenantService.getTenantVersion(tenantId);
        TenantDecisionModel decisionModel = findTenantDecisionModel(evlModelCode);
        if (null == decisionModel) {
            decisionModel = new TenantDecisionModel();
            decisionModel.setTenantId(tenantId);
            decisionModel.setVersion(version);
        }
        decisionModel.setUseAI(useAI);
        decisionModel.setCode(evlModelCode);
        mongoTemplateUser.save(decisionModel);
        return decisionModel;
    }

    /**
     * 翻译评价方案名称
     * @param decisionScheme 评价方案
     * @throws DWBusinessException 业务异常
     */
    private void translateDecisionSchemeName(DecisionScheme decisionScheme) throws DWBusinessException {
        String name = decisionScheme.getName();
        JSONObject transResult = translateUtils.translateContent("name", name, new String[]{"zh2Hant", "zh2Hans"});
        if (transResult == null) {
            decisionScheme.setLang(null);
            return;
        }
        JSONObject nameObject = transResult.getJSONObject("name");
        if (nameObject == null) {
            decisionScheme.setLang(null);
            return;
        }
        Map<String, Map<String, String>> lang = new HashMap<>();
        Map<String, String> nameMap = new HashMap<>();
        lang.put("name", nameMap);

        if (nameObject.containsKey("zh_CN")) {
            nameMap.put("zh_CN", nameObject.getString("zh_CN"));
        }
        if (nameObject.containsKey("zh_TW")) {
            nameMap.put("zh_TW", nameObject.getString("zh_TW"));
        }
        decisionScheme.setLang(lang);
    }

}
