package com.digiwin.athena.knowledgegraph.service.impl;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import com.alibaba.fastjson.JSONPath;
import com.digiwin.app.resource.DWResourceBundleUtils;
import com.digiwin.athena.kmservice.locale.Lang;
import com.digiwin.athena.kg.activity.Activity;
import com.digiwin.athena.domain.core.app.Application;
import com.digiwin.athena.domain.core.app.ApplicationRelation;
import com.digiwin.athena.knowledgegraph.domain.digitalstaff.DigitalStaff;
import com.digiwin.athena.knowledgegraph.domain.digitalstaff.DigitalStaffCatgInfoDto;
import com.digiwin.athena.knowledgegraph.domain.digitalstaff.DigitalStaffDto;
import com.digiwin.athena.knowledgegraph.domain.digitalstaff.DigitalStaffSumInfoDto;
import com.digiwin.athena.knowledgegraph.domain.digitalstaff.TaskHistory;
import com.digiwin.athena.knowledgegraph.service.DataMapService;
import com.digiwin.athena.knowledgegraph.service.IDigitalStaffService;
import com.digiwin.athena.knowledgegraph.service.KgInnerService;
import com.digiwin.athena.knowledgegraph.service.inner.DataPickService;
import com.digiwin.athena.knowledgegraph.utils.AthenaUtils;
import com.digiwin.athena.knowledgegraph.clients.AtmcUtils;
import com.digiwin.athena.knowledgegraph.clients.CACUtils;
import com.digiwin.athena.knowledgegraph.utils.LanguageUtil;
import com.digiwin.athena.knowledgegraph.utils.ObjectUseUtil;
import com.luhuiguo.chinese.ChineseUtils;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.beanutils.BeanUtils;
import org.apache.commons.collections.CollectionUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.data.mongodb.core.MongoTemplate;
import org.springframework.data.mongodb.core.query.Criteria;
import org.springframework.data.mongodb.core.query.Query;
import org.springframework.stereotype.Service;
import org.springframework.util.ObjectUtils;
import org.springframework.util.StringUtils;

import java.io.IOException;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Comparator;
import java.util.Date;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

/**
 * <Description> <br>
 *
 * @author wang.xinyuan<br>
 * @version 1.0<br>
 * @taskId <br>
 * @CreateDate 2021/11/11 <br>
 * @see: com.digiwin.athena.knowledgegraph.service.impl.DigitalStaffService <br>
 */
@Lang
@Slf4j
@Service
public class DigitalStaffService implements IDigitalStaffService {

    @Autowired
    private DataPickService dataPickService;

    public enum GradeCode {ASSISTANT, EXPERT, MASTER}

    private static final String EOCDUTYLIST = "/api/eoc/v2/duty/list";
    private static final String ENHANCE = "enhance"; //增强应用标识码
    private static final String DIGITALSTAFF = "digitalStaff"; //增强应用扩展类型是数字员工
    private static final String APPCODE = "appCode"; //增强应用扩展类型是数字员工
    private static final String APPNAME = "appName"; //增强应用扩展类型是数字员工
    private static final String DESCRIPTION = "description"; //增强应用扩展类型是数字员工
    private static final String TASK = "task"; //项目标识
    private static final String ACTIVITY = "activity"; //任务标识

    @Autowired
    @Qualifier("knowledgegraphSystem")
    private MongoTemplate mongoTemplateSystem;

    @Autowired
    CACUtils cacUtils;
    @Autowired
    AtmcUtils atmcUtils;
    @Autowired
    TenantService tenantService;
    @Autowired
    DataMapService dataMapService;
    @Autowired
    TaskService taskService;
    @Autowired
    AppService appService;
    @Autowired
    KgInnerService kgInnerService;

    @Value("${iamApToken}")
    String appToken;

    @Override
    public List<DigitalStaffDto> getList() throws Exception {
        // 获取登录的租户信息
        String tenantId = AthenaUtils.getTenantId();
        String tenantVersion = kgInnerService.getTenantVersion(tenantId);
        // 语言信息
        String locale = DWResourceBundleUtils.getCurrentLocale().toString();
        log.info("language:" + locale);
        // 获取当前租户对应的增强应用信息和数字员工编码
        Map<String, Map<String, String>> appDigiStaffInfo = getDigitalStaffCodeListByTenantId(locale);
        if (appDigiStaffInfo.isEmpty()) {
            log.info("当前租户无法获取增强应用信息");
            return null;
        }
        List<String> digitalStaffCodeList = new ArrayList(appDigiStaffInfo.keySet());
        // 利用数字员工编码查询所有数字员工信息
        List<DigitalStaff> digitalStaffList = getDigitalStaffInfoList(digitalStaffCodeList, locale, tenantVersion);
        // 获取职能信息列表
        JSONArray functionList = getFunctionList();

        // 数字员工信息转存变量，为了存储输出的信息
        List<DigitalStaffDto> tempDigiStaffInfoList = new ArrayList<>();
        // 列表视图信息组装
        for (DigitalStaff digitalStaff : digitalStaffList) {
            DigitalStaffDto digitalStaffInfo = new DigitalStaffDto();
            BeanUtils.copyProperties(digitalStaffInfo, digitalStaff);
            // 1.获取职级名称
            String gradeName = getGradeNameByLang(digitalStaffInfo.getGradeCode(), locale);
            digitalStaffInfo.setGradeName(gradeName);
            // 2.获取职能名称
            String functionName = getFunctionName(functionList, digitalStaffInfo.getFunctionCode());
            digitalStaffInfo.setFunctionName(functionName);
            // 3.获取有效期
            // 3.1数字员工对应的增强应用的code
            Map<String, String> appCodeAndNameAndDesc = appDigiStaffInfo.get(digitalStaffInfo.getCode());
            String appCode = appCodeAndNameAndDesc.get(APPCODE);
            String appName = appCodeAndNameAndDesc.get(APPNAME);
            String appDesc = appCodeAndNameAndDesc.get(DESCRIPTION);
            // 3.2获取权限信息
            Map<String, String> appInfo = cacUtils.getAppAuthDate(tenantId, appCode);
            String startDate = appInfo.get("effectiveTime");
            String endDate = appInfo.get("expiredTime");
            String startDate2 = startDate.replace("-", "/");
            String endDate2 = endDate.replace("-", "/");
            digitalStaffInfo.setStartDate(startDate);
            digitalStaffInfo.setEndDate(endDate);
            digitalStaffInfo.setValidPeriod(startDate2 + " - " + endDate2);
            // 4.职称名称-使用增强应用的名称
            digitalStaffInfo.setTitleName(appName);
            // 5.工作目标描述
            digitalStaffInfo.setWorkTargetDescribe(appDesc);
            // 6.项目描述，项目名称的拼接(同时获取任务名称)
            // 6.1获取应用相关的项目code
            List<ApplicationRelation> applicationRelaInfo = getTaskInfoByAppCode(appCode, tenantId);
            // 6.2获取应用相关的任务code列表
            List<String> activitiesCodeList = getActivityInfoByAppCode(appCode, tenantId);

            String taskNameStr = "";
            Set<String> activityNameList = new HashSet<>();
            Map<String,Object> findActivityCode = new HashMap<>();
            if (CollectionUtils.isNotEmpty(applicationRelaInfo)) {
                Map<String, Object> taskAndActivity = getTaskAndActivityInfo(applicationRelaInfo, activitiesCodeList, locale, findActivityCode);
                Object taskNameObj = taskAndActivity.get("taskName");
                Object activitiesCodeNameObj = taskAndActivity.get("activitiesCodeNameMap");
                taskNameStr = ObjectUtils.isEmpty(taskNameObj) ? "" : (String) taskNameObj;
                findActivityCode = ObjectUtils.isEmpty(activitiesCodeNameObj) ? new HashMap<>() : (Map<String,Object>) activitiesCodeNameObj;
            }
            // 6.3 applicationRelation表中没有应用相关的项目信息时，直接走getActivityDefinition方法获取任务名称，taskId设为""，这里是直接走datamap去查询任务数据了
            getActivityNameDirect(activitiesCodeList, "", findActivityCode);

            digitalStaffInfo.setProjectDescribe(taskNameStr);
            // 6.4 组合任务名称列表：将findActivityCode中的value值取出来放入set表中去重
            for (Map.Entry<String,Object> entry : findActivityCode.entrySet()) {
                Object nameObj = entry.getValue();
                if(null != nameObj && !ObjectUtils.isEmpty(nameObj)){
                    activityNameList.add((String)nameObj);
                }
            }
            // 7.已完成工作总数量
            //7.1 统计已完成工作状态信息
            Integer finishJobNum = getWholeFinishJobNum(activityNameList);
            digitalStaffInfo.setFinishJobNum(finishJobNum);
            // 8.完成工作的前部分拼接字段
            String spliceStr = appName + " " + finishJobNum.toString() + " " + handleLang("次", locale);
            digitalStaffInfo.setSpliceStr(spliceStr);
            // 9.是否设定（这里统一给否，到deliverydesigner里再去查真实值）
            digitalStaffInfo.setEducateSetting(false);
            // 10.传到delivery去查数据
            digitalStaffInfo.setAppCode(appCode);
            // 11.存储增强应用对应的应用code和name
            String relaAppCode = "";
            String relaAppName = "";
            List<ApplicationRelation> applicationRelationList = appService.getAppCodeByTypeAndCode("enhance",appCode);
            if(CollectionUtils.isNotEmpty(applicationRelationList)){
                // 增强应用对应的应用只有一个，增强应用和应用的对应关系是1对多的关系
                ApplicationRelation appRela = applicationRelationList.get(0);
                relaAppCode = appRela.getAppCode();
                relaAppName = appRela.getAppName();
            }
            digitalStaffInfo.setRelaAppCode(relaAppCode);
            digitalStaffInfo.setRelaAppName(relaAppName);

            tempDigiStaffInfoList.add(digitalStaffInfo);
        }
        return tempDigiStaffInfoList;
    }

    @Override
    public DigitalStaffSumInfoDto getVisualizeList() throws Exception {
        String tenantId = AthenaUtils.getTenantId();
        String tenantVersion = kgInnerService.getTenantVersion(tenantId);
        String locale = DWResourceBundleUtils.getCurrentLocale().toString();
        // 获取当前租户对应的增强应用信息和数字员工编码
        Map<String, Map<String, String>> appDigiStaffInfo = getDigitalStaffCodeListByTenantId(locale);
        List<String> digitalStaffCodeList = new ArrayList(appDigiStaffInfo.keySet());
        if (appDigiStaffInfo.isEmpty()) {
            log.info("当前租户无法获取增强应用信息");
            return null;
        }
        // 利用数字员工编码查询所有数字员工信息
        List<DigitalStaff> digitalStaffList = getDigitalStaffInfoList(digitalStaffCodeList, locale, tenantVersion);

        // 数字员工信息转存变量，为了存储输出的信息
        List<DigitalStaffDto> tempDigiStaffInfoList = new ArrayList<>();

        DigitalStaffSumInfoDto result = new DigitalStaffSumInfoDto();
        List<DigitalStaffCatgInfoDto> digitalStaffCatgInfo = new ArrayList<>();
        // 全部数字员工的数量
        int digitalStaffNum = 0;
        // 全部完成工作数量
        int totalFinishJobNum = 0;
        // 图例视图信息组装
        for (DigitalStaff digitalStaff : digitalStaffList) {
            digitalStaffNum = digitalStaffNum + 1;
            DigitalStaffDto digitalStaffInfo = new DigitalStaffDto();
            BeanUtils.copyProperties(digitalStaffInfo, digitalStaff);
            // 数字员工对应的增强应用的code,name
            Map<String, String> appCodeAndNameAndDesc = appDigiStaffInfo.get(digitalStaffInfo.getCode());
            String appCode = appCodeAndNameAndDesc.get(APPCODE);
            // 1.获取职级名称
            String gradeName = getGradeNameByLang(digitalStaffInfo.getGradeCode(), locale);
            digitalStaffInfo.setGradeName(gradeName);
            // 2.获取code对应的任务名称列表
            // 2.1获取应用相关的项目code
            List<ApplicationRelation> applicationRelaInfo = getTaskInfoByAppCode(appCode, tenantId);
            // 2.2获取应用相关的任务code列表
            List<String> activitiesCodeList = getActivityInfoByAppCode(appCode, tenantId);
            Set<String> activityNameList = new HashSet<>();
            Map<String,Object> findActivityCode = new HashMap<>();
            if (CollectionUtils.isNotEmpty(applicationRelaInfo)) {
                Map<String, Object> taskAndActivity = getTaskAndActivityInfo(applicationRelaInfo, activitiesCodeList, locale, findActivityCode);
                Object activitiesCodeNameObj = taskAndActivity.get("activitiesCodeNameMap");
                findActivityCode = ObjectUtils.isEmpty(activitiesCodeNameObj) ? new HashMap<>() : (Map<String,Object>) activitiesCodeNameObj;
            }
            // 2.3 applicationRelation表中没有应用相关的项目信息，或者项目相关信息中没有任务相关的信息时，直接走getActivityDefinition方法获取任务名称，
            // taskId设为""，这里是直接走datamap去查询任务数据了
            getActivityNameDirect(activitiesCodeList, "", findActivityCode);
            // 2.4 组合任务名称列表：将findActivityCode中的value值取出来放入set表中去重
            for (Map.Entry<String,Object> entry : findActivityCode.entrySet()) {
                Object nameObj = entry.getValue();
                if(null != nameObj && !ObjectUtils.isEmpty(nameObj)){
                    activityNameList.add((String)nameObj);
                }
            }
            // 3.统计已完成工作状态信息
            Map<String, Object> jobInfo = getFinishJobInfo(activityNameList, locale);
            int finishJobNum = (int) jobInfo.get("totalFinishjobNum");
            totalFinishJobNum = totalFinishJobNum + finishJobNum;
            // 4.完成工作总数
            digitalStaffInfo.setFinishJobNum(finishJobNum);
            // 5.已完成工作详情
            digitalStaffInfo.setFinishJobList((List<String>) jobInfo.get("taskFinishList"));
            // 6.工作范围
            digitalStaffInfo.setJobList((String) jobInfo.get("jobList"));
            // 7.传到delivery去查数据
            digitalStaffInfo.setAppCode(appCode);
            // 8.存储增强应用对应的应用code
            String relaAppCode = "";
            String relaAppName = "";
            List<ApplicationRelation> applicationRelationList = appService.getAppCodeByTypeAndCode("enhance",appCode);
            if(CollectionUtils.isNotEmpty(applicationRelationList)){
                // 增强应用对应的应用只有一个，增强应用和应用的对应关系是1对多的关系
                ApplicationRelation appRela = applicationRelationList.get(0);
                relaAppCode = appRela.getAppCode();
                relaAppName = appRela.getAppName();
            }
            digitalStaffInfo.setRelaAppCode(relaAppCode);
            digitalStaffInfo.setRelaAppName(relaAppName);
            tempDigiStaffInfoList.add(digitalStaffInfo);
        }

        // 按类别对数字员工分类
        Map<String, List<DigitalStaffDto>> classify = new HashMap<>();
        // 统计所有类别
        for (DigitalStaffDto digitalStaffInfo : tempDigiStaffInfoList) {
            List<DigitalStaffDto> temp = new ArrayList<>();
            String category = digitalStaffInfo.getCategoryCode();
            if (!classify.containsKey(category)) {
                classify.put(digitalStaffInfo.getCategoryCode(), temp);
            }
        }
        // 统计每个类别的数字员工信息
        for (DigitalStaffDto digitalStaffInfo : tempDigiStaffInfoList) {
            String catgCode = digitalStaffInfo.getCategoryCode();
            List<DigitalStaffDto> temp = classify.get(catgCode);
            temp.add(digitalStaffInfo);
            classify.put(catgCode, temp);
        }
        // 组装结果数据
        for (Map.Entry<String, List<DigitalStaffDto>> entry : classify.entrySet()) {
            DigitalStaffCatgInfoDto tempDto = new DigitalStaffCatgInfoDto();
            tempDto.setCategoryCode(entry.getKey());
            tempDto.setCategoryName(entry.getValue().get(0).getCategoryName());
            tempDto.setEmployee(entry.getValue());
            // 设定数量（这里统一给0，到deliverydesigner里再去查真实值）
            Map<String, Integer> categoryNoSettingNum = new HashMap<>();
            categoryNoSettingNum.put("ASSISTANT", 0);
            categoryNoSettingNum.put("EXPERT", 0);
            categoryNoSettingNum.put("MASTER", 0);
            tempDto.setCategoryNoSettingNum(categoryNoSettingNum);

            digitalStaffCatgInfo.add(tempDto);
        }
        result.setDigitalStaffNum(digitalStaffNum);
        result.setTotalFinishJobNum(totalFinishJobNum);
        result.setDigital(digitalStaffCatgInfo);
        result.setNoSettingTotalNum(0);
        return result;
    }

    // 获取应用相关的项目code
    private List<ApplicationRelation> getTaskInfoByAppCode(String appCode, String tenantId) throws Exception {
        return appService.getApplicationRelationByTypeAndAppCode(tenantId, TASK, appCode);
    }

    // 6.2获取应用相关的任务code
    private List<String> getActivityInfoByAppCode(String appCode, String tenantId) throws Exception {
        List<String> activitiesCodeList = new ArrayList<>();
        List<ApplicationRelation> applicationRelation = appService.getApplicationRelationByTypeAndAppCode(tenantId, ACTIVITY, appCode);
        if(null != applicationRelation) {
            // 取出任务code数组中的code值
            activitiesCodeList = applicationRelation.stream().map(ApplicationRelation::getCode).collect(Collectors.toList());
        }
        return activitiesCodeList;
    }

    /* 获取项目名称拼接字段和任务名称列表
     * apprelaList:从ApplicationRelation获取的应用相关的项目信息
     * activitiesCodeList：applicationRelation表中应用对应的任务code列表
     * locale：多语言标识
     * findActivityCode：存储已获取任务名称的变量
     */
    private Map<String, Object> getTaskAndActivityInfo(List<ApplicationRelation> apprelaList, List<String> activitiesCodeList,
                                                       String locale, Map<String,Object> findActivityCode) throws Exception {
        Map<String, Object> result = new HashMap<>();
        StringBuilder proName = new StringBuilder();
        for (ApplicationRelation temp : apprelaList) {
            String taskId = temp.getCode();
            // 1.根据项目的Id，查询项目的名称和项目下所有的任务
            // 2.使用增强应用对应的任务code，在1返回的activities中匹配，如果匹配到则获取任务名称，没有匹配到任务名称的 增强任务code，再调用getActivityDefinition方法获取
            Object taskInfo = taskService.getActivitiesWithMultiLanguage(taskId);
            if (!ObjectUtils.isEmpty(taskInfo)) {
                String combinName = "taskName." + locale;
                Object taskName = JSONPath.eval(taskInfo, combinName);
                if (null != taskName && !StringUtils.isEmpty(taskName)) {
                    proName = proName.append(taskName).append(" ");
                }
                Object activitieList = JSONPath.eval(taskInfo, "activities");
                if (null != activitieList && !ObjectUtils.isEmpty(activitieList)) {
                    // applicationRelation表中有项目信息，getActivitiesWithMultiLanguage接口能查到项目信息并且存在任务信息
                    getActivityName(activitieList, activitiesCodeList, locale, findActivityCode);
                }
            }
        }
        result.put("activitiesCodeNameMap", findActivityCode);
        String taskNameStr = proName.toString();
        String nameStr = "";
        if (StringUtils.isEmpty(taskNameStr)) {
            result.put("taskName", nameStr);
        } else {
            nameStr = taskNameStr.substring(0, taskNameStr.length() - 1);
            result.put("taskName", nameStr);
        }
        return result;
    }

    /* 获取任务名称：从activitieList中匹配，如果匹配不到，最后外面会进行一次补缺
     * activitieList:获取项目信息中的任务list
     * activitiesCodeList：applicationRelation表中应用对应的任务code列表
     * locale：多语言标识
     * findActivityCode：存储已获取任务名称的变量
     */
    private void getActivityName( Object activitieList, List<String> activitiesCodeList, String locale, Map<String,Object> findActivityCode) {

        List<Activity> activityInfo = (List<Activity>) activitieList;
        for(Activity temp : activityInfo){
            String code = temp.getCode();
            if(activitiesCodeList.contains(code)){
                if(findActivityCode.containsKey(code)){
                    // 如果当前任务code在之前已经查出过了，这里直接跳过
                    continue;
                }
                Object activityNameMap = temp.getActivityName();
                Object activityName = ObjectUseUtil.getValueFromObject(activityNameMap, locale);
                findActivityCode.put(code, activityName);
            }
        }
    }

    /* 直接用getActivityDefinition方法获取任务名称
     * activitiesCodeList：applicationRelation表中应用对应的任务code列表
     * taskId：项目Code
     * findActivityCode：存储已获取任务名称的变量
     */
    private void getActivityNameDirect(List<String> activitiesCodeList, String taskId, Map<String,Object> findActivityCode) throws Exception {
        for (String activitiesCode : activitiesCodeList) {
            if(findActivityCode.containsKey(activitiesCode)){
                // 如果当前任务code在之前已经查出过了，这里直接跳过
                continue;
            }
            Object taskInfo = taskService.getActivityDefinition(taskId, activitiesCode, "");
            Object taskName = ObjectUseUtil.getValueFromObject(taskInfo, "name");
            if (!ObjectUtils.isEmpty(taskInfo)) {
                findActivityCode.put(activitiesCode,taskName);
            }
        }
    }

    // 获取当前租户下的增强应用对应的数字员工的信息
    private Map<String, Map<String, String>> getDigitalStaffCodeListByTenantId(String locale) throws Exception {
        Map<String, Map<String, String>> appDigiStaffInfo = new HashMap<>();
        String tenantId = AthenaUtils.getTenantId();
        // 获取当前租户的所有应用信息
        List<String> appCodes = dataPickService.tenantInitializedAppCodes(AthenaUtils.getTenantId());
        if (CollectionUtils.isEmpty(appCodes)) {
            log.info("当前租户没有相关应用信息");
            return appDigiStaffInfo;
        }
        // 过滤应用中的增强应用Code并取出对应数字员工的Code
        for (String appCode : appCodes) {
            Criteria criteria = Criteria.where("code").is(appCode).and("category").is(ENHANCE)
                    .and("extendType").is(DIGITALSTAFF);
            Application applicationInfo = dataPickService.findOneByCondition(criteria, Application.class, "application");
            if (applicationInfo != null) {
                // 判断当前租户的该应用是否已授权
                Boolean authOrNot = cacUtils.getAppAuth(tenantId, appCode);
                if (authOrNot) {
                    // 处理多语言
                    LanguageUtil.processLocaleLanguage(applicationInfo, locale);
                    // 这里默认从neo4j取出的增强应用的Code在Application表里一定会有记录，并且Application表里是增强应用标识的一定有数字员工的code
                    Map<String, String> appCodeAndName = new HashMap<>();
                    appCodeAndName.put(APPCODE, appCode);
                    appCodeAndName.put(APPNAME, applicationInfo.getName());
                    appCodeAndName.put(DESCRIPTION, applicationInfo.getDescription());
                    appDigiStaffInfo.put(applicationInfo.getExtendData(), appCodeAndName);
                }
            }
        }
        return appDigiStaffInfo;
    }

    private List<DigitalStaff> getDigitalStaffInfoList(List<String> dsCodeList, String locale, String tenantVersion) throws NoSuchFieldException, IllegalAccessException {
        Query query = new Query();
        query.addCriteria(Criteria.where("code").in(dsCodeList)
                .and("version").is(tenantVersion));
        List<DigitalStaff> digitalStaffList = mongoTemplateSystem.find(query, DigitalStaff.class, "digitalStaff");
        if (CollectionUtils.isEmpty(digitalStaffList)) {
            log.info("暂无数字员工信息");
            return new ArrayList<>();
        }
        // 处理类中存在的多语言
        LanguageUtil.processLocaleLanguage(digitalStaffList, locale);
        return digitalStaffList;
    }

    // 获得职级名称
    public String getGradeNameByLang(String code, String language) {
        String gradeName = "";
        switch (language) {
            case "zh_CN":
            case "zh_TW":
                if (GradeCode.ASSISTANT.name().equals(code)) {
                    gradeName = "助手型";
                } else if (GradeCode.EXPERT.name().equals(code)) {
                    gradeName = "能手型";
                } else if (GradeCode.MASTER.name().equals(code)) {
                    gradeName = "高手型";
                }
                break;
            case "en_US":
                if (GradeCode.ASSISTANT.name().equals(code)) {
                    gradeName = "Assistant";
                } else if (GradeCode.EXPERT.name().equals(code)) {
                    gradeName = "Expert";
                } else if (GradeCode.MASTER.name().equals(code)) {
                    gradeName = "Master";
                }
                break;
            default:
                break;
        }
        return gradeName;
    }

    // 获取职能列表（现在从EOC中取，后期可能要改为从neo4j中获取）
    private JSONArray getFunctionList() throws IOException {
        JSONObject resp = AthenaUtils.getEoc(EOCDUTYLIST, null, new HashMap<>(), appToken);
        Object data = resp.get("data");
        JSONArray functionList = ((JSONObject) data).getJSONArray("list");
        return functionList;
    }

    // 从职能列表中获取职能名称
    private String getFunctionName(JSONArray functionList, String functionCode) {
        for (Object functionInfo : functionList) {
            if (functionCode.equals(((JSONObject) functionInfo).get("id"))) {
                return (String) ((JSONObject) functionInfo).get("name");
            }
        }
        return "";
    }

    // 统计当前数字员工已完成工作数量
    public Integer getWholeFinishJobNum(Set<String> jobNameList) {
        int wholeFinishJobNum = 0;
        for (String jobName : jobNameList) {
            Map<String, Object> param = new HashMap<>();
            param.put("name", jobName); // type: 0项目、1任务
            param.put("type", "1"); // state: 0逾时完成、2准时完成
            param.put("state", "0");
            param.put("startTime", null);
            param.put("endTime", null);
            param.put("page", 1);
            param.put("rows", 10000);
            HashMap resp = atmcUtils.postAtmcTask(null, JSON.toJSONString(param));
            // 逾期完成的任务数量
            Object response = resp.get("response");
            int overTaskNum = (int) ((JSONObject) response).get("total");
            wholeFinishJobNum = wholeFinishJobNum + overTaskNum;
            // 准时完成的任务数量
            param.put("state", "2");
            HashMap resp2 = atmcUtils.postAtmcTask(null, JSON.toJSONString(param));
            Object response2 = resp2.get("response");
            int onTimeTaskNum = (int) ((JSONObject) response2).get("total");
            wholeFinishJobNum = wholeFinishJobNum + onTimeTaskNum;
        }
        return wholeFinishJobNum;
    }

    // 统计当前数字员工当天已完成工作信息（总完成工作数量、按时间存储的完成任务名称和拼接任务名称）
    public Map<String, Object> getFinishJobInfo(Set<String> jobNameList, String language) {
        Map<String, Object> result = new HashMap<>();
        List<TaskHistory> taskLastTimeList = new ArrayList<>();
        List<String> taskFinishList = new ArrayList<>();
        StringBuilder jobList = new StringBuilder().append(handleLang("包括", language)).append(":");
        Date date = new Date();
        SimpleDateFormat formatter = new SimpleDateFormat("yyyy-MM-dd");
        String curDate = formatter.format(date);
        String startTime = curDate + " 00:00:00";
        String endTime = curDate + " 59:59:59";
        Integer totalFinishjobNum = 0;
        for (String jobName : jobNameList) {
            List<TaskHistory> taskHistoryList = new ArrayList<>();
            // 逾期完成的任务数量
            Map<String, Object> param = new HashMap<>();
            param.put("name", jobName);
            param.put("type", "1");// type: 0项目、1任务
            param.put("state", "0");// state: 0逾时完成、2准时完成
            param.put("startTime", startTime);
            param.put("endTime", endTime);
            param.put("page", 1);
            param.put("rows", 10000);
            HashMap resp0 = atmcUtils.postAtmcTask(null, JSON.toJSONString(param));
            Object response0 = resp0.get("response");
            int overTaskNum = (int) ((JSONObject) response0).get("total");
            param.put("state", "2");
            HashMap resp2 = atmcUtils.postAtmcTask(null, JSON.toJSONString(param));
            Object response2 = resp2.get("response");
            int onTimeTaskNum = (int) ((JSONObject) response2).get("total");

            int taskTotalNum = overTaskNum + onTimeTaskNum;
            totalFinishjobNum = totalFinishjobNum + taskTotalNum;
            if (taskTotalNum > 0) {
                JSONArray records1 = ((JSONObject) response0).getJSONArray("records");
                for (Object records : records1) {
                    TaskHistory temp = JSONObject.toJavaObject((JSON) records, TaskHistory.class);
                    taskHistoryList.add(temp);
                }
                JSONArray records2 = ((JSONObject) response2).getJSONArray("records");
                for (Object records : records2) {
                    TaskHistory temp = JSONObject.toJavaObject((JSON) records, TaskHistory.class);
                    taskHistoryList.add(temp);
                }
                // 对当前任务名称下的已完成任务列表排序，并保留时间最新的那条记录
                List<TaskHistory> taskListSorted = taskHistoryList.stream().sorted(Comparator.comparing(TaskHistory::getEndTime).reversed()).collect(Collectors.toList());
                TaskHistory latstedTask = taskListSorted.get(0);
                latstedTask.setTotalJobNum(taskTotalNum);
                taskLastTimeList.add(latstedTask);
            }
        }
        // 对所有最新时间的不同任务列表排序，按时间倒序
        List<TaskHistory> taskListSorted = taskLastTimeList.stream().sorted(Comparator.comparing(TaskHistory::getEndTime).reversed()).collect(Collectors.toList());
        // 组装返回数据
        result.put("totalFinishjobNum", totalFinishjobNum);
        for (TaskHistory taskInfo : taskListSorted) {
            // 格式化时间和完成任务
            String time = taskInfo.getEndTime().substring(11, 16);
            String temp = time + " " + handleLang("完成", language) + taskInfo.getTotalJobNum() + handleLang("项", language) + taskInfo.getName();
            taskFinishList.add(temp);
            // 组装完成的任务名称
            jobList = jobList.append(taskInfo.getName()).append("、");
        }
        result.put("taskFinishList", taskFinishList);
        String jobListStr = jobList.toString();
        jobListStr = jobListStr.substring(0, jobListStr.length() - 1);
        result.put("jobList", jobListStr);

        return result;
    }

    public String handleLang(String name, String language) {
        String result = name;
        switch (language) {
            case "zh_CN":
                ChineseUtils.toSimplified(name);
                break;
            case "zh_TW":
                ChineseUtils.toTraditional(name);
                break;
            case "en_US":
                if ("次".equals(name)) {
                    result = "second";
                } else if ("包括".equals(name)) {
                    result = "include";
                } else if ("完成".equals(name)) {
                    result = " complete";
                } else if ("项".equals(name)) {
                    result = "term";
                }
                break;
            default:
                break;
        }
        return result;
    }

}