package com.digiwin.athena.knowledgegraph.service.impl;

import com.digiwin.app.container.exceptions.DWArgumentException;
import com.digiwin.app.container.exceptions.DWBusinessException;
import com.digiwin.app.module.DWModuleConfigUtils;
import com.digiwin.athena.kmservice.locale.Lang;
import com.digiwin.athena.knowledgegraph.data.MongoDBManager;
import com.digiwin.athena.knowledgegraph.mq.RabbitMQManager;
import com.digiwin.athena.knowledgegraph.service.IScheduleRuleConfigService;
import com.digiwin.athena.knowledgegraph.service.model.ScheduleRuleConfigDTO;
import com.digiwin.athena.kg.monitorRule.TriggerDTO;
import com.digiwin.athena.knowledgegraph.utils.I18nUtils;
import com.mongodb.client.MongoCollection;
import com.mongodb.client.MongoCursor;
import com.mongodb.client.model.Updates;
import org.apache.commons.lang3.StringUtils;
import org.bson.Document;
import org.bson.conversions.Bson;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.springframework.stereotype.Service;

import java.lang.reflect.Field;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Map;

import static com.mongodb.client.model.Filters.and;
import static com.mongodb.client.model.Filters.eq;
@Lang
@Service
public class ScheduleRuleConfigService implements IScheduleRuleConfigService {
    private static String scheduleRuleConfigColName = "scheduleRuleConfigs";
    protected static String scheduleRuleConfigKeyName = "configId";

    private static String rabbitQueueName = "utScheduleRuleConfigQueue";
    private static String rabbitRoutingKey = "utScheduleRuleConfig";

    private static Log log = LogFactory.getLog(ScheduleRuleConfigService.class);

    static {
        if (!UnitTestHelper.isUnitTest) {
            //必须通过 DWModuleConfigUtils.getProperty 方法来获取参数
            //使用DWModuleConfigUtils.getCurrentModuleProperty部署到云端的情况会取不到当前模块 无法获取参数
            rabbitQueueName = DWModuleConfigUtils.getProperty("knowledgegraph","monitorRuleConfigQueueName");
            rabbitRoutingKey = DWModuleConfigUtils.getProperty("knowledgegraph","monitorRuleConfigRoutingKey");
        }
    }
    @Override
    public Object put(ScheduleRuleConfigDTO config) throws Exception {
        String configId = config.getConfigId();
        if (StringUtils.isEmpty(configId)) {
            throw new DWArgumentException("config", I18nUtils.getValue("knowledgegraph.config.configIdEmpty"));
        }
        String tenantId = config.getTenantId();
        if (StringUtils.isEmpty(tenantId)) {
            throw new DWArgumentException("config", I18nUtils.getValue("knowledgegraph.config.tenantIdEmpty"));
        }
        String ruleId = config.getRuleId();
        if (StringUtils.isEmpty(ruleId)) {
            throw new DWArgumentException("config", I18nUtils.getValue("knowledgegraph.config.ruleIdEmpty"));
        }

        Map<String, Object> configMap = new HashMap<>();
        configMap.put(scheduleRuleConfigKeyName, config.getConfigId());
        configMap.put("tenantId", tenantId);
        configMap.put("ruleId", ruleId);
        configMap.put("status", config.getStatus());
        configMap.put("trigger", convertToMap(config.getTrigger()));

        try {
            MongoCollection<Document> col = MongoDBManager.getMongoDBManager().getCollection(scheduleRuleConfigColName, scheduleRuleConfigKeyName);
            col.insertOne(new Document(configMap));

            SendMessage(tenantId, ruleId, ConfigChangeTypeEnum.CREATED);
        } catch (Exception e) {
            throw new DWBusinessException(e.getLocalizedMessage());
        }

      //  return DWServiceResultBuilder.build(true, null);
        return null;
    }

    private Map<String, Object> convertToMap(TriggerDTO triggerDTO) throws IllegalAccessException {
        Map<String, Object> map = new HashMap<>();
        boolean hasData = false;
        for (Field field : TriggerDTO.class.getDeclaredFields()) {
            field.setAccessible(true);
            if (field.isSynthetic()) {//跳过由编译器生成的字段 例如$jacocoData
                continue;
            }
            String fieldName = field.getName();
            Object value = field.get(triggerDTO);
            if (value != null) {
                map.put(fieldName, value);
                hasData = true;
            }
        }

        if (!hasData) {
            map = null;
        }
        return map;
    }

    private void SendMessage(String tenantId, String ruleId, ConfigChangeTypeEnum changeType) throws Exception {
        RabbitMQManager mqManager = RabbitMQManager.getRabbitMQManager();
        if (mqManager.getEnabled()) {
            String msg = String.format("{\"tenantId\":\"%s\",\"ruleId\":\"%s\",\"changeType\":\"%s\"}", tenantId, ruleId, changeType);
            try {
                boolean sendSuccess = mqManager.PublishMessage(msg, rabbitQueueName, rabbitRoutingKey);
                if (!sendSuccess) {//发送失败 记录日志
                    log.info("RabbitMQ message was sent failed");
                }
            }
            catch (Exception e) {
                log.info("RabbitMQ message was sent failed");
                throw new DWBusinessException(e.getLocalizedMessage());
            }
        }
    }

    @Override
    public Object get(String ruleId, String tenantId) throws Exception {
        MongoCollection<Document> col = MongoDBManager.getMongoDBManager().getCollection(scheduleRuleConfigColName, scheduleRuleConfigKeyName);
        //获取的配置必须是启用状态
        Document doc = col.find(and(eq("ruleId", ruleId), eq("tenantId", tenantId), eq("status", 1))).first();
        String result = null;
        if (doc != null) {
            result = doc.toJson();
        }

        //return DWServiceResultBuilder.build(true, result);
        return result;
    }

    @Override
    public Object getQuery(String tenantId, int count) throws Exception {
        MongoCollection<Document> col = MongoDBManager.getMongoDBManager().getCollection(scheduleRuleConfigColName, scheduleRuleConfigKeyName);
        //获取的配置必须是启用状态
        Bson condition = eq("status", 1);
        if (tenantId != null && !tenantId.isEmpty()) {//tenantId为空 则查询所有启用的配置
            condition = and(eq("tenantId", tenantId), condition);
        }

        MongoCursor<Document> cursor;
        if (count > 0) {
            cursor = col.find(condition).limit(count).iterator();
        } else {
            cursor = col.find(condition).iterator();
        }
        ArrayList<String> result = new ArrayList<>();
        try {
            while (cursor.hasNext()) {
                result.add(cursor.next().toJson());
            }
        } finally {
            cursor.close();
        }

        Object re;
        if (result.size() > 0) {
            re = result.toArray(new String[0]);
        } else {
            re = null;
        }

        return re;
    }

    private Document get(String configId) throws Exception {
        MongoCollection<Document> col = MongoDBManager.getMongoDBManager().getCollection(scheduleRuleConfigColName, scheduleRuleConfigKeyName);
        Document doc = col.find(eq("configId", configId)).first();
        return doc;
    }

    @Override
    public Object delete(String configId) throws Exception {
        Document doc = this.get(configId);

        if (doc != null) {
            MongoCollection<Document> col = MongoDBManager.getMongoDBManager().getCollection(scheduleRuleConfigColName, scheduleRuleConfigKeyName);
            col.deleteOne(eq(scheduleRuleConfigKeyName, configId));

            //保存时 tennantId ruleId 判断不可为空 这里获取到的数据不应该为空 不做空值判断
            SendMessage((String) doc.get("tenantId"), (String) doc.get("ruleId"), ConfigChangeTypeEnum.DELETED);
        }

        //return DWServiceResultBuilder.build(true, null);
        return null;
    }

    @Override
    public Object putEnable(String configId) throws Exception {
        MongoCollection<Document> col = MongoDBManager.getMongoDBManager().getCollection(scheduleRuleConfigColName, scheduleRuleConfigKeyName);
        col.updateOne(eq(scheduleRuleConfigKeyName, configId), Updates.set("status", 1));

        Document doc = this.get(configId);
        if (doc != null) {
            SendMessage((String) doc.get("tenantId"), (String) doc.get("ruleId"), ConfigChangeTypeEnum.ENABLED);
        }

      //  return DWServiceResultBuilder.build(true, null);
        return null;
    }

    @Override
    public Object putDisable(String configId) throws Exception {
        MongoCollection<Document> col = MongoDBManager.getMongoDBManager().getCollection(scheduleRuleConfigColName, scheduleRuleConfigKeyName);
        col.updateOne(eq(scheduleRuleConfigKeyName, configId), Updates.set("status", 0));

        Document doc = this.get(configId);
        if (doc != null) {
            SendMessage((String) doc.get("tenantId"), (String) doc.get("ruleId"), ConfigChangeTypeEnum.DISABLED);
        }

       // return DWServiceResultBuilder.build(true, null);
        return null;
    }
}